/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.dubbo.common.serialize.kryo;

import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.serialize.ObjectInput;
import org.apache.dubbo.common.serialize.ObjectOutput;
import org.apache.dubbo.common.serialize.Serialization;
import org.apache.dubbo.common.serialize.kryo.optimized.KryoSerialization2;
import org.apache.dubbo.common.serialize.model.person.BigPerson;
import org.apache.dubbo.common.serialize.model.person.FullAddress;
import org.apache.dubbo.common.serialize.model.person.PersonInfo;
import org.apache.dubbo.common.serialize.model.person.PersonStatus;
import org.apache.dubbo.common.serialize.model.person.Phone;

import org.junit.jupiter.api.Test;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.fail;

public class KryoSerialization2Test {

    protected Serialization serialization = new KryoSerialization2();

    protected ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();

    protected URL url = new URL("protocol", "1.1.1.1", 1234);

    // ================ Primitive Type ================
    protected BigPerson bigPerson;

    {
        bigPerson = new BigPerson();
        bigPerson.setPersonId("superman111");
        bigPerson.setLoginName("superman");
        bigPerson.setStatus(PersonStatus.ENABLED);
        bigPerson.setEmail("sm@1.com");
        bigPerson.setPenName("pname");

        ArrayList<Phone> phones = new ArrayList<Phone>();
        Phone phone1 = new Phone("86", "0571", "87654321", "001");
        Phone phone2 = new Phone("86", "0571", "87654322", "002");
        phones.add(phone1);
        phones.add(phone2);

        PersonInfo pi = new PersonInfo();
        pi.setPhones(phones);
        Phone fax = new Phone("86", "0571", "87654321", null);
        pi.setFax(fax);
        FullAddress addr = new FullAddress("CN", "zj", "3480", "wensanlu", "315000");
        pi.setFullAddress(addr);
        pi.setMobileNo("13584652131");
        pi.setMale(true);
        pi.setDepartment("b2b");
        pi.setHomepageUrl("www.capcom.com");
        pi.setJobTitle("qa");
        pi.setName("superman");

        bigPerson.setInfoProfile(pi);
    }

    @Test
    public void testObject() throws IOException, ClassNotFoundException {
        ObjectOutput objectOutput = serialization.serialize(url, byteArrayOutputStream);
        objectOutput.writeObject(bigPerson);
        objectOutput.flushBuffer();

        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(
                byteArrayOutputStream.toByteArray());
        ObjectInput deserialize = serialization.deserialize(url, byteArrayInputStream);

        assertEquals(bigPerson, BigPerson.class.cast(deserialize.readObject(BigPerson.class)));

        try {
            deserialize.readObject(BigPerson.class);
            fail();
        } catch (IOException expected) {
        }
    }

    @Test
    public void testObjectWithAttachments() throws IOException, ClassNotFoundException {
        ObjectOutput objectOutput = serialization.serialize(url, byteArrayOutputStream);
        objectOutput.writeObject(bigPerson);

        Map<String, Object> attachments = new HashMap<>();
        attachments.put("attachments", "attachments");
        objectOutput.writeObject(attachments);

        objectOutput.flushBuffer();

        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(
                byteArrayOutputStream.toByteArray());
        ObjectInput deserialize = serialization.deserialize(url, byteArrayInputStream);

        assertEquals(bigPerson, BigPerson.class.cast(deserialize.readObject(BigPerson.class)));
        assertEquals(attachments, deserialize.readAttachments());

        try {
            deserialize.readObject(BigPerson.class);
            fail();
        } catch (IOException expected) {
        }
    }

}
