/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

plugins {
    id("com.github.autostyle")
    id("com.github.vlsi.ide")
}

// When building build-logic projects, Gradle applies the plugins to "gradle-kotlin-dsl-accessors" project
// And the below logic can't fully work there (e.g. .ratignore file is not available)
// The proper resolution would be to support lazy configuration properties in AutoStyle
val isActualPluginApplication =
    rootProject.name.startsWith("jmeter")

fun com.github.autostyle.gradle.BaseFormatExtension.license() {
    if (isActualPluginApplication) {
        licenseHeader(rootProject.ide.licenseHeader) {
            copyrightStyle("bat", com.github.autostyle.generic.DefaultCopyrightStyle.REM)
            copyrightStyle("cmd", com.github.autostyle.generic.DefaultCopyrightStyle.REM)
            addBlankLineAfter.set(true)
        }
    }
    trimTrailingWhitespace()
    endWithNewline()
}

autostyle {
    kotlinGradle {
        license()
        ktlint("0.40.0")
    }
    format("configs") {
        filter {
            include("**/*.sh", "**/*.bsh", "**/*.cmd", "**/*.bat")
            include("**/*.properties", "**/*.yml")
            include("**/*.xsd", "**/*.xsl", "**/*.xml")
            // Autostyle does not support gitignore yet https://github.com/autostyle/autostyle/issues/13
            exclude("out/**")
            if (project == rootProject) {
                if (isActualPluginApplication) {
                    exclude(rootDir.resolve(".ratignore").readLines())
                }
                exclude("gradlew*")
                // Generated by batch tests. It ignores log4j2.xml, however it is not that important
                // The configuration will be removed when Autostyle will use .gitignore
                exclude("bin/*.xml")
            } else {
                exclude("bin/**")
            }
        }
        license()
    }
    format("markdown") {
        filter.include("**/*.md")
        endWithNewline()
    }
}

plugins.withId("java") {
    autostyle {
        java {
            license()
            importOrder("static ", "java.", "javax", "org", "net", "com", "")
            removeUnusedImports()
            indentWithSpaces(4)
        }
    }
}

plugins.withId("groovy") {
    autostyle {
        groovy {
            license()
            importOrder("static ", "java.", "javax", "org", "net", "com", "")
            indentWithSpaces(4)
        }
    }
}

plugins.withId("org.jetbrains.kotlin.jvm") {
    autostyle {
        kotlin {
            license()
            trimTrailingWhitespace()
            ktlint("0.40.0")
            endWithNewline()
        }
    }
}
