/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.io.Closeable;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.kafka.clients.ApiVersions;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.FetchSessionHandler;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.NodeApiVersions;
import org.apache.kafka.clients.StaleMetadataException;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.LogTruncationException;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.clients.consumer.OffsetOutOfRangeException;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.clients.consumer.internals.ConsumerMetadata;
import org.apache.kafka.clients.consumer.internals.ConsumerNetworkClient;
import org.apache.kafka.clients.consumer.internals.FetcherMetricsRegistry;
import org.apache.kafka.clients.consumer.internals.OffsetsForLeaderEpochClient;
import org.apache.kafka.clients.consumer.internals.RequestFuture;
import org.apache.kafka.clients.consumer.internals.RequestFutureAdapter;
import org.apache.kafka.clients.consumer.internals.RequestFutureListener;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.IsolationLevel;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.errors.InvalidTopicException;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.header.internals.RecordHeaders;
import org.apache.kafka.common.message.ApiVersionsResponseData;
import org.apache.kafka.common.message.ListOffsetsRequestData;
import org.apache.kafka.common.message.ListOffsetsResponseData;
import org.apache.kafka.common.message.OffsetForLeaderEpochResponseData;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Meter;
import org.apache.kafka.common.metrics.stats.Min;
import org.apache.kafka.common.metrics.stats.Value;
import org.apache.kafka.common.metrics.stats.WindowedCount;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.ControlRecordType;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.FetchRequest;
import org.apache.kafka.common.requests.FetchResponse;
import org.apache.kafka.common.requests.ListOffsetsRequest;
import org.apache.kafka.common.requests.ListOffsetsResponse;
import org.apache.kafka.common.requests.MetadataRequest;
import org.apache.kafka.common.requests.MetadataResponse;
import org.apache.kafka.common.requests.OffsetsForLeaderEpochRequest;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.CloseableIterator;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Timer;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.helpers.MessageFormatter;

public class Fetcher<K, V>
implements Closeable {
    private final Logger log;
    private final LogContext logContext;
    private final ConsumerNetworkClient client;
    private final Time time;
    private final int minBytes;
    private final int maxBytes;
    private final int maxWaitMs;
    private final int fetchSize;
    private final long retryBackoffMs;
    private final long requestTimeoutMs;
    private final int maxPollRecords;
    private final boolean checkCrcs;
    private final String clientRackId;
    private final ConsumerMetadata metadata;
    private final FetchManagerMetrics sensors;
    private final SubscriptionState subscriptions;
    private final ConcurrentLinkedQueue<CompletedFetch> completedFetches;
    private final BufferSupplier decompressionBufferSupplier = BufferSupplier.create();
    private final Deserializer<K> keyDeserializer;
    private final Deserializer<V> valueDeserializer;
    private final IsolationLevel isolationLevel;
    private final Map<Integer, FetchSessionHandler> sessionHandlers;
    private final AtomicReference<RuntimeException> cachedListOffsetsException = new AtomicReference();
    private final AtomicReference<RuntimeException> cachedOffsetForLeaderException = new AtomicReference();
    private final OffsetsForLeaderEpochClient offsetsForLeaderEpochClient;
    private final Set<Integer> nodesWithPendingFetchRequests;
    private final ApiVersions apiVersions;
    private final AtomicInteger metadataUpdateVersion = new AtomicInteger(-1);
    private CompletedFetch nextInLineFetch = null;

    public Fetcher(LogContext logContext, ConsumerNetworkClient client, int minBytes, int maxBytes, int maxWaitMs, int fetchSize, int maxPollRecords, boolean checkCrcs, String clientRackId, Deserializer<K> keyDeserializer, Deserializer<V> valueDeserializer, ConsumerMetadata metadata, SubscriptionState subscriptions, Metrics metrics, FetcherMetricsRegistry metricsRegistry, Time time, long retryBackoffMs, long requestTimeoutMs, IsolationLevel isolationLevel, ApiVersions apiVersions) {
        this.log = logContext.logger(Fetcher.class);
        this.logContext = logContext;
        this.time = time;
        this.client = client;
        this.metadata = metadata;
        this.subscriptions = subscriptions;
        this.minBytes = minBytes;
        this.maxBytes = maxBytes;
        this.maxWaitMs = maxWaitMs;
        this.fetchSize = fetchSize;
        this.maxPollRecords = maxPollRecords;
        this.checkCrcs = checkCrcs;
        this.clientRackId = clientRackId;
        this.keyDeserializer = keyDeserializer;
        this.valueDeserializer = valueDeserializer;
        this.completedFetches = new ConcurrentLinkedQueue();
        this.sensors = new FetchManagerMetrics(metrics, metricsRegistry);
        this.retryBackoffMs = retryBackoffMs;
        this.requestTimeoutMs = requestTimeoutMs;
        this.isolationLevel = isolationLevel;
        this.apiVersions = apiVersions;
        this.sessionHandlers = new HashMap<Integer, FetchSessionHandler>();
        this.offsetsForLeaderEpochClient = new OffsetsForLeaderEpochClient(client, logContext);
        this.nodesWithPendingFetchRequests = new HashSet<Integer>();
    }

    protected boolean hasCompletedFetches() {
        return !this.completedFetches.isEmpty();
    }

    public boolean hasAvailableFetches() {
        return this.completedFetches.stream().anyMatch(fetch -> this.subscriptions.isFetchable(((CompletedFetch)fetch).partition));
    }

    public synchronized int sendFetches() {
        this.sensors.maybeUpdateAssignment(this.subscriptions);
        Map<Node, FetchSessionHandler.FetchRequestData> fetchRequestMap = this.prepareFetchRequests();
        for (Map.Entry<Node, FetchSessionHandler.FetchRequestData> entry : fetchRequestMap.entrySet()) {
            final Node fetchTarget = entry.getKey();
            final FetchSessionHandler.FetchRequestData data = entry.getValue();
            FetchRequest.Builder request = FetchRequest.Builder.forConsumer(this.maxWaitMs, this.minBytes, data.toSend()).isolationLevel(this.isolationLevel).setMaxBytes(this.maxBytes).metadata(data.metadata()).toForget(data.toForget()).rackId(this.clientRackId);
            this.log.debug("Sending {} {} to broker {}", new Object[]{this.isolationLevel, data, fetchTarget});
            RequestFuture<ClientResponse> future = this.client.send(fetchTarget, request);
            this.nodesWithPendingFetchRequests.add(entry.getKey().id());
            future.addListener(new RequestFutureListener<ClientResponse>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 * Enabled aggressive block sorting
                 * Enabled unnecessary exception pruning
                 * Enabled aggressive exception aggregation
                 */
                @Override
                public void onSuccess(ClientResponse resp) {
                    Fetcher fetcher = Fetcher.this;
                    synchronized (fetcher) {
                        try {
                            FetchResponse response = (FetchResponse)resp.responseBody();
                            FetchSessionHandler handler = Fetcher.this.sessionHandler(fetchTarget.id());
                            if (handler == null) {
                                Fetcher.this.log.error("Unable to find FetchSessionHandler for node {}. Ignoring fetch response.", (Object)fetchTarget.id());
                                return;
                            }
                            if (!handler.handleResponse(response)) {
                                return;
                            }
                            HashSet<TopicPartition> partitions = new HashSet<TopicPartition>(response.responseData().keySet());
                            FetchResponseMetricAggregator metricAggregator = new FetchResponseMetricAggregator(Fetcher.this.sensors, partitions);
                            for (Map.Entry entry : response.responseData().entrySet()) {
                                TopicPartition partition2 = entry.getKey();
                                FetchRequest.PartitionData requestData = data.sessionPartitions().get(partition2);
                                if (requestData == null) {
                                    String message;
                                    if (data.metadata().isFull()) {
                                        message = MessageFormatter.arrayFormat("Response for missing full request partition: partition={}; metadata={}", new Object[]{partition2, data.metadata()}).getMessage();
                                        throw new IllegalStateException(message);
                                    }
                                    message = MessageFormatter.arrayFormat("Response for missing session request partition: partition={}; metadata={}; toSend={}; toForget={}", new Object[]{partition2, data.metadata(), data.toSend(), data.toForget()}).getMessage();
                                    throw new IllegalStateException(message);
                                }
                                long fetchOffset = requestData.fetchOffset;
                                FetchResponse.PartitionData partitionData = entry.getValue();
                                Fetcher.this.log.debug("Fetch {} at offset {} for partition {} returned fetch data {}", new Object[]{Fetcher.this.isolationLevel, fetchOffset, partition2, partitionData});
                                Iterator<? extends RecordBatch> batches = ((Records)partitionData.records()).batches().iterator();
                                short responseVersion = resp.requestHeader().apiVersion();
                                Fetcher.this.completedFetches.add(new CompletedFetch(partition2, partitionData, metricAggregator, batches, fetchOffset, responseVersion));
                            }
                            Fetcher.this.sensors.fetchLatency.record(resp.requestLatencyMs());
                        }
                        finally {
                            Fetcher.this.nodesWithPendingFetchRequests.remove(fetchTarget.id());
                        }
                        return;
                    }
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onFailure(RuntimeException e) {
                    Fetcher fetcher = Fetcher.this;
                    synchronized (fetcher) {
                        try {
                            FetchSessionHandler handler = Fetcher.this.sessionHandler(fetchTarget.id());
                            if (handler != null) {
                                handler.handleError(e);
                            }
                        }
                        finally {
                            Fetcher.this.nodesWithPendingFetchRequests.remove(fetchTarget.id());
                        }
                    }
                }
            });
        }
        return fetchRequestMap.size();
    }

    public Map<String, List<PartitionInfo>> getAllTopicMetadata(Timer timer) {
        return this.getTopicMetadata(MetadataRequest.Builder.allTopics(), timer);
    }

    public Map<String, List<PartitionInfo>> getTopicMetadata(MetadataRequest.Builder request, Timer timer) {
        if (!request.isAllTopics() && request.emptyTopicList()) {
            return Collections.emptyMap();
        }
        do {
            RequestFuture<ClientResponse> future = this.sendMetadataRequest(request);
            this.client.poll(future, timer);
            if (future.failed() && !future.isRetriable()) {
                throw future.exception();
            }
            if (future.succeeded()) {
                MetadataResponse response = (MetadataResponse)future.value().responseBody();
                Cluster cluster = response.buildCluster();
                Set<String> unauthorizedTopics = cluster.unauthorizedTopics();
                if (!unauthorizedTopics.isEmpty()) {
                    throw new TopicAuthorizationException(unauthorizedTopics);
                }
                boolean shouldRetry = false;
                Map<String, Errors> errors = response.errors();
                if (!errors.isEmpty()) {
                    this.log.debug("Topic metadata fetch included errors: {}", (Object)errors);
                    for (Map.Entry<String, Errors> errorEntry : errors.entrySet()) {
                        String topic = errorEntry.getKey();
                        Errors error = errorEntry.getValue();
                        if (error == Errors.INVALID_TOPIC_EXCEPTION) {
                            throw new InvalidTopicException("Topic '" + topic + "' is invalid");
                        }
                        if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) continue;
                        if (error.exception() instanceof RetriableException) {
                            shouldRetry = true;
                            continue;
                        }
                        throw new KafkaException("Unexpected error fetching metadata for topic " + topic, error.exception());
                    }
                }
                if (!shouldRetry) {
                    HashMap<String, List<PartitionInfo>> topicsPartitionInfos = new HashMap<String, List<PartitionInfo>>();
                    for (String topic : cluster.topics()) {
                        topicsPartitionInfos.put(topic, cluster.partitionsForTopic(topic));
                    }
                    return topicsPartitionInfos;
                }
            }
            timer.sleep(this.retryBackoffMs);
        } while (timer.notExpired());
        throw new TimeoutException("Timeout expired while fetching topic metadata");
    }

    private RequestFuture<ClientResponse> sendMetadataRequest(MetadataRequest.Builder request) {
        Node node = this.client.leastLoadedNode();
        if (node == null) {
            return RequestFuture.noBrokersAvailable();
        }
        return this.client.send(node, request);
    }

    private Long offsetResetStrategyTimestamp(TopicPartition partition2) {
        OffsetResetStrategy strategy = this.subscriptions.resetStrategy(partition2);
        if (strategy == OffsetResetStrategy.EARLIEST) {
            return -2L;
        }
        if (strategy == OffsetResetStrategy.LATEST) {
            return -1L;
        }
        return null;
    }

    private OffsetResetStrategy timestampToOffsetResetStrategy(long timestamp) {
        if (timestamp == -2L) {
            return OffsetResetStrategy.EARLIEST;
        }
        if (timestamp == -1L) {
            return OffsetResetStrategy.LATEST;
        }
        return null;
    }

    public void resetOffsetsIfNeeded() {
        RuntimeException exception = this.cachedListOffsetsException.getAndSet(null);
        if (exception != null) {
            throw exception;
        }
        Set<TopicPartition> partitions = this.subscriptions.partitionsNeedingReset(this.time.milliseconds());
        if (partitions.isEmpty()) {
            return;
        }
        HashMap<TopicPartition, Long> offsetResetTimestamps = new HashMap<TopicPartition, Long>();
        for (TopicPartition partition2 : partitions) {
            Long timestamp = this.offsetResetStrategyTimestamp(partition2);
            if (timestamp == null) continue;
            offsetResetTimestamps.put(partition2, timestamp);
        }
        this.resetOffsetsAsync(offsetResetTimestamps);
    }

    public void validateOffsetsIfNeeded() {
        RuntimeException exception = this.cachedOffsetForLeaderException.getAndSet(null);
        if (exception != null) {
            throw exception;
        }
        this.validatePositionsOnMetadataChange();
        Map<TopicPartition, SubscriptionState.FetchPosition> partitionsToValidate = this.subscriptions.partitionsNeedingValidation(this.time.milliseconds()).stream().filter(tp -> this.subscriptions.position((TopicPartition)tp) != null).collect(Collectors.toMap(Function.identity(), this.subscriptions::position));
        this.validateOffsetsAsync(partitionsToValidate);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<TopicPartition, OffsetAndTimestamp> offsetsForTimes(Map<TopicPartition, Long> timestampsToSearch, Timer timer) {
        this.metadata.addTransientTopics(this.topicsForPartitions(timestampsToSearch.keySet()));
        try {
            Map fetchedOffsets = this.fetchOffsetsByTimes(timestampsToSearch, timer, true).fetchedOffsets;
            HashMap<TopicPartition, OffsetAndTimestamp> offsetsByTimes = new HashMap<TopicPartition, OffsetAndTimestamp>(timestampsToSearch.size());
            for (Map.Entry<TopicPartition, Long> entry : timestampsToSearch.entrySet()) {
                offsetsByTimes.put(entry.getKey(), null);
            }
            for (Map.Entry<TopicPartition, Long> entry : fetchedOffsets.entrySet()) {
                ListOffsetData offsetData = (ListOffsetData)((Object)entry.getValue());
                offsetsByTimes.put(entry.getKey(), new OffsetAndTimestamp(offsetData.offset, offsetData.timestamp, offsetData.leaderEpoch));
            }
            HashMap<TopicPartition, OffsetAndTimestamp> hashMap = offsetsByTimes;
            return hashMap;
        }
        finally {
            this.metadata.clearTransientTopics();
        }
    }

    private ListOffsetResult fetchOffsetsByTimes(Map<TopicPartition, Long> timestampsToSearch, Timer timer, boolean requireTimestamps) {
        ListOffsetResult result = new ListOffsetResult();
        if (timestampsToSearch.isEmpty()) {
            return result;
        }
        HashMap<TopicPartition, Long> remainingToSearch = new HashMap<TopicPartition, Long>(timestampsToSearch);
        do {
            RequestFuture<ListOffsetResult> future = this.sendListOffsetsRequests(remainingToSearch, requireTimestamps);
            this.client.poll(future, timer);
            if (!future.isDone()) break;
            if (future.succeeded()) {
                ListOffsetResult value = future.value();
                result.fetchedOffsets.putAll(value.fetchedOffsets);
                remainingToSearch.keySet().retainAll(value.partitionsToRetry);
            } else if (!future.isRetriable()) {
                throw future.exception();
            }
            if (remainingToSearch.isEmpty()) {
                return result;
            }
            this.client.awaitMetadataUpdate(timer);
        } while (timer.notExpired());
        throw new TimeoutException("Failed to get offsets by times in " + timer.elapsedMs() + "ms");
    }

    public Map<TopicPartition, Long> beginningOffsets(Collection<TopicPartition> partitions, Timer timer) {
        return this.beginningOrEndOffset(partitions, -2L, timer);
    }

    public Map<TopicPartition, Long> endOffsets(Collection<TopicPartition> partitions, Timer timer) {
        return this.beginningOrEndOffset(partitions, -1L, timer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<TopicPartition, Long> beginningOrEndOffset(Collection<TopicPartition> partitions, long timestamp, Timer timer) {
        this.metadata.addTransientTopics(this.topicsForPartitions(partitions));
        try {
            Map<TopicPartition, Long> timestampsToSearch = partitions.stream().distinct().collect(Collectors.toMap(Function.identity(), tp -> timestamp));
            ListOffsetResult result = this.fetchOffsetsByTimes(timestampsToSearch, timer, false);
            Map<TopicPartition, Long> map = result.fetchedOffsets.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> ((ListOffsetData)entry.getValue()).offset));
            return map;
        }
        finally {
            this.metadata.clearTransientTopics();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<TopicPartition, List<ConsumerRecord<K, V>>> fetchedRecords() {
        HashMap<TopicPartition, List<ConsumerRecord<TopicPartition, Object>>> fetched = new HashMap<TopicPartition, List<ConsumerRecord<TopicPartition, Object>>>();
        ArrayDeque<CompletedFetch> pausedCompletedFetches = new ArrayDeque<CompletedFetch>();
        int recordsRemaining = this.maxPollRecords;
        try {
            while (recordsRemaining > 0) {
                List<ConsumerRecord<K, V>> records;
                if (this.nextInLineFetch == null || this.nextInLineFetch.isConsumed) {
                    records = this.completedFetches.peek();
                    if (records == null) {
                        break;
                    }
                    if (((CompletedFetch)((Object)records)).notInitialized()) {
                        try {
                            this.nextInLineFetch = this.initializeCompletedFetch((CompletedFetch)((Object)records));
                        }
                        catch (Exception e) {
                            FetchResponse.PartitionData partition2 = ((CompletedFetch)((Object)records)).partitionData;
                            if (fetched.isEmpty() && (partition2.records() == null || ((Records)partition2.records()).sizeInBytes() == 0)) {
                                this.completedFetches.poll();
                            }
                            throw e;
                        }
                    } else {
                        this.nextInLineFetch = records;
                    }
                    this.completedFetches.poll();
                    continue;
                }
                if (this.subscriptions.isPaused(this.nextInLineFetch.partition)) {
                    this.log.debug("Skipping fetching records for assigned partition {} because it is paused", (Object)this.nextInLineFetch.partition);
                    pausedCompletedFetches.add(this.nextInLineFetch);
                    this.nextInLineFetch = null;
                    continue;
                }
                records = this.fetchRecords(this.nextInLineFetch, recordsRemaining);
                if (records.isEmpty()) continue;
                TopicPartition partition3 = this.nextInLineFetch.partition;
                List currentRecords = (List)fetched.get(partition3);
                if (currentRecords == null) {
                    fetched.put(partition3, records);
                } else {
                    ArrayList newRecords = new ArrayList(records.size() + currentRecords.size());
                    newRecords.addAll(currentRecords);
                    newRecords.addAll(records);
                    fetched.put(partition3, newRecords);
                }
                recordsRemaining -= records.size();
            }
        }
        catch (KafkaException e) {
            if (fetched.isEmpty()) {
                throw e;
            }
        }
        finally {
            this.completedFetches.addAll(pausedCompletedFetches);
        }
        return fetched;
    }

    private List<ConsumerRecord<K, V>> fetchRecords(CompletedFetch completedFetch, int maxRecords) {
        if (!this.subscriptions.isAssigned(completedFetch.partition)) {
            this.log.debug("Not returning fetched records for partition {} since it is no longer assigned", (Object)completedFetch.partition);
        } else if (!this.subscriptions.isFetchable(completedFetch.partition)) {
            this.log.debug("Not returning fetched records for assigned partition {} since it is no longer fetchable", (Object)completedFetch.partition);
        } else {
            SubscriptionState.FetchPosition position = this.subscriptions.position(completedFetch.partition);
            if (position == null) {
                throw new IllegalStateException("Missing position for fetchable partition " + completedFetch.partition);
            }
            if (completedFetch.nextFetchOffset == position.offset) {
                Long lead;
                Long partitionLag;
                List partRecords = completedFetch.fetchRecords(maxRecords);
                this.log.trace("Returning {} fetched records at offset {} for assigned partition {}", partRecords.size(), position, completedFetch.partition);
                if (completedFetch.nextFetchOffset > position.offset) {
                    SubscriptionState.FetchPosition nextPosition = new SubscriptionState.FetchPosition(completedFetch.nextFetchOffset, completedFetch.lastEpoch, position.currentLeader);
                    this.log.trace("Update fetching position to {} for partition {}", (Object)nextPosition, (Object)completedFetch.partition);
                    this.subscriptions.position(completedFetch.partition, nextPosition);
                }
                if ((partitionLag = this.subscriptions.partitionLag(completedFetch.partition, this.isolationLevel)) != null) {
                    this.sensors.recordPartitionLag(completedFetch.partition, partitionLag);
                }
                if ((lead = this.subscriptions.partitionLead(completedFetch.partition)) != null) {
                    this.sensors.recordPartitionLead(completedFetch.partition, lead);
                }
                return partRecords;
            }
            this.log.debug("Ignoring fetched records for {} at offset {} since the current position is {}", completedFetch.partition, completedFetch.nextFetchOffset, position);
        }
        this.log.trace("Draining fetched records for partition {}", (Object)completedFetch.partition);
        completedFetch.drain();
        return Collections.emptyList();
    }

    void resetOffsetIfNeeded(TopicPartition partition2, OffsetResetStrategy requestedResetStrategy, ListOffsetData offsetData) {
        SubscriptionState.FetchPosition position = new SubscriptionState.FetchPosition(offsetData.offset, Optional.empty(), this.metadata.currentLeader(partition2));
        offsetData.leaderEpoch.ifPresent(epoch -> this.metadata.updateLastSeenEpochIfNewer(partition2, (int)epoch));
        this.subscriptions.maybeSeekUnvalidated(partition2, position, requestedResetStrategy);
    }

    private void resetOffsetsAsync(Map<TopicPartition, Long> partitionResetTimestamps) {
        Map<Node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>> timestampsToSearchByNode = this.groupListOffsetRequests(partitionResetTimestamps, new HashSet<TopicPartition>());
        for (Map.Entry<Node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>> entry : timestampsToSearchByNode.entrySet()) {
            Node node = entry.getKey();
            final Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition> resetTimestamps = entry.getValue();
            this.subscriptions.setNextAllowedRetry(resetTimestamps.keySet(), this.time.milliseconds() + this.requestTimeoutMs);
            RequestFuture<ListOffsetResult> future = this.sendListOffsetRequest(node, resetTimestamps, false);
            future.addListener(new RequestFutureListener<ListOffsetResult>(){

                @Override
                public void onSuccess(ListOffsetResult result) {
                    if (!result.partitionsToRetry.isEmpty()) {
                        Fetcher.this.subscriptions.requestFailed(result.partitionsToRetry, Fetcher.this.time.milliseconds() + Fetcher.this.retryBackoffMs);
                        Fetcher.this.metadata.requestUpdate();
                    }
                    for (Map.Entry fetchedOffset : result.fetchedOffsets.entrySet()) {
                        TopicPartition partition2 = (TopicPartition)fetchedOffset.getKey();
                        ListOffsetData offsetData = (ListOffsetData)fetchedOffset.getValue();
                        ListOffsetsRequestData.ListOffsetsPartition requestedReset = (ListOffsetsRequestData.ListOffsetsPartition)resetTimestamps.get(partition2);
                        Fetcher.this.resetOffsetIfNeeded(partition2, Fetcher.this.timestampToOffsetResetStrategy(requestedReset.timestamp()), offsetData);
                    }
                }

                @Override
                public void onFailure(RuntimeException e) {
                    Fetcher.this.subscriptions.requestFailed(resetTimestamps.keySet(), Fetcher.this.time.milliseconds() + Fetcher.this.retryBackoffMs);
                    Fetcher.this.metadata.requestUpdate();
                    if (!(e instanceof RetriableException) && !Fetcher.this.cachedListOffsetsException.compareAndSet(null, e)) {
                        Fetcher.this.log.error("Discarding error in ListOffsetResponse because another error is pending", e);
                    }
                }
            });
        }
    }

    static boolean hasUsableOffsetForLeaderEpochVersion(NodeApiVersions nodeApiVersions) {
        ApiVersionsResponseData.ApiVersion apiVersion = nodeApiVersions.apiVersion(ApiKeys.OFFSET_FOR_LEADER_EPOCH);
        if (apiVersion == null) {
            return false;
        }
        return OffsetsForLeaderEpochRequest.supportsTopicPermission(apiVersion.maxVersion());
    }

    private void validateOffsetsAsync(Map<TopicPartition, SubscriptionState.FetchPosition> partitionsToValidate) {
        Map<Node, Map<TopicPartition, SubscriptionState.FetchPosition>> regrouped = this.regroupFetchPositionsByLeader(partitionsToValidate);
        long nextResetTimeMs = this.time.milliseconds() + this.requestTimeoutMs;
        regrouped.forEach((node, fetchPositions) -> {
            if (node.isEmpty()) {
                this.metadata.requestUpdate();
                return;
            }
            NodeApiVersions nodeApiVersions = this.apiVersions.get(node.idString());
            if (nodeApiVersions == null) {
                this.client.tryConnect((Node)node);
                return;
            }
            if (!Fetcher.hasUsableOffsetForLeaderEpochVersion(nodeApiVersions)) {
                this.log.debug("Skipping validation of fetch offsets for partitions {} since the broker does not support the required protocol version (introduced in Kafka 2.3)", (Object)fetchPositions.keySet());
                for (TopicPartition partition2 : fetchPositions.keySet()) {
                    this.subscriptions.completeValidation(partition2);
                }
                return;
            }
            this.subscriptions.setNextAllowedRetry(fetchPositions.keySet(), nextResetTimeMs);
            RequestFuture future = this.offsetsForLeaderEpochClient.sendAsyncRequest((Node)node, fetchPositions);
            future.addListener(new RequestFutureListener<OffsetsForLeaderEpochClient.OffsetForEpochResult>(){

                @Override
                public void onSuccess(OffsetsForLeaderEpochClient.OffsetForEpochResult offsetsResult) {
                    ArrayList truncations = new ArrayList();
                    if (!offsetsResult.partitionsToRetry().isEmpty()) {
                        Fetcher.this.subscriptions.setNextAllowedRetry(offsetsResult.partitionsToRetry(), Fetcher.this.time.milliseconds() + Fetcher.this.retryBackoffMs);
                        Fetcher.this.metadata.requestUpdate();
                    }
                    offsetsResult.endOffsets().forEach((topicPartition, respEndOffset) -> {
                        SubscriptionState.FetchPosition requestPosition = (SubscriptionState.FetchPosition)fetchPositions.get(topicPartition);
                        Optional<SubscriptionState.LogTruncation> truncationOpt = Fetcher.this.subscriptions.maybeCompleteValidation((TopicPartition)topicPartition, requestPosition, (OffsetForLeaderEpochResponseData.EpochEndOffset)respEndOffset);
                        truncationOpt.ifPresent(truncations::add);
                    });
                    if (!truncations.isEmpty()) {
                        Fetcher.this.maybeSetOffsetForLeaderException(Fetcher.this.buildLogTruncationException(truncations));
                    }
                }

                @Override
                public void onFailure(RuntimeException e) {
                    Fetcher.this.subscriptions.requestFailed(fetchPositions.keySet(), Fetcher.this.time.milliseconds() + Fetcher.this.retryBackoffMs);
                    Fetcher.this.metadata.requestUpdate();
                    if (!(e instanceof RetriableException)) {
                        Fetcher.this.maybeSetOffsetForLeaderException(e);
                    }
                }
            });
        });
    }

    private LogTruncationException buildLogTruncationException(List<SubscriptionState.LogTruncation> truncations) {
        HashMap<TopicPartition, OffsetAndMetadata> divergentOffsets = new HashMap<TopicPartition, OffsetAndMetadata>();
        HashMap<TopicPartition, Long> truncatedFetchOffsets = new HashMap<TopicPartition, Long>();
        for (SubscriptionState.LogTruncation truncation : truncations) {
            truncation.divergentOffsetOpt.ifPresent(divergentOffset -> divergentOffsets.put(truncation.topicPartition, (OffsetAndMetadata)divergentOffset));
            truncatedFetchOffsets.put(truncation.topicPartition, truncation.fetchPosition.offset);
        }
        return new LogTruncationException("Detected truncated partitions: " + truncations, truncatedFetchOffsets, divergentOffsets);
    }

    private void maybeSetOffsetForLeaderException(RuntimeException e) {
        if (!this.cachedOffsetForLeaderException.compareAndSet(null, e)) {
            this.log.error("Discarding error in OffsetsForLeaderEpoch because another error is pending", e);
        }
    }

    private RequestFuture<ListOffsetResult> sendListOffsetsRequests(Map<TopicPartition, Long> timestampsToSearch, boolean requireTimestamps) {
        final HashSet<TopicPartition> partitionsToRetry = new HashSet<TopicPartition>();
        Map<Node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>> timestampsToSearchByNode = this.groupListOffsetRequests(timestampsToSearch, partitionsToRetry);
        if (timestampsToSearchByNode.isEmpty()) {
            return RequestFuture.failure(new StaleMetadataException());
        }
        final RequestFuture<ListOffsetResult> listOffsetRequestsFuture = new RequestFuture<ListOffsetResult>();
        final HashMap fetchedTimestampOffsets = new HashMap();
        final AtomicInteger remainingResponses = new AtomicInteger(timestampsToSearchByNode.size());
        for (Map.Entry<Node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>> entry : timestampsToSearchByNode.entrySet()) {
            RequestFuture<ListOffsetResult> future = this.sendListOffsetRequest(entry.getKey(), entry.getValue(), requireTimestamps);
            future.addListener(new RequestFutureListener<ListOffsetResult>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onSuccess(ListOffsetResult partialResult) {
                    RequestFuture requestFuture = listOffsetRequestsFuture;
                    synchronized (requestFuture) {
                        fetchedTimestampOffsets.putAll(partialResult.fetchedOffsets);
                        partitionsToRetry.addAll(partialResult.partitionsToRetry);
                        if (remainingResponses.decrementAndGet() == 0 && !listOffsetRequestsFuture.isDone()) {
                            ListOffsetResult result = new ListOffsetResult(fetchedTimestampOffsets, partitionsToRetry);
                            listOffsetRequestsFuture.complete(result);
                        }
                    }
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onFailure(RuntimeException e) {
                    RequestFuture requestFuture = listOffsetRequestsFuture;
                    synchronized (requestFuture) {
                        if (!listOffsetRequestsFuture.isDone()) {
                            listOffsetRequestsFuture.raise(e);
                        }
                    }
                }
            });
        }
        return listOffsetRequestsFuture;
    }

    private Map<Node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>> groupListOffsetRequests(Map<TopicPartition, Long> timestampsToSearch, Set<TopicPartition> partitionsToRetry) {
        HashMap<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition> partitionDataMap = new HashMap<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition>();
        for (Map.Entry<TopicPartition, Long> entry : timestampsToSearch.entrySet()) {
            TopicPartition tp = entry.getKey();
            Long offset = entry.getValue();
            Metadata.LeaderAndEpoch leaderAndEpoch = this.metadata.currentLeader(tp);
            if (!leaderAndEpoch.leader.isPresent()) {
                this.log.debug("Leader for partition {} is unknown for fetching offset {}", (Object)tp, (Object)offset);
                this.metadata.requestUpdate();
                partitionsToRetry.add(tp);
                continue;
            }
            Node leader = leaderAndEpoch.leader.get();
            if (this.client.isUnavailable(leader)) {
                this.client.maybeThrowAuthFailure(leader);
                this.log.debug("Leader {} for partition {} is unavailable for fetching offset until reconnect backoff expires", (Object)leader, (Object)tp);
                partitionsToRetry.add(tp);
                continue;
            }
            int currentLeaderEpoch = leaderAndEpoch.epoch.orElse(-1);
            partitionDataMap.put(tp, new ListOffsetsRequestData.ListOffsetsPartition().setPartitionIndex(tp.partition()).setTimestamp(offset).setCurrentLeaderEpoch(currentLeaderEpoch));
        }
        return this.regroupPartitionMapByNode(partitionDataMap);
    }

    private RequestFuture<ListOffsetResult> sendListOffsetRequest(final Node node, Map<TopicPartition, ListOffsetsRequestData.ListOffsetsPartition> timestampsToSearch, boolean requireTimestamp) {
        ListOffsetsRequest.Builder builder = ListOffsetsRequest.Builder.forConsumer(requireTimestamp, this.isolationLevel).setTargetTimes(ListOffsetsRequest.toListOffsetsTopics(timestampsToSearch));
        this.log.debug("Sending ListOffsetRequest {} to broker {}", (Object)builder, (Object)node);
        return this.client.send(node, builder).compose(new RequestFutureAdapter<ClientResponse, ListOffsetResult>(){

            @Override
            public void onSuccess(ClientResponse response, RequestFuture<ListOffsetResult> future) {
                ListOffsetsResponse lor = (ListOffsetsResponse)response.responseBody();
                Fetcher.this.log.trace("Received ListOffsetResponse {} from broker {}", (Object)lor, (Object)node);
                Fetcher.this.handleListOffsetResponse(lor, future);
            }
        });
    }

    private void handleListOffsetResponse(ListOffsetsResponse listOffsetsResponse, RequestFuture<ListOffsetResult> future) {
        HashMap<TopicPartition, ListOffsetData> fetchedOffsets = new HashMap<TopicPartition, ListOffsetData>();
        HashSet<TopicPartition> partitionsToRetry = new HashSet<TopicPartition>();
        HashSet<String> unauthorizedTopics = new HashSet<String>();
        for (ListOffsetsResponseData.ListOffsetsTopicResponse topic : listOffsetsResponse.topics()) {
            block8: for (ListOffsetsResponseData.ListOffsetsPartitionResponse partition2 : topic.partitions()) {
                TopicPartition topicPartition = new TopicPartition(topic.name(), partition2.partitionIndex());
                Errors error = Errors.forCode(partition2.errorCode());
                switch (error) {
                    case NONE: {
                        if (!partition2.oldStyleOffsets().isEmpty()) {
                            if (partition2.oldStyleOffsets().size() > 1) {
                                future.raise(new IllegalStateException("Unexpected partitionData response of length " + partition2.oldStyleOffsets().size()));
                                return;
                            }
                            long offset = partition2.oldStyleOffsets().get(0);
                            this.log.debug("Handling v0 ListOffsetResponse response for {}. Fetched offset {}", (Object)topicPartition, (Object)offset);
                            if (offset == -1L) continue block8;
                            ListOffsetData offsetData = new ListOffsetData(offset, null, Optional.empty());
                            fetchedOffsets.put(topicPartition, offsetData);
                            break;
                        }
                        this.log.debug("Handling ListOffsetResponse response for {}. Fetched offset {}, timestamp {}", topicPartition, partition2.offset(), partition2.timestamp());
                        if (partition2.offset() == -1L) continue block8;
                        Optional<Integer> leaderEpoch = partition2.leaderEpoch() == -1 ? Optional.empty() : Optional.of(partition2.leaderEpoch());
                        ListOffsetData offsetData = new ListOffsetData(partition2.offset(), partition2.timestamp(), leaderEpoch);
                        fetchedOffsets.put(topicPartition, offsetData);
                        break;
                    }
                    case UNSUPPORTED_FOR_MESSAGE_FORMAT: {
                        this.log.debug("Cannot search by timestamp for partition {} because the message format version is before 0.10.0", (Object)topicPartition);
                        break;
                    }
                    case NOT_LEADER_OR_FOLLOWER: 
                    case REPLICA_NOT_AVAILABLE: 
                    case KAFKA_STORAGE_ERROR: 
                    case OFFSET_NOT_AVAILABLE: 
                    case LEADER_NOT_AVAILABLE: 
                    case FENCED_LEADER_EPOCH: 
                    case UNKNOWN_LEADER_EPOCH: {
                        this.log.debug("Attempt to fetch offsets for partition {} failed due to {}, retrying.", (Object)topicPartition, (Object)error);
                        partitionsToRetry.add(topicPartition);
                        break;
                    }
                    case UNKNOWN_TOPIC_OR_PARTITION: {
                        this.log.warn("Received unknown topic or partition error in ListOffset request for partition {}", (Object)topicPartition);
                        partitionsToRetry.add(topicPartition);
                        break;
                    }
                    case TOPIC_AUTHORIZATION_FAILED: {
                        unauthorizedTopics.add(topicPartition.topic());
                        break;
                    }
                    default: {
                        this.log.warn("Attempt to fetch offsets for partition {} failed due to unexpected exception: {}, retrying.", (Object)topicPartition, (Object)error.message());
                        partitionsToRetry.add(topicPartition);
                    }
                }
            }
        }
        if (!unauthorizedTopics.isEmpty()) {
            future.raise(new TopicAuthorizationException(unauthorizedTopics));
        } else {
            future.complete(new ListOffsetResult(fetchedOffsets, partitionsToRetry));
        }
    }

    private List<TopicPartition> fetchablePartitions() {
        HashSet<TopicPartition> exclude = new HashSet<TopicPartition>();
        if (this.nextInLineFetch != null && !this.nextInLineFetch.isConsumed) {
            exclude.add(this.nextInLineFetch.partition);
        }
        for (CompletedFetch completedFetch : this.completedFetches) {
            exclude.add(completedFetch.partition);
        }
        return this.subscriptions.fetchablePartitions(tp -> !exclude.contains(tp));
    }

    Node selectReadReplica(TopicPartition partition2, Node leaderReplica, long currentTimeMs) {
        Optional<Integer> nodeId = this.subscriptions.preferredReadReplica(partition2, currentTimeMs);
        if (nodeId.isPresent()) {
            Optional node = nodeId.flatMap(id -> this.metadata.fetch().nodeIfOnline(partition2, (int)id));
            if (node.isPresent()) {
                return (Node)node.get();
            }
            this.log.trace("Not fetching from {} for partition {} since it is marked offline or is missing from our metadata, using the leader instead.", (Object)nodeId, (Object)partition2);
            this.subscriptions.clearPreferredReadReplica(partition2);
            return leaderReplica;
        }
        return leaderReplica;
    }

    private void validatePositionsOnMetadataChange() {
        int newMetadataUpdateVersion = this.metadata.updateVersion();
        if (this.metadataUpdateVersion.getAndSet(newMetadataUpdateVersion) != newMetadataUpdateVersion) {
            this.subscriptions.assignedPartitions().forEach(topicPartition -> {
                Metadata.LeaderAndEpoch leaderAndEpoch = this.metadata.currentLeader((TopicPartition)topicPartition);
                this.subscriptions.maybeValidatePositionForCurrentLeader(this.apiVersions, (TopicPartition)topicPartition, leaderAndEpoch);
            });
        }
    }

    private Map<Node, FetchSessionHandler.FetchRequestData> prepareFetchRequests() {
        LinkedHashMap<Node, FetchSessionHandler.Builder> fetchable = new LinkedHashMap<Node, FetchSessionHandler.Builder>();
        this.validatePositionsOnMetadataChange();
        long currentTimeMs = this.time.milliseconds();
        for (TopicPartition partition2 : this.fetchablePartitions()) {
            SubscriptionState.FetchPosition position = this.subscriptions.position(partition2);
            if (position == null) {
                throw new IllegalStateException("Missing position for fetchable partition " + partition2);
            }
            Optional<Node> leaderOpt = position.currentLeader.leader;
            if (!leaderOpt.isPresent()) {
                this.log.debug("Requesting metadata update for partition {} since the position {} is missing the current leader node", (Object)partition2, (Object)position);
                this.metadata.requestUpdate();
                continue;
            }
            Node node = this.selectReadReplica(partition2, leaderOpt.get(), currentTimeMs);
            if (this.client.isUnavailable(node)) {
                this.client.maybeThrowAuthFailure(node);
                this.log.trace("Skipping fetch for partition {} because node {} is awaiting reconnect backoff", (Object)partition2, (Object)node);
                continue;
            }
            if (this.nodesWithPendingFetchRequests.contains(node.id())) {
                this.log.trace("Skipping fetch for partition {} because previous request to {} has not been processed", (Object)partition2, (Object)node);
                continue;
            }
            FetchSessionHandler.Builder builder = (FetchSessionHandler.Builder)fetchable.get(node);
            if (builder == null) {
                int id = node.id();
                FetchSessionHandler handler = this.sessionHandler(id);
                if (handler == null) {
                    handler = new FetchSessionHandler(this.logContext, id);
                    this.sessionHandlers.put(id, handler);
                }
                builder = handler.newBuilder();
                fetchable.put(node, builder);
            }
            builder.add(partition2, new FetchRequest.PartitionData(position.offset, -1L, this.fetchSize, position.currentLeader.epoch, Optional.empty()));
            this.log.debug("Added {} fetch request for partition {} at position {} to node {}", new Object[]{this.isolationLevel, partition2, position, node});
        }
        LinkedHashMap<Node, FetchSessionHandler.FetchRequestData> reqs = new LinkedHashMap<Node, FetchSessionHandler.FetchRequestData>();
        for (Map.Entry entry : fetchable.entrySet()) {
            reqs.put((Node)entry.getKey(), ((FetchSessionHandler.Builder)entry.getValue()).build());
        }
        return reqs;
    }

    private Map<Node, Map<TopicPartition, SubscriptionState.FetchPosition>> regroupFetchPositionsByLeader(Map<TopicPartition, SubscriptionState.FetchPosition> partitionMap) {
        return partitionMap.entrySet().stream().filter(entry -> ((SubscriptionState.FetchPosition)entry.getValue()).currentLeader.leader.isPresent()).collect(Collectors.groupingBy(entry -> ((SubscriptionState.FetchPosition)entry.getValue()).currentLeader.leader.get(), Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    private <T> Map<Node, Map<TopicPartition, T>> regroupPartitionMapByNode(Map<TopicPartition, T> partitionMap) {
        return partitionMap.entrySet().stream().collect(Collectors.groupingBy(entry -> this.metadata.fetch().leaderFor((TopicPartition)entry.getKey()), Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CompletedFetch initializeCompletedFetch(CompletedFetch nextCompletedFetch) {
        CompletedFetch completedFetch;
        block26: {
            TopicPartition tp = nextCompletedFetch.partition;
            FetchResponse.PartitionData partition2 = nextCompletedFetch.partitionData;
            long fetchOffset = nextCompletedFetch.nextFetchOffset;
            completedFetch = null;
            Errors error = partition2.error();
            try {
                if (!this.subscriptions.hasValidPosition(tp)) {
                    this.log.debug("Ignoring fetched records for partition {} since it no longer has valid position", (Object)tp);
                    break block26;
                }
                if (error == Errors.NONE) {
                    SubscriptionState.FetchPosition position = this.subscriptions.position(tp);
                    if (position == null || position.offset != fetchOffset) {
                        this.log.debug("Discarding stale fetch response for partition {} since its offset {} does not match the expected offset {}", tp, fetchOffset, position);
                        CompletedFetch completedFetch2 = null;
                        return completedFetch2;
                    }
                    this.log.trace("Preparing to read {} bytes of data for partition {} with offset {}", ((Records)partition2.records()).sizeInBytes(), tp, position);
                    Iterator<? extends RecordBatch> batches = ((Records)partition2.records()).batches().iterator();
                    completedFetch = nextCompletedFetch;
                    if (!batches.hasNext() && ((Records)partition2.records()).sizeInBytes() > 0) {
                        if (completedFetch.responseVersion < 3) {
                            Map<TopicPartition, Long> recordTooLargePartitions = Collections.singletonMap(tp, fetchOffset);
                            throw new RecordTooLargeException("There are some messages at [Partition=Offset]: " + recordTooLargePartitions + " whose size is larger than the fetch size " + this.fetchSize + " and hence cannot be returned. Please considering upgrading your broker to 0.10.1.0 or newer to avoid this issue. Alternately, increase the fetch size on the client (using " + "max.partition.fetch.bytes" + ")", recordTooLargePartitions);
                        }
                        throw new KafkaException("Failed to make progress reading messages at " + tp + "=" + fetchOffset + ". Received a non-empty fetch response from the server, but no complete records were found.");
                    }
                    if (partition2.highWatermark() >= 0L) {
                        this.log.trace("Updating high watermark for partition {} to {}", (Object)tp, (Object)partition2.highWatermark());
                        this.subscriptions.updateHighWatermark(tp, partition2.highWatermark());
                    }
                    if (partition2.logStartOffset() >= 0L) {
                        this.log.trace("Updating log start offset for partition {} to {}", (Object)tp, (Object)partition2.logStartOffset());
                        this.subscriptions.updateLogStartOffset(tp, partition2.logStartOffset());
                    }
                    if (partition2.lastStableOffset() >= 0L) {
                        this.log.trace("Updating last stable offset for partition {} to {}", (Object)tp, (Object)partition2.lastStableOffset());
                        this.subscriptions.updateLastStableOffset(tp, partition2.lastStableOffset());
                    }
                    if (partition2.preferredReadReplica().isPresent()) {
                        this.subscriptions.updatePreferredReadReplica(completedFetch.partition, partition2.preferredReadReplica().get(), () -> {
                            long expireTimeMs = this.time.milliseconds() + this.metadata.metadataExpireMs();
                            this.log.debug("Updating preferred read replica for partition {} to {}, set to expire at {}", tp, partition2.preferredReadReplica().get(), expireTimeMs);
                            return expireTimeMs;
                        });
                    }
                    nextCompletedFetch.initialized = true;
                    break block26;
                }
                if (error == Errors.NOT_LEADER_OR_FOLLOWER || error == Errors.REPLICA_NOT_AVAILABLE || error == Errors.KAFKA_STORAGE_ERROR || error == Errors.FENCED_LEADER_EPOCH || error == Errors.OFFSET_NOT_AVAILABLE) {
                    this.log.debug("Error in fetch for partition {}: {}", (Object)tp, (Object)error.exceptionName());
                    this.metadata.requestUpdate();
                    break block26;
                }
                if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) {
                    this.log.warn("Received unknown topic or partition error in fetch for partition {}", (Object)tp);
                    this.metadata.requestUpdate();
                    break block26;
                }
                if (error == Errors.OFFSET_OUT_OF_RANGE) {
                    Optional<Integer> clearedReplicaId = this.subscriptions.clearPreferredReadReplica(tp);
                    if (!clearedReplicaId.isPresent()) {
                        SubscriptionState.FetchPosition position = this.subscriptions.position(tp);
                        if (position == null || fetchOffset != position.offset) {
                            this.log.debug("Discarding stale fetch response for partition {} since the fetched offset {} does not match the current offset {}", tp, fetchOffset, position);
                        } else {
                            this.handleOffsetOutOfRange(position, tp);
                        }
                    } else {
                        this.log.debug("Unset the preferred read replica {} for partition {} since we got {} when fetching {}", new Object[]{clearedReplicaId.get(), tp, error, fetchOffset});
                    }
                    break block26;
                }
                if (error == Errors.TOPIC_AUTHORIZATION_FAILED) {
                    this.log.warn("Not authorized to read from partition {}.", (Object)tp);
                    throw new TopicAuthorizationException(Collections.singleton(tp.topic()));
                }
                if (error == Errors.UNKNOWN_LEADER_EPOCH) {
                    this.log.debug("Received unknown leader epoch error in fetch for partition {}", (Object)tp);
                    break block26;
                }
                if (error == Errors.UNKNOWN_SERVER_ERROR) {
                    this.log.warn("Unknown server error while fetching offset {} for topic-partition {}", (Object)fetchOffset, (Object)tp);
                    break block26;
                }
                if (error == Errors.CORRUPT_MESSAGE) {
                    throw new KafkaException("Encountered corrupt message when fetching offset " + fetchOffset + " for topic-partition " + tp);
                }
                throw new IllegalStateException("Unexpected error code " + error.code() + " while fetching at offset " + fetchOffset + " from topic-partition " + tp);
            }
            finally {
                if (completedFetch == null) {
                    nextCompletedFetch.metricAggregator.record(tp, 0, 0);
                }
                if (error != Errors.NONE) {
                    this.subscriptions.movePartitionToEnd(tp);
                }
            }
        }
        return completedFetch;
    }

    private void handleOffsetOutOfRange(SubscriptionState.FetchPosition fetchPosition, TopicPartition topicPartition) {
        String errorMessage = "Fetch position " + fetchPosition + " is out of range for partition " + topicPartition;
        if (!this.subscriptions.hasDefaultOffsetResetPolicy()) {
            this.log.info("{}, raising error to the application since no reset policy is configured", (Object)errorMessage);
            throw new OffsetOutOfRangeException(errorMessage, Collections.singletonMap(topicPartition, fetchPosition.offset));
        }
        this.log.info("{}, resetting offset", (Object)errorMessage);
        this.subscriptions.requestOffsetReset(topicPartition);
    }

    private ConsumerRecord<K, V> parseRecord(TopicPartition partition2, RecordBatch batch, Record record) {
        try {
            long offset = record.offset();
            long timestamp = record.timestamp();
            Optional<Integer> leaderEpoch = this.maybeLeaderEpoch(batch.partitionLeaderEpoch());
            TimestampType timestampType = batch.timestampType();
            RecordHeaders headers = new RecordHeaders(record.headers());
            ByteBuffer keyBytes = record.key();
            byte[] keyByteArray = keyBytes == null ? null : Utils.toArray(keyBytes);
            Object key = keyBytes == null ? null : (Object)this.keyDeserializer.deserialize(partition2.topic(), headers, keyByteArray);
            ByteBuffer valueBytes = record.value();
            byte[] valueByteArray = valueBytes == null ? null : Utils.toArray(valueBytes);
            Object value = valueBytes == null ? null : (Object)this.valueDeserializer.deserialize(partition2.topic(), headers, valueByteArray);
            return new ConsumerRecord<Object, Object>(partition2.topic(), partition2.partition(), offset, timestamp, timestampType, record.checksumOrNull(), keyByteArray == null ? -1 : keyByteArray.length, valueByteArray == null ? -1 : valueByteArray.length, key, value, headers, leaderEpoch);
        }
        catch (RuntimeException e) {
            throw new SerializationException("Error deserializing key/value for partition " + partition2 + " at offset " + record.offset() + ". If needed, please seek past the record to continue consumption.", e);
        }
    }

    private Optional<Integer> maybeLeaderEpoch(int leaderEpoch) {
        return leaderEpoch == -1 ? Optional.empty() : Optional.of(leaderEpoch);
    }

    public void clearBufferedDataForUnassignedPartitions(Collection<TopicPartition> assignedPartitions) {
        Iterator<CompletedFetch> completedFetchesItr = this.completedFetches.iterator();
        while (completedFetchesItr.hasNext()) {
            CompletedFetch records = completedFetchesItr.next();
            TopicPartition tp = records.partition;
            if (assignedPartitions.contains(tp)) continue;
            records.drain();
            completedFetchesItr.remove();
        }
        if (this.nextInLineFetch != null && !assignedPartitions.contains(this.nextInLineFetch.partition)) {
            this.nextInLineFetch.drain();
            this.nextInLineFetch = null;
        }
    }

    public void clearBufferedDataForUnassignedTopics(Collection<String> assignedTopics) {
        HashSet<TopicPartition> currentTopicPartitions = new HashSet<TopicPartition>();
        for (TopicPartition tp : this.subscriptions.assignedPartitions()) {
            if (!assignedTopics.contains(tp.topic())) continue;
            currentTopicPartitions.add(tp);
        }
        this.clearBufferedDataForUnassignedPartitions(currentTopicPartitions);
    }

    protected FetchSessionHandler sessionHandler(int node) {
        return this.sessionHandlers.get(node);
    }

    public static Sensor throttleTimeSensor(Metrics metrics, FetcherMetricsRegistry metricsRegistry) {
        Sensor fetchThrottleTimeSensor = metrics.sensor("fetch-throttle-time");
        fetchThrottleTimeSensor.add(metrics.metricInstance(metricsRegistry.fetchThrottleTimeAvg, new String[0]), new Avg());
        fetchThrottleTimeSensor.add(metrics.metricInstance(metricsRegistry.fetchThrottleTimeMax, new String[0]), new Max());
        return fetchThrottleTimeSensor;
    }

    @Override
    public void close() {
        if (this.nextInLineFetch != null) {
            this.nextInLineFetch.drain();
        }
        this.decompressionBufferSupplier.close();
    }

    private Set<String> topicsForPartitions(Collection<TopicPartition> partitions) {
        return partitions.stream().map(TopicPartition::topic).collect(Collectors.toSet());
    }

    private static class FetchManagerMetrics {
        private final Metrics metrics;
        private FetcherMetricsRegistry metricsRegistry;
        private final Sensor bytesFetched;
        private final Sensor recordsFetched;
        private final Sensor fetchLatency;
        private final Sensor recordsFetchLag;
        private final Sensor recordsFetchLead;
        private int assignmentId = 0;
        private Set<TopicPartition> assignedPartitions = Collections.emptySet();

        private FetchManagerMetrics(Metrics metrics, FetcherMetricsRegistry metricsRegistry) {
            this.metrics = metrics;
            this.metricsRegistry = metricsRegistry;
            this.bytesFetched = metrics.sensor("bytes-fetched");
            this.bytesFetched.add(metrics.metricInstance(metricsRegistry.fetchSizeAvg, new String[0]), new Avg());
            this.bytesFetched.add(metrics.metricInstance(metricsRegistry.fetchSizeMax, new String[0]), new Max());
            this.bytesFetched.add(new Meter(metrics.metricInstance(metricsRegistry.bytesConsumedRate, new String[0]), metrics.metricInstance(metricsRegistry.bytesConsumedTotal, new String[0])));
            this.recordsFetched = metrics.sensor("records-fetched");
            this.recordsFetched.add(metrics.metricInstance(metricsRegistry.recordsPerRequestAvg, new String[0]), new Avg());
            this.recordsFetched.add(new Meter(metrics.metricInstance(metricsRegistry.recordsConsumedRate, new String[0]), metrics.metricInstance(metricsRegistry.recordsConsumedTotal, new String[0])));
            this.fetchLatency = metrics.sensor("fetch-latency");
            this.fetchLatency.add(metrics.metricInstance(metricsRegistry.fetchLatencyAvg, new String[0]), new Avg());
            this.fetchLatency.add(metrics.metricInstance(metricsRegistry.fetchLatencyMax, new String[0]), new Max());
            this.fetchLatency.add(new Meter(new WindowedCount(), metrics.metricInstance(metricsRegistry.fetchRequestRate, new String[0]), metrics.metricInstance(metricsRegistry.fetchRequestTotal, new String[0])));
            this.recordsFetchLag = metrics.sensor("records-lag");
            this.recordsFetchLag.add(metrics.metricInstance(metricsRegistry.recordsLagMax, new String[0]), new Max());
            this.recordsFetchLead = metrics.sensor("records-lead");
            this.recordsFetchLead.add(metrics.metricInstance(metricsRegistry.recordsLeadMin, new String[0]), new Min());
        }

        private void recordTopicFetchMetrics(String topic, int bytes, int records) {
            String name = "topic." + topic + ".bytes-fetched";
            Sensor bytesFetched = this.metrics.getSensor(name);
            if (bytesFetched == null) {
                Map<String, String> metricTags = Collections.singletonMap("topic", topic.replace('.', '_'));
                bytesFetched = this.metrics.sensor(name);
                bytesFetched.add(this.metrics.metricInstance(this.metricsRegistry.topicFetchSizeAvg, metricTags), new Avg());
                bytesFetched.add(this.metrics.metricInstance(this.metricsRegistry.topicFetchSizeMax, metricTags), new Max());
                bytesFetched.add(new Meter(this.metrics.metricInstance(this.metricsRegistry.topicBytesConsumedRate, metricTags), this.metrics.metricInstance(this.metricsRegistry.topicBytesConsumedTotal, metricTags)));
            }
            bytesFetched.record(bytes);
            name = "topic." + topic + ".records-fetched";
            Sensor recordsFetched = this.metrics.getSensor(name);
            if (recordsFetched == null) {
                HashMap<String, String> metricTags = new HashMap<String, String>(1);
                metricTags.put("topic", topic.replace('.', '_'));
                recordsFetched = this.metrics.sensor(name);
                recordsFetched.add(this.metrics.metricInstance(this.metricsRegistry.topicRecordsPerRequestAvg, metricTags), new Avg());
                recordsFetched.add(new Meter(this.metrics.metricInstance(this.metricsRegistry.topicRecordsConsumedRate, metricTags), this.metrics.metricInstance(this.metricsRegistry.topicRecordsConsumedTotal, metricTags)));
            }
            recordsFetched.record(records);
        }

        private void maybeUpdateAssignment(SubscriptionState subscription) {
            int newAssignmentId = subscription.assignmentId();
            if (this.assignmentId != newAssignmentId) {
                Set<TopicPartition> newAssignedPartitions = subscription.assignedPartitions();
                for (TopicPartition tp : this.assignedPartitions) {
                    if (newAssignedPartitions.contains(tp)) continue;
                    this.metrics.removeSensor(FetchManagerMetrics.partitionLagMetricName(tp));
                    this.metrics.removeSensor(FetchManagerMetrics.partitionLeadMetricName(tp));
                    this.metrics.removeMetric(this.partitionPreferredReadReplicaMetricName(tp));
                }
                for (TopicPartition tp : newAssignedPartitions) {
                    MetricName metricName;
                    if (this.assignedPartitions.contains(tp) || this.metrics.metric(metricName = this.partitionPreferredReadReplicaMetricName(tp)) != null) continue;
                    this.metrics.addMetric(metricName, (config, now) -> subscription.preferredReadReplica(tp, 0L).orElse(-1));
                }
                this.assignedPartitions = newAssignedPartitions;
                this.assignmentId = newAssignmentId;
            }
        }

        private void recordPartitionLead(TopicPartition tp, long lead) {
            this.recordsFetchLead.record(lead);
            String name = FetchManagerMetrics.partitionLeadMetricName(tp);
            Sensor recordsLead = this.metrics.getSensor(name);
            if (recordsLead == null) {
                Map<String, String> metricTags = this.topicPartitionTags(tp);
                recordsLead = this.metrics.sensor(name);
                recordsLead.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLead, metricTags), new Value());
                recordsLead.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLeadMin, metricTags), new Min());
                recordsLead.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLeadAvg, metricTags), new Avg());
            }
            recordsLead.record(lead);
        }

        private void recordPartitionLag(TopicPartition tp, long lag) {
            this.recordsFetchLag.record(lag);
            String name = FetchManagerMetrics.partitionLagMetricName(tp);
            Sensor recordsLag = this.metrics.getSensor(name);
            if (recordsLag == null) {
                Map<String, String> metricTags = this.topicPartitionTags(tp);
                recordsLag = this.metrics.sensor(name);
                recordsLag.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLag, metricTags), new Value());
                recordsLag.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLagMax, metricTags), new Max());
                recordsLag.add(this.metrics.metricInstance(this.metricsRegistry.partitionRecordsLagAvg, metricTags), new Avg());
            }
            recordsLag.record(lag);
        }

        private static String partitionLagMetricName(TopicPartition tp) {
            return tp + ".records-lag";
        }

        private static String partitionLeadMetricName(TopicPartition tp) {
            return tp + ".records-lead";
        }

        private MetricName partitionPreferredReadReplicaMetricName(TopicPartition tp) {
            Map<String, String> metricTags = this.topicPartitionTags(tp);
            return this.metrics.metricInstance(this.metricsRegistry.partitionPreferredReadReplica, metricTags);
        }

        private Map<String, String> topicPartitionTags(TopicPartition tp) {
            HashMap<String, String> metricTags = new HashMap<String, String>(2);
            metricTags.put("topic", tp.topic().replace('.', '_'));
            metricTags.put("partition", String.valueOf(tp.partition()));
            return metricTags;
        }
    }

    private static class FetchResponseMetricAggregator {
        private final FetchManagerMetrics sensors;
        private final Set<TopicPartition> unrecordedPartitions;
        private final FetchMetrics fetchMetrics = new FetchMetrics();
        private final Map<String, FetchMetrics> topicFetchMetrics = new HashMap<String, FetchMetrics>();

        private FetchResponseMetricAggregator(FetchManagerMetrics sensors, Set<TopicPartition> partitions) {
            this.sensors = sensors;
            this.unrecordedPartitions = partitions;
        }

        public void record(TopicPartition partition2, int bytes, int records) {
            this.unrecordedPartitions.remove(partition2);
            this.fetchMetrics.increment(bytes, records);
            String topic = partition2.topic();
            FetchMetrics topicFetchMetric = this.topicFetchMetrics.get(topic);
            if (topicFetchMetric == null) {
                topicFetchMetric = new FetchMetrics();
                this.topicFetchMetrics.put(topic, topicFetchMetric);
            }
            topicFetchMetric.increment(bytes, records);
            if (this.unrecordedPartitions.isEmpty()) {
                this.sensors.bytesFetched.record(this.fetchMetrics.fetchBytes);
                this.sensors.recordsFetched.record(this.fetchMetrics.fetchRecords);
                for (Map.Entry<String, FetchMetrics> entry : this.topicFetchMetrics.entrySet()) {
                    FetchMetrics metric = entry.getValue();
                    this.sensors.recordTopicFetchMetrics(entry.getKey(), metric.fetchBytes, metric.fetchRecords);
                }
            }
        }

        private static class FetchMetrics {
            private int fetchBytes;
            private int fetchRecords;

            private FetchMetrics() {
            }

            protected void increment(int bytes, int records) {
                this.fetchBytes += bytes;
                this.fetchRecords += records;
            }
        }
    }

    private class CompletedFetch {
        private final TopicPartition partition;
        private final Iterator<? extends RecordBatch> batches;
        private final Set<Long> abortedProducerIds;
        private final PriorityQueue<FetchResponse.AbortedTransaction> abortedTransactions;
        private final FetchResponse.PartitionData<Records> partitionData;
        private final FetchResponseMetricAggregator metricAggregator;
        private final short responseVersion;
        private int recordsRead;
        private int bytesRead;
        private RecordBatch currentBatch;
        private Record lastRecord;
        private CloseableIterator<Record> records;
        private long nextFetchOffset;
        private Optional<Integer> lastEpoch;
        private boolean isConsumed = false;
        private Exception cachedRecordException = null;
        private boolean corruptLastRecord = false;
        private boolean initialized = false;

        private CompletedFetch(TopicPartition partition2, FetchResponse.PartitionData<Records> partitionData, FetchResponseMetricAggregator metricAggregator, Iterator<? extends RecordBatch> batches, Long fetchOffset, short responseVersion) {
            this.partition = partition2;
            this.partitionData = partitionData;
            this.metricAggregator = metricAggregator;
            this.batches = batches;
            this.nextFetchOffset = fetchOffset;
            this.responseVersion = responseVersion;
            this.lastEpoch = Optional.empty();
            this.abortedProducerIds = new HashSet<Long>();
            this.abortedTransactions = this.abortedTransactions(partitionData);
        }

        private void drain() {
            if (!this.isConsumed) {
                this.maybeCloseRecordStream();
                this.cachedRecordException = null;
                this.isConsumed = true;
                this.metricAggregator.record(this.partition, this.bytesRead, this.recordsRead);
                if (this.bytesRead > 0) {
                    Fetcher.this.subscriptions.movePartitionToEnd(this.partition);
                }
            }
        }

        private void maybeEnsureValid(RecordBatch batch) {
            if (Fetcher.this.checkCrcs && this.currentBatch.magic() >= 2) {
                try {
                    batch.ensureValid();
                }
                catch (CorruptRecordException e) {
                    throw new KafkaException("Record batch for partition " + this.partition + " at offset " + batch.baseOffset() + " is invalid, cause: " + e.getMessage());
                }
            }
        }

        private void maybeEnsureValid(Record record) {
            if (Fetcher.this.checkCrcs) {
                try {
                    record.ensureValid();
                }
                catch (CorruptRecordException e) {
                    throw new KafkaException("Record for partition " + this.partition + " at offset " + record.offset() + " is invalid, cause: " + e.getMessage());
                }
            }
        }

        private void maybeCloseRecordStream() {
            if (this.records != null) {
                this.records.close();
                this.records = null;
            }
        }

        private Record nextFetchedRecord() {
            while (true) {
                if (this.records == null || !this.records.hasNext()) {
                    this.maybeCloseRecordStream();
                    if (!this.batches.hasNext()) {
                        if (this.currentBatch != null) {
                            this.nextFetchOffset = this.currentBatch.nextOffset();
                        }
                        this.drain();
                        return null;
                    }
                    this.currentBatch = this.batches.next();
                    this.lastEpoch = this.currentBatch.partitionLeaderEpoch() == -1 ? Optional.empty() : Optional.of(this.currentBatch.partitionLeaderEpoch());
                    this.maybeEnsureValid(this.currentBatch);
                    if (Fetcher.this.isolationLevel == IsolationLevel.READ_COMMITTED && this.currentBatch.hasProducerId()) {
                        this.consumeAbortedTransactionsUpTo(this.currentBatch.lastOffset());
                        long producerId = this.currentBatch.producerId();
                        if (this.containsAbortMarker(this.currentBatch)) {
                            this.abortedProducerIds.remove(producerId);
                        } else if (this.isBatchAborted(this.currentBatch)) {
                            Fetcher.this.log.debug("Skipping aborted record batch from partition {} with producerId {} and offsets {} to {}", this.partition, producerId, this.currentBatch.baseOffset(), this.currentBatch.lastOffset());
                            this.nextFetchOffset = this.currentBatch.nextOffset();
                            continue;
                        }
                    }
                    this.records = this.currentBatch.streamingIterator(Fetcher.this.decompressionBufferSupplier);
                    continue;
                }
                Record record = (Record)this.records.next();
                if (record.offset() < this.nextFetchOffset) continue;
                this.maybeEnsureValid(record);
                if (!this.currentBatch.isControlBatch()) {
                    return record;
                }
                this.nextFetchOffset = record.offset() + 1L;
            }
        }

        private List<ConsumerRecord<K, V>> fetchRecords(int maxRecords) {
            ArrayList records;
            block9: {
                if (this.corruptLastRecord) {
                    throw new KafkaException("Received exception when fetching the next record from " + this.partition + ". If needed, please seek past the record to continue consumption.", this.cachedRecordException);
                }
                if (this.isConsumed) {
                    return Collections.emptyList();
                }
                records = new ArrayList();
                try {
                    for (int i = 0; i < maxRecords; ++i) {
                        if (this.cachedRecordException == null) {
                            this.corruptLastRecord = true;
                            this.lastRecord = this.nextFetchedRecord();
                            this.corruptLastRecord = false;
                        }
                        if (this.lastRecord != null) {
                            records.add(Fetcher.this.parseRecord(this.partition, this.currentBatch, this.lastRecord));
                            ++this.recordsRead;
                            this.bytesRead += this.lastRecord.sizeInBytes();
                            this.nextFetchOffset = this.lastRecord.offset() + 1L;
                            this.cachedRecordException = null;
                            continue;
                        }
                        break;
                    }
                }
                catch (SerializationException se) {
                    this.cachedRecordException = se;
                    if (records.isEmpty()) {
                        throw se;
                    }
                }
                catch (KafkaException e) {
                    this.cachedRecordException = e;
                    if (!records.isEmpty()) break block9;
                    throw new KafkaException("Received exception when fetching the next record from " + this.partition + ". If needed, please seek past the record to continue consumption.", e);
                }
            }
            return records;
        }

        private void consumeAbortedTransactionsUpTo(long offset) {
            if (this.abortedTransactions == null) {
                return;
            }
            while (!this.abortedTransactions.isEmpty() && this.abortedTransactions.peek().firstOffset <= offset) {
                FetchResponse.AbortedTransaction abortedTransaction = this.abortedTransactions.poll();
                this.abortedProducerIds.add(abortedTransaction.producerId);
            }
        }

        private boolean isBatchAborted(RecordBatch batch) {
            return batch.isTransactional() && this.abortedProducerIds.contains(batch.producerId());
        }

        private PriorityQueue<FetchResponse.AbortedTransaction> abortedTransactions(FetchResponse.PartitionData<?> partition2) {
            if (partition2.abortedTransactions() == null || partition2.abortedTransactions().isEmpty()) {
                return null;
            }
            PriorityQueue<FetchResponse.AbortedTransaction> abortedTransactions = new PriorityQueue<FetchResponse.AbortedTransaction>(partition2.abortedTransactions().size(), Comparator.comparingLong(o -> o.firstOffset));
            abortedTransactions.addAll(partition2.abortedTransactions());
            return abortedTransactions;
        }

        private boolean containsAbortMarker(RecordBatch batch) {
            if (!batch.isControlBatch()) {
                return false;
            }
            Iterator batchIterator = batch.iterator();
            if (!batchIterator.hasNext()) {
                return false;
            }
            Record firstRecord = (Record)batchIterator.next();
            return ControlRecordType.ABORT == ControlRecordType.parse(firstRecord.key());
        }

        private boolean notInitialized() {
            return !this.initialized;
        }
    }

    static class ListOffsetResult {
        private final Map<TopicPartition, ListOffsetData> fetchedOffsets;
        private final Set<TopicPartition> partitionsToRetry;

        ListOffsetResult(Map<TopicPartition, ListOffsetData> fetchedOffsets, Set<TopicPartition> partitionsNeedingRetry) {
            this.fetchedOffsets = fetchedOffsets;
            this.partitionsToRetry = partitionsNeedingRetry;
        }

        ListOffsetResult() {
            this.fetchedOffsets = new HashMap<TopicPartition, ListOffsetData>();
            this.partitionsToRetry = new HashSet<TopicPartition>();
        }
    }

    static class ListOffsetData {
        final long offset;
        final Long timestamp;
        final Optional<Integer> leaderEpoch;

        ListOffsetData(long offset, Long timestamp, Optional<Integer> leaderEpoch) {
            this.offset = offset;
            this.timestamp = timestamp;
            this.leaderEpoch = leaderEpoch;
        }
    }
}

