/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.core.messages;

import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.nio.ByteBuffer;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.websocket.core.CoreSession;
import org.eclipse.jetty.websocket.core.Frame;
import org.eclipse.jetty.websocket.core.messages.MessageSink;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageInputStream
extends InputStream
implements MessageSink {
    private static final Logger LOG = LoggerFactory.getLogger(MessageInputStream.class);
    private static final Entry EOF = new Entry(null, Callback.NOOP);
    private static final Entry CLOSED = new Entry(null, Callback.NOOP);
    private static final Entry FAILED = new Entry(null, Callback.NOOP);
    private final AutoLock.WithCondition lock = new AutoLock.WithCondition();
    private final ArrayDeque<Entry> buffers = new ArrayDeque();
    private final CoreSession session;
    private Entry currentEntry;
    private Throwable failure;
    private boolean closed;
    private long timeoutMs = -1L;

    public MessageInputStream(CoreSession session) {
        this.session = session;
    }

    @Override
    public void accept(Frame frame, Callback callback) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("accepting {}", (Object)frame);
        }
        if (!frame.isFin() && !frame.hasPayload()) {
            callback.succeeded();
            this.session.demand();
            return;
        }
        Runnable action = null;
        try (AutoLock.WithCondition l = this.lock.lock();){
            if (this.failure != null) {
                Throwable cause = this.failure;
                action = () -> callback.failed(cause);
            } else if (this.closed) {
                action = () -> {
                    callback.succeeded();
                    if (!frame.isFin()) {
                        this.session.demand();
                    }
                };
            } else {
                this.buffers.offer(new Entry(frame, callback));
                if (frame.isFin()) {
                    this.buffers.offer(EOF);
                }
            }
            l.signal();
        }
        if (action != null) {
            action.run();
        }
    }

    @Override
    public int read() throws IOException {
        int len;
        byte[] buf = new byte[1];
        do {
            if ((len = this.read(buf, 0, 1)) >= 0) continue;
            return -1;
        } while (len <= 0);
        return buf[0] & 0xFF;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        ByteBuffer buffer = ByteBuffer.wrap(b, off, len).slice();
        BufferUtil.clear((ByteBuffer)buffer);
        return this.read(buffer);
    }

    public int read(ByteBuffer buffer) throws IOException {
        Entry currentEntry = this.getCurrentEntry();
        if (LOG.isDebugEnabled()) {
            LOG.debug("currentEntry = {}", (Object)currentEntry);
        }
        if (currentEntry == FAILED) {
            throw IO.rethrow((Throwable)this.getFailure());
        }
        if (currentEntry == CLOSED) {
            throw new IOException("Closed");
        }
        if (currentEntry == EOF) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Read EOF");
            }
            return -1;
        }
        ByteBuffer payload = currentEntry.frame.getPayload();
        if (currentEntry.frame.isFin() && !payload.hasRemaining()) {
            this.succeedCurrentEntry();
            return this.read(buffer);
        }
        int length = BufferUtil.append((ByteBuffer)buffer, (ByteBuffer)payload);
        if (!payload.hasRemaining()) {
            this.succeedCurrentEntry();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("filled {} bytes from {}", (Object)length, (Object)currentEntry);
        }
        return length;
    }

    @Override
    public void fail(Throwable failure) {
        if (LOG.isDebugEnabled()) {
            LOG.atDebug().setCause(failure).log("fail()");
        }
        ArrayList<Entry> entries = new ArrayList<Entry>();
        try (AutoLock.WithCondition l = this.lock.lock();){
            if (this.failure != null) {
                return;
            }
            this.failure = failure;
            this.drainInto(entries);
            this.buffers.offer(FAILED);
            l.signal();
        }
        entries.forEach(e -> e.callback.failed(failure));
    }

    @Override
    public void close() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("close()");
        }
        ArrayList<Entry> entries = new ArrayList<Entry>();
        try (AutoLock.WithCondition l = this.lock.lock();){
            if (this.closed) {
                return;
            }
            this.closed = true;
            this.drainInto(entries);
            this.buffers.offer(CLOSED);
            l.signal();
        }
        boolean demandRequired = false;
        for (Entry e : entries) {
            e.callback.succeeded();
            if (e.frame == null || e.frame.isFin()) continue;
            demandRequired = true;
        }
        if (demandRequired) {
            this.session.demand();
        }
    }

    private void drainInto(ArrayList<Entry> entries) {
        assert (this.lock.isHeldByCurrentThread());
        if (this.currentEntry != null) {
            entries.add(this.currentEntry);
            this.currentEntry = null;
        }
        entries.addAll(this.buffers);
        this.buffers.clear();
    }

    public void setTimeout(long timeoutMs) {
        this.timeoutMs = timeoutMs;
    }

    private void succeedCurrentEntry() {
        Entry current;
        try (AutoLock.WithCondition ignored = this.lock.lock();){
            current = this.currentEntry;
            this.currentEntry = null;
        }
        if (current != null) {
            current.callback.succeeded();
            if (!current.frame.isFin()) {
                this.session.demand();
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private Entry getCurrentEntry() throws IOException {
        try (AutoLock.WithCondition l = this.lock.lock();){
            if (this.currentEntry != null) {
                Entry entry2 = this.currentEntry;
                return entry2;
            }
            long timeout = this.timeoutMs;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Waiting {} ms to read", (Object)timeout);
            }
            while (true) {
                Entry result;
                if ((result = this.buffers.poll()) != null) {
                    Entry entry = this.currentEntry = result;
                    return entry;
                }
                if (timeout < 0L) {
                    l.await();
                    continue;
                }
                if (!l.await(timeout, TimeUnit.MILLISECONDS)) break;
            }
            throw new IOException(String.format("Read timeout: %,dms expired", timeout));
        }
        catch (InterruptedException e) {
            this.close();
            throw new InterruptedIOException();
        }
    }

    private Throwable getFailure() {
        try (AutoLock.WithCondition ignored = this.lock.lock();){
            Throwable throwable = this.failure;
            return throwable;
        }
    }

    private record Entry(Frame frame, Callback callback) {
    }
}

