/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.qpid.protonj2.codec;

import java.util.concurrent.TimeUnit;

import org.apache.qpid.protonj2.buffer.ProtonBuffer;
import org.apache.qpid.protonj2.buffer.ProtonBufferAllocator;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.profile.GCProfiler;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

@State(Scope.Benchmark)
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
@Warmup(iterations = 5, time = 1)
@Measurement(iterations = 5, time = 1)
public abstract class CodecBenchmarkBase {

    public static final int DEFAULT_BUFFER_SIZE = 8192;

    protected ProtonBuffer buffer;
    protected Encoder encoder = CodecFactory.getDefaultEncoder();
    protected EncoderState encoderState = encoder.newEncoderState();
    protected Decoder decoder = CodecFactory.getDefaultDecoder();
    protected DecoderState decoderState = decoder.newDecoderState();

    /**
     * It could be overridden to allow encoding/decoding buffer to be sized
     * differently from {@link #DEFAULT_BUFFER_SIZE}
     *
     * @return the buffer size to use in tests when creating encoding buffers.
     */
    protected int bufferSize() {
        return DEFAULT_BUFFER_SIZE;
    }

    private void initProtonBuffer() {
        buffer = ProtonBufferAllocator.defaultAllocator().allocate(bufferSize());
    }

    public void init() {
        initProtonBuffer();
    }

    public static void main(String[] args) throws RunnerException {
        runBenchmark(CodecBenchmarkBase.class);
    }

    public static void runBenchmark(Class<?> benchmarkClass) throws RunnerException {
        final Options opt = new OptionsBuilder()
            .include(benchmarkClass.getSimpleName())
            .addProfiler(GCProfiler.class)
            .shouldDoGC(true)
            .warmupIterations(5)
            .measurementIterations(5)
            .forks(1)
            .build();

        new Runner(opt).run();
    }
}
