import { MatrixDirectoryVisibility } from "../bridge/IrcHandler";
import { IrcRoom } from "../models/IrcRoom";
import { IrcClientConfig } from "../models/IrcClientConfig";
import { MatrixRoom, MatrixUser, UserBridgeStore, UserActivityStore, RoomBridgeStore, RoomBridgeStoreEntry as Entry, UserActivity, UserActivitySet, ProvisionSession } from "matrix-appservice-bridge";
import { DataStore, RoomOrigin, ChannelMappings, UserFeatures } from "./DataStore";
import { IrcServer, IrcServerConfig } from "../irc/IrcServer";
export declare class NeDBDataStore implements DataStore {
    private userStore;
    private userActivityStore;
    private roomStore;
    private bridgeDomain;
    private serverMappings;
    private cryptoStore?;
    constructor(userStore: UserBridgeStore, userActivityStore: UserActivityStore, roomStore: RoomBridgeStore, bridgeDomain: string, pkeyPath?: string);
    runMigrations(): Promise<void>;
    setServerFromConfig(server: IrcServer, serverConfig: IrcServerConfig): Promise<void>;
    /**
     * Persists an IRC <--> Matrix room mapping in the database.
     * @param {IrcRoom} ircRoom : The IRC room to store.
     * @param {MatrixRoom} matrixRoom : The Matrix room to store.
     * @param {string} origin : "config" if this mapping is from the config yaml,
     * "provision" if this mapping was provisioned, "alias" if it was created via
     * aliasing and "join" if it was created during a join.
     * @return {Promise}
     */
    storeRoom(ircRoom: IrcRoom, matrixRoom: MatrixRoom, origin: RoomOrigin): Promise<void>;
    /**
     * Get an IRC <--> Matrix room mapping from the database.
     * @param {string} roomId : The Matrix room ID.
     * @param {string} ircDomain : The IRC server domain.
     * @param {string} ircChannel : The IRC channel.
     * @param {string} origin : (Optional) "config" if this mapping was from the config yaml,
     * "provision" if this mapping was provisioned, "alias" if it was created via aliasing and
     * "join" if it was created during a join.
     * @return {Promise} A promise which resolves to a room entry, or null if one is not found.
     */
    getRoom(roomId: string, ircDomain: string, ircChannel: string, origin?: RoomOrigin): Promise<Entry | null>;
    /**
     * Get all Matrix <--> IRC room mappings from the database.
     * @return {Promise} A promise which resolves to a map:
     *      $roomId => [{networkId: 'server #channel1', channel: '#channel2'} , ...]
     */
    getAllChannelMappings(): Promise<ChannelMappings>;
    /**
     * Get provisioned IRC <--> Matrix room mappings from the database where
     * the matrix room ID is roomId.
     * @param {string} roomId : The Matrix room ID.
     * @return {Promise} A promise which resolves to a list
     * of entries.
     */
    getProvisionedMappings(roomId: string): Promise<Entry[]>;
    /**
     * Remove an IRC <--> Matrix room mapping from the database.
     * @param {string} roomId : The Matrix room ID.
     * @param {string} ircDomain : The IRC server domain.
     * @param {string} ircChannel : The IRC channel.
     * @param {string} origin : "config" if this mapping was from the config yaml,
     * "provision" if this mapping was provisioned, "alias" if it was created via
     * aliasing and "join" if it was created during a join.
     * @return {Promise}
     */
    removeRoom(roomId: string, ircDomain: string, ircChannel: string, origin?: RoomOrigin): Promise<void>;
    /**
     * Retrieve a list of IRC rooms for a given room ID.
     * @param {string} roomId : The room ID to get mapped IRC channels.
     * @return {Promise<Array<IrcRoom>>} A promise which resolves to a list of
     * rooms.
     */
    getIrcChannelsForRoomId(roomId: string): Promise<IrcRoom[]>;
    /**
     * Retrieve a list of IRC rooms for a given list of room IDs. This is significantly
     * faster than calling getIrcChannelsForRoomId for each room ID.
     * @param {string[]} roomIds : The room IDs to get mapped IRC channels.
     * @return {Promise<Map<string, IrcRoom[]>>} A promise which resolves to a map of
     * room ID to an array of IRC rooms.
     */
    getIrcChannelsForRoomIds(roomIds: string[]): Promise<{
        [roomId: string]: IrcRoom[];
    }>;
    /**
     * Retrieve a list of Matrix rooms for a given server and channel.
     * @param {IrcServer} server : The server to get rooms for.
     * @param {string} channel : The channel to get mapped rooms for.
     * @return {Promise<Array<MatrixRoom>>} A promise which resolves to a list of rooms.
     */
    getMatrixRoomsForChannel(server: IrcServer, channel: string): Promise<MatrixRoom[]>;
    getMappingsForChannelByOrigin(server: IrcServer, channel: string, origin: RoomOrigin | RoomOrigin[], allowUnset: boolean): Promise<Entry[]>;
    getModesForChannel(server: IrcServer, channel: string): Promise<{
        [id: string]: string[];
    }>;
    setModeForRoom(roomId: string, mode: string, enabled?: boolean): Promise<void>;
    setPmRoom(ircRoom: IrcRoom, matrixRoom: MatrixRoom, userId: string, virtualUserId: string): Promise<void>;
    removePmRoom(roomId: string): Promise<void>;
    getMatrixPmRoom(realUserId: string, virtualUserId: string): Promise<MatrixRoom | null>;
    getMatrixPmRoomById(roomId: string): Promise<MatrixRoom | null>;
    getTrackedChannelsForServer(domain: string): Promise<string[]>;
    getRoomIdsFromConfig(): Promise<string[]>;
    removeConfigMappings(): Promise<void>;
    getIpv6Counter(server: IrcServer, homeserver: string | null): Promise<number>;
    setIpv6Counter(counter: number, server: IrcServer, homeserver: string | null): Promise<void>;
    /**
     * Retrieve a stored admin room based on the room's ID.
     * @param {String} roomId : The room ID of the admin room.
     * @return {Promise} Resolved when the room is retrieved.
     */
    getAdminRoomById(roomId: string): Promise<MatrixRoom | null>;
    /**
     * Stores a unique admin room for a given user ID.
     * @param {MatrixRoom} room : The matrix room which is the admin room for this user.
     * @param {String} userId : The user ID who is getting an admin room.
     * @return {Promise} Resolved when the room is stored.
     */
    storeAdminRoom(room: MatrixRoom, userId: string): Promise<void>;
    removeAdminRoom(room: MatrixRoom): Promise<void>;
    upsertMatrixRoom(room: MatrixRoom): Promise<void>;
    getAdminRoomByUserId(userId: string): Promise<MatrixRoom | null>;
    storeMatrixUser(matrixUser: MatrixUser): Promise<void>;
    getIrcClientConfig(userId: string, domain: string): Promise<IrcClientConfig | null>;
    getMatrixUserByLocalpart(localpart: string): Promise<MatrixUser | null>;
    storeIrcClientConfig(config: IrcClientConfig): Promise<void>;
    ensurePasskeyCanDecrypt(): Promise<void>;
    getUserFeatures(userId: string): Promise<UserFeatures>;
    storeUserFeatures(userId: string, features: UserFeatures): Promise<void>;
    getUserActivity(): Promise<UserActivitySet>;
    storeUserActivity(userId: string, activity: UserActivity): Promise<void>;
    storePass(userId: string, domain: string, pass: string): Promise<void>;
    removePass(userId: string, domain: string): Promise<void>;
    getMatrixUserByUsername(domain: string, username: string): Promise<MatrixUser | undefined>;
    getCountForUsernamePrefix(domain: string, usernamePrefix: string): Promise<number>;
    updateLastSeenTimeForUser(userId: string): Promise<void>;
    getLastSeenTimeForUsers(): Promise<{
        user_id: string;
        ts: number;
    }[]>;
    getAllUserIds(): Promise<string[]>;
    getRoomVisibility(roomId: string): Promise<MatrixDirectoryVisibility>;
    getRoomsVisibility(roomIds: string[]): Promise<Map<string, MatrixDirectoryVisibility>>;
    setRoomVisibility(roomId: string, visibility: MatrixDirectoryVisibility): Promise<void>;
    deactivateUser(userId: string): Promise<void>;
    isUserDeactivated(userId: string): Promise<boolean>;
    getRoomCount(): Promise<number>;
    roomUpgradeOnRoomMigrated(oldRoomId: string, newRoomId: string): Promise<void>;
    getSessionForToken(): Promise<ProvisionSession | null>;
    createSession(): Promise<void>;
    deleteSession(): Promise<void>;
    deleteAllSessions(): Promise<void>;
    destroy(): Promise<void>;
    private static createPmId;
    private static createAdminId;
    static createMappingId(roomId: string, ircDomain: string, ircChannel: string): string;
}
