/***************************************************************************
*   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
***************************************************************************/
/** @file
* This file defines classes SKGBudgetObject.
*
* @author Stephane MANKOWSKI / Guillaume DE BURE
*/
#include "skgbudgetobject.h"

#include <KLocale>

#include "skgdocumentbank.h"
#include "skgtraces.h"
#include "skgcategoryobject.h"
#include "skgbudgetruleobject.h"
#include "skgtransactionmng.h"

SKGBudgetObject::SKGBudgetObject(SKGDocument* iDocument, int iID)
    : SKGObjectBase(iDocument, "v_budget", iID)
{
}

SKGBudgetObject::~SKGBudgetObject()
{
}

SKGBudgetObject::SKGBudgetObject(const SKGBudgetObject& iObject)
    : SKGObjectBase(iObject)
{
}

SKGBudgetObject::SKGBudgetObject(const SKGObjectBase& iObject)
{
    if (iObject.getRealTable() == "budget") {
        copyFrom(iObject);
    } else {
        *this = SKGObjectBase(iObject.getDocument(), "v_budget", iObject.getID());
    }
}

const SKGBudgetObject& SKGBudgetObject::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

QString SKGBudgetObject::getWhereclauseId() const
{
    // Could we use the id
    QString output = SKGObjectBase::getWhereclauseId();
    if (output.isEmpty()) {
        QString y = getAttribute("i_year");
        if (!y.isEmpty()) {
            output = "i_year=" % y;
        }

        QString m = getAttribute("i_month");
        if (!m.isEmpty()) {
            if (!output.isEmpty()) {
                output = output % " AND ";
            }
            output = output % "i_month=" % m;
        }

        QString r = getAttribute("rc_category_id");
        if (!output.isEmpty()) {
            output = output % " AND ";
        }
        output = output % "rc_category_id=" % (r.isEmpty() ? "0" : r);
    }
    return output;
}

SKGError SKGBudgetObject::setBudgetedAmount(double iAmount)
{
    SKGError err = setAttribute("f_budgeted", SKGServices::doubleToString(iAmount));
    IFOKDO(err, setAttribute("f_budgeted_modified", SKGServices::doubleToString(iAmount)))
    return err;
}

double SKGBudgetObject::getBudgetedAmount() const
{
    return SKGServices::stringToDouble(getAttribute("f_budgeted"));
}

double SKGBudgetObject::getBudgetedModifiedAmount() const
{
    return SKGServices::stringToDouble(getAttribute("f_budgeted_modified"));
}

double SKGBudgetObject::getDelta() const
{
    return SKGServices::stringToDouble(getAttribute("f_DELTABEFORETRANSFER"));
}

SKGError SKGBudgetObject::setYear(int iYear)
{
    return setAttribute("i_year", SKGServices::intToString(iYear));
}

int SKGBudgetObject::getYear() const
{
    return SKGServices::stringToInt(getAttribute("i_year"));
}

SKGError SKGBudgetObject::setMonth(int iMonth)
{
    return setAttribute("i_month", SKGServices::intToString(iMonth));
}

int SKGBudgetObject::getMonth() const
{
    return SKGServices::stringToInt(getAttribute("i_month"));
}

SKGError SKGBudgetObject::setCategory(const SKGCategoryObject& iCategory)
{
    return setAttribute("rc_category_id", SKGServices::intToString(iCategory.getID()));;
}

SKGError SKGBudgetObject::getCategory(SKGCategoryObject& oCategory) const
{
    return getDocument()->getObject("v_category", "id=" % getAttribute("rc_category_id"), oCategory);
}

SKGError SKGBudgetObject::enableSubCategoriesInclusion(bool iEnable)
{
    return setAttribute("t_including_subcategories", iEnable ? "Y" : "N");
}

int SKGBudgetObject::isSubCategoriesInclusionEnabled() const
{
    return (getAttribute("t_including_subcategories") == "Y");
}

SKGError SKGBudgetObject::removeCategory()
{
    return setAttribute("rc_category_id", "0");
}

SKGError SKGBudgetObject::process()
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    // Main values
    int m = getMonth();
    int y = getYear();
    double transferred = SKGServices::stringToDouble(getAttribute("f_transferred"));

    // Get budgets rules ordered
    SKGObjectBase::SKGListSKGObjectBase budgetsRules;
    QString sql = "(t_year_condition='N' OR i_year=" % SKGServices::intToString(y) % ") AND "
                  "(t_month_condition='N' OR i_month=" % SKGServices::intToString(m) % ") AND "
                  "(t_category_condition='N' OR rc_category_id=" % getAttribute("rc_category_id") % ") "
                  "ORDER BY t_absolute DESC, id";
    err = getDocument()->getObjects("v_budgetrule", sql, budgetsRules);

    int nb = budgetsRules.count();
    if (!err && nb) {
        err = getDocument()->beginTransaction("#INTERNAL#" % i18nc("Progression step", "Apply budget rules"), nb);
        for (int i = 0; !err && i < nb; ++i) {
            SKGBudgetRuleObject rule(budgetsRules.at(i));

            // Do we have something to do
            SKGBudgetRuleObject::Condition cond = rule.getCondition();
            double delta = getDelta();
            double quantity = rule.getQuantity();
            if (delta) {
                if (cond == SKGBudgetRuleObject::ALL || (delta < 0 && cond == SKGBudgetRuleObject::NEGATIVE) || (delta > 0 && cond == SKGBudgetRuleObject::POSITIVE)) {
                    // Compute value to transfer
                    double value = (rule.isAbolute() ? (cond == SKGBudgetRuleObject::NEGATIVE ? qMax(delta, quantity) : qMin(delta, quantity)) : quantity * (delta - transferred) / 100.0);

                    // Get impacted budget
                    SKGBudgetObject impactedBudget = *this;
                    impactedBudget.resetID();

                    SKGBudgetRuleObject::Mode mode = rule.getTransferMode();
                    if (mode == SKGBudgetRuleObject::NEXT) {
                        // Next
                        int mi = m;
                        int yi = y;
                        if (mi == 0) {
                            // Yearly budget
                            ++yi;
                        } else {
                            // Monthly budget
                            ++mi;
                            if (mi == 13) {
                                mi = 1;
                                ++yi;
                            }
                        }
                        IFOKDO(err, impactedBudget.setYear(yi))
                        IFOKDO(err, impactedBudget.setMonth(mi))
                    } else if (mode == SKGBudgetRuleObject::YEAR) {
                        // Year
                        IFOKDO(err, impactedBudget.setYear(y))
                        IFOKDO(err, impactedBudget.setMonth(0))
                    }

                    // Change category
                    if (!err && rule.isCategoryChangeEnabled()) {
                        SKGCategoryObject transferCategory;
                        rule.getTransferCategory(transferCategory);
                        err = impactedBudget.setCategory(transferCategory);
                    }

                    IFOK(err) {
                        if (impactedBudget.exist()) {
                            err = impactedBudget.load();
                            QString newBudget = SKGServices::doubleToString(impactedBudget.getBudgetedModifiedAmount() - value);
                            IFOKDO(err, impactedBudget.setAttribute("f_budgeted_modified", newBudget))
                            IFOKDO(err, impactedBudget.save())
                            SKGTRACEL(1) << "Transfer " << value << " from '" << getDisplayName() << "' to '" << impactedBudget.getDisplayName() << "' due to '" << rule.getDisplayName() << "'" << endl;

                            transferred += value;
                            IFOKDO(err, setAttribute("f_transferred", SKGServices::doubleToString(transferred)))
                            IFOKDO(err, save())
                        } else {
                            getDocument()->sendMessage(i18nc("", "Impossible to apply the rule '%1' for budget '%2' because the impacted budget does not exist", rule.getDisplayName(), this->getDisplayName()), SKGDocument::Warning);
                        }
                    }
                }
            }
            IFOKDO(err, getDocument()->stepForward(i + 1))
        }

        SKGENDTRANSACTION(getDocument(),  err);
    }

    return err;
}

SKGError SKGBudgetObject::createAutomaticBudget(SKGDocumentBank* iDocument, int iYear, int iBaseYear, bool iUseScheduledOperation, bool iRemovePreviousBudget)
{
    Q_UNUSED(iUseScheduledOperation);
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    QString baseYear = SKGServices::intToString(iBaseYear);
    int fistMonth = 0;
    if (iDocument) {
        SKGStringListList listTmp;
        err = iDocument->executeSelectSqliteOrder(
                  "SELECT MIN(STRFTIME('%m', d_date)) FROM operation WHERE i_group_id = 0 AND STRFTIME('%Y', d_date) = '" % baseYear %
                  "' AND t_template='N'",
                  listTmp);
        if (listTmp.count() == 2) {
            fistMonth = SKGServices::stringToInt(listTmp.at(1).at(0));
        }
    }
    if (!err && iDocument) {
        SKGStringListList listTmp;
        QString sql = "SELECT t_REALCATEGORY, COUNT(TOT),"
                      "100*COUNT(TOT)/((CASE WHEN STRFTIME('%Y', date('now'))<>'" % baseYear % "' THEN 12 ELSE STRFTIME('%m', date('now'))-1 END)-" %
                      SKGServices::intToString(fistMonth) % "+1) AS CPOUR,"
                      "ROUND(TOTAL(TOT)/COUNT(TOT)), MAX(MONTH), TOTAL(TOT) FROM ("
                      "SELECT t_REALCATEGORY, STRFTIME('%m', d_date) AS MONTH, TOTAL(f_REALCURRENTAMOUNT) AS TOT "
                      "FROM v_suboperation_consolidated WHERE i_group_id = 0 AND d_DATEYEAR = '" % baseYear % "' AND d_DATEMONTH<STRFTIME('%Y-%m', date('now')) "
                      "GROUP BY t_REALCATEGORY, d_DATEMONTH"
                      ") GROUP BY t_REALCATEGORY ORDER BY COUNT(TOT) DESC, (MAX(TOT)-MIN(TOT))/ABS(ROUND(TOTAL(TOT)/COUNT(TOT))) ASC, ROUND(TOTAL(TOT)/COUNT(TOT)) ASC";
        err = iDocument->executeSelectSqliteOrder(sql, listTmp);


        // SELECT r.d_date,r.i_period_increment,r.t_period_unit, r.i_nb_times, r. t_times, r.t_CATEGORY, r.f_CURRENTAMOUNT  FROM v_recurrentoperation_display r WHERE r.t_TRANSFER='N'

        /*double sumBudgeted = 0;
        double sumOps = 0;*/

        int nbval = listTmp.count();
        if (!err) {
            int step = 0;
            err = iDocument->beginTransaction("#INTERNAL#" % i18nc("Progression step", "Create automatic budget"), nbval - 1 + 1 + (iRemovePreviousBudget ? 1 : 0));

            // Remove previous budgets
            if (iRemovePreviousBudget) {
                IFOKDO(err, iDocument->executeSqliteOrder("DELETE FROM budget WHERE i_year=" % SKGServices::intToString(iYear)))
                ++step;
                IFOKDO(err, iDocument->stepForward(step))
            }

            // Create budgets
            for (int i = 1; !err && i < nbval; ++i) {  // Ignore header
                // Get values
                QString catName = listTmp.at(i).at(0);
                int count = SKGServices::stringToInt(listTmp.at(i).at(1));
                int countPercent = SKGServices::stringToInt(listTmp.at(i).at(2));
                double amount = SKGServices::stringToDouble(listTmp.at(i).at(3));
                int month = SKGServices::stringToInt(listTmp.at(i).at(4));
                // sumOps += SKGServices::stringToDouble(listTmp.at(i).at(5));

                if (!catName.isEmpty() && (countPercent > 85 || count == 1)) {
                    SKGCategoryObject cat;;
                    err = iDocument->getObject("v_category", "t_fullname = '" % SKGServices::stringToSqlString(catName) % '\'', cat);
                    for (int m = fistMonth; !err && m <= (count == 1 ? fistMonth : 12); ++m) {
                        SKGBudgetObject budget(iDocument);
                        err = budget.setBudgetedAmount(amount);
                        IFOKDO(err, budget.setYear(iYear))
                        IFOKDO(err, budget.setMonth(count == 1 ? month : m))
                        IFOKDO(err, budget.setCategory(cat))
                        IFOKDO(err, budget.save())

                        // sumBudgeted += amount;
                    }
                }
                ++step;
                IFOKDO(err, iDocument->stepForward(step))
            }

            // Analyze
            IFOKDO(err, iDocument->executeSqliteOrder("ANALYZE");)
            ++step;
            IFOKDO(err, iDocument->stepForward(step))

            SKGENDTRANSACTION(iDocument,  err);
        }
    }
    return err;
}

SKGError SKGBudgetObject::balanceBudget(SKGDocumentBank* iDocument, int iYear, int iMonth, bool iBalanceYear)
{
    SKGError err;
    SKGTRACEINFUNCRC(10, err);
    if (iDocument) {
        err = iDocument->beginTransaction("#INTERNAL#" % i18nc("Progression step", "Balance budgets"), 2);

        // Monthly balancing
        if (!err && iMonth != -1) {
            for (int m = (iMonth == 0 ? 1 : iMonth); !err && m <= (iMonth == 0 ? 12 : iMonth); ++m) {
                SKGStringListList listTmp;
                err = iDocument->executeSelectSqliteOrder("SELECT TOTAL(f_budgeted) FROM budget WHERE i_year=" % SKGServices::intToString(iYear) % " AND i_month=" % SKGServices::intToString(m) % " AND rc_category_id<>0", listTmp);
                if (!err && listTmp.count() == 2) {
                    SKGBudgetObject budget(iDocument);
                    double amount = -SKGServices::stringToDouble(listTmp.at(1).at(0));
                    err = budget.setBudgetedAmount(amount);
                    IFOKDO(err, budget.setYear(iYear))
                    IFOKDO(err, budget.setMonth(m))
                    IFOKDO(err, budget.save())
                }
            }
        }
        IFOKDO(err, iDocument->stepForward(1))

        // Annual balancing
        if (!err && iBalanceYear) {
            SKGStringListList listTmp;
            err = iDocument->executeSelectSqliteOrder("SELECT TOTAL(f_budgeted) FROM budget WHERE i_year=" % SKGServices::intToString(iYear) % " AND (i_month<>0 OR rc_category_id<>0)", listTmp);
            if (!err && listTmp.count() == 2) {
                SKGBudgetObject budget(iDocument);
                double amount = -SKGServices::stringToDouble(listTmp.at(1).at(0));
                err = budget.setBudgetedAmount(amount);
                IFOKDO(err, budget.setYear(iYear))
                IFOKDO(err, budget.setMonth(0))
                IFOKDO(err, budget.save())
            }
        }
        IFOKDO(err, iDocument->stepForward(2))

        SKGENDTRANSACTION(iDocument,  err);
    }
    return err;
}

#include "skgbudgetobject.moc"
