# compositions.py
#
# Copyright 2025 Hari Rana (TheEvilSkeleton)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later


import logging

from typing import Any, cast
from collections.abc import Callable

from gi.repository import Gio, Gtk, GObject


class RefineCompositions:
    """
    Class providing decorators and methods for common widget properties and behavior.
    """

    @staticmethod
    def value(func: Callable[[Any, str], None]) -> Callable[[Any, str], None]:
        """`@Refine.value` decorator implementing the `GObject:value` property."""

        @GObject.Property(type=str, default="")
        def value(self) -> str:
            return self._value

        @value.setter  # type: ignore [no-redef]
        def value(self, value) -> None:
            if not self.is_valid_setting:
                return

            func(self, value)

            self._value = value

        return cast(Callable[[Any, str], None], value)

    @staticmethod
    def key(func: Callable[[Any, str], None]) -> Callable[[Any, str], None]:
        """`@Refine.key` decorator implementing the `GObject:key` property."""

        @GObject.Property(
            type=str,
            default="",
            flags=GObject.ParamFlags.READWRITE | GObject.ParamFlags.EXPLICIT_NOTIFY,
        )
        def key(self) -> str:
            return self._key

        @key.setter  # type: ignore [no-redef]
        def key(self, key) -> None:
            if not self.is_valid_setting:
                return

            try:
                if not self.schema.has_key(key):
                    raise TypeError

                func(self, key)
            except Exception:
                self.is_valid_setting = False
                logging.warning(f"Key “{key}” does not exist. Skipping…")

            self._key = key
            self.notify("key")

        return cast(Callable[[Any, str], None], key)

    @staticmethod
    def schema_id(func: Callable[[Any, str], None]) -> Callable[[Any, str], None]:
        """`@Refine.schema_id` decorator implementing the `GObject:schema-id` property."""

        @GObject.Property(type=str, default="")
        def schema_id(self) -> str:
            return self._schema_id

        @schema_id.setter  # type: ignore [no-redef]
        def schema_id(self, schema_id: str) -> None:
            toplevels = Gtk.Window.get_toplevels()

            if not (window := toplevels.get_item(0)):
                return

            application = window.get_application()

            try:
                self.settings = application.settings[schema_id]["settings"]
                self.schema = application.settings[schema_id]["schema"]
                self.is_valid_setting = bool(self.settings)
                return
            except KeyError:
                pass

            try:
                for source in application.sources:
                    self.schema = source.lookup(schema_id, True)
                    if self.schema:
                        break

                if not self.schema:
                    raise TypeError
            except (AttributeError, TypeError):
                self.is_valid_setting = False
                application.settings[schema_id] = {"settings": None, "schema": None}
                logging.warning(f"Schema ID “{schema_id}” does not exist. Skipping…")
                return

            self.settings = Gio.Settings.new(schema_id)
            application.settings[schema_id] = {
                "settings": self.settings,
                "schema": self.schema,
            }

            self._schema_id = schema_id

        return cast(Callable[[Any, str], None], schema_id)

    @staticmethod
    def set_up_separator_revealer(
        revealer: Gtk.Revealer, widget: Gtk.Widget, widget_property: str
    ) -> None:
        """Initialize and set up the `Gtk.Revealer`."""

        def state_flags_changed(
            widget: Gtk.Widget, _flags: Gtk.StateFlags, revealer: Gtk.Revealer
        ) -> None:
            flags = widget.get_state_flags()

            is_prelight = flags & (Gtk.StateFlags.PRELIGHT)
            is_activated = flags & (Gtk.StateFlags.SELECTED | Gtk.StateFlags.CHECKED)
            is_focused = flags & Gtk.StateFlags.FOCUSED & Gtk.StateFlags.FOCUS_VISIBLE

            revealer.set_reveal_child(not (is_prelight or is_activated or is_focused))

        revealer.set_reveal_child(True)
        widget.connect("state-flags-changed", state_flags_changed, revealer)
        widget.bind_property(
            "visible", revealer, "visible", GObject.BindingFlags.SYNC_CREATE
        )
        widget.bind_property(
            "opacity", revealer, "opacity", GObject.BindingFlags.SYNC_CREATE
        )
