"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearSplitCookies = clearSplitCookies;
exports.getExtraAuthStorageValue = getExtraAuthStorageValue;
exports.setExtraAuthStorage = setExtraAuthStorage;
exports.splitValueIntoCookies = splitValueIntoCookies;
exports.unsplitCookiesIntoValue = unsplitCookiesIntoValue;

var _router = require("../../../../src/core/server/http/router");

var _compression = require("../utils/compression");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function getExtraAuthStorageValue(request, options) {
  let compressedContent = '';
  let content = '';

  if (options.additionalCookies > 0) {
    compressedContent = unsplitCookiesIntoValue(request, options.cookiePrefix, options.additionalCookies);
  }

  try {
    content = (0, _compression.inflateValue)(Buffer.from(compressedContent, 'base64')).toString();
  } catch (error) {
    throw error;
  }

  return content;
}
/**
 * Compress and split up the given value into multiple cookies
 * @param request
 * @param cookie
 * @param options
 */


function setExtraAuthStorage(request, content, options) {
  const compressedAuthorizationHeaderValue = (0, _compression.deflateValue)(content);
  const compressedContent = compressedAuthorizationHeaderValue.toString('base64');
  splitValueIntoCookies(request, options.cookiePrefix, compressedContent, options.additionalCookies, options.logger);
}

function splitValueIntoCookies(request, // @todo Should be OpenSearchDashboardsRequest, I believe?
cookiePrefix, value, additionalCookies, logger) {
  /**
   * Assume that Iron adds around 50%.
   * Remember that an empty cookie is around 30 bytes
   */
  const maxLengthPerCookie = Math.floor(_common.MAX_LENGTH_OF_COOKIE_BYTES / _common.ESTIMATED_IRON_COOKIE_OVERHEAD);
  const cookiesNeeded = value.length / maxLengthPerCookie; // Assume 1 bit per character since this value is encoded
  // If the amount of additional cookies aren't enough for our logic, we try to write the value anyway
  // TODO We could also consider throwing an error, since a failed cookie leads to weird redirects.
  // But throwing would probably also lead to a weird redirect, since we'd get the token from the IdP again and again

  let splitValueAt = maxLengthPerCookie;

  if (cookiesNeeded > additionalCookies) {
    splitValueAt = Math.ceil(value.length / additionalCookies);

    if (logger) {
      logger.warn('The payload may be too large for the cookies. To be safe, we would need ' + Math.ceil(cookiesNeeded) + ' cookies in total, but we only have ' + additionalCookies + '. This can be changed with opensearch_security.openid.extra_storage.additional_cookies.');
    }
  }

  const rawRequest = (0, _router.ensureRawRequest)(request);
  const values = [];

  for (let i = 1; i <= additionalCookies; i++) {
    values.push(value.substring((i - 1) * splitValueAt, i * splitValueAt));
  }

  values.forEach(async (cookieSplitValue, index) => {
    const cookieName = cookiePrefix + (index + 1);

    if (cookieSplitValue === '') {
      // Make sure we clean up cookies that are not needed for the given value
      rawRequest.cookieAuth.h.unstate(cookieName);
    } else {
      rawRequest.cookieAuth.h.state(cookieName, cookieSplitValue);
    }
  });
}

function unsplitCookiesIntoValue(request, cookiePrefix, additionalCookies) {
  const rawRequest = (0, _router.ensureRawRequest)(request);
  let fullCookieValue = '';

  for (let i = 1; i <= additionalCookies; i++) {
    const cookieName = cookiePrefix + i;

    if (rawRequest.state[cookieName]) {
      fullCookieValue = fullCookieValue + rawRequest.state[cookieName];
    }
  }

  return fullCookieValue;
}

function clearSplitCookies(request, options) {
  const rawRequest = (0, _router.ensureRawRequest)(request);

  for (let i = 1; i <= options.additionalCookies; i++) {
    const cookieName = options.cookiePrefix + i;

    if (rawRequest.state[cookieName]) {
      rawRequest.cookieAuth.h.unstate(cookieName);
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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