/*
 * Decompiled with CFR 0.152.
 */
package org.locationtech.jts.io.twkb;

import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Objects;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.CoordinateSequenceFactory;
import org.locationtech.jts.geom.CoordinateSequences;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.impl.PackedCoordinateSequenceFactory;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.twkb.TWKBHeader;
import org.locationtech.jts.io.twkb.Varint;

public class TWKBReader {
    private static final GeometryFactory DEFAULT_FACTORY = new GeometryFactory((CoordinateSequenceFactory)PackedCoordinateSequenceFactory.DOUBLE_FACTORY);
    private GeometryFactory geometryFactory;

    public TWKBReader() {
        this(DEFAULT_FACTORY);
    }

    public TWKBReader(GeometryFactory geometryFactory) {
        this.geometryFactory = geometryFactory;
    }

    public TWKBReader setGeometryFactory(GeometryFactory geometryFactory) {
        this.geometryFactory = geometryFactory;
        return this;
    }

    public Geometry read(byte[] bytes) throws ParseException {
        return this.read(new ByteArrayInputStream(bytes));
    }

    public Geometry read(InputStream in) throws ParseException {
        return this.read(new DataInputStream(in));
    }

    public Geometry read(DataInput in) throws ParseException {
        try {
            return TWKBReader.read(this.geometryFactory, in);
        }
        catch (IOException ex) {
            throw new ParseException("Unexpected IOException caught: " + ex.getMessage());
        }
    }

    private static Geometry read(GeometryFactory factory2, DataInput in) throws IOException {
        Objects.requireNonNull(factory2, "GeometryFactory is null");
        Objects.requireNonNull(in, "DataInput is null");
        TWKBHeader header = TWKBReader.readHeader(in);
        return TWKBReader.readGeometryBody(factory2, header, in);
    }

    private static TWKBHeader readHeader(DataInput in) throws IOException {
        Objects.requireNonNull(in);
        int typeAndPrecisionHeader = in.readByte() & 0xFF;
        int geometryTypeCode = typeAndPrecisionHeader & 0xF;
        TWKBHeader.GeometryType geometryType = TWKBHeader.GeometryType.valueOf(geometryTypeCode);
        int precision = Varint.zigzagDecode((typeAndPrecisionHeader & 0xF0) >> 4);
        int metadata_header = in.readByte() & 0xFF;
        boolean hasBBOX = (metadata_header & 1) > 0;
        boolean hasSize = (metadata_header & 2) > 0;
        boolean hasIdList = (metadata_header & 4) > 0;
        boolean hasExtendedPrecision = (metadata_header & 8) > 0;
        boolean isEmpty = (metadata_header & 0x10) > 0;
        boolean hasZ = false;
        boolean hasM = false;
        int zprecision = 0;
        int mprecision = 0;
        if (hasExtendedPrecision) {
            int extendedDimsHeader = in.readByte() & 0xFF;
            hasZ = (extendedDimsHeader & 1) > 0;
            hasM = (extendedDimsHeader & 2) > 0;
            zprecision = (extendedDimsHeader & 0x1C) >> 2;
            mprecision = (extendedDimsHeader & 0xE0) >> 5;
        }
        int geometryBodySize = -1;
        if (hasSize) {
            geometryBodySize = Varint.readUnsignedVarInt(in);
        }
        return new TWKBHeader().setGeometryType(geometryType).setXyPrecision(precision).setHasZ(hasZ).setZPrecision(zprecision).setHasM(hasM).setMPrecision(mprecision).setHasIdList(hasIdList).setEmpty(isEmpty).setHasSize(hasSize).setHasBBOX(hasBBOX).setGeometryBodySize(geometryBodySize);
    }

    private static Geometry readGeometryBody(GeometryFactory factory2, TWKBHeader header, DataInput in) throws IOException {
        TWKBHeader.GeometryType geometryType = header.geometryType();
        if (header.isEmpty()) {
            return geometryType.createEmpty(factory2);
        }
        if (header.hasBBOX()) {
            TWKBReader.skipBbox(header, in);
        }
        switch (geometryType) {
            case POINT: {
                return TWKBReader.readPoint(factory2, in, header);
            }
            case LINESTRING: {
                return TWKBReader.readLineString(factory2, in, header, new long[header.getDimensions()]);
            }
            case POLYGON: {
                return TWKBReader.readPolygon(factory2, in, header, new long[header.getDimensions()]);
            }
            case MULTIPOINT: {
                return TWKBReader.readMultiPoint(factory2, in, header);
            }
            case MULTILINESTRING: {
                return TWKBReader.readMultiLineString(factory2, in, header);
            }
            case MULTIPOLYGON: {
                return TWKBReader.readMultiPolygon(factory2, in, header);
            }
            case GEOMETRYCOLLECTION: {
                return TWKBReader.readGeometryCollection(factory2, in, header);
            }
        }
        throw new IllegalStateException();
    }

    private static Point readPoint(GeometryFactory factory2, DataInput in, TWKBHeader header) throws IOException {
        CoordinateSequence seq = TWKBReader.createCoordinateSequence(factory2, 1, header);
        int dimensions = header.getDimensions();
        for (int d = 0; d < dimensions; ++d) {
            long preciseOrdinate = Varint.readSignedVarLong(in);
            int precision = header.getPrecision(d);
            double ordinate = (double)preciseOrdinate / Math.pow(10.0, precision);
            seq.setOrdinate(0, d, ordinate);
        }
        return factory2.createPoint(seq);
    }

    private static LineString readLineString(GeometryFactory factory2, DataInput in, TWKBHeader header, long[] prev) throws IOException {
        CoordinateSequence coordinates = TWKBReader.readCoordinateSequence(factory2, in, header, prev);
        return factory2.createLineString(coordinates);
    }

    private static LinearRing readLinearRing(GeometryFactory factory2, DataInput in, TWKBHeader header, long[] prev) throws IOException {
        CoordinateSequence seq = TWKBReader.readCoordinateSequence(factory2, in, header, prev);
        if (!CoordinateSequences.isRing((CoordinateSequence)seq)) {
            seq = CoordinateSequences.ensureValidRing((CoordinateSequenceFactory)factory2.getCoordinateSequenceFactory(), (CoordinateSequence)seq);
        }
        return factory2.createLinearRing(seq);
    }

    private static Polygon readPolygon(GeometryFactory factory2, DataInput in, TWKBHeader header, long[] prev) throws IOException {
        int nrings = Varint.readUnsignedVarInt(in);
        if (nrings == 0) {
            return factory2.createPolygon();
        }
        LinearRing shell = TWKBReader.readLinearRing(factory2, in, header, prev);
        LinearRing[] holes = new LinearRing[nrings - 1];
        for (int h2 = 0; h2 < nrings - 1; ++h2) {
            holes[h2] = TWKBReader.readLinearRing(factory2, in, header, prev);
        }
        return factory2.createPolygon(shell, holes);
    }

    private static MultiPoint readMultiPoint(GeometryFactory factory2, DataInput in, TWKBHeader header) throws IOException {
        int nmembers = Varint.readUnsignedVarInt(in);
        if (header.hasIdList()) {
            TWKBReader.skipIdList(nmembers, in);
        }
        CoordinateSequence coordinates = TWKBReader.readCoordinateSequence(factory2, in, nmembers, header, new long[header.getDimensions()]);
        return factory2.createMultiPoint(coordinates);
    }

    private static MultiLineString readMultiLineString(GeometryFactory factory2, DataInput in, TWKBHeader header) throws IOException {
        int nmembers = Varint.readUnsignedVarInt(in);
        if (header.hasIdList()) {
            TWKBReader.skipIdList(nmembers, in);
        }
        LineString[] lineStrings = new LineString[nmembers];
        long[] prev = new long[header.getDimensions()];
        for (int mN = 0; mN < nmembers; ++mN) {
            lineStrings[mN] = TWKBReader.readLineString(factory2, in, header, prev);
        }
        return factory2.createMultiLineString(lineStrings);
    }

    private static Geometry readMultiPolygon(GeometryFactory factory2, DataInput in, TWKBHeader header) throws IOException {
        int nmembers = Varint.readUnsignedVarInt(in);
        if (header.hasIdList()) {
            TWKBReader.skipIdList(nmembers, in);
        }
        long[] prev = new long[header.getDimensions()];
        Polygon[] polygons = new Polygon[nmembers];
        for (int mN = 0; mN < nmembers; ++mN) {
            polygons[mN] = TWKBReader.readPolygon(factory2, in, header, prev);
        }
        return factory2.createMultiPolygon(polygons);
    }

    private static Geometry readGeometryCollection(GeometryFactory factory2, DataInput in, TWKBHeader header) throws IOException {
        int nmembers = Varint.readUnsignedVarInt(in);
        if (header.hasIdList()) {
            TWKBReader.skipIdList(nmembers, in);
        }
        Geometry[] geometries = new Geometry[nmembers];
        for (int geomN = 0; geomN < nmembers; ++geomN) {
            geometries[geomN] = TWKBReader.read(factory2, in);
        }
        return factory2.createGeometryCollection(geometries);
    }

    private static void skipIdList(int nmembers, DataInput in) throws IOException {
        TWKBReader.readIdList(nmembers, null, in);
    }

    private static void readIdList(int nmembers, long[] target, DataInput in) throws IOException {
        for (int i = 0; i < nmembers; ++i) {
            long id = Varint.readUnsignedVarLong(in);
            if (target == null) continue;
            target[i] = id;
        }
    }

    private static void skipBbox(TWKBHeader header, DataInput in) throws IOException {
        int dimensions = header.getDimensions();
        for (int coord = 0; coord < dimensions; ++coord) {
            Varint.readSignedVarLong(in);
            Varint.readSignedVarLong(in);
        }
    }

    private static CoordinateSequence readCoordinateSequence(GeometryFactory factory2, DataInput in, TWKBHeader header, long[] prev) throws IOException {
        int size = Varint.readUnsignedVarInt(in);
        return TWKBReader.readCoordinateSequence(factory2, in, size, header, prev);
    }

    private static CoordinateSequence readCoordinateSequence(GeometryFactory factory2, DataInput in, int size, TWKBHeader header, long[] prev) throws IOException {
        CoordinateSequence sequence2 = TWKBReader.createCoordinateSequence(factory2, size, header);
        int dimensions = header.getDimensions();
        for (int coordIndex = 0; coordIndex < size; ++coordIndex) {
            for (int ordinateIndex = 0; ordinateIndex < dimensions; ++ordinateIndex) {
                int precision = header.getPrecision(ordinateIndex);
                long prevValue = prev[ordinateIndex];
                long delta = Varint.readSignedVarLong(in);
                long preciseOrdinate = delta + prevValue;
                double ordinate = (double)preciseOrdinate / Math.pow(10.0, precision);
                prev[ordinateIndex] = preciseOrdinate;
                sequence2.setOrdinate(coordIndex, ordinateIndex, ordinate);
            }
        }
        return sequence2;
    }

    private static CoordinateSequence createCoordinateSequence(GeometryFactory factory2, int size, TWKBHeader header) {
        int dim = header.getDimensions();
        int measures = header.hasM() ? 1 : 0;
        CoordinateSequence sequence2 = factory2.getCoordinateSequenceFactory().create(size, dim, measures);
        if (sequence2.getDimension() != dim) {
            throw new IllegalStateException("Provided CoordinateSequenceFactory does not support the required dimension. Requested " + header + ", returned " + sequence2.getDimension());
        }
        if (measures != sequence2.getMeasures()) {
            throw new IllegalStateException("CoordinateSequenceFactory error: requested " + measures + " measures, returned " + sequence2.getMeasures());
        }
        return sequence2;
    }
}

