// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.apache.doris.nereids.trees.expressions.functions.scalar;

import org.apache.doris.catalog.FunctionSignature;
import org.apache.doris.nereids.exceptions.AnalysisException;
import org.apache.doris.nereids.trees.expressions.Expression;
import org.apache.doris.nereids.trees.expressions.functions.AlwaysNullable;
import org.apache.doris.nereids.trees.expressions.visitor.ExpressionVisitor;
import org.apache.doris.nereids.types.ArrayType;
import org.apache.doris.nereids.types.BooleanType;

import com.google.common.collect.ImmutableList;

import java.util.List;

/**
 * ScalarFunction 'array_match_all'.
 */
public class ArrayMatchAll extends ScalarFunction
        implements HighOrderFunction, AlwaysNullable {

    public static final List<FunctionSignature> SIGNATURES = ImmutableList.of(
            FunctionSignature.ret(BooleanType.INSTANCE).args(ArrayType.of(BooleanType.INSTANCE))
    );

    /**
     * constructor with arguments.
     * array_match_all(lambda, a1, ...) = array_match(a1, array_map(lambda, a1, ...))
     */
    public ArrayMatchAll(Expression arg) {
        super("array_match_all", arg instanceof Lambda ? new ArrayMap(arg) : arg);
    }

    /** constructor for withChildren and reuse signature */
    private ArrayMatchAll(ScalarFunctionParams functionParams) {
        super(functionParams);
    }

    @Override
    public ArrayMatchAll withChildren(List<Expression> children) {
        if (children.size() != 1) {
            throw new AnalysisException(
                    String.format("The number of args of %s must be 1 but is %d", getName(), children.size()));
        }
        return new ArrayMatchAll(getFunctionParams(children));
    }

    @Override
    public <R, C> R accept(ExpressionVisitor<R, C> visitor, C context) {
        return visitor.visitArrayMatchAll(this, context);
    }

    @Override
    public List<FunctionSignature> getImplSignature() {
        return SIGNATURES;
    }
}
