/*
 * Decompiled with CFR 0.152.
 */
package org.apache.geode.internal.cache;

import java.io.File;
import java.text.DecimalFormat;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.apache.geode.cache.DiskAccessException;
import org.apache.geode.internal.cache.DirectoryHolder;
import org.apache.geode.internal.cache.DiskStoreImpl;
import org.apache.geode.internal.logging.log4j.LogMarker;
import org.apache.geode.logging.internal.executors.LoggingExecutors;
import org.apache.geode.logging.internal.log4j.api.LogService;
import org.apache.logging.log4j.Logger;

public class DiskStoreMonitor {
    private static final Logger logger = LogService.getLogger();
    private static final int USAGE_CHECK_INTERVAL = Integer.getInteger("gemfire.DISK_USAGE_POLLING_INTERVAL_MILLIS", 10000);
    private static final float LOG_WARNING_THRESHOLD_PCT = Integer.getInteger("gemfire.DISK_USAGE_LOG_WARNING_PERCENT", 99).intValue();
    static final String DISK_USAGE_DISABLE_MONITORING = "gemfire.DISK_USAGE_DISABLE_MONITORING";
    private final boolean disableMonitor = Boolean.getBoolean("gemfire.DISK_USAGE_DISABLE_MONITORING");
    private final ScheduledExecutorService exec;
    private final Map<DiskStoreImpl, Set<DirectoryHolderUsage>> disks = new ConcurrentHashMap<DiskStoreImpl, Set<DirectoryHolderUsage>>();
    private final LogUsage logDisk;
    volatile DiskStateAction _testAction;

    public static void checkWarning(float val) {
        if (val < 0.0f || val > 100.0f) {
            throw new IllegalArgumentException(String.format("Disk usage warning percentage must be set to a value between 0-100.  The value %s is invalid.", Float.valueOf(val)));
        }
    }

    public static void checkCritical(float val) {
        if (val < 0.0f || val > 100.0f) {
            throw new IllegalArgumentException(String.format("Disk usage critical percentage must be set to a value between 0-100.  The value %s is invalid.", Float.valueOf(val)));
        }
    }

    public DiskStoreMonitor(File logFile) {
        this.logDisk = new LogUsage(DiskStoreMonitor.getLogDir(logFile));
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Disk monitoring is {}", (Object)(this.disableMonitor ? "disabled" : "enabled"));
            logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Log directory usage warning is set to {}%", (Object)Float.valueOf(LOG_WARNING_THRESHOLD_PCT));
            logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Scheduling disk usage checks every {} ms", (Object)USAGE_CHECK_INTERVAL);
        }
        if (this.disableMonitor) {
            this.exec = null;
        } else {
            this.exec = LoggingExecutors.newScheduledThreadPool((int)1, (String)"DiskStoreMonitor");
            this.exec.scheduleWithFixedDelay(() -> {
                try {
                    this.checkUsage();
                }
                catch (Exception e) {
                    logger.error("The DiskStore Monitor has encountered an error", (Throwable)e);
                }
            }, 0L, USAGE_CHECK_INTERVAL, TimeUnit.MILLISECONDS);
        }
    }

    LogUsage getLogDisk() {
        return this.logDisk;
    }

    public void addDiskStore(DiskStoreImpl ds) {
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Now monitoring disk store {}", (Object)ds.getName());
        }
        HashSet<DirectoryHolderUsage> du = new HashSet<DirectoryHolderUsage>();
        for (DirectoryHolder dir : ds.getDirectoryHolders()) {
            du.add(new DirectoryHolderUsage(ds, dir));
        }
        this.disks.put(ds, du);
    }

    public void removeDiskStore(DiskStoreImpl ds) {
        if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
            logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "No longer monitoring disk store {}", (Object)ds.getName());
        }
        this.disks.remove(ds);
    }

    public boolean isNormal(DiskStoreImpl ds, DirectoryHolder dir) {
        Set<DirectoryHolderUsage> dirs = this.disks.get(ds);
        if (dirs != null) {
            for (DirectoryHolderUsage du : dirs) {
                if (du.dir != dir) continue;
                return du.getState() == DiskState.NORMAL;
            }
        }
        return true;
    }

    public void close() {
        if (this.exec != null) {
            this.exec.shutdownNow();
        }
        this.disks.clear();
    }

    private void checkUsage() {
        for (Map.Entry<DiskStoreImpl, Set<DirectoryHolderUsage>> entry : this.disks.entrySet()) {
            DiskUsage du;
            DiskState update;
            DiskStoreImpl ds = entry.getKey();
            Iterator<DirectoryHolderUsage> iterator = entry.getValue().iterator();
            while (iterator.hasNext() && (update = (du = (DiskUsage)iterator.next()).update(ds.getDiskUsageWarningPercentage(), ds.getDiskUsageCriticalPercentage())) != DiskState.CRITICAL) {
            }
        }
        this.logDisk.update(LOG_WARNING_THRESHOLD_PCT, 100.0f);
    }

    private static File getLogDir(File logFile) {
        File logDir = null;
        if (logFile != null) {
            logDir = logFile.getParentFile();
        }
        if (logDir == null) {
            logDir = new File(".");
        }
        return logDir;
    }

    class DirectoryHolderUsage
    extends DiskUsage {
        private final DiskStoreImpl disk;
        private final DirectoryHolder dir;

        public DirectoryHolderUsage(DiskStoreImpl disk, DirectoryHolder dir) {
            this.disk = disk;
            this.dir = dir;
        }

        @Override
        protected void handleStateChange(DiskState next, String pct, String criticalMessage) {
            if (DiskStoreMonitor.this._testAction != null) {
                logger.info(LogMarker.DISK_STORE_MONITOR_MARKER, "Invoking test handler for state change to {}", (Object)next);
                DiskStoreMonitor.this._testAction.handleDiskStateChange(next);
            }
            Object[] args = new Object[]{this.dir.getDir(), this.disk.getName(), pct};
            switch (next) {
                case NORMAL: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR_MARKER, "The disk volume {} for disk store {} has returned to normal usage levels and is {} full", args);
                    break;
                }
                case WARN: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR_MARKER, "The disk volume {} for disk store {} has exceeded the warning usage threshold and is {} full", args);
                    break;
                }
                case CRITICAL: {
                    logger.error(LogMarker.DISK_STORE_MONITOR_MARKER, "The disk volume {} for disk store {} has exceeded the critical usage threshold and is {} full", args);
                    String msg = "Critical disk usage threshold exceeded for volume " + this.dir.getDir().getAbsolutePath() + ": " + criticalMessage;
                    this.disk.handleDiskAccessException(new DiskAccessException(msg, this.disk));
                }
            }
        }

        @Override
        protected File dir() {
            return this.dir.getDir();
        }

        @Override
        protected long getMinimumSpace() {
            return (long)DiskStoreImpl.MIN_DISK_SPACE_FOR_LOGS + this.disk.getMaxOplogSize();
        }

        @Override
        protected void recordStats(long total, long free, long elapsed) {
            this.dir.getDiskDirectoryStats().addVolumeCheck(total, free, elapsed);
        }
    }

    static class LogUsage
    extends DiskUsage {
        private final File dir;

        public LogUsage(File dir) {
            this.dir = dir;
        }

        @Override
        protected void handleStateChange(DiskState next, String pct, String critcalMessage) {
            Object[] args = new Object[]{this.dir.getAbsolutePath(), pct};
            switch (next) {
                case NORMAL: {
                    logger.info(LogMarker.DISK_STORE_MONITOR_MARKER, "The disk volume {} for log files has returned to normal usage levels and is {} full.", args);
                    break;
                }
                case WARN: 
                case CRITICAL: {
                    logger.warn(LogMarker.DISK_STORE_MONITOR_MARKER, "The disk volume {} for log files has exceeded the warning usage threshold and is {} full.", args);
                }
            }
        }

        @Override
        protected long getMinimumSpace() {
            return DiskStoreImpl.MIN_DISK_SPACE_FOR_LOGS;
        }

        @Override
        protected File dir() {
            return this.dir;
        }

        @Override
        protected void recordStats(long total, long free, long elapsed) {
        }
    }

    static abstract class DiskUsage {
        private DiskState state = DiskState.NORMAL;

        DiskUsage() {
        }

        public synchronized DiskState getState() {
            return this.state;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public DiskState update(float warning, float critical) {
            boolean belowMin;
            DiskState next;
            DiskState current;
            DiskUsage diskUsage = this;
            synchronized (diskUsage) {
                current = this.state;
            }
            if (!(warning > 0.0f) && !(critical > 0.0f)) {
                return current;
            }
            if (!this.dir().exists()) {
                if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
                    logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Skipping check of non-existent directory {}", (Object)this.dir().getAbsolutePath());
                }
                return current;
            }
            long minMegabytes = this.getMinimumSpace();
            long minBytes = minMegabytes * 1024L * 1024L;
            if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
                logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Checking usage for directory {}, minimum free space is {} MB", (Object)this.dir().getAbsolutePath(), (Object)minMegabytes);
            }
            long start = System.nanoTime();
            long remaining = this.dir().getUsableSpace();
            long total = this.dir().getTotalSpace();
            long elapsed = System.nanoTime() - start;
            double use = 100.0 * (double)(total - remaining) / (double)total;
            this.recordStats(total, remaining, elapsed);
            DecimalFormat decimalFormat = new DecimalFormat("#.#");
            String pct = decimalFormat.format(use) + "%";
            if (logger.isTraceEnabled(LogMarker.DISK_STORE_MONITOR_VERBOSE)) {
                logger.trace(LogMarker.DISK_STORE_MONITOR_VERBOSE, "Directory {} has {} bytes free out of {} ({} usage)", (Object)this.dir().getAbsolutePath(), (Object)remaining, (Object)total, (Object)pct);
            }
            if ((next = DiskState.select(use, warning, critical, belowMin = remaining < minBytes)) == current) {
                return next;
            }
            DiskUsage diskUsage2 = this;
            synchronized (diskUsage2) {
                this.state = next;
            }
            String criticalMessage = null;
            if (next == DiskState.CRITICAL) {
                criticalMessage = belowMin ? "the file system only has " + remaining + " bytes free which is below the minimum of " + minBytes + "." : "the file system is " + pct + " full, which reached the critical threshold of " + decimalFormat.format(critical) + "%.";
            }
            this.handleStateChange(next, pct, criticalMessage);
            return next;
        }

        protected abstract File dir();

        protected abstract long getMinimumSpace();

        protected abstract void recordStats(long var1, long var3, long var5);

        protected abstract void handleStateChange(DiskState var1, String var2, String var3);
    }

    static interface DiskStateAction {
        public void handleDiskStateChange(DiskState var1);
    }

    static enum DiskState {
        NORMAL,
        WARN,
        CRITICAL;


        public static DiskState select(double actual, double warn, double critical, boolean belowMinimum) {
            if (critical > 0.0 && (actual > critical || belowMinimum)) {
                return CRITICAL;
            }
            if (warn > 0.0 && actual > warn) {
                return WARN;
            }
            return NORMAL;
        }
    }
}

