/****************************************************************************
 * libs/libc/pthread/pthread_barriedestroy.c
 *
 *   Copyright (C) 2007, 2009, 2017 Gregory Nutt. All rights reserved.
 *   Author: Gregory Nutt <gnutt@nuttx.org>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 3. Neither the name NuttX nor the names of its contributors may be
 *    used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <pthread.h>
#include <semaphore.h>
#include <errno.h>
#include <debug.h>

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Name: pthread_barrier_destroy
 *
 * Description:
 *   The pthread_barrier_destroy() function destroys the barrier referenced
 *   by 'barrier' and releases any resources used by the barrier. The effect
 *   of subsequent use of the barrier is undefined until the barrier is
 *   reinitialized by another call to pthread_barrier_init(). The result
 *   are undefined if pthread_barrier_destroy() is called when any thread is
 *   blocked on the barrier, or if this function is called with an
 *   uninitialized barrier.
 *
 * Input Parameters:
 *   barrier - barrier to be destroyed.
 *
 * Returned Value:
 *   0 (OK) on success or on of the following error numbers:
 *
 *   EBUSY  The implementation has detected an attempt to destroy a barrier
 *          while it is in use.
 *   EINVAL The value specified by barrier is invalid.
 *
 * Assumptions:
 *
 ****************************************************************************/

int pthread_barrier_destroy(FAR pthread_barrier_t *barrier)
{
  int ret = OK;

  if (!barrier)
    {
      ret = EINVAL;
    }
  else
    {
      sem_destroy(&barrier->sem);
      barrier->count = 0;
    }

  return ret;
}
