/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.io;

import com.google.common.collect.Iterators;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.lucene.util.IOUtils;
import org.elasticsearch.common.logging.ESLogger;

public final class FileSystemUtils {
    private FileSystemUtils() {
    }

    public static boolean hasExtensions(Path root, final String ... extensions) throws IOException {
        final AtomicBoolean retVal = new AtomicBoolean(false);
        Files.walkFileTree(root, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                for (String extension : extensions) {
                    if (!file.getFileName().toString().endsWith(extension)) continue;
                    retVal.set(true);
                    return FileVisitResult.TERMINATE;
                }
                return super.visitFile(file, attrs);
            }
        });
        return retVal.get();
    }

    public static boolean exists(Path ... files) {
        for (Path file : files) {
            if (!Files.exists(file, new LinkOption[0])) continue;
            return true;
        }
        return false;
    }

    public static boolean isHidden(Path path) {
        Path fileName = path.getFileName();
        if (fileName == null) {
            return false;
        }
        return fileName.toString().startsWith(".");
    }

    public static Path append(Path base, Path path, int strip) {
        for (Path subPath : path) {
            if (strip-- > 0) continue;
            base = base.resolve(subPath.toString());
        }
        return base;
    }

    public static void deleteSubDirectories(Path ... paths) throws IOException {
        for (Path path : paths) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(path);){
                for (Path subPath : stream) {
                    if (!Files.isDirectory(subPath, new LinkOption[0])) continue;
                    IOUtils.rm(subPath);
                }
            }
        }
    }

    public static boolean isAccessibleDirectory(Path directory, ESLogger logger) {
        assert (directory != null && logger != null);
        if (!Files.exists(directory, new LinkOption[0])) {
            logger.debug("[{}] directory does not exist.", directory.toAbsolutePath());
            return false;
        }
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            logger.debug("[{}] should be a directory but is not.", directory.toAbsolutePath());
            return false;
        }
        if (!Files.isReadable(directory)) {
            logger.debug("[{}] directory is not readable.", directory.toAbsolutePath());
            return false;
        }
        return true;
    }

    public static BufferedReader newBufferedReader(URL url, Charset cs) throws IOException {
        CharsetDecoder decoder = cs.newDecoder();
        InputStreamReader reader = new InputStreamReader(url.openStream(), decoder);
        return new BufferedReader(reader);
    }

    public static void moveFilesWithoutOverwriting(Path source, final Path destination, final String suffix) throws IOException {
        Files.createDirectories(destination, new FileAttribute[0]);
        final int configPathRootLevel = source.getNameCount();
        Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            private Path buildPath(Path path) {
                return destination.resolve(path);
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                Path subpath;
                Path path;
                if (configPathRootLevel != dir.getNameCount() && !Files.exists(path = this.buildPath(subpath = dir.subpath(configPathRootLevel, dir.getNameCount())), new LinkOption[0])) {
                    FileSystemUtils.move(dir, path);
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Path path;
                Path subpath = null;
                if (configPathRootLevel != file.getNameCount()) {
                    subpath = file.subpath(configPathRootLevel, file.getNameCount());
                }
                if (!Files.exists(path = this.buildPath(subpath), new LinkOption[0])) {
                    FileSystemUtils.move(file, path);
                } else if (suffix != null && !this.isSameFile(file, path)) {
                    path = path.resolveSibling(path.getFileName().toString().concat(suffix));
                    Files.move(file, path, StandardCopyOption.REPLACE_EXISTING);
                }
                return FileVisitResult.CONTINUE;
            }

            private boolean isSameFile(Path first, Path second) throws IOException {
                boolean sameFileSize;
                boolean bl = sameFileSize = Files.size(first) == Files.size(second);
                if (!sameFileSize) {
                    return false;
                }
                byte[] firstBytes = Files.readAllBytes(first);
                byte[] secondBytes = Files.readAllBytes(second);
                return Arrays.equals(firstBytes, secondBytes);
            }
        });
    }

    public static void copyDirectoryRecursively(Path source, Path destination) throws IOException {
        Files.walkFileTree(source, new TreeCopier(source, destination, false));
    }

    public static void move(Path source, Path destination) throws IOException {
        try {
            Files.move(source, destination, new CopyOption[0]);
        }
        catch (DirectoryNotEmptyException e) {
            Files.walkFileTree(source, new TreeCopier(source, destination, true));
        }
    }

    public static Path[] files(Path from, DirectoryStream.Filter<Path> filter) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(from, filter);){
            Path[] pathArray = Iterators.toArray(stream.iterator(), Path.class);
            return pathArray;
        }
    }

    public static Path[] files(Path directory) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(directory);){
            Path[] pathArray = Iterators.toArray(stream.iterator(), Path.class);
            return pathArray;
        }
    }

    public static Path[] files(Path directory, String glob) throws IOException {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(directory, glob);){
            Path[] pathArray = Iterators.toArray(stream.iterator(), Path.class);
            return pathArray;
        }
    }

    static class TreeCopier
    extends SimpleFileVisitor<Path> {
        private final Path source;
        private final Path target;
        private final boolean delete;

        TreeCopier(Path source, Path target, boolean delete) {
            this.source = source;
            this.target = target;
            this.delete = delete;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
            Path newDir = this.target.resolve(this.source.relativize(dir));
            try {
                Files.copy(dir, newDir, new CopyOption[0]);
            }
            catch (FileAlreadyExistsException fileAlreadyExistsException) {
            }
            catch (IOException x) {
                return FileVisitResult.SKIP_SUBTREE;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            if (this.delete) {
                IOUtils.rm(dir);
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Path newFile = this.target.resolve(this.source.relativize(file));
            try {
                Files.copy(file, newFile, new CopyOption[0]);
                if (this.delete) {
                    Files.deleteIfExists(file);
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

