﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/s3-crt/S3Crt_EXPORTS.h>
#include <aws/s3-crt/model/BlockedEncryptionTypes.h>
#include <aws/s3-crt/model/ServerSideEncryptionByDefault.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace S3Crt {
namespace Model {

/**
 * <p>Specifies the default server-side encryption configuration.</p>  <ul>
 * <li> <p> <b>General purpose buckets</b> - If you're specifying a customer
 * managed KMS key, we recommend using a fully qualified KMS key ARN. If you use a
 * KMS key alias instead, then KMS resolves the key within the requester’s account.
 * This behavior can result in data that's encrypted with a KMS key that belongs to
 * the requester, and not the bucket owner.</p> </li> <li> <p> <b>Directory
 * buckets</b> - When you specify an <a
 * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#customer-cmk">KMS
 * customer managed key</a> for encryption in your directory bucket, only use the
 * key ID or key ARN. The key alias format of the KMS key isn't supported.</p>
 * </li> </ul> <p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/s3-2006-03-01/ServerSideEncryptionRule">AWS
 * API Reference</a></p>
 */
class ServerSideEncryptionRule {
 public:
  AWS_S3CRT_API ServerSideEncryptionRule() = default;
  AWS_S3CRT_API ServerSideEncryptionRule(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_S3CRT_API ServerSideEncryptionRule& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_S3CRT_API void AddToNode(Aws::Utils::Xml::XmlNode& parentNode) const;

  ///@{
  /**
   * <p>Specifies the default server-side encryption to apply to new objects in the
   * bucket. If a PUT Object request doesn't specify any server-side encryption, this
   * default encryption will be applied.</p>
   */
  inline const ServerSideEncryptionByDefault& GetApplyServerSideEncryptionByDefault() const { return m_applyServerSideEncryptionByDefault; }
  inline bool ApplyServerSideEncryptionByDefaultHasBeenSet() const { return m_applyServerSideEncryptionByDefaultHasBeenSet; }
  template <typename ApplyServerSideEncryptionByDefaultT = ServerSideEncryptionByDefault>
  void SetApplyServerSideEncryptionByDefault(ApplyServerSideEncryptionByDefaultT&& value) {
    m_applyServerSideEncryptionByDefaultHasBeenSet = true;
    m_applyServerSideEncryptionByDefault = std::forward<ApplyServerSideEncryptionByDefaultT>(value);
  }
  template <typename ApplyServerSideEncryptionByDefaultT = ServerSideEncryptionByDefault>
  ServerSideEncryptionRule& WithApplyServerSideEncryptionByDefault(ApplyServerSideEncryptionByDefaultT&& value) {
    SetApplyServerSideEncryptionByDefault(std::forward<ApplyServerSideEncryptionByDefaultT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether Amazon S3 should use an S3 Bucket Key with server-side
   * encryption using KMS (SSE-KMS) for new objects in the bucket. Existing objects
   * are not affected. Setting the <code>BucketKeyEnabled</code> element to
   * <code>true</code> causes Amazon S3 to use an S3 Bucket Key. </p>  <ul>
   * <li> <p> <b>General purpose buckets</b> - By default, S3 Bucket Key is not
   * enabled. For more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/bucket-key.html">Amazon S3
   * Bucket Keys</a> in the <i>Amazon S3 User Guide</i>.</p> </li> <li> <p>
   * <b>Directory buckets</b> - S3 Bucket Keys are always enabled for
   * <code>GET</code> and <code>PUT</code> operations in a directory bucket and can’t
   * be disabled. S3 Bucket Keys aren't supported, when you copy SSE-KMS encrypted
   * objects from general purpose buckets to directory buckets, from directory
   * buckets to general purpose buckets, or between directory buckets, through <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html">CopyObject</a>,
   * <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/API/API_UploadPartCopy.html">UploadPartCopy</a>,
   * <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/directory-buckets-objects-Batch-Ops">the
   * Copy operation in Batch Operations</a>, or <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/create-import-job">the
   * import jobs</a>. In this case, Amazon S3 makes a call to KMS every time a copy
   * request is made for a KMS-encrypted object.</p> </li> </ul>
   */
  inline bool GetBucketKeyEnabled() const { return m_bucketKeyEnabled; }
  inline bool BucketKeyEnabledHasBeenSet() const { return m_bucketKeyEnabledHasBeenSet; }
  inline void SetBucketKeyEnabled(bool value) {
    m_bucketKeyEnabledHasBeenSet = true;
    m_bucketKeyEnabled = value;
  }
  inline ServerSideEncryptionRule& WithBucketKeyEnabled(bool value) {
    SetBucketKeyEnabled(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A bucket-level setting for Amazon S3 general purpose buckets used to prevent
   * the upload of new objects encrypted with the specified server-side encryption
   * type. For example, blocking an encryption type will block
   * <code>PutObject</code>, <code>CopyObject</code>, <code>PostObject</code>,
   * multipart upload, and replication requests to the bucket for objects with the
   * specified encryption type. However, you can continue to read and list any
   * pre-existing objects already encrypted with the specified encryption type. For
   * more information, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/blocking-unblocking-s3-c-encryption-gpb.html">Blocking
   * or unblocking SSE-C for a general purpose bucket</a>.</p>  <p>Currently,
   * this parameter only supports blocking or unblocking server-side encryption with
   * customer-provided keys (SSE-C). For more information about SSE-C, see <a
   * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/ServerSideEncryptionCustomerKeys.html">Using
   * server-side encryption with customer-provided keys (SSE-C)</a>.</p>
   */
  inline const BlockedEncryptionTypes& GetBlockedEncryptionTypes() const { return m_blockedEncryptionTypes; }
  inline bool BlockedEncryptionTypesHasBeenSet() const { return m_blockedEncryptionTypesHasBeenSet; }
  template <typename BlockedEncryptionTypesT = BlockedEncryptionTypes>
  void SetBlockedEncryptionTypes(BlockedEncryptionTypesT&& value) {
    m_blockedEncryptionTypesHasBeenSet = true;
    m_blockedEncryptionTypes = std::forward<BlockedEncryptionTypesT>(value);
  }
  template <typename BlockedEncryptionTypesT = BlockedEncryptionTypes>
  ServerSideEncryptionRule& WithBlockedEncryptionTypes(BlockedEncryptionTypesT&& value) {
    SetBlockedEncryptionTypes(std::forward<BlockedEncryptionTypesT>(value));
    return *this;
  }
  ///@}
 private:
  ServerSideEncryptionByDefault m_applyServerSideEncryptionByDefault;
  bool m_applyServerSideEncryptionByDefaultHasBeenSet = false;

  bool m_bucketKeyEnabled{false};
  bool m_bucketKeyEnabledHasBeenSet = false;

  BlockedEncryptionTypes m_blockedEncryptionTypes;
  bool m_blockedEncryptionTypesHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
