//// (c) 1992-2024 Intel Corporation.                            
// Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack words    
// and logos are trademarks of Intel Corporation or its subsidiaries in the U.S.  
// and/or other countries. Other marks and brands may be claimed as the property  
// of others. See Trademarks on intel.com for full list of Intel trademarks or    
// the Trademarks & Brands Names Database (if Intel) or See www.Intel.com/legal (if Altera) 
// Your use of Intel Corporation's design tools, logic functions and other        
// software and tools, and its AMPP partner logic functions, and any output       
// files any of the foregoing (including device programming or simulation         
// files), and any associated documentation or information are expressly subject  
// to the terms and conditions of the Altera Program License Subscription         
// Agreement, Intel MegaCore Function License Agreement, or other applicable      
// license agreement, including, without limitation, that your use is for the     
// sole purpose of programming logic devices manufactured by Intel and sold by    
// Intel or its authorized distributors.  Please refer to the applicable          
// agreement for further details.                                                 


// Each time this block receives an input valid, it generates a parameterized
// number of output valids, with the output valids spaced by a parameterized
// number of clock cycles.
//
// External logic must ensure that no new input valid is received until all
// output valids have been generated.  Additional input valids will produce
// undefined behaviour.

`default_nettype none

module hld_valid_generator_with_ii #(
    parameter  int NUM_OUTPUT_VALIDS,       // number of output valids to generate from each input valid
    parameter  int II,                      // Initialtion interval, gap between output valids in clock cycles
    parameter  bit ZERO_LATENCY,            // set to 1 to generate o_valid on same clock cycle as i_valid
    parameter  bit ASYNC_RESET,
    parameter  bit SYNCHRONIZE_RESET
) (
    input  wire     clk,
    input  wire     i_resetn,

    input  wire     i_valid,                // a 1 here causes the block to start generating output valids
    output logic    o_valid,                // output valids, generated as described above
    output logic    o_first_valid           // asserted coincident with the first output valid sent for a new input valid
);
    ///////////////////////////////////////
    // Parameter checking
    //
    // Generate an error if any illegal parameter settings or combinations are used
    ///////////////////////////////////////
    initial /* synthesis enable_verilog_initial_construct */
    begin
        if ( NUM_OUTPUT_VALIDS < 2 )
            $fatal(1, "Illegal parameteriazation, NUM_OUTPUT_VALIDS must be > 1");
        if ( II < 1 )
            $fatal(1, "Illegal parameteriazation, II must be >= 1");
    end

    //derived parameters
    parameter int VALID_COUNT_WIDTH = $clog2(NUM_OUTPUT_VALIDS) + 1;    // extra bit so we can count down to -1
    parameter int II_COUNT_WIDTH = $clog2(II) + 1;


    /////////////
    //  Reset 
    logic aclrn, sclrn, resetn_synchronized;
    acl_reset_handler #(
        .ASYNC_RESET            (ASYNC_RESET),
        .USE_SYNCHRONIZER       (SYNCHRONIZE_RESET),
        .SYNCHRONIZE_ACLRN      (SYNCHRONIZE_RESET),
        .PIPE_DEPTH             (3),
        .NUM_COPIES             (1) 
    ) acl_reset_handler_inst (
        .clk                    (clk),
        .i_resetn               (i_resetn),
        .o_aclrn                (aclrn),
        .o_resetn_synchronized  (resetn_synchronized),
        .o_sclrn                (sclrn)
    );
    
    logic                           active;         // asserted when generating valids, 0 when waiting for a new valid
    logic [VALID_COUNT_WIDTH-1:0]   valid_count;    // count number of valids to send, counts down to -1
    logic [II_COUNT_WIDTH-1:0]      ii_count;       // count pause between output valids, counts down to -1

    logic gen_valid;        // internally generated valid
    logic gen_first_valid;  // internally generated first_valid

    always_ff @(posedge clk or negedge aclrn) begin
        if (!aclrn) begin
            gen_valid <= '0;
            gen_first_valid <= '0;
            active <= '0;
            valid_count <= NUM_OUTPUT_VALIDS - 2 - ZERO_LATENCY;
            ii_count <= ZERO_LATENCY ? II - 2 : '1;     // set to -1 (all 1's), so we trigger an output valid immediately (unless ZERO_LATENCY is set)
        end else begin
    
            // determine active bit
            if (ii_count[II_COUNT_WIDTH-1] && valid_count[VALID_COUNT_WIDTH-1]) begin
                active <= '0;
            end else begin
                if (!active) begin
                    active <= i_valid;
                end
            end



            // we generate an output valid the clock cycle after i_valid is asserted, so
            // gen_first_valid is just a delayed version of i_valid
            gen_first_valid <= i_valid;

            // counter logic
            // only count when active is asserted or i_valid is asserted
            // counters count down to -1 then get reloaded
            if (i_valid || active) begin

                // ii counter
                if (ii_count[II_COUNT_WIDTH-1]) begin   // ii_count has reached -1, reload
                    if (valid_count[VALID_COUNT_WIDTH-1]) begin
                        ii_count <= ZERO_LATENCY ? II - 2 : '1;     // after all output valids are sent, reload with -1 (all 1's) so we are ready to send a new valid immediately
                    end else begin
                        ii_count <= II - 2; // reload counter to count down to -1
                    end
                end else begin      // ii_count has not reached -1, keep counting down
                    ii_count <= ii_count - 1;
                end

                // valid counter
                if (ii_count[II_COUNT_WIDTH-1]) begin   // only change valid counter when ii counter rolls over
                    if (valid_count[VALID_COUNT_WIDTH-1]) begin     // valid counter has reached -1, reload
                        valid_count <= NUM_OUTPUT_VALIDS - 2 - ZERO_LATENCY;
                    end else begin      // valid count has not reached -1, continue counting down
                        valid_count <= valid_count - 1;
                    end
                end

            end

            // output valid logic
            if ( (i_valid || active) && (ii_count[II_COUNT_WIDTH-1]) ) begin
                gen_valid <= '1;
            end else begin
                gen_valid <= '0;
            end


            // synchronous reset logic
            if (!sclrn) begin
                gen_valid <= '0;
                gen_first_valid <= '0;
                active <= '0;
                valid_count <= NUM_OUTPUT_VALIDS - 2 - ZERO_LATENCY;
                ii_count <= ZERO_LATENCY ? II - 2 : '1;     // set to -1 (all 1's), so we trigger an output valid immediately (unless ZERO_LATENCY is set)
            end
        
        end

    end

    if ( ZERO_LATENCY ) begin
        assign o_valid = i_valid | gen_valid;
        assign o_first_valid = i_valid;
    end else begin
        assign o_valid = gen_valid;
        assign o_first_valid = gen_first_valid;
    end

endmodule

`default_nettype wire
