#
# Copyright (C) 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of them
# is governed by the express license under which they were provided to you ("License"). Unless
# the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
# or transmit this software or the related documents without Intel's prior written permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#


# ------------------------------------------------------------------------------
# This example shows how to get flexible cachesim traffic data from the
# bottom-up dataset.
# ------------------------------------------------------------------------------

import sys

try:

    import advisor

except ImportError:

    print(
        """Import error: Python could not resolve path to Advisor's pythonapi directory.
        To fix, either manually add path to the pythonapi directory into PYTHONPATH environment
        variable, or use advixe-vars.* scripts to set up product environment variables automatically."""
    )
    sys.exit(1)

# Check command-line arguments.
if len(sys.argv) < 2:
    print('Usage: "python {} path_to_project_dir [colon_separated_cache_sizes_list]"'.format(__file__))
    print('e.g.: "python {} ./prj 8k:512k:6m"'.format(__file__))
    sys.exit(2)

# Open the Advisor Project and load the data.
project = advisor.open_project(sys.argv[1])
survey = project.load(advisor.SURVEY)

prev_cache_size = 0
cache_sizes = []
if len(sys.argv) > 2:
    for level_size in sys.argv[2].split(":"):
        if level_size.endswith('b'):
            cache_size = int(level_size[:-1]) + prev_cache_size
        elif level_size.endswith('k'):
            cache_size = 1024 * int(level_size[:-1]) + prev_cache_size
        elif level_size.endswith('m'):
            cache_size = 1024 * 1024 * int(level_size[:-1]) + prev_cache_size
        elif level_size.endswith('g'):
            cache_size = 1024 * 1024 * 1024 * int(level_size[:-1]) + prev_cache_size
        else:
            print('Wrong cache level format: {}'.format(level_size))
            sys.exit(2)

        cache_sizes.append(cache_size)
        prev_cache_size = cache_size

key = "function_call_sites_and_loops"
max_length = max(len(key), 12)
for row in survey.bottomup:
    max_length = max(max_length, len(row[key]))
key_fmt = '{{:<{}}}'.format(max_length)
col_fmt = ' {:<12}'

table_header = key_fmt.format(key)
if len(sys.argv) > 2:
    for level_size in sys.argv[2].split(","):
        table_header += col_fmt.format('{} loads'.format(level_size))
        table_header += col_fmt.format('{} stores'.format(level_size))
print(table_header)

# Iterate the data in the bottomup table, then pretty-print it with get_flex_cachesim_data.
for row in survey.bottomup:
    table_row = key_fmt.format(row[key])
    # get_flex_cachesim_data returns a list of tuples [(r,w),(r,w)]
    for bytes_r, bytes_w in row.get_flex_cachesim_data(cache_sizes):
        table_row += col_fmt.format(bytes_r / 1000000000)
        table_row += col_fmt.format(bytes_w / 1000000000)
    print(table_row)
