/****************************************************************************
*                                                                           *
* Copyright (C) 2024 Intel Corporation                                      *
*                                                                           *
*****************************************************************************

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 ***************************************************************************

For more documentation than found here, see

[1] doc/ReadMe_IMB.txt

[2] Intel(R) MPI Benchmarks
    Users Guide and Methodology Description
    In
    doc/IMB_Users_Guide.pdf

 File: IMB_cpu_exploit.c

 Implemented functions:

 IMB_cpu_exploit;


 ***************************************************************************/


#include "IMB_declare.h"
#include "IMB_prototypes.h"

double IMB_cpu_exploit_reworked(float target_secs, int initialize) {
    IMB_cpu_exploit(target_secs, initialize);
    return MFlops;
}

void IMB_cpu_exploit(float target_secs, int initialize) {
/*


                      Runs a CPU intensive code (matrix multiply) for a
                      user defined amount of CPU time



Input variables:

-target_secs          (type float)
                      That many seconds (about) the matrix multiply is run


-initialize           (type int)
                      1/0 for first / subsequent calls. If 1, the program
                      determines how to run in order to achieve target CPU time.



*/

/*
in: target_secs: desired runtime (about) of the current call
    initialize: 1/0 for first/following call with this value of target_secs
*/

#define SIZE 10
    static float a[SIZE][SIZE], x[SIZE], y[SIZE];
    double t1,t2;
    static int Nrep, target_reps;
    int i,j,repeat;

    if (target_secs <= 0.) {
        return;
    }

    if (MFlops < 0.) {
        for (i = 0; i < SIZE; i++) {
            x[i] = y[i] = 0.;
            for (j=0; j< SIZE; j++) {
                a[i][j] = 1.;
            }
        }
        Nrep = (50000000 / (2 * SIZE*SIZE)) + 1;
        t1 = MPI_Wtime();
        for (repeat = 0; repeat < Nrep; repeat++) {
            for (i=0; i<SIZE; i++) {
                for (j=0; j<SIZE; j++) {
                    x[i] = x[i] + a[i][j] * y[j];
                }
            }
        }
        t2 = MPI_Wtime();

        MFlops = (Nrep * 2 * SIZE * SIZE) * 1.e-6 / (t2 - t1);
        Nrep = (int)(1. / (t2 - t1) * Nrep);
        target_reps = 0;
    }

    if (initialize) {
        target_reps = max(1, (int)(target_secs * Nrep));
        t1 = MPI_Wtime();

        for (repeat = 0; repeat < target_reps; repeat++) {
            for (i = 0; i < SIZE; i++) {
                for (j = 0; j < SIZE; j++) {
                    x[i] = x[i] + a[i][j] * y[j];
                }
            }
        }
        t2 = MPI_Wtime();

        tCPU = 1000000. * (t2-t1);
    } else {
        for( repeat=0; repeat < target_reps; repeat++ ) {
            for (i=0; i<SIZE; i++) {
                for (j=0; j<SIZE; j++) {
                    x[i] = x[i] + a[i][j] * y[j];
                }
            }
        }
    }
}

