# Copyright (C) 2023 Intel Corporation
# SPDX-License-Identifier: MIT

import re
from pathlib import Path

from mpp.parsers.system_information_parser import SystemInformationParser, \
    SystemInfoParserAttributes, SectionNames, UnitNames, SymbolNames

unit_names = UnitNames('OS Processor', 'Phys. Package', 'Module', 'Core Type', 'Core', 'Logical Processor')

section_names = SectionNames('Processor Features', 'OS Processor <', 'System Features', 'Uncore Performance Monitoring '
                             'Units', 'EDP Configuration', 'RDT H/W Support', 'GPU Information', 'RAM Features',
                             'QPI Link Features', 'IIO Unit Features', 'PCIeX')

symbol_names = SymbolNames('Number of Packages', 'Threads Per Core', 'Cores Per Package', 'Threads Per Package')


class EmonSystemInformationParser(SystemInformationParser):
    INT_LIKE_RE = re.compile(r'\d+[\d,]*')

    """
    Parse system information stored in EMON data files (emon.dat, emon-v.dat)
    """

    parser_attributes = SystemInfoParserAttributes("Version Info:", '\t', unit_names, section_names, symbol_names)

    def __init__(self, input_file: Path, ref_tsc_hz: int = 0):
        """
        Initialize the EMON system information parser

        :param input_file: the EMON data file to parse
        :param ref_tsc_hz: an optional system frequency value (in Hz). Overrides system information in the input file
                            (if such information exists)

        """

        super().__init__(input_file, ref_tsc_hz)

    def _set_ref_tsc(self):
        tsc_candidates = [key for key in self.attributes.keys() if key.startswith('TSC Freq')]
        if len(tsc_candidates) > 0:
            # Assume the first occurrence inside tsc_candidates is the TSC Freq
            ref_tsc = self.attributes[tsc_candidates[0]].split(' ', 1)[0]
        elif 'Processor Base Freq' in self.attributes:
            ref_tsc = self.attributes['Processor Base Freq'].split(' ', 1)[0]
        else:
            # Unable to determine processor frequency
            ref_tsc = 0
        self._ref_tsc = float(ref_tsc) * 1000000
