/*******************************************************************************
* Copyright 2018-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            Template for VM issues reproducer
 *            performing one function call at defined arguments
 *            and simple results comparison to reference values.
 *
 *******************************************************************************/

#include <stdio.h>
#include <math.h>
#include <omp.h>

#include "mkl.h"
#include "mkl_omp_offload.h"

/* VM function */
#define FUNC    vmdErf
/* Data type */
#define TYPE    double
/* Accuracy */
#define ACC     VML_LA
/* Vector length */
#define LEN     2
/* Input arguments */
#define ARGS    0.70710678118654752, 1.41421356237309504
/* Expected reference results */
#define REFS    0.68268949213708596, 0.95449973610364158
/* Allowed relative error */
#define EPS     1.0e-15

/* String making macro for print */
#define str(x)   #x
#define STR(x)   str(x)

/* Offload device number */
static const int devnum = 0;

/**
 * @brief Main function for VM API testing
 *
 * Main performs call one VM function and simple results check
 *
 * @param[in] argc         Number of arguments
 * @param[in] argv         Pointer to argument strings
 * @return                 -1 for FAIL or 0 for PASS
 *
 */
int main (int argc, char **argv)
{
    int  err      = 0;                     /* Number of errors */
    TYPE arg[LEN] = {ARGS};                /* Input array */
    TYPE ref[LEN] = {REFS};                /* Reference results array */
    TYPE res[LEN] = {0};                   /* Output array */
    TYPE *parg = &arg[0], *pres = &res[0]; /* Array pointers for OMP pragmas */

    fprintf (stdout, "omp offload vm_one_func: started...\n"); fflush (stdout);

    #pragma omp target data map(to:parg[0:LEN]) map(tofrom:pres[0:LEN]) device(devnum)
    #pragma omp dispatch device(devnum)
        FUNC (LEN, parg, pres, ACC);

    for (int i = 0; i < LEN; i = i + 1)
    {
        fprintf (stderr, "\t%s (%.17lg) = \n\t\t%.17lg / computed /\n\t\t%.17lg / expected /\n",
                     STR(FUNC), arg[i], res[i], ref[i]);

        /* Check relative error versus reference result */
        err += (fabs ((res[i] - ref[i]) / ref[i]) > EPS);
    }

    fprintf (stdout, "omp offload vm_one_func: %s\n\n", (err)?"FAIL":"PASS");

    return (err)?-1:0;
} /* main */
