# Copyright (C) 2024 Intel Corporation
# SPDX-License-Identifier: MIT

from mpp.core.data_processor import _DataProcessor, DataProcessorFactory
from cli.timer import timer
from cli.writers.views import ViewWriter
from mpp import ViewCollection, ViewInitializer, ViewGenerator, DataAccumulator
from mpp.core.devices import DeviceCollection
from mpp.core.api_args import ApiArgs, SystemInformation
    

class Mpp:

    def __init__(self, api_args: ApiArgs):
        self.__args = api_args
        self.__device_collection: DeviceCollection = None
        self.__view_collection: ViewCollection = None
        self.__view_generator: ViewGenerator = None
        self.data_accumulator: DataAccumulator = None
        self.__data_processor: _DataProcessor = None
        self.__view_writer: ViewWriter = None
        self.__total_metrics_derived = 0
        self.__summary_views = None

        self.__system_information: SystemInformation = api_args.system_information

    @property
    def device_collection(self) -> DeviceCollection:
        return self.__device_collection

    @property
    def view_collection(self) -> ViewCollection:
        return self.__view_collection

    @property
    def view_generator(self) -> ViewGenerator:
        return self.__view_generator

    @property
    def view_writer(self) -> ViewWriter:
        return self.__view_writer
    
    @property
    def total_metrics_derived(self):
        return self.__total_metrics_derived
    
    @property
    def summary_views(self):
        return self.__summary_views

    @property
    def ref_tsc(self):
        return self.__system_information.ref_tsc

    @property
    def detail_views(self):
        return self.__data_processor.detail_views

    def initialize(self):
        self.__device_collection = DeviceCollection(self.__args)
        self.__get_view_collection()
        self.__view_generator = ViewGenerator(self.view_collection)
        self.data_accumulator = DataAccumulator(self.__view_generator)
        self.__view_writer = ViewWriter(self.__system_information.has_modules, writers=self.__args.output_writers)
        self.__data_processor = DataProcessorFactory().create(self.__args.is_parallel, self.data_accumulator, self.view_generator, self.view_writer)

    def process_data(self, event_reader, partitions, number_of_cores, no_detail_view):
        self.__data_processor.process(event_reader, partitions, number_of_cores, no_detail_view)

    def generate_summary_views(self, first_sample, last_sample):
        with timer() as number_of_seconds:
            self.__summary_views = self.data_accumulator.generate_summary_views()
            self.__total_metrics_derived = self._get_total_derived_metrics()
            if self.view_writer:
                self.view_writer.write(list(self.__summary_views.values()), first_sample, last_sample)
            print(f'Generated all summary tables in {number_of_seconds}')
    
    def _get_total_derived_metrics(self):
        view_names = list(filter(lambda x: 'system_view_summary' in x, list(self.__summary_views.keys())))
        metric_names = set()
        for view in view_names:
            metric_names = metric_names.union(set(list(filter(lambda x: 'metric_' in x, self.__summary_views[view].data.columns))))
        return metric_names

    def __get_view_collection(self):
        view_initializer = ViewInitializer(self.__device_collection.devices, self.__args)
        view_initializer.add_views()
        self.__view_collection = view_initializer.view_collection
