/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/


#ifndef _LWPMUDRV_H_
#define _LWPMUDRV_H_

#include "lwpmudrv_defines.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_types.h"
#include "lwpmudrv_version.h"
#include "lwpmudrv_struct.h"
#include "pebs.h"

#include <sys/param.h>
#include <sys/types.h>
#include <sys/cdefs.h>
#include <sys/systm.h>
#include <sys/proc.h>
#include <sys/pcpu.h>
#include <sys/sched.h>
#include <sys/malloc.h>
#include <machine/cpufunc.h>
#include <vm/vm.h>
#include <vm/pmap.h>
#include <machine/pmap.h>
#include <machine/segments.h>

MALLOC_DECLARE(M_SEP);

/*
 * Print macros for driver messages
 */

#if defined(MYDEBUG)
#define SEP_PRINT_DEBUG(fmt,args...) { printf(SEP_MSG_PREFIX" [DEBUG] " fmt,##args); }
#else
#define SEP_PRINT_DEBUG(fmt,args...) {;}
#endif

#define SEP_PRINT(fmt,args...) { printf(SEP_MSG_PREFIX" " fmt,##args); }

#define SEP_PRINT_WARNING(fmt,args...) { printf(SEP_MSG_PREFIX" [Warning] " fmt,##args); }

#define SEP_PRINT_ERROR(fmt,args...) { printf(SEP_MSG_PREFIX" [ERROR] " fmt,##args); }

// Macro to return the thread group id
#define GET_CURRENT_TGID() (current->tgid)

#if defined(DRV_IA32) || defined(DRV_EM64T)
#define OVERFLOW_ARGS  U64*, U64*
#endif


typedef struct DRV_EVENT_MASK_NODE_S  DRV_EVENT_MASK_NODE;
typedef        DRV_EVENT_MASK_NODE    *DRV_EVENT_MASK;

struct DRV_EVENT_MASK_NODE_S {
    U16 event_idx;    // 0 <= index < MAX_EVENTS
    U16 desc_id;
    union {
        U32 bitFields1;
        struct {
            U32 precise              : 1;
            U32 lbr_capture          : 1;
            U32 dear_capture         : 1;  // Indicates which events need to have additional registers read
                                           // because they are DEAR events.
            U32 iear_capture         : 1;  // Indicates which events need to have additional registers read
                                           // because they are IEAR events.
            U32 btb_capture          : 1;  // Indicates which events need to have additional registers read
                                           // because they are BTB events.
            U32 ipear_capture        : 1;  // Indicates which events need to have additional registers read
                                           // because they are IPEAR events.
            U32 uncore_capture       : 1;
            U32 branch               : 1;  // Indicates whether the event is related to branch opertion or
                                           // not
            U32 perf_metrics_capture : 1;  // Indicates whether the event is related to perf_metrics or not
            U32 trigger              : 1;  // Indicates if the event interrupted is the trigger event, flag used
                                           // in ebc mode and uncore interrupt mode
            U32 collect_on_ctx_sw    : 1;  // Indicates to collect the data only if context switch occurs.
            U32 reserved             : 21;
        } s1;
    } u1;
};

#define DRV_EVENT_MASK_event_idx(d)             (d)->event_idx
#define DRV_EVENT_MASK_desc_id(d)               (d)->desc_id
#define DRV_EVENT_MASK_bitFields1(d)            (d)->u1.bitFields1
#define DRV_EVENT_MASK_precise(d)               (d)->u1.s1.precise
#define DRV_EVENT_MASK_lbr_capture(d)           (d)->u1.s1.lbr_capture
#define DRV_EVENT_MASK_dear_capture(d)          (d)->u1.s1.dear_capture
#define DRV_EVENT_MASK_iear_capture(d)          (d)->u1.s1.iear_capture
#define DRV_EVENT_MASK_btb_capture(d)           (d)->u1.s1.btb_capture
#define DRV_EVENT_MASK_ipear_capture(d)         (d)->u1.s1.ipear_capture
#define DRV_EVENT_MASK_uncore_capture(d)        (d)->u1.s1.uncore_capture
#define DRV_EVENT_MASK_branch(d)                (d)->u1.s1.branch
#define DRV_EVENT_MASK_perf_metrics_capture(d)  (d)->u1.s1.perf_metrics_capture
#define DRV_EVENT_MASK_trigger(d)               (d)->u1.s1.trigger
#define DRV_EVENT_MASK_collect_on_ctx_sw(d)     (d)->u1.s1.collect_on_ctx_sw

#define MAX_OVERFLOW_EVENTS 16    // This defines the maximum number of overflow events per interrupt.
                                  // In order to reduce memory footprint, the value should be at least
                                  // the number of fixed and general PMU registers.
                                  // Sandybridge with HT off has 11 PMUs(3 fixed and 8 generic)

typedef struct DRV_MASKS_NODE_S  DRV_MASKS_NODE;
typedef        DRV_MASKS_NODE    *DRV_MASKS;

/*
 * @macro DRV_EVENT_MASK_NODE_S
 * @brief
 * The structure is used to store overflow events when handling PMU interrupt.
 * This approach should be more efficient than checking all event masks
 * if there are many events to be monitored
 * and only a few events among them have overflow per interrupt.
 */
struct DRV_MASKS_NODE_S {
    DRV_EVENT_MASK_NODE eventmasks[MAX_OVERFLOW_EVENTS];
    U8                  masks_num; // 0 <= mask_num <= MAX_OVERFLOW_EVENTS
};

#define DRV_MASKS_masks_num(d)           (d)->masks_num
#define DRV_MASKS_eventmasks(d)          (d)->eventmasks


/*
 *  Dispatch table for virtualized functions.
 *  Used to enable common functionality for different
 *  processor microarchitectures
 */
typedef struct DISPATCH_NODE_S  DISPATCH_NODE;
typedef        DISPATCH_NODE   *DISPATCH;

struct DISPATCH_NODE_S {
    VOID (*init)(PVOID);
    VOID (*fini)(PVOID);
    VOID (*write)(PVOID);
    VOID (*freeze)(PVOID);
    VOID (*restart)(PVOID);
    VOID (*read_data)(PVOID, U32);
    VOID (*check_overflow)(DRV_MASKS);
    VOID (*swap_group)(DRV_BOOL);
    U64  (*read_lbrs)(PVOID);
    VOID (*clean_up)(PVOID);
    VOID (*hw_errata)(VOID);
    VOID (*read_power)(PVOID);
    U64  (*check_overflow_errata)(ECB, U32, U64);
    VOID (*read_counts)(PVOID, U32);
    U64  (*check_overflow_gp_errata)(ECB,U64*);
    VOID (*read_ro)(PVOID, U32, U32);
    VOID (*platform_info)(PVOID);
    VOID (*trigger_read)(PVOID, U32, U32);    // Counter reads triggered/initiated by User mode timer
    VOID (*scan_for_uncore)(PVOID);
    VOID (*read_metrics)(PVOID);
};

extern VOID         **PMU_register_data;
extern VOID         **desc_data;

/*!
 * @struct LWPMU_DEVICE_NODE_S
 * @brief  Struct to hold fields per device
 *           PMU_register_data_unc - MSR info
 *           dispatch_unc          - dispatch table
 *           em_groups_counts_unc  - # groups
 *           pcfg_unc              - config struct
 */
typedef struct LWPMU_DEVICE_NODE_S  LWPMU_DEVICE_NODE;
typedef        LWPMU_DEVICE_NODE   *LWPMU_DEVICE;

struct LWPMU_DEVICE_NODE_S {
    VOID         **PMU_register_data;
    DISPATCH       dispatch;
    S32            em_groups_count;
    VOID          *pcfg;
    U64          **unc_prev_value;
    U64         ***unc_acc_value;
    U64            counter_mask;
    U64            num_events;
    U32            num_units;
    VOID          *ec;
    S32           *cur_group;
    S32            pci_dev_node_index;
    U32            device_type;
    LBR            lbr;
    PEBS_INFO_NODE pebs_info_node;
};

#define LWPMU_DEVICE_PMU_register_data(dev)   (dev)->PMU_register_data
#define LWPMU_DEVICE_dispatch(dev)            (dev)->dispatch
#define LWPMU_DEVICE_em_groups_count(dev)     (dev)->em_groups_count
#define LWPMU_DEVICE_pcfg(dev)                (dev)->pcfg
#define LWPMU_DEVICE_prev_value(dev)          (dev)->unc_prev_value
#define LWPMU_DEVICE_acc_value(dev)           (dev)->unc_acc_value
#define LWPMU_DEVICE_counter_mask(dev)        (dev)->counter_mask
#define LWPMU_DEVICE_num_events(dev)          (dev)->num_events
#define LWPMU_DEVICE_num_units(dev)           (dev)->num_units
#define LWPMU_DEVICE_ec(dev)                  (dev)->ec
#define LWPMU_DEVICE_cur_group(dev)           (dev)->cur_group
#define LWPMU_DEVICE_pci_dev_node_index(dev)  (dev)->pci_dev_node_index
#define LWPMU_DEVICE_device_type(dev)         (dev)->device_type
#define LWPMU_DEVICE_lbr(dev)                 (dev)->lbr
#define LWPMU_DEVICE_pebs_dispatch(dev)       (dev)->pebs_info_node.pebs_dispatch
#define LWPMU_DEVICE_pebs_record_size(dev)    (dev)->pebs_info_node.pebs_record_size
#define LWPMU_DEVICE_apebs_basic_offset(dev)  (dev)->pebs_info_node.apebs_basic_offset
#define LWPMU_DEVICE_apebs_mem_offset(dev)    (dev)->pebs_info_node.apebs_mem_offset
#define LWPMU_DEVICE_apebs_gpr_offset(dev)    (dev)->pebs_info_node.apebs_gpr_offset
#define LWPMU_DEVICE_apebs_xmm_offset(dev)    (dev)->pebs_info_node.apebs_xmm_offset
#define LWPMU_DEVICE_apebs_lbr_offset(dev)    (dev)->pebs_info_node.apebs_lbr_offset
#define LWPMU_DEVICE_apebs_css_offset(dev)    (dev)->pebs_info_node.apebs_css_offset


extern U32            num_devices;
extern U32            cur_devices;
extern LWPMU_DEVICE   devices;
extern U64           *pmu_state;
extern U32            num_core_devs;

// Handy macro
#define TSC_SKEW(this_cpu)     (cpu_tsc[this_cpu] - cpu_tsc[0])

/*
 *  The IDT / GDT descriptor for use in identifying code segments
 */
#if defined(DRV_EM64T)
#pragma pack(push,1)
typedef struct _idtgdtDesc {
    U16    idtgdt_limit;
    PVOID  idtgdt_base;
} IDTGDT_DESC;
#pragma pack(pop)

extern IDTGDT_DESC         gdt_desc;
#endif

#define preempt_enable sched_unpin
#define preempt_disable sched_pin

#define copy_from_user(kaddr, uaddr, size) copyin((uaddr), (kaddr), (size))
#define copy_to_user(uaddr, kaddr, size)   copyout((kaddr), (uaddr), (size))
#define get_user(kvar, uaddr)              copyin((uaddr), &(kvar), sizeof(*(uaddr)))
#define put_user(kvar, uaddr)              copyout(&(kvar), (uaddr), sizeof(*(uaddr)))

#define _IOC_TYPE(x)(((x) >> 8) & 0xff)
#define _IOC_NR(x)  ((x) & 0xff)
#endif
