/*
 * Copyright (C) 2005 Intel Corporation
 *
 * This software and the related documents are Intel copyrighted materials, and your use of them
 * is governed by the express license under which they were provided to you ("License"). Unless
 * the License provides otherwise, you may not use, modify, copy, publish, distribute, disclose
 * or transmit this software or the related documents without Intel's prior written permission.
 *
 * This software and the related documents are provided as is, with no express or implied
 * warranties, other than those that are expressly stated in the License.
*/

#include "lwpmudrv_defines.h"
#include "lwpmudrv_version.h"

#include "lwpmudrv_types.h"
#include "rise_errors.h"
#include "lwpmudrv_ecb.h"
#include "lwpmudrv_ioctl.h"
#include "lwpmudrv_struct.h"
#include "ecb_iterators.h"
#include "unc_common.h"
#include "pci.h"

#if defined(DRV_IA32) || defined(DRV_EM64T)
#include "apic.h"
#endif
#include "cpumon.h"
#include "lwpmudrv.h"
#include "utility.h"
#include "control.h"
#if defined(DRV_IA32) || defined(DRV_EM64T)
#include "core2.h"
#endif
#include "pmi.h"

#include "output.h"
#include "fbsd_os.h"
#include "sys_info.h"
#include "eventmux.h"
#include "unc_common.h"
#if defined(DRV_IA32) || defined(DRV_EM64T)
#include "pebs.h"
#endif

#include <sys/unistd.h>
#include <sys/kthread.h>
#include <sys/param.h>
#include <sys/conf.h>
#include <sys/ioccom.h>
#include <sys/kernel.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/pmc.h>
#include <sys/pmckern.h>
#include <sys/smp.h>
#include <sys/systm.h>
#include <sys/uio.h>
#ifdef DRV_USE_NMI
#include <sys/bus.h>
#include <sys/pmckern.h>

#include <machine/intr_machdep.h>
#if __FreeBSD_version >= 1100006
#include <x86/apicvar.h>
#else
#include <machine/apicvar.h>
#endif
#endif
#include "pmu_info_struct.h"
#include "pmu_list.h"


MALLOC_DEFINE(M_SEP, "sep", "sep driver memory allocations");

static int lwpmu_Load(void);
static void lwpmu_Unload(void);
int LWPMUDRV_Abnormal_Terminate(void);

static int sep_modevent(module_t mod, int type, void *arg)
{
    int error = 0;

    switch (type) {
    case MOD_LOAD:
        if (pmc_kernel_version != PMC_VERSION) {
            if (pmc_kernel_version == 0) {
                SEP_DRV_LOG_LOAD("Kernel not compiled with 'options HWPMC_HOOKS'.");
            }
            else {
                SEP_DRV_LOG_LOAD("Kernel PMC version (0x%x) doesn't match module version (0x%x).",
                                  pmc_kernel_version,
                                  PMC_VERSION);
            }
            error = EPROGMISMATCH;
            break;
        }
        error = lwpmu_Load();
        break;
    case MOD_UNLOAD:
        lwpmu_Unload();
        break;
    default:
        break;
    }

    return (error);
}

moduledata_t sep_mod = {
       "sep",
       (modeventhand_t)sep_modevent,
       0
};

DECLARE_MODULE(sep, sep_mod, SI_SUB_SMP, SI_ORDER_ANY);
MODULE_VERSION(sep, 1);

typedef struct LWPMU_DEV_NODE_S  LWPMU_DEV_NODE;
typedef        LWPMU_DEV_NODE   *LWPMU_DEV;

struct LWPMU_DEV_NODE_S {
  struct cdev       *cdev;
};

#define LWPMU_DEV_cdev(dev)        (dev)->cdev

/* Global variables of the driver */
SEP_VERSION_NODE          drv_version;
U64                      *read_counter_info         = NULL;
U64                      *prev_counter_data         = NULL;
U32                       emon_buffer_size          = 0;
U64                       prev_counter_size         = 0;
#if defined(DRV_IA32) || defined(DRV_EM64T)
#endif
VOID                    **desc_data                 = NULL;
#if defined(DRV_IA32) || defined(DRV_EM64T)
#endif
U64                       total_ram                 = 0;
U32                       output_buffer_size        = OUTPUT_LARGE_BUFFER;
#if defined(DRV_IA32) || defined(DRV_EM64T)
#endif
static  S32               desc_count                = 0;
uid_t                     uid                       = 0;
DRV_CONFIG                drv_cfg                   = NULL;
DEV_CONFIG                cur_pcfg                  = NULL;
volatile pid_t            control_pid               = 0;
LWPMU_DEV                 lwpmu_control             = NULL;
LWPMU_DEV                 lwmod_control             = NULL;
LWPMU_DEV                 lwemon_control            = NULL;
LWPMU_DEV                 lwsamp_control            = NULL;
LWPMU_DEV                 lwunc_control             = NULL;
EMON_BUFFER_DRIVER_HELPER emon_buffer_driver_helper = NULL;

/* needed for multiple devices (core/uncore) */
U32                     num_devices                 = 0;
U32                     num_core_devs               = 0;
U32                     cur_device                  = 0;
LWPMU_DEVICE            devices                     = NULL;
static U32              uncore_em_factor            = 0;
struct callout         *unc_read_timer              = NULL;
S32                     max_groups_unc              = 0;
DRV_BOOL                unc_buf_init                = FALSE;
DRV_SETUP_INFO_NODE     req_drv_setup_info;

#if !defined(DRV_USE_UNLOCKED_IOCTL)
static struct sx        ioctl_lock;
#endif

#define  PMU_DEVICES            2   // pmu, mod
#define  OTHER_PMU_DEVICES      1   // mod

static S8              *cpu_mask_bits     = NULL;

/*
 *  Global data: Buffer control structure
 */
BUFFER_DESC      cpu_buf    = NULL;
BUFFER_DESC      unc_buf    = NULL;
BUFFER_DESC      module_buf = NULL;
BUFFER_DESC      emon_buf   = NULL;

extern volatile int      config_done;

CPU_STATE          pcb                    = NULL;
size_t             pcb_size               = 0;
U32               *core_to_package_map    = NULL;
U32               *core_to_module_map     = NULL;
U32               *core_to_phys_core_map  = NULL;
U32               *core_to_thread_map     = NULL;
U32               *core_to_dev_map        = NULL;
S32               *module_id_index        = NULL;
U32               *cores_per_module       = NULL;
U32               *threads_per_core       = NULL;
U32                max_threads_per_core   = 0;
U32                max_cores_per_module   = 0;
U32                module_enum_supported  = 0;
U32                num_packages           = 0;
U32                num_pci_domains        = 0;
U64               *pmu_state              = NULL;
U64               *cpu_tsc                = NULL;
U64               *prev_cpu_tsc           = NULL;
U64               *diff_cpu_tsc           = NULL;
U64               *temp_tsc              = NULL;
U64                max_rmid               = 0;
UNCORE_TOPOLOGY_INFO_NODE              uncore_topology;
PLATFORM_TOPOLOGY_PROG_NODE            platform_topology_prog_node;
static PLATFORM_TOPOLOGY_PROG_NODE     req_platform_topology_prog_node;
UNCORE_DISCOVERY_TABLE_LIST_NODE       uncore_discovery_tables;

static U8              *prev_set_CR4   = 0;

struct thread          *abnormal_handler = NULL;
static volatile int     abnormal_end     = 0;
static struct mtx       abnormal_mutex;
static struct cv        abnormal_event;
static S32              allowlist_index  = -1;

#if defined(DRV_USE_UNLOCKED_IOCTL)
#define MTX_INIT(lock, desc)
#define MTX_LOCK(lock)
#define MTX_UNLOCK(lock)
#define MTX_DESTROY(lock)
#else
#define MTX_INIT(lock, desc)    sx_init((lock), (desc))
#define MTX_LOCK(lock)          sx_xlock((lock));
#define MTX_UNLOCK(lock)        sx_xunlock((lock));
#define MTX_DESTROY(lock)       sx_destroy((lock))
#endif


/*
 * @fn void lwpmudrv_Allocate_Uncore_Buffer
 *
 * @param
 *
 * @return   OS_STATUE
 *
 * @brief    allocate buffer space to save/restore the data before collection
 */
static OS_STATUS
lwpmudrv_Allocate_Uncore_Buffer (
    VOID
)
{
    U32            i, j, k, l;
    U32            max_entries = 0;
    U32            num_entries;
    DEV_UNC_CONFIG pcfg_unc;
    ECB            ecb;

    SEP_DRV_LOG_TRACE_IN("");

    for (i = num_core_devs; i < num_devices; i++) {
        pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
        if (!pcfg_unc || !DEV_UNC_CONFIG_num_events(pcfg_unc)) {
            SEP_DRV_LOG_TRACE("Skipping uncore buffer allocation for device %u with no events.", i);
            continue;
        }
        LWPMU_DEVICE_acc_value(&devices[i]) = CONTROL_Allocate_Memory(num_packages * sizeof(U64 **));
        LWPMU_DEVICE_prev_value(&devices[i]) = CONTROL_Allocate_Memory(num_packages * sizeof(U64 *));
        for (j = 0; j < num_packages; j++) {
            // Allocate memory and zero out accumulator array (one per group)
            LWPMU_DEVICE_acc_value(&devices[i])[j] = CONTROL_Allocate_Memory(LWPMU_DEVICE_em_groups_count(&devices[i]) * sizeof(U64 *));
            for (k = 0; k < LWPMU_DEVICE_em_groups_count(&devices[i]); k++) {
                ecb = LWPMU_DEVICE_PMU_register_data(&devices[i])[k];
                num_entries = ECB_num_events(ecb) * LWPMU_DEVICE_num_units(&devices[i]);
                LWPMU_DEVICE_acc_value(&devices[i])[j][k] = CONTROL_Allocate_Memory(num_entries * sizeof(U64));
                for (l = 0; l < num_entries; l++) {
                    LWPMU_DEVICE_acc_value(&devices[i])[j][k][l] = 0LL;
                }
                if (max_entries < num_entries) {
                    max_entries = num_entries;
                }
            }
            // Allocate memory and zero out prev_value array (one across groups)
            LWPMU_DEVICE_prev_value(&devices[i])[j] = CONTROL_Allocate_Memory(max_entries * sizeof(U64));
            for (k = 0; k < max_entries; k++) {
                LWPMU_DEVICE_prev_value(&devices[i])[j][k] = 0LL;
            }
        }
        max_entries = 0;
    }

    SEP_DRV_LOG_TRACE_OUT("Success");
    return OS_SUCCESS;
}

static OS_STATUS
lwpmudrv_Free_Uncore_Buffer (
    U32  i
)
{
    U32  j, k;

    SEP_DRV_LOG_TRACE_IN("");

    if (LWPMU_DEVICE_prev_value(&devices[i])) {
        for (j = 0; j < num_packages; j++) {
            LWPMU_DEVICE_prev_value(&devices[i])[j] = CONTROL_Free_Memory(LWPMU_DEVICE_prev_value(&devices[i])[j]);
        }
        LWPMU_DEVICE_prev_value(&devices[i]) = CONTROL_Free_Memory(LWPMU_DEVICE_prev_value(&devices[i]));
    }
    if (LWPMU_DEVICE_acc_value(&devices[i])) {
        for (j = 0; j < num_packages; j++) {
            if (LWPMU_DEVICE_acc_value(&devices[i])[j]) {
                for (k = 0; k < LWPMU_DEVICE_em_groups_count(&devices[i]); k++) {
                    LWPMU_DEVICE_acc_value(&devices[i])[j][k] = CONTROL_Free_Memory(LWPMU_DEVICE_acc_value(&devices[i])[j][k]);
                }
                LWPMU_DEVICE_acc_value(&devices[i])[j] = CONTROL_Free_Memory(LWPMU_DEVICE_acc_value(&devices[i])[j]);
            }
        }
        LWPMU_DEVICE_acc_value(&devices[i]) = CONTROL_Free_Memory(LWPMU_DEVICE_acc_value(&devices[i]));
    }

    SEP_DRV_LOG_TRACE_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Initialize_State(void)
 *
 * @param none
 *
 * @return OS_STATUS
 *
 * @brief  Allocates the memory needed at load time.  Initializes all the
 * @brief  necessary state variables with the default values.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Initialize_State (
    VOID
)
{
    SEP_DRV_LOG_INIT_IN("");

    /*
     *  Machine Initializations
     *  Abstract this information away into a separate entry point
     *
     *  Question:  Should we allow for the use of Hot-cpu
     *    add/subtract functionality while the driver is executing?
     */
    GLOBAL_STATE_num_cpus(driver_state)          = smp_cpus;
    GLOBAL_STATE_active_cpus(driver_state)       = smp_cpus;
    GLOBAL_STATE_cpu_count(driver_state)         = 0;
    GLOBAL_STATE_dpc_count(driver_state)         = 0;
    GLOBAL_STATE_num_em_groups(driver_state)     = 0;
    CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_UNINITIALIZED);

    SEP_DRV_LOG_INIT_OUT("Success: num_cpus=%d, active_cpus=%d.",
                    GLOBAL_STATE_num_cpus(driver_state),
                    GLOBAL_STATE_active_cpus(driver_state));
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn  static void lwpmudrv_Fill_TSC_Info (PVOID param)
 *
 * @param param - pointer the buffer to fill in.
 *
 * @return none
 *
 * @brief  Read the TSC and write into the correct array slot.
 *
 * <I>Special Notes</I>
 */
atomic_t read_now;
static struct cv read_tsc_now_cv;
static struct mtx read_tsc_now_mtx;
static VOID
lwpmudrv_Fill_TSC_Info (
    PVOID   param
)
{
    U32      this_cpu;

    SEP_DRV_LOG_TRACE_IN("");

    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();
    //
    // Wait until all CPU's are ready to proceed
    // This will serve as a synchronization point to compute tsc skews.
    //
    atomic_subtract_int(&read_now, 1);
    while (atomic_load_acq_int(&read_now) > 0);

    UTILITY_Read_TSC(&cpu_tsc[this_cpu]);
    SEP_DRV_LOG_TRACE("This cpu %d --- tsc --- 0x%llx.",
                    this_cpu, cpu_tsc[this_cpu]);

    SEP_DRV_LOG_TRACE_OUT("Success");
    return;
}


/*********************************************************************
 *  Internal Driver functions
 *     Should be called only from the lwpmudrv_DeviceControl routine
 *********************************************************************/

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Version(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_VERSION call.
 * @brief  Returns the version number of the kernel mode sampling.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Version (
    IOCTL_ARGS   arg
)
{
    OS_STATUS status;

    SEP_DRV_LOG_FLOW_IN("");

    // Check if enough space is provided for collecting the data
    if ((arg->len_drv_to_usr != sizeof(U32))  || (arg->buf_drv_to_usr == NULL)) {
        return OS_FAULT;
    }

    status = put_user(SEP_VERSION_NODE_sep_version(&drv_version), (U32 *)arg->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Reserve(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief
 * @brief  Local function that handles the LWPMU_IOCTL_RESERVE call.
 * @brief  Sets the state to RESERVED if possible.  Returns BUSY if unable
 * @brief  to reserve the PMU.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Reserve (
    IOCTL_ARGS    arg
)
{
    OS_STATUS  status = OS_SUCCESS;
    S32        reserve_success, reserve_status;

    SEP_DRV_LOG_FLOW_IN("");

    // Check if enough space is provided for collecting the data
    if ((arg->len_drv_to_usr != sizeof(S32))  || (arg->buf_drv_to_usr == NULL)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    reserve_success = CHANGE_DRIVER_STATE(STATE_BIT_UNINITIALIZED, DRV_STATE_RESERVED);

    /* reserve_success == 1 if the cmpset was successful, but we want to
     *  return 0 on success.
     */
    reserve_status = !reserve_success;
    status = put_user(reserve_status, (int*)arg->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Finish_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Finalize PMU after collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Finish_Op (
    PVOID param
)
{
    U32        this_cpu = CONTROL_THIS_CPU();
    U32        dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->fini != NULL) {
        dispatch->fini(&dev_idx);
    }

    SEP_DRV_LOG_TRACE_OUT("");
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static VOID lwpmudrv_Clean_Up(DRV_BOOL)
 *
 * @param  DRV_BOOL finish - Flag to call finish
 *
 * @return VOID
 *
 * @brief  Cleans up the memory allocation.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Clean_Up (
    DRV_BOOL finish
)
{
    U32  i;

    SEP_DRV_LOG_FLOW_IN("");

    if (DRV_CONFIG_use_pcl(drv_cfg) == TRUE) {
        drv_cfg = CONTROL_Free_Memory(drv_cfg);
        goto signal_end;
    }

#if defined(DRV_IA32) || defined(DRV_EM64T)
    if (devices) {
        U32            id;
        U32            num_groups;
        EVENT_CONFIG   ec;
        DEV_UNC_CONFIG pcfg_unc;
        DISPATCH       dispatch_unc = NULL;

        // first call destroy function for each device, then deallocate.
        for (id = 0; id < num_devices; id++) {
            if (LWPMU_DEVICE_pcfg(&devices[id])) {
                if (LWPMU_DEVICE_device_type(&devices[id]) == DEVICE_INFO_UNCORE) {
                    pcfg_unc     = LWPMU_DEVICE_pcfg(&devices[id]);
                    dispatch_unc = LWPMU_DEVICE_dispatch(&devices[id]);
                    if (DEV_UNC_CONFIG_num_events(pcfg_unc) && dispatch_unc && dispatch_unc->fini) {
                        SEP_DRV_LOG_TRACE("LWP: calling UNC Init.");
                        dispatch_unc->fini((PVOID *)&id);
                    }
                    lwpmudrv_Free_Uncore_Buffer(id);
                }
                else if (finish) {
                    CONTROL_Invoke_Parallel(lwpmudrv_Finish_Op, NULL);
                }
            }

            if (LWPMU_DEVICE_PMU_register_data(&devices[id])) {
                ec =  LWPMU_DEVICE_ec(&devices[id]);
                if (LWPMU_DEVICE_device_type(&devices[id]) == DEVICE_INFO_CORE) {
                    num_groups = EVENT_CONFIG_num_groups(ec);
                }
                else {
                    num_groups = EVENT_CONFIG_num_groups_unc(ec);
                }
                for (i = 0; i < num_groups; i++) {
                    CONTROL_Free_Memory(LWPMU_DEVICE_PMU_register_data(&devices[id])[i]);
                }
                LWPMU_DEVICE_PMU_register_data(&devices[id]) = CONTROL_Free_Memory(LWPMU_DEVICE_PMU_register_data(&devices[id]));
            }
            LWPMU_DEVICE_pcfg(&devices[id]) = CONTROL_Free_Memory(LWPMU_DEVICE_pcfg(&devices[id]));
            LWPMU_DEVICE_ec(&devices[id])   = CONTROL_Free_Memory(LWPMU_DEVICE_ec(&devices[id]));
            if (LWPMU_DEVICE_lbr(&devices[id])) {
                LWPMU_DEVICE_lbr(&devices[id])  = CONTROL_Free_Memory(LWPMU_DEVICE_lbr(&devices[id]));
            }
            if (LWPMU_DEVICE_cur_group(&devices[id])) {
                LWPMU_DEVICE_cur_group(&devices[id])  = CONTROL_Free_Memory(LWPMU_DEVICE_cur_group(&devices[id]));
            }
        }
        devices = CONTROL_Free_Memory(devices);
    }
#endif

    if (desc_data) {
        for (i = 0; i < GLOBAL_STATE_num_descriptors(driver_state); i++) {
            CONTROL_Free_Memory(desc_data[i]);
        }
    }
    desc_data               = CONTROL_Free_Memory(desc_data);
    drv_cfg                 = CONTROL_Free_Memory(drv_cfg);
    pmu_state               = CONTROL_Free_KMemory(pmu_state, GLOBAL_STATE_num_cpus(driver_state)*sizeof(U64)*2);
    cpu_mask_bits           = CONTROL_Free_Memory(cpu_mask_bits);
    core_to_dev_map         = CONTROL_Free_Memory(core_to_dev_map);

signal_end:
    GLOBAL_STATE_num_em_groups(driver_state)   = 0;
    GLOBAL_STATE_num_descriptors(driver_state) = 0;
    num_devices                                = 0;
    num_core_devs                              = 0;
    max_groups_unc                             = 0;
    control_pid                                = 0;
    unc_buf_init                               = FALSE;

    memset(pcb, 0, pcb_size);

    SEP_DRV_LOG_FLOW_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Initialize_Driver (PVOID buf_drv_to_usr, size_t len_drv_to_usr)
 *
 * @param  buf_drv_to_usr       - pointer to the input buffer
 * @param  len_drv_to_usr   - size of the input buffer
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_INIT call.
 * @brief  Sets up the interrupt handler.
 * @brief  Set up the output buffers/files needed to make the driver
 * @brief  operational.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Initialize_Driver (
    PVOID         buf_drv_to_usr,
    size_t        len_drv_to_usr
)
{
    U32        initialize;
    S32        cpu_num;
    int        status    = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("Entering Initialize driver");

    if (buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    initialize = CHANGE_DRIVER_STATE(STATE_BIT_RESERVED, DRV_STATE_IDLE);

    if (!initialize) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Unexpected driver state!");
        return OS_IN_PROGRESS;
    }

    /*
     *   Program State Initializations
     */
    drv_cfg = CONTROL_Allocate_Memory(len_drv_to_usr);
    if (!drv_cfg) {
        status = OS_NO_MEM;
        SEP_DRV_LOG_ERROR("Memory allocation failure for drv_cfg!");
        goto clean_return;
    }

    if (copy_from_user(drv_cfg, buf_drv_to_usr, len_drv_to_usr)) {
        status = OS_FAULT;
        SEP_DRV_LOG_ERROR("Memory copy failure for drv_cfg!");
        goto clean_return;
    }

    pmu_state = CONTROL_Allocate_KMemory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(U64)*2);
    if (!pmu_state) {
        status = OS_NO_MEM;
        SEP_DRV_LOG_ERROR("Memory allocation failure for pmu_state!");
    }

    uncore_em_factor = 0;
    for (cpu_num = 0; cpu_num < GLOBAL_STATE_num_cpus(driver_state); cpu_num++) {
        CPU_STATE_accept_interrupt(&pcb[cpu_num]) = 1;
        CPU_STATE_initial_mask(&pcb[cpu_num])     = 1;
        CPU_STATE_group_swap(&pcb[cpu_num])       = 1;
        CPU_STATE_reset_mask(&pcb[cpu_num])       = 0;
#if (defined(DRV_IA32) || defined(DRV_EM64T))
        CPU_STATE_last_p_state_valid(&pcb[cpu_num]) = FALSE;
#endif
    }
    SEP_DRV_LOG_TRACE("Config : size = %d.", DRV_CONFIG_size(drv_cfg));
    SEP_DRV_LOG_TRACE("Config : counting_mode = %d.", DRV_CONFIG_counting_mode(drv_cfg));

    if (core_to_dev_map == NULL) {
        core_to_dev_map = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));
        if (!core_to_dev_map) {
            status = OS_NO_MEM;
            SEP_DRV_LOG_ERROR("Memory allocation failure for core_to_dev_map!");
            goto clean_return;
        }
    }

    control_pid = curproc->p_pid;
    SEP_DRV_LOG_TRACE("Control PID = %d.", control_pid);

    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
        if (cpu_buf == NULL) {
            cpu_buf = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(BUFFER_DESC_NODE));
        }
        /*
         * Allocate the output and control buffers for each CPU in the system
         * Allocate and set up the temp output files for each CPU in the system
         * Allocate and set up the temp outout file for detailing the Modules in the system
         */
        DRV_CONFIG_seed_name(drv_cfg)     = NULL;
        DRV_CONFIG_seed_name_len(drv_cfg) = 0;
        status = OUTPUT_Initialize();
        if (status != OS_SUCCESS) {
            SEP_DRV_LOG_ERROR("OUTPUT_Initialize failed!");
            goto clean_return;
        }
        SEP_DRV_LOG_TRACE("lwpmudrv_Initialize: After OUTPUT_Initialize.");

        /*
         * Program the APIC and set up the interrupt handler
         */
        CPUMON_Install_Cpuhooks();
        SEP_DRV_LOG_TRACE("lwpmudrv_Initialize: Finished Installing cpu hooks.");
        /*
         * Set up the call back routines based on architecture.
         */
#if defined(DRV_EM64T)
        SYS_Get_GDT_Base((PVOID*)&gdt_desc);
#endif
        SEP_DRV_LOG_TRACE("lwpmudrv_Initialize: about to install module notification");
        FBSD_OS_Install_Hooks();
    }
    else if (DRV_CONFIG_emon_timer_interval(drv_cfg)) {
        output_buffer_size = OUTPUT_EMON_BUFFER;
        if (emon_buf == NULL) {
            emon_buf = CONTROL_Allocate_Memory(sizeof(BUFFER_DESC_NODE));
            if (!emon_buf) {
                status = OS_NO_MEM;
                SEP_DRV_LOG_ERROR("Memory allocation failure for emon_buf!");
                goto clean_return;
            }
        }
        status = OUTPUT_Initialize_EMON();
        if (status != OS_SUCCESS) {
            SEP_DRV_LOG_ERROR("OUTPUT_Initialize_EMON failed!");
            goto clean_return;
        }
    }

clean_return:
    if (status != OS_SUCCESS) {
        // release all memory allocated in this function:
        drv_cfg          = CONTROL_Free_Memory(drv_cfg);
        pmu_state        = CONTROL_Free_Memory(pmu_state);
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Initialize (PVOID buf_drv_to_usr, size_t len_drv_to_usr)
 *
 * @param  buf_drv_to_usr       - pointer to the input buffer
 * @param  len_drv_to_usr   - size of the input buffer
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_INIT call.
 * @brief  Sets up the interrupt handler.
 * @brief  Set up the output buffers/files needed to make the driver
 * @brief  operational.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Initialize (
    PVOID         buf_drv_to_usr,
    size_t        len_drv_to_usr
)
{
    int        status    = OS_SUCCESS;
    S32        cpu_num;
    DEV_CONFIG pcfg;

    SEP_DRV_LOG_FLOW_IN("Entering Initialize?");

    if (buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    if (!devices) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("No devices allocated!");
        return OS_INVALID;
    }

    pcfg = CONTROL_Allocate_Memory(len_drv_to_usr);
    if (!pcfg) {
        status = OS_NO_MEM;
        SEP_DRV_LOG_ERROR("Memory allocation failure for pcfg!");
        goto clean_return;
    }

    if (copy_from_user(pcfg, buf_drv_to_usr, len_drv_to_usr)) {
        status = OS_FAULT;
        SEP_DRV_LOG_ERROR("Memory copy failure for pcfg!");
        goto clean_return;
    }

    cur_device = DEV_CONFIG_device_index(pcfg);
    if (cur_device >= num_devices) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("No more devices to allocate! Wrong lwpmudrv_Init_Num_Devices.");
        return OS_FAULT;
    }

    LWPMU_DEVICE_pcfg(&devices[cur_device]) = pcfg;
    cur_pcfg = (DEV_CONFIG)LWPMU_DEVICE_pcfg(&devices[cur_device]);
#if __FreeBSD_version < 1101511 || defined(DISABLE_PEBS)
    // FIXME: disable pebs for FreeBSD meltdown patch
    DEV_CONFIG_pebs_mode((DEV_CONFIG)LWPMU_DEVICE_pcfg(&devices[cur_device])) = 0;
#endif
    LWPMU_DEVICE_dispatch(&devices[cur_device]) = UTILITY_Configure_CPU(DEV_CONFIG_dispatch_id(cur_pcfg));
    if (LWPMU_DEVICE_dispatch(&devices[cur_device]) == NULL) {
        SEP_DRV_LOG_ERROR("Dispatch pointer is NULL!");
        status = OS_INVALID;
        goto clean_return;
    }

    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
#if defined(DRV_IA32) || defined(DRV_EM64T)
        PEBS_Initialize(cur_device);
#endif
    }

    /* Create core to device ID map */
    for (cpu_num = 0; cpu_num < GLOBAL_STATE_num_cpus(driver_state); cpu_num++) {
        if (CPU_STATE_core_type(&pcb[cpu_num]) == DEV_CONFIG_core_type(cur_pcfg) &&
            (DEV_CONFIG_core_model_id(cur_pcfg) == 0 ||
            (DEV_CONFIG_core_model_id(cur_pcfg) == CPU_STATE_core_model_id(&pcb[cpu_num])))) {
            core_to_dev_map[cpu_num] = cur_device;
        }
    }
    num_core_devs++; //New core device
    LWPMU_DEVICE_device_type(&devices[cur_device]) = DEVICE_INFO_CORE;

clean_return:
    if (status != OS_SUCCESS) {
        // release all memory allocated in this function:
        PEBS_Destroy();
        lwpmudrv_Clean_Up(FALSE);
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Initialize_Num_Devices(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief
 * @brief  Local function that handles the LWPMU_IOCTL_INIT_NUM_DEV call.
 * @brief  Init # uncore devices.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Initialize_Num_Devices (
    IOCTL_ARGS arg
)
{
    SEP_DRV_LOG_FLOW_IN("");

    // Check if enough space is provided for collecting the data
    if ((arg->len_usr_to_drv != sizeof(U32))  || (arg->buf_usr_to_drv == NULL)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    if (copy_from_user(&num_devices, arg->buf_usr_to_drv, arg->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure");
        return OS_FAULT;
    }
    /*
     *   Allocate memory for number of devices
     */
    if (num_devices != 0) {
        devices = CONTROL_Allocate_Memory(num_devices * sizeof(LWPMU_DEVICE_NODE));
        if (!devices) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Unable to allocate memory for devices!");
            return OS_NO_MEM;
        }
    }
    cur_device = 0;

    SEP_DRV_LOG_FLOW_OUT("Success: num_devices=%d, devices=%p.", num_devices, devices);
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Initialize_UNC(PVOID buf_drv_to_usr, U32 len_drv_to_usr)
 *
 * @param  buf_drv_to_usr       - pointer to the input buffer
 * @param  len_drv_to_usr   - size of the input buffer
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_INIT call.
 * @brief  Sets up the interrupt handler.
 * @brief  Set up the output buffers/files needed to make the driver
 * @brief  operational.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Initialize_UNC (
    PVOID         buf_drv_to_usr,
    U32           len_drv_to_usr
)
{
    DEV_UNC_CONFIG  pcfg_unc;
    U32             i;
    int             status = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: current state is not IDLE.");
        return OS_IN_PROGRESS;
    }

    if (!devices) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("No devices allocated!");
        return OS_INVALID;
    }
    if (buf_drv_to_usr == NULL) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }
    if (len_drv_to_usr != sizeof(DEV_UNC_CONFIG_NODE)) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Got len_drv_to_usr=%d, expecting size=%d", len_drv_to_usr, (int)sizeof(DEV_UNC_CONFIG_NODE));
        return OS_FAULT;
    }

    /*
     *   Program State Initializations:
     *   Foreach device, copy over pcfg and configure dispatch table
     */
    // allocate memory
    pcfg_unc = CONTROL_Allocate_Memory(sizeof(DEV_UNC_CONFIG_NODE));
    if (pcfg_unc == NULL) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("pcfg_unc allocation failed!");
        return OS_NO_MEM;
    }
    // copy over pcfg
    if (copy_from_user(pcfg_unc, buf_drv_to_usr, len_drv_to_usr)) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Failed to copy from user!");
        return OS_FAULT;
    }
    cur_device = DEV_UNC_CONFIG_device_index(pcfg_unc);
    if (cur_device >= num_devices) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("No more devices to allocate! Wrong lwpmudrv_Init_Num_Devices.");
        return OS_FAULT;
    }
    LWPMU_DEVICE_pcfg(&devices[cur_device]) = pcfg_unc;
    if (DEV_UNC_CONFIG_num_events(pcfg_unc) > 0) {
        // configure dispatch from dispatch_id
        LWPMU_DEVICE_dispatch(&devices[cur_device]) = UTILITY_Configure_CPU(DEV_UNC_CONFIG_dispatch_id(pcfg_unc));
        if (LWPMU_DEVICE_dispatch(&devices[cur_device]) == NULL) {
            CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
            SEP_DRV_LOG_ERROR_FLOW_OUT("Unable to configure CPU!");
            return OS_FAULT;
        }
    }

    LWPMU_DEVICE_cur_group(&devices[cur_device]) = CONTROL_Allocate_Memory(num_packages * sizeof(S32));
    if (LWPMU_DEVICE_cur_group(&devices[cur_device]) == NULL) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Cur_grp allocation failed for device %u!", cur_device);
        return OS_NO_MEM;
    }
    for (i = 0; i < num_packages; i++) {
        LWPMU_DEVICE_cur_group(&devices[cur_device])[i] = 0;
    }

    LWPMU_DEVICE_em_groups_count(&devices[cur_device]) = 0;
    LWPMU_DEVICE_num_units(&devices[cur_device])       = DEV_UNC_CONFIG_num_units(pcfg_unc);
    LWPMU_DEVICE_device_type(&devices[cur_device])     = DEVICE_INFO_UNCORE;

    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE && DEV_UNC_CONFIG_num_events(pcfg_unc)) {
        if (unc_buf == NULL) {
            unc_buf = CONTROL_Allocate_Memory(num_packages*sizeof(BUFFER_DESC_NODE));
            if (!unc_buf) {
                CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
                SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure.");
                return OS_NO_MEM;
            }
        }

        if (!unc_buf_init) {
            status = OUTPUT_Initialize_UNC();
            if (status != OS_SUCCESS) {
                CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
                SEP_DRV_LOG_ERROR_FLOW_OUT("OUTPUT_Initialize failed!");
                return status;
            }
            unc_buf_init = TRUE;
        }
    }

    SEP_DRV_LOG_FLOW_OUT("unc dispatch id = %d.", DEV_UNC_CONFIG_dispatch_id(pcfg_unc));
    return status;
}
#endif


/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Terminate(void)
 *
 * @param  none
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the DRV_OPERATION_TERMINATE call.
 * @brief  Cleans up the interrupt handler and resets the PMU state.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Terminate (
    VOID
)
{
    U32  terminate_success;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() == DRV_STATE_UNINITIALIZED) {
        SEP_DRV_LOG_FLOW_OUT("Success (already uninitialized).");
        return OS_SUCCESS;
    }

    terminate_success = CHANGE_DRIVER_STATE(STATE_BIT_STOPPED | STATE_BIT_TERMINATING | STATE_BIT_IDLE, DRV_STATE_UNINITIALIZED);
    if (!terminate_success) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Unexpected state!");
        return OS_IN_PROGRESS;
    }

    lwpmudrv_Clean_Up(TRUE);

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Switch_To_Next_Group(param)
 *
 * @param  none
 *
 * @return none
 *
 * @brief  Switch to the next event group for both core and uncore.
 * @brief  This function assumes an active collection is frozen
 * @brief  or no collection is active.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Switch_To_Next_Group (
    VOID
)
{
    S32            cpuid;
    U32            i, j;
    CPU_STATE      pcpu;
    EVENT_CONFIG   ec;
    DEV_UNC_CONFIG pcfg_unc;
    DISPATCH       dispatch_unc;

    SEP_DRV_LOG_FLOW_IN("");

    for (cpuid = 0; cpuid < GLOBAL_STATE_num_cpus(driver_state); cpuid++) {
        pcpu     = &pcb[cpuid];
        ec       = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[core_to_dev_map[cpuid]]);
        if (!ec) {
            continue;
        }
        CPU_STATE_current_group(pcpu)++;
        // make the event group list circular
        CPU_STATE_current_group(pcpu) %= EVENT_CONFIG_num_groups(ec);
    }

    if (num_devices) {
        for(i = num_core_devs; i < num_devices; i++) {
            pcfg_unc     = LWPMU_DEVICE_pcfg(&devices[i]);
            dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);
            if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && dispatch_unc) {
                for (j = 0; j < num_packages; j++) {
                    LWPMU_DEVICE_cur_group(&devices[i])[j]++;
                    if (CPU_STATE_current_group(&pcb[0]) == 0) {
                        LWPMU_DEVICE_cur_group(&devices[i])[j] = 0;
                    }
                    LWPMU_DEVICE_cur_group(&devices[i])[j] %= LWPMU_DEVICE_em_groups_count(&devices[i]);
                }
            }
        }
    }

    SEP_DRV_LOG_FLOW_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwmpudrv_Get_Driver_State(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_GET_Driver_State call.
 * @brief  Returns the current driver state.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Driver_State (
    IOCTL_ARGS    arg
)
{
    OS_STATUS  status = OS_SUCCESS;
    S32        current_phase;

    SEP_DRV_LOG_TRACE_IN("");

    // Check if enough space is provided for collecting the data
    if ((arg->len_drv_to_usr != sizeof(U32)) || (arg->buf_drv_to_usr == NULL)) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("Invalid arguments!");
        return OS_FAULT;
    }

    current_phase = GET_DRIVER_STATE();
    status = put_user(current_phase, (U32*)arg->buf_drv_to_usr);

    SEP_DRV_LOG_TRACE_OUT("Return value: %d.", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Cleanup_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Clean up registers after collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Cleanup_Op (
    PVOID param
)
{
    U32        this_cpu = CONTROL_THIS_CPU();
    U32        dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->clean_up != NULL) {
        dispatch->clean_up(NULL);
    }

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Pause_Uncore(void)
 *
 * @param - 1 if switching group, 0 otherwise
 *
 * @return OS_STATUS
 *
 * @brief Pause the uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Pause_Uncore (
    PVOID param
)
{
    U32 i;
    U32 switch_grp = *((U32*)param);
    DEV_UNC_CONFIG pcfg_unc = NULL;
    DISPATCH   dispatch_unc = NULL;

    SEP_DRV_LOG_TRACE_IN("");

    for (i = num_core_devs; i < num_devices; i++) {
         pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
         dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);

         if (pcfg_unc                                &&
             DEV_UNC_CONFIG_num_events(pcfg_unc)     &&
             dispatch_unc                            &&
             dispatch_unc->freeze) {
                SEP_DRV_LOG_TRACE("LWP: calling UNC Pause.");
                if (switch_grp) {
                    if (LWPMU_DEVICE_em_groups_count(&devices[i]) > 1) {
                        dispatch_unc->freeze(&i);
                    }
                }
                else {
                    dispatch_unc->freeze(&i);
                }
         }
    }

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Pause_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Pause the core/uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Pause_Op (
    PVOID param
)
{
    U32        this_cpu   = CONTROL_THIS_CPU();
    U32        dev_idx    = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg       = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch   = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U32        switch_grp = 0;

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->freeze != NULL) {
        dispatch->freeze(NULL);
    }

    lwpmudrv_Pause_Uncore((PVOID)&switch_grp);

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Pause(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Pause the collection
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Pause (
    VOID
)
{
    U32  pause_success;
    int  i;
    int  done = FALSE;

    SEP_DRV_LOG_FLOW_IN("");

    if (!pcb || !drv_cfg) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Pcb or drv_cfg pointer is NULL!");
        return OS_INVALID;
    }

    pause_success = CHANGE_DRIVER_STATE(STATE_BIT_RUNNING, DRV_STATE_PAUSING);

    if (pause_success) {
        for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
            CPU_STATE_accept_interrupt(&pcb[i]) = 0;
        }
        while (!done) {
            done = TRUE;
            for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
                if (atomic_load_acq_int(&CPU_STATE_in_interrupt(&pcb[i]))) {
                    done = FALSE;
                }
            }
        }
        CONTROL_Invoke_Parallel(lwpmudrv_Pause_Op, NULL);
        /*
         * This means that the PAUSE state has been reached.
         */
        CHANGE_DRIVER_STATE(STATE_BIT_PAUSING, DRV_STATE_PAUSED);
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Resume_Uncore(void)
 *
 * @param - 1 if switching group, 0 otherwise
 *
 * @return OS_STATUS
 *
 * @brief Resume the uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Resume_Uncore (
    PVOID param
)
{
    U32 i;
    U32 switch_grp;
    DEV_UNC_CONFIG pcfg_unc = NULL;
    DISPATCH   dispatch_unc = NULL;

    SEP_DRV_LOG_TRACE_IN("");

    switch_grp = *((U32*)param);

    for (i = num_core_devs; i < num_devices; i++) {
         pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
         dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);

         if (pcfg_unc                                &&
             DEV_UNC_CONFIG_num_events(pcfg_unc)     &&
             dispatch_unc                            &&
             dispatch_unc->restart) {
                SEP_DRV_LOG_TRACE("LWP: calling UNC Resume.");
                if (switch_grp) {
                    if (LWPMU_DEVICE_em_groups_count(&devices[i]) > 1) {
                        dispatch_unc->restart(&i);
                    }
                }
                else {
                    dispatch_unc->restart(&i);
                }
         }
    }

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Resume_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Resume the core/uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Resume_Op (
    PVOID param
)
{
    U32        this_cpu   = CONTROL_THIS_CPU();
    U32        dev_idx    = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg       = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch   = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U32        switch_grp = 0;

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->restart != NULL) {
        dispatch->restart((VOID *)(size_t)0);
    }

    lwpmudrv_Resume_Uncore((PVOID)&switch_grp);

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Resume(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Resume the collection
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Resume (
    VOID
)
{
    U32        resume_success;
    int        i;

    SEP_DRV_LOG_FLOW_IN("");

    if (!pcb || !drv_cfg) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Pcb or drv_cfg pointer is NULL!");
        return OS_INVALID;
    }

    /*
     * If we are in the process of pausing sampling, wait until the pause has been
     * completed.  Then start the Resume process.
     */

    while (GET_DRIVER_STATE() == DRV_STATE_PAUSING) {
        SYS_IO_Delay();
        SYS_IO_Delay();
        continue;
    }

    resume_success = CHANGE_DRIVER_STATE(STATE_BIT_PAUSED, DRV_STATE_RUNNING);
    if (resume_success) {
        if (cpu_mask_bits) {
            for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
                CPU_STATE_accept_interrupt(&pcb[i]) = cpu_mask_bits[i] ? 1 : 0;
                CPU_STATE_group_swap(&pcb[i])       = 1;
            }
        }
        else {
            for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
                CPU_STATE_accept_interrupt(&pcb[i]) = 1;
                CPU_STATE_group_swap(&pcb[i])       = 1;
            }
        }
        CONTROL_Invoke_Parallel(lwpmudrv_Resume_Op, NULL);
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Write_Uncore(void)
 *
 * @param - 1 if switching group, 0 otherwise
 *
 * @return OS_STATUS
 *
 * @brief Program the uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Write_Uncore (
    PVOID param
)
{
    U32 i;
    U32 switch_grp;
    DEV_UNC_CONFIG pcfg_unc = NULL;
    DISPATCH   dispatch_unc = NULL;

    SEP_DRV_LOG_TRACE_IN("");

    switch_grp = *((U32*)param);

    for (i = num_core_devs; i < num_devices; i++) {
         pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
         dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);

         if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && dispatch_unc && dispatch_unc->write) {
                SEP_DRV_LOG_TRACE("LWP: calling UNC Write.");
                if (switch_grp) {
                    if (LWPMU_DEVICE_em_groups_count(&devices[i]) > 1) {
                        dispatch_unc->write(&i);
                    }
                }
                else {
                    dispatch_unc->write(&i);
                }
         }
    }

    SEP_DRV_LOG_TRACE_OUT("");
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Write_Op(void)
 *
 * @param - Do operation for Core only
 *
 * @return OS_STATUS
 *
 * @brief Program the core/uncore collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Write_Op (
    PVOID param
)
{
    U32        this_cpu   = CONTROL_THIS_CPU();
    U32        dev_idx    = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg       = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch   = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    U32        switch_grp = 0;

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->write != NULL) {
        dispatch->write((VOID *)(size_t)0);
    }

    if (param == NULL) {
        lwpmudrv_Write_Uncore((PVOID)&switch_grp);
    }

    SEP_DRV_LOG_TRACE_OUT("");
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Switch_Group(void)
 *
 * @param none
 *
 * @return OS_STATUS
 *
 * @brief Switch the current group that is being collected.
 *
 * <I>Special Notes</I>
 *     This routine is called from the user mode code to handle the multiple group
 *     situation.  4 distinct steps are taken:
 *     Step 1: Pause the sampling
 *     Step 2: Increment the current group count
 *     Step 3: Write the new group to the PMU
 *     Step 4: Resume sampling
 */
static OS_STATUS
lwpmudrv_Switch_Group (
    VOID
)
{
    S32            idx;
    CPU_STATE      pcpu;
    EVENT_CONFIG   ec;
    OS_STATUS      status        = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (!pcb || !drv_cfg) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Pcb or drv_cfg pointer is NULL!");
        return OS_INVALID;
    }

    if (!DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_RUNNING | STATE_BIT_PAUSED)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Return value: %d (invalid driver state!).", status);
        return status;
    }

    status = lwpmudrv_Pause();

    for (idx = 0; idx < GLOBAL_STATE_num_cpus(driver_state); idx++) {
        pcpu = &pcb[idx];
        ec   = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[core_to_dev_map[idx]]);
        CPU_STATE_current_group(pcpu)++;
        // make the event group list circular
        CPU_STATE_current_group(pcpu) %= EVENT_CONFIG_num_groups(ec);
    }
    CONTROL_Invoke_Parallel(lwpmudrv_Write_Op, (VOID *)(size_t)CONTROL_THIS_CPU());
    if (drv_cfg && (DRV_CONFIG_start_paused(drv_cfg) == FALSE)) {
        lwpmudrv_Resume();
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwmudrv_Read_Specific_TSC (PVOID param)
 *
 * @param param - pointer to the result
 *
 * @return none
 *
 * @brief  Read the tsc value in the current processor and
 * @brief  write the result into param.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Read_Specific_TSC (
    PVOID  param
)
{
    U32 this_cpu;

    SEP_DRV_LOG_TRACE_IN("");

    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    if (this_cpu == 0) {
        UTILITY_Read_TSC((U64*)param);
    }
    preempt_enable();

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Uncore_Switch_Group(void)
 *
 * @param none
 *
 * @return OS_STATUS
 *
 * @brief Switch the current group that is being collected.
 *
 * <I>Special Notes</I>
 *     This routine is called from the user mode code to handle the multiple group
 *     situation.  4 distinct steps are taken:
 *     Step 1: Pause the sampling
 *     Step 2: Increment the current group count
 *     Step 3: Write the new group to the PMU
 *     Step 4: Resume sampling
 */
static OS_STATUS
lwpmudrv_Uncore_Switch_Group (
    VOID
)
{
    OS_STATUS      status        = OS_SUCCESS;
    U32            i             = 0;
    U32            j, k;
    DEV_UNC_CONFIG pcfg_unc;
    DISPATCH       dispatch_unc;
    ECB            ecb_unc;
    U32            cur_grp;
    U32            num_units;
    U32            switch_grp    = 1;

    SEP_DRV_LOG_FLOW_IN("");

    if (!devices || !drv_cfg) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Devices or drv_cfg pointer is NULL!");
        return OS_INVALID;
    }

    if (!DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_RUNNING | STATE_BIT_PAUSED)) {
        SEP_DRV_LOG_FLOW_OUT("Driver state is not RUNNING or PAUSED!");
        return OS_INVALID;
    }

    if (max_groups_unc > 1) {
        CONTROL_Invoke_Parallel(lwpmudrv_Pause_Uncore, (PVOID)&switch_grp);
        for(i = num_core_devs; i < num_devices; i++) {
            pcfg_unc     = LWPMU_DEVICE_pcfg(&devices[i]);
            dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);
            num_units    = LWPMU_DEVICE_num_units(&devices[i]);
            if (!pcfg_unc || !DEV_UNC_CONFIG_num_events(pcfg_unc) || !dispatch_unc) {
                continue;
            }
            if (LWPMU_DEVICE_em_groups_count(&devices[i]) > 1) {
                for (j = 0; j < num_packages; j++) {
                    // Switch group
                    LWPMU_DEVICE_cur_group(&devices[i])[j]++;
                    LWPMU_DEVICE_cur_group(&devices[i])[j] %= LWPMU_DEVICE_em_groups_count(&devices[i]);
                    // Post group switch
                    cur_grp = LWPMU_DEVICE_cur_group(&devices[i])[j];
                    ecb_unc = LWPMU_DEVICE_PMU_register_data(&devices[i])[cur_grp];
                    for (k = 0; k < (ECB_num_events(ecb_unc)*num_units); k++) {
                        LWPMU_DEVICE_prev_value(&devices[i])[j][k] = 0LL;  //zero out prev_value for new collection
                    }
                }
            }
        }
        CONTROL_Invoke_Parallel(lwpmudrv_Write_Uncore, (PVOID)&switch_grp);
        CONTROL_Invoke_Parallel(lwpmudrv_Resume_Uncore, (PVOID)&switch_grp);
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Trigger_Read_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Read uncore data
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Trigger_Read_Op (
    PVOID param
)
{
    DEV_UNC_CONFIG        pcfg_unc     = NULL;
    DISPATCH              dispatch_unc = NULL;
    U32                   this_cpu;
    CPU_STATE             pcpu;
    U32                   package_num;
    U64                   tsc;
    BUFFER_DESC           bd;
    EVENT_DESC            evt_desc;
    U32                   cur_grp;
    ECB                   pecb;
    boolean_t             signal_full  = FALSE;
    U32                   sample_size  = 0;
    U32                   offset       = 0;
    PVOID                 buf;
    UncoreSampleRecordPC *psamp;
    U32                   i;

    SEP_DRV_LOG_TRACE_IN("");

    this_cpu     = CONTROL_THIS_CPU();
    pcpu         = &pcb[this_cpu];
    package_num  = core_to_package_map[this_cpu];

    if (!DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_RUNNING | STATE_BIT_PAUSED)) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("State is not RUNNING or PAUSED!");
        return;
    }

    if (!CPU_STATE_socket_master(pcpu)) {
        SEP_DRV_LOG_TRACE_OUT("Not socket master.");
        return;
    }

    UTILITY_Read_TSC(&tsc);
    bd = &unc_buf[package_num];

    for (i = num_core_devs; i < num_devices; i++) {
        pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
        if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && !DEV_UNC_CONFIG_device_with_intr_events(pcfg_unc)) {
            cur_grp = LWPMU_DEVICE_cur_group(&devices[i])[package_num];
            pecb = LWPMU_DEVICE_PMU_register_data(&devices[i])[cur_grp];
            evt_desc = desc_data[ECB_descriptor_id(pecb)];
            sample_size += EVENT_DESC_sample_size(evt_desc);
        }
    }

    buf = OUTPUT_Reserve_Buffer_Space(bd, sample_size, &signal_full, !SEP_IN_NOTIFICATION);
    if (signal_full) {
        mtx_lock(&BUFFER_DESC_mtx(bd));
        cv_signal(&BUFFER_DESC_cv(bd));
        mtx_unlock(&BUFFER_DESC_mtx(bd));
    }

    if (buf) {
        for (i = num_core_devs; i < num_devices; i++) {
            pcfg_unc = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
            dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);
            if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && !DEV_UNC_CONFIG_device_with_intr_events(pcfg_unc) && dispatch_unc && dispatch_unc->trigger_read) {
                cur_grp = LWPMU_DEVICE_cur_group(&devices[i])[package_num];
                pecb = LWPMU_DEVICE_PMU_register_data(&devices[i])[cur_grp];
                evt_desc = desc_data[ECB_descriptor_id(pecb)];

                psamp = (UncoreSampleRecordPC *)(((S8 *)buf) + offset);
                UNCORE_SAMPLE_RECORD_descriptor_id(psamp)  = ECB_descriptor_id(pecb);
                UNCORE_SAMPLE_RECORD_tsc(psamp)            = tsc;
                UNCORE_SAMPLE_RECORD_uncore_valid(psamp)   = 1;
                UNCORE_SAMPLE_RECORD_cpu_num(psamp)        = (U16)this_cpu;
                UNCORE_SAMPLE_RECORD_pkg_num(psamp)        = (U16)package_num;

                dispatch_unc->trigger_read(psamp, i, 0);
                offset += EVENT_DESC_sample_size(evt_desc);
            }
        }
    }
    else {
        SEP_DRV_LOG_WARNING("Buffer space reservation failed; some samples will be dropped.");
    }

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static VOID lwpmudrv_Trigger_Read(void)
 *
 * @param - none
 *
 * @return - none
 *
 * @brief Read the Counter Data.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Trigger_Read (
    PVOID arg
)
{
    unsigned long delay = DRV_CONFIG_unc_timer_interval(drv_cfg);

    SEP_DRV_LOG_TRACE_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_RUNNING) {
        callout_schedule(unc_read_timer, delay*hz/1000);
        SEP_DRV_LOG_TRACE_OUT("Success: driver state is not RUNNING");
        return;
    }

    CONTROL_Invoke_Parallel(lwpmudrv_Trigger_Read_Op, NULL);

    uncore_em_factor++;
    if (uncore_em_factor == DRV_CONFIG_unc_em_factor(drv_cfg)) {
        SEP_DRV_LOG_TRACE("Switching Uncore Group...");
        lwpmudrv_Uncore_Switch_Group();
        uncore_em_factor = 0;
    }

    callout_schedule(unc_read_timer, delay*hz/1000);

    SEP_DRV_LOG_TRACE_OUT("Success.");
    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID lwpmudrv_Uncore_Stop_Timer (void)
 *
 * @brief       Stop the uncore read timer
 *
 * @param       none
 *
 * @return      none
 *
 * <I>Special Notes:</I>
 */
static VOID
lwpmudrv_Uncore_Stop_Timer (
    VOID
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (unc_read_timer == NULL) {
        return;
    }

    callout_drain(unc_read_timer);
    unc_read_timer = CONTROL_Free_Memory(unc_read_timer);

    SEP_DRV_LOG_FLOW_OUT("");

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          OS_STATUS lwpmudrv_Uncore_Start_Timer (void)
 *
 * @brief       Start the uncore read timer
 *
 * @param       none
 *
 * @return      OS_STATUS
 *
 * <I>Special Notes:</I>
 */
static VOID
lwpmudrv_Uncore_Start_Timer (
    VOID
)
{
    unsigned long delay = DRV_CONFIG_unc_timer_interval(drv_cfg);

    SEP_DRV_LOG_FLOW_IN("");

    unc_read_timer = CONTROL_Allocate_Memory(sizeof(struct callout));
    if (unc_read_timer == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for unc_read_timer!");
        return;
    }

    callout_init(unc_read_timer, CALLOUT_MPSAFE);
    callout_reset(unc_read_timer, delay*hz/1000, lwpmudrv_Trigger_Read, NULL);

    SEP_DRV_LOG_FLOW_OUT("");

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Read_Data_Op(PVOID param)
 *
 * @param param   - dummy
 *
 * @return void
 *
 * @brief  Read all the core/uncore data counters at one shot
 *
 * <I>Special Notes</I>
 */
static void
lwpmudrv_Read_Data_Op (
    VOID*    param
)
{
    U32            this_cpu = CONTROL_THIS_CPU();
    DISPATCH       dispatch;
    U32            dev_idx;
    DEV_CONFIG     pcfg;
    DEV_UNC_CONFIG pcfg_unc;

    SEP_DRV_LOG_TRACE_IN("");

    if (devices == NULL) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("Devices is null!");
        return;
    }
    dev_idx  = core_to_dev_map[this_cpu];
    pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->read_data != NULL) {
        dispatch->read_data(param, dev_idx);
    }
    for (dev_idx = num_core_devs; dev_idx < num_devices; dev_idx++) {
        pcfg_unc      = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[dev_idx]);
        if (!pcfg_unc || !DEV_UNC_CONFIG_num_events(pcfg_unc)) {
            continue;
        }
        if (!DRV_CONFIG_emon_mode(drv_cfg)) {
            continue;
        }
        dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);
        if (dispatch == NULL) {
            continue;
        }
        if (dispatch->read_data == NULL) {
            continue;
        }
        dispatch->read_data(param, dev_idx);
    }

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Emon_Switch_Group(param)
 *
 * @param  none
 *
 * @return none
 *
 * @brief  Switch to the next event group for both core and uncore.
 * @brief  This function assumes an active collection is frozen
 * @brief  or no collection is active.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Emon_Switch_Group (
    VOID
)
{
    U32            this_cpu = CONTROL_THIS_CPU();
    CPU_STATE      pcpu     = &pcb[this_cpu];
    U32            dev_idx  = core_to_dev_map[this_cpu];
    EVENT_CONFIG   ec       = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[dev_idx]);
    U32            i, j;
    DEV_UNC_CONFIG pcfg_unc;
    DISPATCH       dispatch_unc;

    SEP_DRV_LOG_FLOW_IN("");

    CPU_STATE_current_group(pcpu)++;
    // make the event group list circular
    CPU_STATE_current_group(pcpu) %= EVENT_CONFIG_num_groups(ec);

    if (CPU_STATE_socket_master(pcpu)) {
        for(i = num_core_devs; i < num_devices; i++) {
            pcfg_unc     = LWPMU_DEVICE_pcfg(&devices[i]);
            dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);
            if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && dispatch_unc) {
                for (j = 0; j < num_packages; j++) {
                    LWPMU_DEVICE_cur_group(&devices[i])[j]++;
                    if (CPU_STATE_current_group(&pcb[0]) == 0) {
                        LWPMU_DEVICE_cur_group(&devices[i])[j] = 0;
                    }
                    LWPMU_DEVICE_cur_group(&devices[i])[j] %= LWPMU_DEVICE_em_groups_count(&devices[i]);
                }
            }
        }
    }

    SEP_DRV_LOG_FLOW_OUT("");
    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Emon_Read_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Read EMON uncore data
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Emon_Read_Op (
    PVOID param
)
{
    U32       this_cpu             = CONTROL_THIS_CPU();
    PVOID     buf                  = param;
    U64      *tsc                  = NULL;
    DRV_BOOL  enter_in_pause_state = FALSE;

    SEP_DRV_LOG_TRACE_IN("");

    if (GET_DRIVER_STATE() == DRV_STATE_PAUSED) {
        SEP_DRV_LOG_TRACE("Entering in pause state.");
        enter_in_pause_state = 1;
    }

    prev_cpu_tsc[this_cpu] = cpu_tsc[this_cpu];
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg) || DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg) || (this_cpu == 0)) {
        UTILITY_Read_TSC(&cpu_tsc[this_cpu]);
    }

    // Counters should be frozen right after time stamped.
    if (!enter_in_pause_state) {
        lwpmudrv_Pause_Op(NULL);
    }

    buf = (PVOID)(((U64 *)buf) + GLOBAL_STATE_num_cpus(driver_state));
    diff_cpu_tsc[this_cpu] = cpu_tsc[this_cpu] - prev_cpu_tsc[this_cpu];
    tsc = ((U64 *)buf) + this_cpu;
    *tsc = diff_cpu_tsc[this_cpu];

    buf = (PVOID)(((U64 *)buf) + GLOBAL_STATE_num_cpus(driver_state));
    lwpmudrv_Read_Data_Op(buf);

    lwpmudrv_Emon_Switch_Group();

    lwpmudrv_Write_Op(NULL);

    if (DRV_CONFIG_per_cpu_tsc(drv_cfg) || DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg) || (this_cpu == 0)) {
        UTILITY_Read_TSC(&cpu_tsc[this_cpu]);
    }

    if (!enter_in_pause_state) {
        CPU_STATE_group_swap(&pcb[this_cpu]) = 1;
        lwpmudrv_Resume_Op(NULL);
    }

    SEP_DRV_LOG_TRACE_OUT("");

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static VOID lwpmudrv_Emon_Read(void)
 *
 * @param - none
 *
 * @return - none
 *
 * @brief Read the EMON Counter Data.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Emon_Read (
    PVOID arg
)
{
    unsigned long delay = DRV_CONFIG_emon_timer_interval(drv_cfg);
    boolean_t     signal_full  = FALSE;
    PVOID         buf;

    SEP_DRV_LOG_TRACE_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_RUNNING) {
        callout_schedule(unc_read_timer, delay*hz/1000);
        SEP_DRV_LOG_TRACE_OUT("Success: driver state is not RUNNING");
        return;
    }

    buf = OUTPUT_Reserve_Buffer_Space(emon_buf, emon_buffer_size, &signal_full, !SEP_IN_NOTIFICATION);
    if (signal_full) {
        mtx_lock(&BUFFER_DESC_mtx(emon_buf));
        cv_signal(&BUFFER_DESC_cv(emon_buf));
        mtx_unlock(&BUFFER_DESC_mtx(emon_buf));
    }

    if (buf) {
        CONTROL_Invoke_Parallel(lwpmudrv_Emon_Read_Op, buf);
    }
    else {
        SEP_DRV_LOG_WARNING("Buffer space reservation failed; some samples will be dropped.");
    }

    callout_schedule(unc_read_timer, delay*hz/1000);

    SEP_DRV_LOG_TRACE_OUT("Success.");

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          VOID lwpmudrv_Emon_Stop_Timer (void)
 *
 * @brief       Stop the EMON read timer
 *
 * @param       none
 *
 * @return      none
 *
 * <I>Special Notes:</I>
 */
static VOID
lwpmudrv_Emon_Stop_Timer (
    VOID
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (unc_read_timer == NULL) {
        return;
    }

    callout_drain(unc_read_timer);
    unc_read_timer = CONTROL_Free_Memory(unc_read_timer);

    SEP_DRV_LOG_FLOW_OUT("");

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          OS_STATUS lwpmudrv_Emon_Start_Timer (void)
 *
 * @brief       Start the Emon read timer
 *
 * @param       none
 *
 * @return      OS_STATUS
 *
 * <I>Special Notes:</I>
 */
static VOID
lwpmudrv_Emon_Start_Timer (
    VOID
)
{
    unsigned long delay = DRV_CONFIG_emon_timer_interval(drv_cfg);

    SEP_DRV_LOG_FLOW_IN("");

    unc_read_timer = CONTROL_Allocate_Memory(sizeof(struct callout));
    if (unc_read_timer == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for unc_read_timer!");
        return;
    }

    callout_init(unc_read_timer, CALLOUT_MPSAFE);
    callout_reset(unc_read_timer, delay*hz/1000, lwpmudrv_Emon_Read, NULL);

    SEP_DRV_LOG_FLOW_OUT("");

    return;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Init_Op(void)
 *
 * @param - none
 *
 * @return OS_STATUS
 *
 * @brief Initialize PMU before collection
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Init_Op (
    PVOID param
)
{
    U32        this_cpu = CONTROL_THIS_CPU();
    U32        dev_idx  = core_to_dev_map[this_cpu];
    DEV_CONFIG pcfg     = LWPMU_DEVICE_pcfg(&devices[dev_idx]);
    DISPATCH   dispatch = LWPMU_DEVICE_dispatch(&devices[dev_idx]);

    SEP_DRV_LOG_TRACE_IN("");

    if (pcfg && DEV_CONFIG_num_events(pcfg) && dispatch != NULL && dispatch->init != NULL) {
        dispatch->init(&dev_idx);
    }

    SEP_DRV_LOG_TRACE_OUT("");
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Init_PMU(void)
 *
 * @param - none
 *
 * @return - OS_STATUS
 *
 * @brief Initialize the PMU and the driver state in preparation for data collection.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Init_PMU (
    IOCTL_ARGS args
)
{
    OS_STATUS  status                 = OS_SUCCESS;
    S32        i;
    DISPATCH   dispatch_unc;
    DEV_UNC_CONFIG pcfg_unc;
    EVENT_CONFIG ec;

    SEP_DRV_LOG_FLOW_IN("");

    if (args->len_usr_to_drv != sizeof(U32) || args->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    if (copy_from_user(&emon_buffer_size, args->buf_usr_to_drv, sizeof(U32))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure");
        return OS_FAULT;
    }

    if (!drv_cfg) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("drv_cfg not set!");
        return OS_FAULT;
    }

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Discarded: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        ec = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[core_to_dev_map[i]]);
        if (!ec) {
            continue;
        }
        CPU_STATE_trigger_count(&pcb[i])     = EVENT_CONFIG_em_factor(ec);
        CPU_STATE_trigger_event_num(&pcb[i]) = EVENT_CONFIG_em_event_num(ec);
    }

    // set cur_device's total groups to max groups of all devices
    max_groups_unc = 0;
    for (i = num_core_devs; i < num_devices; i++) {
        if (max_groups_unc < LWPMU_DEVICE_em_groups_count(&devices[i])) {
            max_groups_unc = LWPMU_DEVICE_em_groups_count(&devices[i]);
        }
    }
    // now go back and up total groups for all devices
    if (DRV_CONFIG_emon_mode(drv_cfg) == TRUE) {
        for (i = num_core_devs; i < num_devices; i++) {
            if (LWPMU_DEVICE_em_groups_count(&devices[i]) < max_groups_unc) {
                LWPMU_DEVICE_em_groups_count(&devices[i]) = max_groups_unc;
            }
        }
    }

    // allocate uncore read buffers
    if (unc_buf_init && !DRV_CONFIG_emon_mode(drv_cfg)) {
        lwpmudrv_Allocate_Uncore_Buffer();
    }

    // must be done after pcb is created and before PMU is first written to
    CONTROL_Invoke_Parallel(lwpmudrv_Init_Op, NULL);

    for (i = num_core_devs; i < num_devices; i++) {
        pcfg_unc     = (DEV_UNC_CONFIG)LWPMU_DEVICE_pcfg(&devices[i]);
        dispatch_unc = LWPMU_DEVICE_dispatch(&devices[i]);
        if (pcfg_unc && DEV_UNC_CONFIG_num_events(pcfg_unc) && dispatch_unc && dispatch_unc->init) {
            dispatch_unc->init((VOID *)&i);
        }
    }

    // Allocate PEBS buffers
    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
        PEBS_Allocate();
    }
    //
    // Transfer the data into the PMU registers
    //
    CONTROL_Invoke_Parallel(lwpmudrv_Write_Op, NULL);

    SEP_DRV_LOG_TRACE("lwpmudrv_Init_PMU: IOCTL_Init_PMU - finished initial Write.");

    if (DRV_CONFIG_counting_mode(drv_cfg) == TRUE || DRV_CONFIG_emon_mode(drv_cfg) == TRUE) {
        prev_counter_size     = emon_buffer_size;

        if (!read_counter_info) {
            read_counter_info = CONTROL_Allocate_Memory(emon_buffer_size);
            if (!read_counter_info) {
                SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
                return OS_NO_MEM;
            }
        }
        if (!prev_counter_data) {
            prev_counter_data = CONTROL_Allocate_Memory(emon_buffer_size);
            if (!prev_counter_data) {
                read_counter_info = CONTROL_Free_Memory(read_counter_info);
                SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
                return OS_NO_MEM;
            }
        }
        if (!emon_buffer_driver_helper) {
            // allocate size = size of EMON_BUFFER_DRIVER_HELPER_NODE + the number of entries in core_index_to_thread_offset_map, which is num of cpu
            emon_buffer_driver_helper = CONTROL_Allocate_Memory(sizeof(EMON_BUFFER_DRIVER_HELPER_NODE) + sizeof(U32) * GLOBAL_STATE_num_cpus(driver_state));
            if (!emon_buffer_driver_helper) {
                SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
                return OS_NO_MEM;
            }
        }
    }

    return status;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Read_MSR(pvoid param)
 *
 * @param param - pointer to the buffer to store the MSR counts
 *
 * @return none
 *
 * @brief
 * @brief  Read the U64 value at address in buf_drv_to_usr and
 * @brief  write the result into buf_usr_to_drv.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Read_MSR (
    PVOID param
)
{
#if defined(DRV_IA32) || defined(DRV_EM64T)
    U32       this_cpu;
    MSR_DATA  this_node;
    U32       reg_num;
    S32       status = 0;

    SEP_DRV_LOG_TRACE_IN("");

    preempt_disable();
    this_cpu  = CONTROL_THIS_CPU();
    this_node = &msr_data[this_cpu];
    reg_num   = MSR_DATA_addr(this_node);

    if (reg_num == 0) {
        preempt_enable();
        SEP_DRV_LOG_ERROR_TRACE_OUT("Error: tried to read MSR 0");
        return;
    }

    MSR_DATA_value(this_node) = SYS_Read_MSR(reg_num);
    MSR_DATA_status(this_node) = status;
    preempt_enable();

    SEP_DRV_LOG_TRACE_OUT("");
#endif

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Get_Perf_Capab(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read the U64 value at IA32_PERF_CAPABILITIES and write
 * @brief  the result into buf_usr_to_drv.
 * @brief  Returns OS_SUCCESS if the read across all cores succeed,
 * @brief  otherwise OS_FAULT.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Perf_Capab (
    IOCTL_ARGS    arg
)
{
    U64            *val;
    S32             i;
    MSR_DATA        node;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments!");
        return OS_FAULT;
    }

    val     = (U64 *)arg->buf_drv_to_usr;

    msr_data = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(MSR_DATA_NODE));
    if (!msr_data) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
        return OS_NO_MEM;
    }

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        node                = &msr_data[i];
        MSR_DATA_addr(node) = IA32_PERF_CAPABILITIES;
    }

    // To do hetero - needs to be invoked for each core type
    CONTROL_Invoke_Parallel(lwpmudrv_Read_MSR, (VOID *)(size_t)0);

    /* copy values to arg array? */
    if (arg->len_drv_to_usr < GLOBAL_STATE_num_cpus(driver_state)*sizeof(U64)) {
        msr_data = CONTROL_Free_Memory(msr_data);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Not enough memory allocated in output buffer!");
        return OS_FAULT;
    }
    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        node = &msr_data[i];
        if (MSR_DATA_status(node)) {
            SEP_DRV_LOG_ERROR("Reading the MSR 0x%x failed", IA32_PERF_CAPABILITIES);
        }
        if (copy_to_user(&val[i], (U64*)&MSR_DATA_value(node), sizeof(U64))) {
            msr_data = CONTROL_Free_Memory(msr_data);
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
            return OS_FAULT;
        }
    }

    msr_data = CONTROL_Free_Memory(msr_data);

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Read_Allowlist_MSR_All_Cores(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read the U64 value at address into buf_drv_to_usr and write
 * @brief  the result into buf_usr_to_drv.
 * @brief  Returns OS_SUCCESS if the read across all cores succeed,
 * @brief  otherwise OS_FAULT.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Read_Allowlist_MSR_All_Cores (
    IOCTL_ARGS    arg
)
{
    static DRV_MSR_OP  req_msr_ops = NULL;
    S32                i;
    MSR_DATA           node;
    OS_STATUS          status = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_usr_to_drv is NULL)!");
        return OS_INVALID;
    }
    if (arg->len_usr_to_drv != sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_usr_to_drv value)!");
        return OS_INVALID;
    }
    if (arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_drv_to_usr is NULL)!");
        return OS_INVALID;
    }
    if (arg->len_drv_to_usr != sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_drv_to_usr value)!");
        return OS_INVALID;
    }

    req_msr_ops = (DRV_MSR_OP)CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(DRV_MSR_OP_NODE));
    if (!req_msr_ops) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
        return OS_NO_MEM;
    }

    if (copy_from_user(req_msr_ops, arg->buf_usr_to_drv, sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state))) {
        SEP_DRV_LOG_ERROR("Memory copy failure!");
        status = OS_FAULT;
        goto clean_return;
    }

    if (!msr_data) {
        msr_data = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(MSR_DATA_NODE));
        if (!msr_data) {
            SEP_DRV_LOG_ERROR("Memory allocation failure!");
            status = OS_NO_MEM;
            goto clean_return;
        }
    }

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        if (!PMU_LIST_Check_MSR(req_msr_ops[i].reg_id)) {
            SEP_DRV_LOG_ERROR("Invalid MSR information! 0x%x", req_msr_ops[i].reg_id);
            status = OS_INVALID;
            goto clean_return;
        }
        else {
            SEP_DRV_LOG_TRACE("Verified the MSR 0x%x", msr_ops[i].reg_id);
        }

        node                = &msr_data[i];
        MSR_DATA_addr(node) = req_msr_ops[i].reg_id;
    }

    CONTROL_Invoke_Parallel(lwpmudrv_Read_MSR, (VOID *)(size_t)0);

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        node = &msr_data[i];
        req_msr_ops[i].reg_read_val = MSR_DATA_value(node);
        req_msr_ops[i].status = MSR_DATA_status(node);
    }

    if (copy_to_user(arg->buf_drv_to_usr, req_msr_ops, sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state))) {
        SEP_DRV_LOG_ERROR("Memory copy failure!");
        status = OS_FAULT;
        goto clean_return;
    }

clean_return:
    req_msr_ops = CONTROL_Free_Memory(req_msr_ops);
    msr_data = CONTROL_Free_Memory(msr_data);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Write_MSR(pvoid iaram)
 *
 * @param param - pointer to array containing the MSR address and the value to be written
 *
 * @return none
 *
 * @brief
 * @brief  Read the U64 value at address in buf_drv_to_usr and
 * @brief  write the result into buf_usr_to_drv.
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Write_MSR (
    PVOID param
)
{
#if defined(DRV_IA32) || defined(DRV_EM64T)
    U32       this_cpu;
    MSR_DATA  this_node;
    U32       reg_num;
    U64       val;
    S32       status = 0;

    SEP_DRV_LOG_TRACE_IN("");

    preempt_disable();
    this_cpu  = CONTROL_THIS_CPU();
    this_node = &msr_data[this_cpu];
    reg_num   = (U32)MSR_DATA_addr(this_node);
    val       = (U64)MSR_DATA_value(this_node);
    // don't attempt to write MSR 0
    if (reg_num == 0) {
        preempt_enable();
        SEP_DRV_LOG_ERROR_TRACE_OUT("Error: tried to write MSR 0!");
        return;
    }

    SYS_Write_MSR(reg_num, val);
    MSR_DATA_status(this_node) = status;
    preempt_enable();

    SEP_DRV_LOG_TRACE_OUT("");
#endif

    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Write_Allowlist_MSR_All_Cores(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read the U64 value at address into buf_drv_to_usr and write
 * @brief  the result into buf_usr_to_drv.
 * @brief  Returns OS_SUCCESS if the write across all cores succeed,
 * @brief  otherwise OS_FAULT.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Write_Allowlist_MSR_All_Cores (
    IOCTL_ARGS    arg
)
{
    static DRV_MSR_OP  req_msr_ops = NULL;
    S32                i;
    MSR_DATA           node;
    OS_STATUS          status = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_usr_to_drv is NULL)!");
        return OS_INVALID;
    }
    if (arg->len_usr_to_drv != sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_usr_to_drv value)!");
        return OS_INVALID;
    }
    if (arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_drv_to_usr is NULL)!");
        return OS_INVALID;
    }
    if (arg->len_drv_to_usr != sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_drv_to_usr value)!");
        return OS_INVALID;
    }

    req_msr_ops = (DRV_MSR_OP)CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(DRV_MSR_OP_NODE));
    if (!req_msr_ops) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure!");
        return OS_NO_MEM;
    }

    if (copy_from_user(req_msr_ops, arg->buf_usr_to_drv, sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state))) {
        SEP_DRV_LOG_ERROR("Memory copy failure!");
        status = OS_FAULT;
        goto clean_return;
    }

    if (!msr_data) {
        msr_data = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state)*sizeof(MSR_DATA_NODE));
        if (!msr_data) {
            SEP_DRV_LOG_ERROR("Memory allocation failure!");
            status = OS_NO_MEM;
            goto clean_return;
        }
    }

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        if (!PMU_LIST_Check_MSR(req_msr_ops[i].reg_id)) {
            SEP_DRV_LOG_ERROR("Invalid MSR information! 0x%x", req_msr_ops[i].reg_id);
            status = OS_INVALID;
            goto clean_return;
        }
        else {
            SEP_DRV_LOG_TRACE("Verified the MSR 0x%x", msr_ops[i].reg_id);
        }

        node                 = &msr_data[i];
        MSR_DATA_addr(node)  = req_msr_ops[i].reg_id;
        MSR_DATA_value(node) = req_msr_ops[i].reg_write_val;
    }

    CONTROL_Invoke_Parallel(lwpmudrv_Write_MSR, (VOID *)(size_t)0);

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        node                 = &msr_data[i];
        req_msr_ops[i].status = MSR_DATA_status(node);
    }

    if (copy_to_user(arg->buf_drv_to_usr, req_msr_ops, sizeof(DRV_MSR_OP_NODE)*GLOBAL_STATE_num_cpus(driver_state))) {
        SEP_DRV_LOG_ERROR("Memory copy failure!");
        status = OS_FAULT;
        goto clean_return;
    }

clean_return:
    req_msr_ops = CONTROL_Free_Memory(req_msr_ops);
    msr_data = CONTROL_Free_Memory(msr_data);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Read_Counters(IOCTL_ARG arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read all the programmed data counters and accumulate them
 * @brief  into a single buffer.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Read_Counters (
    IOCTL_ARGS    arg
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_drv_to_usr == 0 || arg->buf_drv_to_usr == NULL ) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_SUCCESS;
    }
    //
    // Transfer the data in the PMU registers to the output buffer
    //
    if (!read_counter_info) {
        read_counter_info = CONTROL_Allocate_Memory(arg->len_drv_to_usr);
        if (!read_counter_info) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure");
            return OS_NO_MEM;
        }
    }
    if (!prev_counter_data) {
        prev_counter_data = CONTROL_Allocate_Memory(arg->len_drv_to_usr);
        if (!prev_counter_data) {
            read_counter_info = CONTROL_Free_Memory(read_counter_info);
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure");
            return OS_NO_MEM;
        }
    }
    memset(read_counter_info, 0, arg->len_drv_to_usr);

    CONTROL_Invoke_Parallel(lwpmudrv_Read_Data_Op, (VOID *)read_counter_info);

    if (copy_to_user(arg->buf_drv_to_usr, read_counter_info, arg->len_drv_to_usr)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure");
        return OS_FAULT;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Read_Counters_And_Switch_Group(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read / Store the counters and switch to the next valid group.
 *
 * <I>Special Notes</I>
 *     This routine is called from the user mode code to handle the multiple group
 *     situation.  10 distinct steps are taken:
 *     Step 1:  Save the previous cpu's tsc
 *     Step 2:  Read the current cpu's tsc
 *     Step 3:  Pause the counting PMUs
 *     Step 4:  Calculate the difference between the current and previous cpu's tsc
 *     Step 5:  Save original buffer ptr and copy cpu's tsc into the output buffer
 *              Increment the buffer position by number of CPU
 *     Step 6:  Read the currently programmed data PMUs and copy the data into the output buffer
 *              Restore the original buffer ptr.
 *     Step 7:  Write the new group to the PMU
 *     Step 8:  Write the new group to the PMU
 *     Step 9:  Read the current cpu's tsc for next collection (so read MSRs time not included in report)
 *     Step 10: Resume the counting PMUs
 */
static OS_STATUS
lwpmudrv_Read_Counters_And_Switch_Group (
    IOCTL_ARGS arg
)
{
    U64           *p_buffer             = NULL;
    char          *orig_r_buf_ptr       = NULL;
    U64            orig_r_buf_len       = 0;
    OS_STATUS      status               = OS_SUCCESS;
    DRV_BOOL       enter_in_pause_state = 0;
    U32            i                    = 0;
    U64           *tmp                  = NULL;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->buf_drv_to_usr == NULL || arg->len_drv_to_usr == 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }
    if (!DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_RUNNING | STATE_BIT_PAUSED)) {
        SEP_DRV_LOG_FLOW_OUT("'Success'/error: driver state is not RUNNING or PAUSED!");
        return OS_SUCCESS;
    }
    if (GET_DRIVER_STATE() == DRV_STATE_PAUSED) {
        enter_in_pause_state = 1;
    }
    // step 1
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        // swap cpu_tsc and prev_cpu_tsc, so that cpu_tsc is saved in prev_cpu_tsc.
        tmp          = prev_cpu_tsc;
        prev_cpu_tsc = cpu_tsc;
        cpu_tsc      = tmp;
    }
    else {
        prev_cpu_tsc[0] = cpu_tsc[0];
    }

    // step 2
    // if per_cpu_tsc is not defined, read cpu0's tsc and save in var cpu_tsc[0]
    // if per_cpu_tsc is defined, read all cpu's tsc and save in var cpu_tsc by lwpmudrv_Fill_TSC_Info
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg) || DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg)) {
        atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));
        CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);
    }
    else {
        CONTROL_Invoke_Cpu (0, lwpmudrv_Read_Specific_TSC, &cpu_tsc[0]);
    }

    // step 3
    if (!enter_in_pause_state) {
        status = lwpmudrv_Pause();
    }

    // step 4
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
            diff_cpu_tsc[i] = cpu_tsc[i] - prev_cpu_tsc[i];
        }
    }
    else {
        diff_cpu_tsc[0] = cpu_tsc[0] - prev_cpu_tsc[0];
    }

    // step 5
    orig_r_buf_ptr = arg->buf_drv_to_usr;
    orig_r_buf_len = arg->len_drv_to_usr;

    if (DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg)) {
        temp_tsc = cpu_tsc;
    } else {
        temp_tsc = diff_cpu_tsc;
    }

    if (copy_to_user(arg->buf_drv_to_usr, temp_tsc, GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    p_buffer   = (U64 *)(arg->buf_drv_to_usr);
    p_buffer   += GLOBAL_STATE_num_cpus(driver_state);
    arg->buf_drv_to_usr = (char *)p_buffer;
    arg->len_drv_to_usr -= GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64);

    // step 6
    status = lwpmudrv_Read_Counters(arg);

    arg->buf_drv_to_usr = orig_r_buf_ptr;
    arg->len_drv_to_usr = orig_r_buf_len;

    // step 7
    // for each processor, increment its current group number
    lwpmudrv_Switch_To_Next_Group();

    // step 8
    CONTROL_Invoke_Parallel(lwpmudrv_Write_Op, NULL);

    // step 9
    // if per_cpu_tsc is defined, read all cpu's tsc and save in cpu_tsc for next run
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));
        CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);
    }
    else {
        CONTROL_Invoke_Cpu (0, lwpmudrv_Read_Specific_TSC, &cpu_tsc[0]);
    }

    // step 10
    if (!enter_in_pause_state) {
        status = lwpmudrv_Resume();
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/*
 * @fn  static OS_STATUS lwpmudrv_Read_And_Reset_Counters(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Read the current value of the counters, and reset them all to 0.
 *
 * <I>Special Notes</I>
 *     This routine is called from the user mode code to handle the multiple group
 *     situation. 9 distinct steps are taken:
 *     Step 1: Save the previous cpu's tsc
 *     Step 2: Read the current cpu's tsc
 *     Step 3: Pause the counting PMUs
 *     Step 4: Calculate the difference between the current and previous cpu's tsc
 *     Step 5: Save original buffer ptr and copy cpu's tsc into the output buffer
 *             Increment the buffer position by number of CPU
 *     Step 6: Read the currently programmed data PMUs and copy the data into the output buffer
 *             Restore the original buffer ptr.
 *     Step 7: Write the new group to the PMU
 *     Step 8: Read the current cpu's tsc for next collection (so read MSRs time not included in report)
 *     Step 9: Resume the counting PMUs
 */
static OS_STATUS
lwpmudrv_Read_And_Reset_Counters (
    IOCTL_ARGS arg
)
{
    U64           *p_buffer             = NULL;
    char          *orig_r_buf_ptr       = NULL;
    U64            orig_r_buf_len       = 0;
    OS_STATUS      status               = OS_SUCCESS;
    DRV_BOOL       enter_in_pause_state = 0;
    U32            i                    = 0;
    U64           *tmp                  = NULL;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->buf_drv_to_usr == NULL || arg->len_drv_to_usr == 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        return OS_FAULT;
    }

    if (!DRIVER_STATE_IN(GET_DRIVER_STATE(), STATE_BIT_RUNNING | STATE_BIT_PAUSED)) {
        SEP_DRV_LOG_FLOW_OUT("'Success'/error: driver state is not RUNNING or PAUSED!");
        return OS_SUCCESS;
    }

    if (GET_DRIVER_STATE() == DRV_STATE_PAUSED) {
        enter_in_pause_state = 1;
    }

    // step 1
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        // swap cpu_tsc and prev_cpu_tsc, so that cpu_tsc is saved in prev_cpu_tsc.
        tmp          = prev_cpu_tsc;
        prev_cpu_tsc = cpu_tsc;
        cpu_tsc      = tmp;
    }
    else {
        prev_cpu_tsc[0] = cpu_tsc[0];
    }

    // step 2
    // if per_cpu_tsc is not defined, read cpu0's tsc and save in var cpu_tsc[0]
    // if per_cpu_tsc is defined, read all cpu's tsc and save in var cpu_tsc by lwpmudrv_Fill_TSC_Info
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg) || DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg)) {
        atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));
        CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);
    }
    else {
        CONTROL_Invoke_Cpu (0, lwpmudrv_Read_Specific_TSC, &cpu_tsc[0]);
    }

    // step 3
    // Counters should be frozen right after time stamped.
    if (!enter_in_pause_state) {
        status = lwpmudrv_Pause();
    }

    // step 4
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
            diff_cpu_tsc[i] = cpu_tsc[i] - prev_cpu_tsc[i];
        }
    }
    else {
        diff_cpu_tsc[0] = cpu_tsc[0] - prev_cpu_tsc[0];
    }

    // step 5
    orig_r_buf_ptr = arg->buf_drv_to_usr;
    orig_r_buf_len = arg->len_drv_to_usr;

    if (DRV_CONFIG_per_cpu_absolute_tsc(drv_cfg)) {
        temp_tsc = cpu_tsc;
    } else {
        temp_tsc = diff_cpu_tsc;
    }

    if (copy_to_user(arg->buf_drv_to_usr, temp_tsc, GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    p_buffer   = (U64 *)(arg->buf_drv_to_usr);
    p_buffer   += GLOBAL_STATE_num_cpus(driver_state);
    arg->buf_drv_to_usr = (char *)p_buffer;
    arg->len_drv_to_usr -= GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64);

    // step 6
    status = lwpmudrv_Read_Counters(arg);

    arg->buf_drv_to_usr = orig_r_buf_ptr;
    arg->len_drv_to_usr = orig_r_buf_len;

    // step 7
    CONTROL_Invoke_Parallel(lwpmudrv_Write_Op, NULL);

    // step 8
    // if per_cpu_tsc is defined, read all cpu's tsc and save in cpu_tsc for next run
    if (DRV_CONFIG_per_cpu_tsc(drv_cfg)) {
        atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));
        CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);
    }
    else {
        CONTROL_Invoke_Cpu (0, lwpmudrv_Read_Specific_TSC, &cpu_tsc[0]);
    }

    // step 9
    if (!enter_in_pause_state) {
        status = lwpmudrv_Resume();
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Set_Num_EM_Groups(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief Configure the event multiplexing group.
 *
 * <I>Special Notes</I>
 *     None
 */
static OS_STATUS
lwpmudrv_Set_EM_Config (
    IOCTL_ARGS arg
)
{
    EVENT_CONFIG ec;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: Driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }
    if (arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Set_num_em_groups got null pointer.");
        return OS_NO_MEM;
    }
    if (arg->len_usr_to_drv != sizeof(EVENT_CONFIG_NODE)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Unknown size of value passed to IOCTL_EM_CONFIG: %lld.", arg->len_usr_to_drv);
        return OS_NO_MEM;
    }
    LWPMU_DEVICE_ec(&devices[cur_device]) = CONTROL_Allocate_Memory(sizeof(EVENT_CONFIG_NODE));
    if (!LWPMU_DEVICE_ec(&devices[cur_device])) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for ec!");
        return OS_NO_MEM;
    }

    if (copy_from_user(LWPMU_DEVICE_ec(&devices[cur_device]), arg->buf_usr_to_drv, sizeof(EVENT_CONFIG_NODE))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure (event config)!");
        return OS_FAULT;
    }

    ec = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[cur_device]);
    LWPMU_DEVICE_PMU_register_data(&devices[cur_device]) = CONTROL_Allocate_Memory(EVENT_CONFIG_num_groups(ec) *
                                                                                   sizeof(VOID *));
    if (!LWPMU_DEVICE_PMU_register_data(&devices[cur_device])) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for PMU_register_data!");
        return OS_NO_MEM;
    }

    EVENTMUX_Initialize();

    SEP_DRV_LOG_FLOW_OUT("OS_SUCCESS.");
    return OS_SUCCESS;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Set_EM_Config_UNC(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Set the number of em groups in the global state node.
 * @brief  Also, copy the EVENT_CONFIG struct that has been passed in,
 * @brief  into a global location for now.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Set_EM_Config_UNC (
    IOCTL_ARGS arg
)
{
    EVENT_CONFIG    ec;
    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    // allocate memory
    LWPMU_DEVICE_ec(&devices[cur_device]) = CONTROL_Allocate_Memory(sizeof(EVENT_CONFIG_NODE));
    if (copy_from_user(LWPMU_DEVICE_ec(&devices[cur_device]), arg->buf_usr_to_drv, arg->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for LWPMU_device_ec!");
        return OS_FAULT;
    }
    // configure num_groups from ec of the specific device
    ec = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[cur_device]);

    SEP_DRV_LOG_TRACE("Num Groups UNCORE: %d.", EVENT_CONFIG_num_groups_unc(ec));

    LWPMU_DEVICE_PMU_register_data(&devices[cur_device]) = CONTROL_Allocate_Memory(EVENT_CONFIG_num_groups_unc(ec) *
                                                                                   sizeof(VOID *));
    if (!LWPMU_DEVICE_PMU_register_data(&devices[cur_device])) {
        LWPMU_DEVICE_ec(&devices[cur_device]) = CONTROL_Free_Memory(LWPMU_DEVICE_ec(&devices[cur_device]));
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for LWPMU_DEVICE_PMU_register_data");
        return OS_NO_MEM;
    }

    LWPMU_DEVICE_em_groups_count(&devices[cur_device]) = 0;

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}
#endif

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Configure_events(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Copies one group of events into kernel space at
 * @brief  PMU_register_data[em_groups_count].
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Configure_Events (
    IOCTL_ARGS arg
)
{
    U32              group_id;
    ECB              ecb;
    U32              em_groups_count;
    EVENT_CONFIG     ec;
    int              uncopied;
    U32              idx, reg_id;
    OS_STATUS        status = OS_SUCCESS;
    DRV_IOCTL_STATUS reg_check_status = NULL;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        return OS_IN_PROGRESS;
    }

    ec              = (EVENT_CONFIG)LWPMU_DEVICE_ec(&devices[cur_device]);
    em_groups_count = LWPMU_DEVICE_em_groups_count(&devices[cur_device]);

    if (em_groups_count >= EVENT_CONFIG_num_groups(ec)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: EM groups number exceeded initial configuration!");
        status = OS_SUCCESS;
        goto clean_return;
    }
    if (arg->buf_usr_to_drv == NULL || arg->len_usr_to_drv == 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        status = OS_FAULT;
        goto clean_return;
    }
    ecb = CONTROL_Allocate_Memory(arg->len_usr_to_drv);
    if (!ecb) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for ecb!");
        status = OS_NO_MEM;
        goto clean_return;
    }
    //
    // Make a copy of the data for global use.
    //
    uncopied = copy_from_user(ecb, arg->buf_usr_to_drv, arg->len_usr_to_drv);
    if (uncopied > 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for ecb data!");
        status = OS_FAULT;
        goto clean_return;
    }

    reg_check_status = CONTROL_Allocate_Memory(sizeof(DRV_IOCTL_STATUS_NODE));
    if (!reg_check_status) {
        SEP_DRV_LOG_ERROR("Memory allocation failure for reg_check_status");
        CONTROL_Free_Memory(ecb);
        status = OS_NO_MEM;
        goto clean_return;
    }

    // Validation check from PMU list
    for ((idx) = 0; (idx) < ECB_num_entries(ecb); (idx)++) {
        reg_id = ECB_entries_reg_id((ecb),(idx));
        if (reg_id == 0) {
            continue;
        }
        if (!PMU_LIST_Check_MSR(reg_id)) {
            SEP_DRV_LOG_ERROR("Invalid MSR information! 0x%x", reg_id);
            status = OS_INVALID;
            DRV_IOCTL_STATUS_drv_status(reg_check_status) = VT_INVALID_PROG_INFO;
            DRV_IOCTL_STATUS_reg_prog_type(reg_check_status) = PMU_REG_PROG_MSR;
            DRV_IOCTL_STATUS_reg_key1(reg_check_status) = reg_id;
            goto clean_return;
        }
        else {
            SEP_DRV_LOG_TRACE("Verified the msr 0x%x, idx=%u", reg_id, idx);
        }
    }

    group_id = ECB_group_id(ecb);

    if (group_id >= EVENT_CONFIG_num_groups(ec)) {
        CONTROL_Free_Memory(ecb);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Group_id is larger than total number of groups!");
        return OS_INVALID;
    }

    LWPMU_DEVICE_PMU_register_data(&devices[cur_device])[group_id] = ecb;
    LWPMU_DEVICE_em_groups_count(&devices[cur_device])             = group_id + 1;

clean_return:
    if (reg_check_status) {
        if (copy_to_user(arg->buf_drv_to_usr, reg_check_status, sizeof(DRV_IOCTL_STATUS_NODE))) {
            SEP_DRV_LOG_ERROR("Memory copy to user failure for reg_check_status data!");
        }
        reg_check_status = CONTROL_Free_Memory(reg_check_status);
    }

    if (status != OS_SUCCESS) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return status;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Configure_events_UNC(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Make a copy of the uncore registers that need to be programmed
 * @brief  for the next event set used for event multiplexing
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Configure_Events_UNC (
    IOCTL_ARGS arg
)
{
    VOID                 **PMU_register_data_unc;
    S32                    em_groups_count_unc;
    ECB                    ecb;
    EVENT_CONFIG           ec_unc;
    DEV_UNC_CONFIG         pcfg_unc;
    U32                    group_id = 0;
    ECB                    in_ecb   = NULL;
    PMU_MMIO_BAR_INFO_NODE primary;
    PMU_MMIO_BAR_INFO_NODE secondary;
    U32                    idx, reg_id;
    U32                    bar_idx;
    U32                    itr;
    MMIO_BAR_INFO          mmio_bar_list;
    OS_STATUS              status = OS_SUCCESS;
    DRV_IOCTL_STATUS       reg_check_status = NULL;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        status = OS_IN_PROGRESS;
        goto clean_return;
    }

    em_groups_count_unc = LWPMU_DEVICE_em_groups_count(&devices[cur_device]);
    PMU_register_data_unc = LWPMU_DEVICE_PMU_register_data(&devices[cur_device]);
    ec_unc                = LWPMU_DEVICE_ec(&devices[cur_device]);
    pcfg_unc              = LWPMU_DEVICE_pcfg(&devices[cur_device]);

    if (!pcfg_unc || !DEV_UNC_CONFIG_num_events(pcfg_unc) || ec_unc == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Pcfg_unc or ec_unc NULL!");
        status = OS_INVALID;
        goto clean_return;
    }

    if (em_groups_count_unc >= (S32)EVENT_CONFIG_num_groups_unc(ec_unc)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Uncore EM groups number exceeded initial configuration!");
        status = OS_SUCCESS;
        goto clean_return;
    }
    if (arg->buf_usr_to_drv == NULL || arg->len_usr_to_drv < sizeof(ECB_NODE)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments.");
        status = OS_INVALID;
        goto clean_return;
    }
    //       size is in len_usr_to_drv, data is pointed to by buf_usr_to_drv
    //
    in_ecb = CONTROL_Allocate_Memory(arg->len_usr_to_drv);
    if (!in_ecb) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for uncore ecb!");
        status = OS_NO_MEM;
        goto clean_return;
    }
    if (copy_from_user(in_ecb, arg->buf_usr_to_drv, arg->len_usr_to_drv)) {
        CONTROL_Free_Memory(in_ecb);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for uncore ecb data!");
        status = OS_FAULT;
        goto clean_return;
    }

    reg_check_status = CONTROL_Allocate_Memory(sizeof(DRV_IOCTL_STATUS_NODE));
    if (!reg_check_status) {
        SEP_DRV_LOG_ERROR("Memory allocation failure for reg_check_status");
        CONTROL_Free_Memory(in_ecb);
        status = OS_NO_MEM;
        goto clean_return;
    }

    // Validation check from PMU list
    for ((idx) = 0; (idx) < ECB_num_entries(in_ecb); (idx)++) {
        reg_id = ECB_entries_reg_id((in_ecb),(idx));
        if (reg_id == 0) {
            continue;
        }
        switch (ECB_entries_reg_prog_type((in_ecb),(idx))) {
            case PMU_REG_PROG_MSR:
                if (!PMU_LIST_Check_MSR(reg_id)) {
                    SEP_DRV_LOG_ERROR("Invalid MSR information! 0x%x", reg_id);
                    status = OS_INVALID;
                    DRV_IOCTL_STATUS_drv_status(reg_check_status) = VT_INVALID_PROG_INFO;
                    DRV_IOCTL_STATUS_reg_prog_type(reg_check_status) = PMU_REG_PROG_MSR;
                    DRV_IOCTL_STATUS_reg_key1(reg_check_status) = (U64)reg_id;
                    goto clean_return;
                }
                else {
                    SEP_DRV_LOG_TRACE("Verified the msr 0x%x", reg_id);
                }
                break;
            case PMU_REG_PROG_PCI:
                if (!PMU_LIST_Check_PCI((U8)ECB_entries_bus_no((in_ecb),(idx)),
                                        (U8)ECB_entries_dev_no((in_ecb),(idx)),
                                        (U8)ECB_entries_func_no((in_ecb),(idx)),
                                        reg_id)) {
                    SEP_DRV_LOG_ERROR("Invalid PCI information! B%d.D%d.F%d.O0x%x",
                                       ECB_entries_bus_no((in_ecb),(idx)),
                                       ECB_entries_dev_no((in_ecb),(idx)),
                                       ECB_entries_func_no((in_ecb),(idx)),
                                       reg_id);
                    status = OS_INVALID;
                    DRV_IOCTL_STATUS_drv_status(reg_check_status) = VT_INVALID_PROG_INFO;
                    DRV_IOCTL_STATUS_reg_prog_type(reg_check_status) = PMU_REG_PROG_PCI;
                    DRV_IOCTL_STATUS_bus(reg_check_status) = ECB_entries_bus_no((in_ecb),(idx));
                    DRV_IOCTL_STATUS_dev(reg_check_status) = ECB_entries_dev_no((in_ecb),(idx));
                    DRV_IOCTL_STATUS_func(reg_check_status) = ECB_entries_func_no((in_ecb),(idx));
                    DRV_IOCTL_STATUS_offset(reg_check_status) = reg_id;
                    goto clean_return;
                }
                else {
                    SEP_DRV_LOG_TRACE("Verified the PCI B%d.D%d.F%d.O0x%x",
                                       ECB_entries_bus_no((in_ecb),(idx)),
                                       ECB_entries_dev_no((in_ecb),(idx)),
                                       ECB_entries_func_no((in_ecb),(idx)),
                                       reg_id);
                }
                break;
            case PMU_REG_PROG_MMIO:
                memset(&primary, 0, sizeof(PMU_MMIO_BAR_INFO_NODE));
                memset(&secondary, 0, sizeof(PMU_MMIO_BAR_INFO_NODE));
                if (ECB_device_type(in_ecb) == DEVICE_UNC_SOCPERF) {
                    continue;
                }

                if (ECB_device_type(in_ecb) == DEVICE_PMEM_FC || ECB_device_type(in_ecb) == DEVICE_PMEM_MC) {
                    for (itr = 0; itr < num_packages; itr++) {
                        mmio_bar_list = &ECB_mmio_bar_list((in_ecb), (itr));
                        primary.u.s.bus = MMIO_BAR_INFO_bus_no(mmio_bar_list);
                        primary.u.s.dev = MMIO_BAR_INFO_dev_no(mmio_bar_list);
                        primary.u.s.func = MMIO_BAR_INFO_func_no(mmio_bar_list);
                        primary.u.s.offset = MMIO_BAR_INFO_main_bar_offset(mmio_bar_list);
                        primary.mask = MMIO_BAR_INFO_main_bar_mask(mmio_bar_list);
                        primary.shift = MMIO_BAR_INFO_main_bar_shift(mmio_bar_list);
                        if (!MMIO_BAR_INFO_secondary_bar_offset(mmio_bar_list)) {
                            primary.bar_prog_type = MMIO_SINGLE_BAR_TYPE;
                        }
                        else {
                            primary.bar_prog_type = MMIO_DUAL_BAR_TYPE;
                            secondary.bar_prog_type = MMIO_DUAL_BAR_TYPE;
                            secondary.u.s.bus = MMIO_BAR_INFO_bus_no(mmio_bar_list);
                            secondary.u.s.dev = MMIO_BAR_INFO_dev_no(mmio_bar_list);
                            secondary.u.s.func = MMIO_BAR_INFO_func_no(mmio_bar_list);
                            secondary.u.s.offset = MMIO_BAR_INFO_secondary_bar_offset(mmio_bar_list);
                            secondary.mask = MMIO_BAR_INFO_secondary_bar_mask(mmio_bar_list);
                            secondary.shift = MMIO_BAR_INFO_secondary_bar_shift(mmio_bar_list);
                        }
                        if (!PMU_LIST_Check_MMIO(primary, secondary, reg_id)) {
                            SEP_DRV_LOG_ERROR("Invalid MMIO information! Offset:0x%x, B%d.D%d.F%d.O0x%x, M0x%llx.S%d, Sec O0x%x, M0x%llx.S%d", \
                                                reg_id, \
                                                primary.u.s.bus, \
                                                primary.u.s.dev, \
                                                primary.u.s.func, \
                                                primary.u.s.offset, \
                                                primary.mask, \
                                                primary.shift, \
                                                secondary.u.s.offset, \
                                                secondary.mask, \
                                                secondary.shift);
                            status = OS_INVALID;
                            DRV_IOCTL_STATUS_drv_status(reg_check_status) = VT_INVALID_PROG_INFO;
                            DRV_IOCTL_STATUS_reg_prog_type(reg_check_status) = PMU_REG_PROG_MMIO;
                            DRV_IOCTL_STATUS_bus(reg_check_status) = primary.u.s.bus;
                            DRV_IOCTL_STATUS_dev(reg_check_status) = primary.u.s.dev;
                            DRV_IOCTL_STATUS_func(reg_check_status) = primary.u.s.func;
                            DRV_IOCTL_STATUS_offset(reg_check_status) = primary.u.s.offset;
                            DRV_IOCTL_STATUS_reg_key2(reg_check_status) = reg_id;
                            goto clean_return;
                        }
                        else {
                            SEP_DRV_LOG_TRACE("Verified the MMIO Offset:0x%x, B%d.D%d.F%d.O0x%x, M0x%llx.S%d, Sec O0x%x, M0x%llx.S%d", \
                                                reg_id, \
                                                primary.u.s.bus, \
                                                primary.u.s.dev, \
                                                primary.u.s.func, \
                                                primary.u.s.offset, \
                                                primary.mask, \
                                                primary.shift, \
                                                secondary.u.s.offset, \
                                                secondary.mask, \
                                                secondary.shift);
                        }
                    }
                }
                else {
                    bar_idx = ECB_entries_reg_bar_index((in_ecb), (idx));
                    mmio_bar_list = &ECB_mmio_bar_list((in_ecb), (bar_idx));
                    primary.u.s.bus = MMIO_BAR_INFO_bus_no(mmio_bar_list);
                    primary.u.s.dev = MMIO_BAR_INFO_dev_no(mmio_bar_list);
                    primary.u.s.func = MMIO_BAR_INFO_func_no(mmio_bar_list);
                    primary.u.s.offset = MMIO_BAR_INFO_main_bar_offset(mmio_bar_list);
                    primary.mask = MMIO_BAR_INFO_main_bar_mask(mmio_bar_list);
                    primary.shift = MMIO_BAR_INFO_main_bar_shift(mmio_bar_list);
                    if (!MMIO_BAR_INFO_secondary_bar_offset(mmio_bar_list)) {
                        primary.bar_prog_type = MMIO_SINGLE_BAR_TYPE;
                    }
                    else {
                        primary.bar_prog_type = MMIO_DUAL_BAR_TYPE;
                        secondary.bar_prog_type = MMIO_DUAL_BAR_TYPE;
                        secondary.u.s.bus = MMIO_BAR_INFO_bus_no(mmio_bar_list);
                        secondary.u.s.dev = MMIO_BAR_INFO_dev_no(mmio_bar_list);
                        secondary.u.s.func = MMIO_BAR_INFO_func_no(mmio_bar_list);
                        secondary.u.s.offset = MMIO_BAR_INFO_secondary_bar_offset(mmio_bar_list);
                        secondary.mask = MMIO_BAR_INFO_secondary_bar_mask(mmio_bar_list);
                        secondary.shift = MMIO_BAR_INFO_secondary_bar_shift(mmio_bar_list);
                    }
                    if (!PMU_LIST_Check_MMIO(primary, secondary, reg_id)) {
                        SEP_DRV_LOG_ERROR("Invalid MMIO information! Offset:0x%x, B%d.D%d.F%d.O0x%x, M0x%llx.S%d, Sec O0x%x, M0x%llx.S%d", \
                                            reg_id, \
                                            primary.u.s.bus, \
                                            primary.u.s.dev, \
                                            primary.u.s.func, \
                                            primary.u.s.offset, \
                                            primary.mask, \
                                            primary.shift, \
                                            secondary.u.s.offset, \
                                            secondary.mask, \
                                            secondary.shift);
                        status = OS_INVALID;
                        DRV_IOCTL_STATUS_drv_status(reg_check_status) = VT_INVALID_PROG_INFO;
                        DRV_IOCTL_STATUS_reg_prog_type(reg_check_status) = PMU_REG_PROG_MMIO;
                        DRV_IOCTL_STATUS_bus(reg_check_status) = primary.u.s.bus;
                        DRV_IOCTL_STATUS_dev(reg_check_status) = primary.u.s.dev;
                        DRV_IOCTL_STATUS_func(reg_check_status) = primary.u.s.func;
                        DRV_IOCTL_STATUS_offset(reg_check_status) = primary.u.s.offset;
                        DRV_IOCTL_STATUS_reg_key2(reg_check_status) = reg_id;
                        goto clean_return;
                    }
                    else {
                        SEP_DRV_LOG_TRACE("Verified the MMIO Offset:0x%x, B%d.D%d.F%d.O0x%x, M0x%llx.S%d, Sec O0x%x, M0x%llx.S%d", \
                                            reg_id, \
                                            primary.u.s.bus, \
                                            primary.u.s.dev, \
                                            primary.u.s.func, \
                                            primary.u.s.offset, \
                                            primary.mask, \
                                            primary.shift, \
                                            secondary.u.s.offset, \
                                            secondary.mask, \
                                            secondary.shift);
                    }
                }
                break;
            default:
                SEP_DRV_LOG_ERROR("Invalid reg_prog_type! %u, idx=%u",
                                   ECB_entries_reg_prog_type((in_ecb),(idx)), idx);

                status = OS_INVALID;
                goto clean_return;
        }
    }

    group_id = ECB_group_id(in_ecb);

    if (group_id >= EVENT_CONFIG_num_groups_unc(ec_unc)) {
        CONTROL_Free_Memory(in_ecb);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Group_id is larger than total number of groups!");
        status = OS_INVALID;
        goto clean_return;
    }

    PMU_register_data_unc[group_id] = in_ecb;
    // at this point, we know the number of uncore events for this device,
    // so allocate the results buffer per thread for uncore
    ecb = PMU_register_data_unc[group_id];
    if (ecb == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Encountered NULL ECB!");
        status = OS_INVALID;
        goto clean_return;
    }
    LWPMU_DEVICE_num_events(&devices[cur_device])      = ECB_num_events(ecb);
    LWPMU_DEVICE_em_groups_count(&devices[cur_device]) = group_id + 1;

clean_return:
    if (reg_check_status) {
        if (copy_to_user(arg->buf_drv_to_usr, reg_check_status, sizeof(DRV_IOCTL_STATUS_NODE))) {
            SEP_DRV_LOG_ERROR("Memory copy to user failure for reg_check_status data!");
        }
        reg_check_status = CONTROL_Free_Memory(reg_check_status);
    }

    if (status != OS_SUCCESS) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return status;
}
#endif

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Set_Sample_Descriptors(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 *
 * @return OS_STATUS
 *
 * @brief  Set the number of descriptor groups in the global state node.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Set_Sample_Descriptors (
    IOCTL_ARGS    arg
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }
    if (arg->len_usr_to_drv != sizeof(U32) || arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (Unknown size of Sample Descriptors!).");
        return OS_INVALID;
    }

    desc_count = 0;
    if (copy_from_user(&GLOBAL_STATE_num_descriptors(driver_state),
                       arg->buf_usr_to_drv,
                       sizeof(U32))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure");
        return OS_FAULT;
    }

    desc_data  = CONTROL_Allocate_Memory(GLOBAL_STATE_num_descriptors(driver_state) *
                                                sizeof(VOID *));
    if (desc_data == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for desc_data!");
        return OS_NO_MEM;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Configure_Descriptors(IOCTL_ARGS arg)
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 * @return OS_STATUS
 *
 * @brief Make a copy of the descriptors that need to be read in order
 * @brief to configure a sample record.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Configure_Descriptors (
    IOCTL_ARGS    arg
)
{
    int uncopied;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    if (desc_count >= GLOBAL_STATE_num_descriptors(driver_state)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Descriptor groups number exceeded initial configuration!");
        return OS_SUCCESS;
    }

    if (arg->len_usr_to_drv == 0 || arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arg value!");
        return OS_FAULT;
    }

    if (desc_data == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("NULL desc_data!");
        return OS_INVALID;
    }

    //
    // First things first: Make a copy of the data for global use.
    //
    desc_data[desc_count] = CONTROL_Allocate_Memory(arg->len_usr_to_drv);
    uncopied = copy_from_user(desc_data[desc_count], arg->buf_usr_to_drv, arg->len_usr_to_drv);
    if (uncopied > 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Unable to copy desc_data from user!");
        return OS_NO_MEM;
    }
    SEP_DRV_LOG_TRACE("Added descriptor # %d.", desc_count);
    desc_count++;

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_LBR_Info(IOCTL_ARGS arg)
 *
 *
 * @param arg - pointer to the IOCTL_ARGS structure
 * @return OS_STATUS
 *
 * @brief Make a copy of the LBR information that is passed in.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_LBR_Info (
    IOCTL_ARGS    arg
)
{
    OS_STATUS    status = OS_SUCCESS;
    LBR          lbr_list;
    U32          idx, reg_id;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        status = OS_IN_PROGRESS;
        goto clean_return;
    }

    if (DEV_CONFIG_collect_lbrs(cur_pcfg) == FALSE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("LBR capture has not been configured!");
        status = OS_INVALID;
        goto clean_return;
    }

    if (arg->len_usr_to_drv == 0 || arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments!");
        status = OS_FAULT;
        goto clean_return;
    }

    //
    // First things first: Make a copy of the data for global use.
    //

    LWPMU_DEVICE_lbr(&devices[cur_device]) = CONTROL_Allocate_Memory((int)arg->len_usr_to_drv);
    if (!LWPMU_DEVICE_lbr(&devices[cur_device])) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Memory allocation failure for lbr!");
        status = OS_NO_MEM;
        goto clean_return;
    }

    if (copy_from_user(LWPMU_DEVICE_lbr(&devices[cur_device]), arg->buf_usr_to_drv, arg->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for lbr struct!");
        status = OS_FAULT;
        goto clean_return;
    }

    lbr_list = LWPMU_DEVICE_lbr(&devices[cur_device]);
    if (lbr_list) {
        for (idx = 0; idx < LBR_num_entries(lbr_list); idx++) {
            reg_id = LBR_entries_reg_id(lbr_list, idx);
            if (reg_id == 0) {
                continue;
            }
            if (!PMU_LIST_Check_MSR(reg_id)) {
                LBR_entries_reg_id(lbr_list, idx) = 0;
                SEP_DRV_LOG_ERROR("Invalid MSR information! 0x%x", reg_id);
                status = OS_INVALID;
                goto clean_return;
            }
            else {
                SEP_DRV_LOG_TRACE("Verified the msr 0x%x\n", reg_id);
            }
        }
    }

clean_return:
    if (status != OS_SUCCESS) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return status;
}

#define CR4_PCE  0x00000100    //Performance-monitoring counter enable RDPMC
/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Set_CR4_PCE_Bit(PVOID param)
 *
 * @param param - dummy parameter
 *
 * @return NONE
 *
 * @brief Set CR4's PCE bit on the logical processor
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Set_CR4_PCE_Bit (
    PVOID  param
)
{
    U32 this_cpu;
#if defined(DRV_IA32)
    U32 prev_CR4_value = 0;

    SEP_DRV_LOG_TRACE_IN("");

    // remember if RDPMC bit previously set
    // and then enabled it
    __asm__("movl %%cr4,%%eax\n\t"
            "movl %%eax,%0\n"
            "orl  %1,%%eax\n\t"
            "movl %%eax,%%cr4\n"
            :"=r" (prev_CR4_value)
            :"i" (CR4_PCE)
            :"eax");
#endif
#if defined(DRV_EM64T)
    U64 prev_CR4_value = 0;
    // remember if RDPMC bit previously set
    // and then enabled it
    __asm__("movq %%cr4,%%rax\n\t"
            "movq %%rax,%0\n\t"
            "orq  %1,%%rax\n\t"
            "movq %%rax,%%cr4\n"
            :"=r" (prev_CR4_value)
            :"i" (CR4_PCE)
            :"rax");
#endif
    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();

    // if bit RDPMC bit was set before,
    // set flag for when we clear it
    if (prev_CR4_value & CR4_PCE) {
        prev_set_CR4[this_cpu] = 1;
    }

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn static void lwpmudrv_Clear_CR4_PCE_Bit(PVOID param)
 *
 * @param param - dummy parameter
 *
 * @return NONE
 *
 * @brief ClearSet CR4's PCE bit on the logical processor
 *
 * <I>Special Notes</I>
 */
static VOID
lwpmudrv_Clear_CR4_PCE_Bit (
    PVOID  param
)
{
    U32 this_cpu;

    SEP_DRV_LOG_TRACE_IN("");

    preempt_disable();
    this_cpu = CONTROL_THIS_CPU();
    preempt_enable();

    // only clear the CR4 bit if it wasn't set
    // before we started
    if (prev_set_CR4 && !prev_set_CR4[this_cpu]) {
#if defined(DRV_IA32)
        __asm__("movl %%cr4,%%eax\n\t"
                "andl %0,%%eax\n\t"
                "movl %%eax,%%cr4\n"
                :
                :"irg" (~CR4_PCE)
                :"eax");
#endif
#if defined(DRV_EM64T)
        __asm__("movq %%cr4,%%rax\n\t"
                "andq %0,%%rax\n\t"
                "movq %%rax,%%cr4\n"
                :
                :"irg" (~CR4_PCE)
                :"rax");
#endif
    }

    SEP_DRV_LOG_TRACE_OUT("");
    return;
}
#endif

/* ------------------------------------------------------------------------- */
/*!
 * @fn static OS_STATUS lwpmudrv_Start(void)
 *
 * @param none
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the LWPMU_IOCTL_START call.
 * @brief  Set up the OS hooks for process/thread/load notifications.
 * @brief  Write the initial set of MSRs.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Start (
    VOID
)
{
    OS_STATUS  status       = OS_SUCCESS;
    U32        start_success;

    SEP_DRV_LOG_FLOW_IN("");

    /*
     * To Do: Check for state == STATE_IDLE and only then enable sampling
     */
    start_success = CHANGE_DRIVER_STATE(STATE_BIT_IDLE, DRV_STATE_RUNNING);
    if (!start_success) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    if (drv_cfg == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("NULL drv_cfg!");
        return OS_INVALID;
    }

    atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));

#if defined(DRV_IA32) || defined(DRV_EM64T)
    prev_set_CR4 = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U8));
    CONTROL_Invoke_Parallel(lwpmudrv_Set_CR4_PCE_Bit, (PVOID)(size_t)0);
#endif // (DRV_IA32 || DRV_EM64T)
    CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);

    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
        lapic_enable_pmc();
    }
    if (DRV_CONFIG_start_paused(drv_cfg)) {
        SEP_DRV_LOG_INIT("Start in Paused mode.");
        CHANGE_DRIVER_STATE(STATE_BIT_RUNNING, DRV_STATE_PAUSED);
    }
    else {
        CONTROL_Invoke_Parallel(lwpmudrv_Resume_Op, NULL);
        EVENTMUX_Start();
    }
    if (unc_buf_init) {
        lwpmudrv_Uncore_Start_Timer();
    }
    else if (DRV_CONFIG_emon_timer_interval(drv_cfg)) {
    	lwpmudrv_Emon_Start_Timer();
    }

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/*
 * @fn lwpmudrv_Prepare_Stop();
 *
 * @param        NONE
 * @return       OS_STATUS
 *
 * @brief  Local function that handles the DRV_OPERATION_STOP call.
 * @brief  Cleans up the interrupt handler.
 */
static OS_STATUS
lwpmudrv_Prepare_Stop (
    VOID
)
{
    S32 i;
    S32 done                = FALSE;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_TERMINATING) {
        if (!CHANGE_DRIVER_STATE(STATE_BIT_RUNNING | STATE_BIT_PAUSED, DRV_STATE_PREPARE_STOP)) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Unexpected driver state.");
            return OS_INVALID;
        }
    }
    else {
        SEP_DRV_LOG_WARNING("Abnormal termination path.");
    }

    if (drv_cfg == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("drv_cfg is NULL!");
        return OS_SUCCESS;
    }

    if (pcb) {
        for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
            CPU_STATE_accept_interrupt(&pcb[i]) = 0;
        }
        while (!done) {
            done = TRUE;
            for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
                if (atomic_load_acq_int(&CPU_STATE_in_interrupt(&pcb[i]))) {
                    done = FALSE;
                }
            }
        }
        CONTROL_Invoke_Parallel(lwpmudrv_Pause_Op, NULL);
        if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
			lapic_disable_pmc();
		}
        SEP_DRV_LOG_TRACE("lwpmudrv_Prepare_Stop: Outside of all interrupts.");

        if (unc_buf_init) {
            lwpmudrv_Uncore_Stop_Timer();
        }
        else if (DRV_CONFIG_emon_timer_interval(drv_cfg)) {
    	    lwpmudrv_Emon_Stop_Timer();
        }
    }

    if (drv_cfg == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("drv_cfg is NULL!");
        return OS_SUCCESS;
    }

    /*
     * Clean up all the control registers
     */
    CONTROL_Invoke_Parallel(lwpmudrv_Cleanup_Op, (VOID *)NULL);
    SEP_DRV_LOG_TRACE("Cleanup finished.");

    CONTROL_Invoke_Parallel(lwpmudrv_Clear_CR4_PCE_Bit, (VOID *)(size_t)0);
    prev_set_CR4 = CONTROL_Free_Memory(prev_set_CR4);

    SEP_DRV_LOG_FLOW_OUT("Success.");
    return OS_SUCCESS;
}

/*
 * @fn lwpmudrv_Finish_Stop();
 *
 * @param  NONE
 * @return OS_STATUS
 *
 * @brief  Local function that handles the DRV_OPERATION_STOP call.
 * @brief  Cleans up the interrupt handler.
 */
static OS_STATUS
lwpmudrv_Finish_Stop (
    VOID
)
{
    U32        current_state = GET_DRIVER_STATE();
    OS_STATUS  status        = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_TERMINATING) {
        if (!CHANGE_DRIVER_STATE(STATE_BIT_PREPARE_STOP, DRV_STATE_STOPPED)) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Unexpected driver state!");
            return OS_FAULT;
        }
    }
    else {
        SEP_DRV_LOG_WARNING("Abnormal termination path.");
    }

    if (drv_cfg == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("drv_cfg is NULL!");
        return OS_INVALID;
    }

    if (DRV_CONFIG_counting_mode(drv_cfg) == FALSE) {
        FBSD_OS_Uninstall_Hooks();
        if (current_state != DRV_STATE_TERMINATING && module_buf) {

            /*
             *  Make sure that the module buffers are not deallocated and that the module flush
             *  thread has not been terminated.
             */
            status = FBSD_OS_Enum_Process_Modules(TRUE);
            OUTPUT_Flush();
        }

        /*
         * Clean up the interrupt handler via the IDT
         */
        CPUMON_Remove_Cpuhooks();
        PEBS_Destroy();
        EVENTMUX_Destroy();
    }
    else if (DRV_CONFIG_emon_timer_interval(drv_cfg)) {
    	OUTPUT_Flush_EMON();
    }

    read_counter_info         = CONTROL_Free_Memory(read_counter_info);
    prev_counter_data         = CONTROL_Free_Memory(prev_counter_data);
    emon_buffer_driver_helper = CONTROL_Free_Memory(emon_buffer_driver_helper);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_Normalized_TSC(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Return the current value of the normalized TSC.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Normalized_TSC (
    IOCTL_ARGS arg
)
{
    U64    tsc          = 0;
    size_t size_to_copy = sizeof(U64);

    SEP_DRV_LOG_TRACE_IN("");

    if (arg->len_drv_to_usr < size_to_copy || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("Invalid arguments!");
        return OS_FAULT;
    }

    preempt_disable();
    UTILITY_Read_TSC(&tsc);
    tsc -= TSC_SKEW(CONTROL_THIS_CPU());
    preempt_enable();

    if (copy_to_user(arg->buf_drv_to_usr, (VOID *)&tsc, size_to_copy)) {
        SEP_DRV_LOG_ERROR_TRACE_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    SEP_DRV_LOG_TRACE_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_Num_Cores(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Quickly return the (total) number of cpus in the system.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Num_Cores (
    IOCTL_ARGS   arg
)
{
    OS_STATUS status = OS_SUCCESS;
    int num = GLOBAL_STATE_num_cpus(driver_state);

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_drv_to_usr < sizeof(int) || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    SEP_DRV_LOG_TRACE("lwpmudrv_Get_Num_Cores: Num_Cores is %d, buf_usr_to_drv is %p.", num, arg->buf_drv_to_usr);
    status = put_user(num, (int*)arg->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Set_CPU_Mask(PVOID buf_drv_to_usr, U32 len_drv_to_usr)
 *
 * @param buf_drv_to_usr      - pointer to the CPU mask buffer
 * @param len_drv_to_usr - size of the CPU mask buffer
 *
 * @return OS_STATUS
 *
 * @brief  process the CPU mask as requested by the user
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Set_CPU_Mask (
    PVOID         buf_drv_to_usr,
    size_t        len_drv_to_usr
)
{
    U32     cpu_count     = 0;

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    if (len_drv_to_usr == 0 || buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("In_buf_len == 0 or buf_drv_to_usr is NULL!");
        return OS_INVALID;
    }

    cpu_mask_bits = CONTROL_Allocate_Memory((int)len_drv_to_usr);
    if (!cpu_mask_bits) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for cpu_mask_bits!");
        return OS_NO_MEM;
    }

    if (copy_from_user(cpu_mask_bits, (S8*)buf_drv_to_usr, (int)len_drv_to_usr)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    for (cpu_count = 0; cpu_count < (U32)GLOBAL_STATE_num_cpus(driver_state); cpu_count++) {
        CPU_STATE_accept_interrupt(&pcb[cpu_count]) = cpu_mask_bits[cpu_count] ? 1 : 0;
        CPU_STATE_initial_mask(&pcb[cpu_count    ]) = cpu_mask_bits[cpu_count] ? 1 : 0;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}


#if defined(DRV_IA32) || defined(DRV_EM64T)
/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_KERNEL_CS(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Return the value of the Kernel symbol KERNEL_CS.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_KERNEL_CS (
    IOCTL_ARGS   arg
)
{
    OS_STATUS status = OS_SUCCESS;
    int       num    = GCODE_SEL;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_drv_to_usr < sizeof(int) || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    SEP_DRV_LOG_TRACE("lwpmudrv_Get_KERNEL_CS is %d, buf_usr_to_drv is %p.", num, arg->buf_drv_to_usr);
    status = put_user(num, (int*)arg->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}
#endif

/*
 * @fn lwpmudrv_Set_UID
 *
 * @param     IN   arg      - pointer to the output buffer
 * @return   OS_STATUS
 *
 * @brief  Receive the value of the UID of the collector process.
 */
static OS_STATUS
lwpmudrv_Set_UID (
    IOCTL_ARGS   arg
)
{
    OS_STATUS status = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_usr_to_drv < sizeof(uid_t) || arg->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }

    status = get_user(uid, (int*)arg->buf_usr_to_drv);
    SEP_DRV_LOG_TRACE("lwpmudrv_Set_UID is %d.", uid);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_TSC_Skew_Info(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 * @brief  Return the current value of the TSC skew data
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_TSC_Skew_Info (
    IOCTL_ARGS arg
)
{
    S64    *skew_array;
    size_t  skew_array_len;
    S32     i;

    SEP_DRV_LOG_FLOW_IN("");

    skew_array_len = GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64);

    if (arg->len_drv_to_usr < skew_array_len || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Input buffer too small or NULL!");
        return OS_FAULT;
    }

    if (GET_DRIVER_STATE() != DRV_STATE_STOPPED) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: Driver is not STOPPED!");
        return OS_IN_PROGRESS;
    }

    SEP_DRV_LOG_TRACE("lwpmudrv_Get_TSC_Skew_Info dispatched with len_drv_to_usr=%lld.", arg->len_drv_to_usr);

    skew_array = CONTROL_Allocate_Memory(skew_array_len);
    if (skew_array == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for skew_array!");
        return OS_NO_MEM;
    }

    for (i = 0; i < GLOBAL_STATE_num_cpus(driver_state); i++) {
        skew_array[i] = TSC_SKEW(i);
    }

    if (copy_to_user(arg->buf_drv_to_usr, skew_array, skew_array_len)) {
        skew_array = CONTROL_Free_Memory(skew_array);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for skew_array!");
        return OS_FAULT;
    }

    skew_array = CONTROL_Free_Memory(skew_array);

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Collect_Sys_Config(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Local function that handles the COLLECT_SYS_CONFIG call.
 * @brief  Builds and collects the SYS_INFO data needed.
 * @brief  Writes the result into the argument.
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Collect_Sys_Config (
    IOCTL_ARGS   arg
)
{
    OS_STATUS  status = OS_SUCCESS;
    int num;

    SEP_DRV_LOG_FLOW_IN("");

    num = SYS_INFO_Build();

    if (arg->len_drv_to_usr < sizeof(int) || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    SEP_DRV_LOG_TRACE("lwpmudrv_Collect_Sys_Config: size of sys info is %d.", num);
    status = put_user(num, (int*)arg->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Sys_Config(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Return the current value of the normalized TSC.
 *
 * @brief  Transfers the VTSA_SYS_INFO data back to the abstraction layer.
 * @brief  The buf_usr_to_drv should have enough space to handle the transfer.
 */
static OS_STATUS
lwpmudrv_Sys_Config (
    IOCTL_ARGS   arg
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_drv_to_usr == 0 || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    SYS_INFO_Transfer(arg->buf_drv_to_usr, arg->len_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Samp_Read_Num_Of_Core_Counters(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief  Read memory mapped i/o physical location
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Samp_Read_Num_Of_Core_Counters (
    IOCTL_ARGS   arg
)
{
#if defined(DRV_IA32) || defined(DRV_EM64T)
    U64           rax, rbx, rcx, rdx,num_basic_functions;
    U32           val    = 0;
    OS_STATUS     status = OS_SUCCESS;

    SEP_DRV_LOG_FLOW_IN("");

    if (arg->len_drv_to_usr == 0 || arg->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_FAULT;
    }

    UTILITY_Read_Cpuid(0x0,&num_basic_functions,&rbx, &rcx, &rdx);

    if (num_basic_functions >= 0xA) {
         UTILITY_Read_Cpuid(0xA,&rax,&rbx, &rcx, &rdx);
         val    = ((U32)(rax >> 8)) & 0xFF;
    }
    status = put_user(val, (int*)arg->buf_drv_to_usr);
    SEP_DRV_LOG_TRACE("num of counter is %d.",val);
    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
#else
    SEP_DRV_LOG_FLOW_IN("");
    SEP_DRV_LOG_FLOW_OUT("Dummy success.");
    return OS_SUCCESS;
#endif
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_Platform_Info(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief       Reads the MSR_PLATFORM_INFO register if present
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Platform_Info (
    IOCTL_ARGS args
)
{
    U32                    size          = sizeof(DRV_PLATFORM_INFO_NODE);
    OS_STATUS              status        = OS_SUCCESS;
    DRV_PLATFORM_INFO      platform_data = NULL;
    U32                   *dispatch_ids  = NULL;
    DISPATCH               dispatch_ptr  = NULL;
    U32                    i             = 0;
    U32                    num_entries   = args->len_usr_to_drv/sizeof(U32); // # dispatch ids to process

    SEP_DRV_LOG_FLOW_IN("");

    if (!platform_data) {
        platform_data = CONTROL_Allocate_Memory(sizeof(DRV_PLATFORM_INFO_NODE));
        if (!platform_data) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for platform_data!");
            return OS_NO_MEM;
        }
    }

    memset(platform_data, 0, sizeof(DRV_PLATFORM_INFO_NODE));
    if (args->len_usr_to_drv > 0 && args->buf_usr_to_drv != NULL) {
        dispatch_ids = CONTROL_Allocate_Memory(args->len_usr_to_drv);
        if (!dispatch_ids) {
            platform_data = CONTROL_Free_Memory(platform_data);
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for dispatch_ids!");
            return OS_NO_MEM;
        }

        status = copy_from_user(dispatch_ids, args->buf_usr_to_drv, args->len_usr_to_drv);
        if (status) {
            platform_data = CONTROL_Free_Memory(platform_data);
            dispatch_ids = CONTROL_Free_Memory(dispatch_ids);
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for dispatch_ids!");
            return status;
        }
        for (i = 0; i < num_entries; i++) {
            if (dispatch_ids[i] > 0) {
                dispatch_ptr = UTILITY_Configure_CPU(dispatch_ids[i]);
                if (dispatch_ptr &&
                    dispatch_ptr->platform_info) {
                    dispatch_ptr->platform_info((PVOID)platform_data);
                }
            }
        }
        dispatch_ids = CONTROL_Free_Memory(dispatch_ids);
    }
    else if (devices) {
        dispatch_ptr = LWPMU_DEVICE_dispatch(&devices[0]);  //placeholder, needs to be fixed
        if (dispatch_ptr && dispatch_ptr->platform_info) {
            dispatch_ptr->platform_info((PVOID)platform_data);
        }
    }

    if (args->len_drv_to_usr < size || args->buf_drv_to_usr == NULL) {
        platform_data = CONTROL_Free_Memory(platform_data);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments!");
        return OS_FAULT;
    }

    status        = copy_to_user(args->buf_drv_to_usr, platform_data, size);
    platform_data = CONTROL_Free_Memory(platform_data);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          void lwpmudrv_Setup_Cpu_Topology (value)
 *
 * @brief       Sets up the per CPU state structures
 *
 * @param       IOCTL_ARGS args
 *
 * @return      OS_STATUS
 *
 * <I>Special Notes:</I>
 *              This function was added to support abstract dll creation.
 */
static OS_STATUS
lwpmudrv_Setup_Cpu_Topology (
    IOCTL_ARGS args
)
{
    S32               cpu_num;
    S32               iter;
    DRV_TOPOLOGY_INFO drv_topology, dt;

    SEP_DRV_LOG_FLOW_IN("");

    if (GET_DRIVER_STATE() != DRV_STATE_IDLE) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Skipped: driver state is not IDLE!");
        return OS_IN_PROGRESS;
    }
    if (args->len_usr_to_drv == 0 || args->buf_usr_to_drv == NULL || pcb == NULL ) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Topology information has been misconfigured!");
        return OS_INVALID;
    }

    drv_topology = CONTROL_Allocate_Memory(args->len_usr_to_drv);
    if (drv_topology == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for drv_topology!");
        return OS_NO_MEM;
    }

    if (copy_from_user(drv_topology, (DRV_TOPOLOGY_INFO)(args->buf_usr_to_drv), args->len_usr_to_drv)) {
        drv_topology = CONTROL_Free_Memory(drv_topology);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for drv_topology!");
        return OS_FAULT;
    }
    /*
     *   Topology Initializations
     */
    num_packages = 0;
    for (iter = 0; iter < GLOBAL_STATE_num_cpus(driver_state); iter++) {
        dt                                         = &drv_topology[iter];
        cpu_num                                    = DRV_TOPOLOGY_INFO_cpu_number(dt);
        CPU_STATE_socket_master(&pcb[cpu_num])     = DRV_TOPOLOGY_INFO_socket_master(dt);
        num_packages                              += CPU_STATE_socket_master(&pcb[cpu_num]);
        CPU_STATE_core_master(&pcb[cpu_num])       = DRV_TOPOLOGY_INFO_core_master(dt);
        CPU_STATE_thr_master(&pcb[cpu_num])        = DRV_TOPOLOGY_INFO_thr_master(dt);
        CPU_STATE_core_type(&pcb[cpu_num])         = DRV_TOPOLOGY_INFO_cpu_core_type(dt);
        CPU_STATE_system_master(&pcb[cpu_num])     = (iter)? 0 : 1;
        CPU_STATE_core_model_id(&pcb[cpu_num])     = 0;
        SEP_DRV_LOG_TRACE("cpu %d sm = %d cm = %d tm = %d.",
                  cpu_num,
                  CPU_STATE_socket_master(&pcb[cpu_num]),
                  CPU_STATE_core_master(&pcb[cpu_num]),
                  CPU_STATE_thr_master(&pcb[cpu_num]));
    }
    drv_topology = CONTROL_Free_Memory(drv_topology);

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_Num_Samples(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief       Returns the number of samples collected during the current
 * @brief       sampling run
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Get_Num_Samples (
    IOCTL_ARGS args
)
{
    S32               cpu_num;
    U64               samples = 0;
    OS_STATUS         status;

    SEP_DRV_LOG_FLOW_IN("");

    if (pcb == NULL) {
        SEP_DRV_LOG_ERROR("PCB was not initialized.");
        return OS_FAULT;
    }

    if (args->len_drv_to_usr == 0 || args->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Topology information has been misconfigured!");
        return OS_NO_MEM;
    }

    for (cpu_num = 0; cpu_num < GLOBAL_STATE_num_cpus(driver_state); cpu_num++) {
        samples += CPU_STATE_num_samples(&pcb[cpu_num]);

        SEP_DRV_LOG_TRACE("Samples for cpu %d = %lld.",
                        cpu_num,
                        CPU_STATE_num_samples(&pcb[cpu_num]));
    }
    SEP_DRV_LOG_TRACE("Total number of samples %lld.", samples);
    status = put_user(samples, (U64*)args->buf_drv_to_usr);

    SEP_DRV_LOG_FLOW_OUT("Return value: %d", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static OS_STATUS lwpmudrv_Get_Num_Samples(IOCTL_ARGS arg)
 *
 * @param arg - Pointer to the IOCTL structure
 *
 * @return OS_STATUS
 *
 * @brief       Dummy function to support backward compatibility with old driver
 *
 * <I>Special Notes</I>
 */
static OS_STATUS
lwpmudrv_Set_Device_Num_Units (
    IOCTL_ARGS args
)
{
    SEP_DRV_LOG_FLOW_IN("");
    SEP_DRV_LOG_FLOW_OUT("Success [but did not do anything]");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Set_Uncore_Topology_Info_And_Scan
 *
 * @brief       Reads the MSR_PLATFORM_INFO register if present
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Set_Uncore_Topology_Info_And_Scan (
    IOCTL_ARGS args
)
{
    SEP_DRV_LOG_FLOW_IN("");
    SEP_DRV_LOG_FLOW_OUT("Success [but did not do anything]");
    return OS_SUCCESS;
}
/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Get_Uncore_Topology
 *
 * @brief       Reads the MSR_PLATFORM_INFO register if present
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Get_Uncore_Topology (
    IOCTL_ARGS args
)
{
    U32                               dev;
    static UNCORE_TOPOLOGY_INFO_NODE  req_uncore_topology;

    SEP_DRV_LOG_FLOW_IN("");

    if (args->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_usr_to_drv is NULL)!");
        return OS_FAULT;
    }
    if (args->len_usr_to_drv != sizeof(UNCORE_TOPOLOGY_INFO_NODE)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_usr_to_drv value)!");
        return OS_FAULT;
    }

    memset((char *)&req_uncore_topology, 0, sizeof(UNCORE_TOPOLOGY_INFO_NODE));
    if (copy_from_user(&req_uncore_topology, args->buf_usr_to_drv, args->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    for (dev = 0; dev < MAX_DEVICES; dev++) {
        // skip if user does not require to scan this device
        if (!UNCORE_TOPOLOGY_INFO_device_scan(&req_uncore_topology, dev)) {
            continue;
        }
        // skip if this device has been discovered
        if (UNCORE_TOPOLOGY_INFO_device_scan(&uncore_topology, dev)) {
            continue;
        }
        memcpy((U8 *)&(UNCORE_TOPOLOGY_INFO_device(&uncore_topology, dev)),
               (U8 *)&(UNCORE_TOPOLOGY_INFO_device(&req_uncore_topology, dev)),
               sizeof(UNCORE_PCIDEV_NODE));
        UNC_COMMON_PCI_Scan_For_Uncore((VOID*)&dev, dev, NULL);
    }

    if (copy_to_user(args->buf_drv_to_usr, &uncore_topology, args->len_drv_to_usr)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Get_Platform_Topology
 *
 * @brief       Reads the MSR or PCI PLATFORM_INFO register if present
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Get_Platform_Topology (
    IOCTL_ARGS args
)
{
    U32                                 dev;
    U32                                 num_topology_devices = 0;

    SEP_DRV_LOG_FLOW_IN("");

    if (args->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_usr_to_drv is NULL)!");
        return OS_INVALID;
    }
    if (args->len_usr_to_drv != sizeof(PLATFORM_TOPOLOGY_PROG_NODE)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_usr_to_drv value)!");
        return OS_INVALID;
    }

    memset((char *)&req_platform_topology_prog_node, 0, sizeof(PLATFORM_TOPOLOGY_PROG_NODE));
    if (copy_from_user(&req_platform_topology_prog_node, args->buf_usr_to_drv, args->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for req_platform_topology_prog_node!");
        return OS_FAULT;
    }

    num_topology_devices = PLATFORM_TOPOLOGY_PROG_num_devices(&req_platform_topology_prog_node);
    for (dev = 0; dev < num_topology_devices; dev++) {
        //skip if we have populated the register values already
        if (PLATFORM_TOPOLOGY_PROG_EXT_topology_device_prog_valid(&platform_topology_prog_node, dev)) {
            continue;
        }
        memcpy((U8 *)&(PLATFORM_TOPOLOGY_PROG_EXT_topology_device(&platform_topology_prog_node, dev)),
               (U8 *)&(PLATFORM_TOPOLOGY_PROG_EXT_topology_device(&req_platform_topology_prog_node, dev)),
               sizeof(PLATFORM_TOPOLOGY_DISCOVERY_NODE_EXT));
        UNC_COMMON_Get_Platform_Topology(dev);
    }

    if (copy_to_user(args->buf_drv_to_usr, &platform_topology_prog_node, args->len_drv_to_usr)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for platform_topology_prog_node!");
        return OS_FAULT;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Set_Emon_Buffer_Driver_Helper
 *
 * @brief       Setup EMON buffer driver helper
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Set_Emon_Buffer_Driver_Helper (
    IOCTL_ARGS args
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (args->len_usr_to_drv == 0 || args->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Error: Invalid arguments.");
        return OS_INVALID;
    }

    if (!emon_buffer_driver_helper) {
        emon_buffer_driver_helper = CONTROL_Allocate_Memory(args->len_usr_to_drv);
        if (emon_buffer_driver_helper == NULL) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for emon_buffer_driver_helper!");
            return OS_NO_MEM;
        }
    }

    if (copy_from_user(emon_buffer_driver_helper,
                       args->buf_usr_to_drv,
                       args->len_usr_to_drv)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure for device num units!");
        return OS_FAULT;
    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Get_Driver_log
 *
 * @brief       Dumps the driver log
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Get_Driver_Log (
    IOCTL_ARGS args
)
{
    SEP_DRV_LOG_FLOW_IN("");

    if (args->buf_drv_to_usr == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_drv_to_usr is NULL)!");
        return OS_INVALID;
    }
    if (args->len_drv_to_usr < sizeof(*DRV_LOG())) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_drv_to_usr value)!");
        return OS_INVALID;
    }

    if (copy_to_user(args->buf_drv_to_usr, DRV_LOG(), sizeof(*DRV_LOG()))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory copy failure!");
        return OS_FAULT;
    }

    SEP_DRV_LOG_DISAMBIGUATE(); // keeps the driver log's footprint unique (has the highest disambiguator field)

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Control_Driver_log
 *
 * @brief       Sets or/and gets the driver log's configuration
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Control_Driver_Log (
    IOCTL_ARGS args
)
{
    DRV_LOG_CONTROL_NODE log_control;
    U32                  i;

    SEP_DRV_LOG_FLOW_IN("");

    if (args->buf_usr_to_drv == NULL) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (buf_usr_to_drv is NULL)!");
        return OS_INVALID;
    }
    if (args->len_usr_to_drv < sizeof(log_control)) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Invalid arguments (unexpected len_usr_to_drv value)!");
        return OS_INVALID;
    }

    if (copy_from_user(&log_control, args->buf_usr_to_drv, sizeof(log_control))) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("IOCTL args copy failure!");
        return OS_FAULT;
    }

    if (DRV_LOG_CONTROL_command(&log_control) == DRV_LOG_CONTROL_COMMAND_ADJUST_VERBOSITY) {
        for (i = 0; i < DRV_NB_LOG_CATEGORIES; i++) {
            if (DRV_LOG_CONTROL_verbosities(&log_control)[i] == LOG_VERBOSITY_UNSET) {
                SEP_DRV_LOG_TRACE("Current verbosity mask for '%s' is 0x%x",
                    (UTILITY_Log_Category_Strings()[i]),
                    ((U32) DRV_LOG_VERBOSITY(i)));
                DRV_LOG_CONTROL_verbosities(&log_control)[i] = DRV_LOG_VERBOSITY(i);
            }
            else if (DRV_LOG_CONTROL_verbosities(&log_control)[i] == LOG_VERBOSITY_DEFAULT) {
                U32 verbosity;
                switch (i) {
                    case DRV_LOG_CATEGORY_LOAD:
                        verbosity = DRV_LOG_DEFAULT_LOAD_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_INIT:
                        verbosity = DRV_LOG_DEFAULT_INIT_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_DETECTION:
                        verbosity = DRV_LOG_DEFAULT_DETECTION_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_ERROR:
                        verbosity = DRV_LOG_DEFAULT_ERROR_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_STATE_CHANGE:
                        verbosity = DRV_LOG_DEFAULT_STATE_CHANGE_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_MARK:
                        verbosity = DRV_LOG_DEFAULT_MARK_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_DEBUG:
                        verbosity = DRV_LOG_DEFAULT_DEBUG_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_FLOW:
                        verbosity = DRV_LOG_DEFAULT_FLOW_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_ALLOC:
                        verbosity = DRV_LOG_DEFAULT_ALLOC_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_INTERRUPT:
                        verbosity = DRV_LOG_DEFAULT_INTERRUPT_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_TRACE:
                        verbosity = DRV_LOG_DEFAULT_TRACE_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_REGISTER:
                        verbosity = DRV_LOG_DEFAULT_REGISTER_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_NOTIFICATION:
                        verbosity = DRV_LOG_DEFAULT_NOTIFICATION_VERBOSITY;
                        break;
                    case DRV_LOG_CATEGORY_WARNING:
                        verbosity = DRV_LOG_DEFAULT_WARNING_VERBOSITY;
                        break;

                    default:
                        SEP_DRV_LOG_ERROR("Unspecified category '%s' when resetting to default!", UTILITY_Log_Category_Strings()[i]);
                        verbosity = LOG_VERBOSITY_NONE;
                        break;
                }
                SEP_DRV_LOG_INIT("Resetting verbosity mask for '%s' from 0x%x to 0x%x.",
                    UTILITY_Log_Category_Strings()[i],
                    (U32) DRV_LOG_VERBOSITY(i),
                    verbosity);
                DRV_LOG_VERBOSITY(i)                         = verbosity;
                DRV_LOG_CONTROL_verbosities(&log_control)[i] = verbosity;
            }
            else {
                SEP_DRV_LOG_INIT("Changing verbosity mask for '%s' from 0x%x to 0x%x.",
                    UTILITY_Log_Category_Strings()[i],
                    (U32) DRV_LOG_VERBOSITY(i),
                    (U32) DRV_LOG_CONTROL_verbosities(&log_control)[i]);
                DRV_LOG_VERBOSITY(i) = DRV_LOG_CONTROL_verbosities(&log_control)[i];
            }
        }

        for (; i < DRV_MAX_NB_LOG_CATEGORIES; i++) {
            DRV_LOG_CONTROL_verbosities(&log_control)[i] = LOG_VERBOSITY_UNSET;
        }

        if (copy_to_user(args->buf_drv_to_usr, &log_control, sizeof(log_control))) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("IOCTL args copy failure!");
            return OS_FAULT;
        }
    }
    else if (DRV_LOG_CONTROL_command(&log_control) == DRV_LOG_CONTROL_COMMAND_MARK) {
        DRV_LOG_CONTROL_message(&log_control)[DRV_LOG_CONTROL_MAX_DATA_SIZE - 1] = 0;
        SEP_DRV_LOG_MARK("Mark: '%s'.", DRV_LOG_CONTROL_message(&log_control));
    }
    else if (DRV_LOG_CONTROL_command(&log_control) == DRV_LOG_CONTROL_COMMAND_QUERY_SIZE) {
        DRV_LOG_CONTROL_log_size(&log_control) = sizeof(*DRV_LOG());
        SEP_DRV_LOG_TRACE("Driver log size is %u bytes.", DRV_LOG_CONTROL_log_size(&log_control));
        if (copy_to_user(args->buf_drv_to_usr, &log_control, sizeof(log_control))) {
            SEP_DRV_LOG_ERROR_FLOW_OUT("IOCTL args copy failure!");
            return OS_FAULT;
        }
    }
    else if (DRV_LOG_CONTROL_command(&log_control) == DRV_LOG_CONTROL_COMMAND_BENCHMARK) {
        U32 nb_iterations = *(U32*)&DRV_LOG_CONTROL_message(&log_control);

        SEP_DRV_LOG_INIT_IN("Starting benchmark (%u iterations)...", nb_iterations);
        for (i = 0; i < nb_iterations; i++) {
            (void) i;
        }
        SEP_DRV_LOG_INIT_OUT("Benchmark complete (%u/%u iterations).", i, nb_iterations);

    }

    SEP_DRV_LOG_FLOW_OUT("Success");
    return OS_SUCCESS;
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn          U64 lwpmudrv_Get_Drv_Setup_Info
 *
 * @brief       Get numerous information of driver
 *
 * @param arg   Pointer to the IOCTL structure
 *
 * @return      status
 *
 * <I>Special Notes:</I>
 *              <NONE>
 */
static OS_STATUS
lwpmudrv_Get_Drv_Setup_Info (
    IOCTL_ARGS args
)
{
#define VMM_VENDOR_STR_LEN 12
    OS_STATUS   result              = OS_SUCCESS;
    U32         pebs_unavailable    = 0;
    U64         rax, rbx, rcx, rdx;
    S8          vmm_vendor_name[VMM_VENDOR_STR_LEN+1];
    S8         *VMM_VMWARE_STR   = "VMwareVMware";
    S8         *VMM_KVM_STR      = "KVMKVMKVM\0\0\0";
    S8         *VMM_MSHYPERV_STR = "Microsoft Hv";

    SEP_DRV_LOG_FLOW_IN("");

    memset((char *)&req_drv_setup_info, 0, sizeof(DRV_SETUP_INFO_NODE));

    UTILITY_Read_Cpuid(0x1, &rax, &rbx, &rcx, &rdx);

    if ((rcx >> 31) & 0x1) {
        UTILITY_Read_Cpuid(0x40000000, &rax, &rbx, &rcx, &rdx);
        memcpy(vmm_vendor_name, &rbx, 4);
        memcpy(vmm_vendor_name+4, &rcx, 4);
        memcpy(vmm_vendor_name+8, &rdx, 4);
        memcpy(vmm_vendor_name+12, "\0", 1);

        DRV_SETUP_INFO_vmm_mode(&req_drv_setup_info)   = 1;
        DRV_SETUP_INFO_vmm_guest_vm(&req_drv_setup_info) = 1;
        DRV_SETUP_INFO_vmm_vendor(&req_drv_setup_info) = DRV_VMM_UNKNOWN;

        if (!strncmp(vmm_vendor_name, VMM_VMWARE_STR, VMM_VENDOR_STR_LEN)) {
            DRV_SETUP_INFO_vmm_vendor(&req_drv_setup_info) = DRV_VMM_VMWARE;
        }
        else if (!strncmp(vmm_vendor_name, VMM_KVM_STR, VMM_VENDOR_STR_LEN)) {
            DRV_SETUP_INFO_vmm_vendor(&req_drv_setup_info) = DRV_VMM_KVM;
        }
        else if (!strncmp(vmm_vendor_name, VMM_MSHYPERV_STR, VMM_VENDOR_STR_LEN)) {
            DRV_SETUP_INFO_vmm_vendor(&req_drv_setup_info) = DRV_VMM_HYPERV;
        }
    }

    rcx = 0;
    UTILITY_Read_Cpuid(0x23, &rax, &rbx, &rcx, &rdx);
    if ((rax >> 5) & 0x1) {
        DRV_SETUP_INFO_pebs_accessible(&req_drv_setup_info) = 1;
    } else {
        pebs_unavailable = (SYS_Read_MSR(IA32_MISC_ENABLE) >> 12) & 0x1;
        if (!pebs_unavailable) {
            DRV_SETUP_INFO_pebs_accessible(&req_drv_setup_info) = 1;
        }
    }

    DRV_SETUP_INFO_pebs_ignored_by_pti(&req_drv_setup_info) = 1;

    if (allowlist_index != -1) {
        DRV_SETUP_INFO_register_allowlist_detected(&req_drv_setup_info) = 1;
    }

    DRV_SETUP_INFO_drv_type(&req_drv_setup_info) = drv_type;

    copy_to_user((PVOID)(args->buf_drv_to_usr), (void*)&req_drv_setup_info, sizeof(DRV_SETUP_INFO_NODE));
    args->len_drv_to_usr = sizeof(DRV_SETUP_INFO_NODE);

    SEP_DRV_LOG_FLOW_OUT("Res: %u.", result);

    return result;
}


/*******************************************************************************
 *  External Driver functions
 *      These functions are registered into the file operations table that
 *      controls this device.
 *      Open, Close, Read, Write, Release
 *******************************************************************************/

/* ------------------------------------------------------------------------- */
/*!
 * @fn  extern IOCTL_OP_TYPE lwpmu_Device_Control(IOCTL_USE_NODE, filp, cmd, arg)
 *
 * @param   IOCTL_USE_INODE       - Used for pre 2.6.32 kernels
 * @param   struct   file   *filp - file pointer
 * @param   unsigned int     cmd  - IOCTL command
 * @param   unsigned long    arg  - args to the IOCTL command
 *
 * @return OS_STATUS
 *
 * @brief  Worker function that handles IOCTL requests from the user mode.
 *
 * <I>Special Notes</I>
 */
static int
lwpmu_Device_Control (
    struct cdev     *cdev,
    u_long           cmd,
    caddr_t          arg,
    int              flag,
    struct thread   *td
)
{
    int              status = OS_SUCCESS;
    IOCTL_ARGS_NODE  local_args;

    if (arg) {
        memcpy(&local_args, (IOCTL_ARGS)arg, sizeof(IOCTL_ARGS_NODE));
    }

    SEP_DRV_LOG_TRACE("type: %d, subcommand: %d.",(U32)_IOC_TYPE(cmd),(U32)_IOC_NR(cmd));

    if (IOCGROUP(cmd) != LWPMU_IOC_MAGIC) {
        return OS_ILLEGAL_IOCTL;
    }

    cmd = (U32)_IOC_NR(cmd);

    if (cmd ==  DRV_OPERATION_GET_DRIVER_STATE) {
        SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_DRIVER_STATE.");
        status = lwpmudrv_Get_Driver_State(&local_args);
        return status;
    }
    if (cmd == DRV_OPERATION_GET_DRIVER_LOG) {
        SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_DRIVER_LOG.");
        status = lwpmudrv_Get_Driver_Log(&local_args);
        SEP_DRV_LOG_TRACE_OUT("Return value for command %u: %d", cmd, status);
        return status;
    }
    if (cmd == DRV_OPERATION_CONTROL_DRIVER_LOG) {
        SEP_DRV_LOG_TRACE("DRV_OPERATION_CONTROL_DRIVER_LOG.");
        status = lwpmudrv_Control_Driver_Log(&local_args);
        SEP_DRV_LOG_TRACE_OUT("Return value for command %u: %d", cmd, status);
        return status;
    }
    if (GET_DRIVER_STATE() == DRV_STATE_PREPARE_STOP) {
        SEP_DRV_LOG_TRACE("skipping ioctl -- processing stop.");
        return status;
    }

    MTX_LOCK(&ioctl_lock);
    UTILITY_Driver_Set_Active_Ioctl(cmd);
    switch (cmd) {

       /*
        * Common IOCTL commands
        */

        case DRV_OPERATION_VERSION:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_VERSION.");
            status = lwpmudrv_Version(&local_args);
            break;

        case DRV_OPERATION_RESERVE:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_RESERVE.");
            status = lwpmudrv_Reserve(&local_args);
            break;

        case DRV_OPERATION_INIT_DRIVER:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_INIT_DRIVER.");
            status = lwpmudrv_Initialize_Driver(local_args.buf_usr_to_drv, local_args.len_usr_to_drv);
            break;

        case DRV_OPERATION_INIT:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_INIT.");
            status = lwpmudrv_Initialize(local_args.buf_usr_to_drv, local_args.len_usr_to_drv);
            break;

        case DRV_OPERATION_INIT_PMU:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_INIT_PMU.");
            status = lwpmudrv_Init_PMU(&local_args);
            break;

        case DRV_OPERATION_SET_CPU_MASK:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_CPU_MASK.");
            status = lwpmudrv_Set_CPU_Mask(local_args.buf_usr_to_drv, local_args.len_usr_to_drv);
            break;

        case DRV_OPERATION_START:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_START.");
            status = lwpmudrv_Start();
            break;

        case DRV_OPERATION_STOP:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_STOP.");
            status = lwpmudrv_Prepare_Stop();
            UTILITY_Driver_Set_Active_Ioctl(0);
            MTX_UNLOCK(&ioctl_lock);

            MTX_LOCK(&ioctl_lock);
            UTILITY_Driver_Set_Active_Ioctl(cmd);
            if (GET_DRIVER_STATE() == DRV_STATE_PREPARE_STOP) {
                status = lwpmudrv_Finish_Stop();
            }
            break;

        case DRV_OPERATION_PAUSE:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_PAUSE.");
            status = lwpmudrv_Pause();
            break;

        case DRV_OPERATION_RESUME:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_RESUME.");
            status = lwpmudrv_Resume();
            break;

        case DRV_OPERATION_EM_GROUPS:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_EM_GROUPS.");
            status = lwpmudrv_Set_EM_Config(&local_args);
            break;

        case DRV_OPERATION_EM_CONFIG_NEXT:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_EM_CONFIG_NEXT.");
            status = lwpmudrv_Configure_Events(&local_args);
            break;

        case DRV_OPERATION_NUM_DESCRIPTOR:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_NUM_DESCRIPTOR.");
            status = lwpmudrv_Set_Sample_Descriptors(&local_args);
            break;

        case DRV_OPERATION_DESC_NEXT:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_DESC_NEXT.");
            status = lwpmudrv_Configure_Descriptors(&local_args);
            break;

        case DRV_OPERATION_GET_NORMALIZED_TSC:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_NORMALIZED_TSC.");
            status = lwpmudrv_Get_Normalized_TSC(&local_args);
            break;

        case DRV_OPERATION_GET_NORMALIZED_TSC_STANDALONE:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_NORMALIZED_TSC_STANDALONE.");
            status = lwpmudrv_Get_Normalized_TSC(&local_args);
            break;

        case DRV_OPERATION_NUM_CORES:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_NUM_CORES.");
            status = lwpmudrv_Get_Num_Cores(&local_args);
            break;

        case DRV_OPERATION_KERNEL_CS:
#if defined(DRV_IA32) || defined(DRV_EM64T)
            SEP_DRV_LOG_TRACE("DRV_OPERATION_KERNEL_CS.");
            status = lwpmudrv_Get_KERNEL_CS(&local_args);
#endif
            break;

        case DRV_OPERATION_SET_UID:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_UID.");
            status = lwpmudrv_Set_UID(&local_args);
            break;

        case DRV_OPERATION_TSC_SKEW_INFO:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_TSC_SKEW_INFO.");
            status = lwpmudrv_Get_TSC_Skew_Info(&local_args);
            break;

        case DRV_OPERATION_COLLECT_SYS_CONFIG:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_COLLECT_SYS_CONFIG.");
            status = lwpmudrv_Collect_Sys_Config(&local_args);
            break;

        case DRV_OPERATION_GET_SYS_CONFIG:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_SYS_CONFIG.");
            status = lwpmudrv_Sys_Config(&local_args);
            break;

        case DRV_OPERATION_TERMINATE:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_TERMINATE.");
            status = lwpmudrv_Terminate();
            break;

        case DRV_OPERATION_SET_CPU_TOPOLOGY:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_CPU_TOPOLOGY.");
            status = lwpmudrv_Setup_Cpu_Topology(&local_args);
            break;

        case DRV_OPERATION_GET_NUM_CORE_CTRS:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_NUM_CORE_CTRS.");
            status = lwpmudrv_Samp_Read_Num_Of_Core_Counters(&local_args);
            break;

        case DRV_OPERATION_GET_PLATFORM_INFO:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_PLATFORM_INFO.");
            status = lwpmudrv_Get_Platform_Info(&local_args);
            break;

        case DRV_OPERATION_SWITCH_GROUP:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SWITCH_GROUP.");
            status = lwpmudrv_Switch_Group();
            break;

        case DRV_OPERATION_GET_PERF_CAPAB:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_PERF_CAPAB.");
            status = lwpmudrv_Get_Perf_Capab(&local_args);
            break;

       /*
        * EMON-specific IOCTL commands
        */
        case DRV_OPERATION_READ_MSR:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_READ_MSR.");
            status = lwpmudrv_Read_Allowlist_MSR_All_Cores(&local_args);
            break;

        case DRV_OPERATION_WRITE_MSR:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_WRITE_MSR.");
            status = lwpmudrv_Write_Allowlist_MSR_All_Cores(&local_args);
            break;

        case DRV_OPERATION_READ_SWITCH_GROUP:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_READ_SWITCH_GROUP.");
            status = lwpmudrv_Read_Counters_And_Switch_Group(&local_args);
            break;

        case DRV_OPERATION_READ_AND_RESET:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_READ_AND_RESET.");
            status = lwpmudrv_Read_And_Reset_Counters(&local_args);
            break;

       /*
        * Platform-specific IOCTL commands (IA32 and Intel64)
        */

#if defined(DRV_IA32) || defined(DRV_EM64T)
        case DRV_OPERATION_INIT_UNC:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_INIT_UNC.");
            status = lwpmudrv_Initialize_UNC(local_args.buf_usr_to_drv, local_args.len_usr_to_drv);
            break;

        case DRV_OPERATION_EM_GROUPS_UNC:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_EM_GROUPS_UNC.");
            status = lwpmudrv_Set_EM_Config_UNC(&local_args);
            break;

        case DRV_OPERATION_EM_CONFIG_NEXT_UNC:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_EM_CONFIG_NEXT_UNC.");
            status = lwpmudrv_Configure_Events_UNC(&local_args);
            break;

        case DRV_OPERATION_LBR_INFO:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_LBR_INFO.");
            status = lwpmudrv_LBR_Info(&local_args);
            break;

        case DRV_OPERATION_INIT_NUM_DEV:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_INIT_NUM_DEV.");
            status = lwpmudrv_Initialize_Num_Devices(&local_args);
            break;
#endif
        case DRV_OPERATION_GET_NUM_SAMPLES:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_NUM_SAMPLES.");
            status = lwpmudrv_Get_Num_Samples(&local_args);
            break;
        case DRV_OPERATION_SET_DEVICE_NUM_UNITS:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_DEVICE_NUM_UNITS.");
            status = lwpmudrv_Set_Device_Num_Units(&local_args);
            break;

        case DRV_OPERATION_SET_SCAN_UNCORE_TOPOLOGY_INFO:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_SCAN_UNCORE_TOPOLOGY_INFO.");
            status = lwpmudrv_Set_Uncore_Topology_Info_And_Scan(&local_args);
            break;

        case DRV_OPERATION_GET_UNCORE_TOPOLOGY:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_UNCORE_TOPOLOGY.");
            status = lwpmudrv_Get_Uncore_Topology(&local_args);
            break;

        case DRV_OPERATION_GET_PLATFORM_TOPOLOGY:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_PLATFORM_TOPOLOGY.");
            status = lwpmudrv_Get_Platform_Topology(&local_args);
            break;

        case DRV_OPERATION_GET_DRV_SETUP_INFO:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_DRV_SETUP_INFO.");
            status = lwpmudrv_Get_Drv_Setup_Info(&local_args);
            break;

        case DRV_OPERATION_SET_EMON_BUFFER_DRIVER_HELPER:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_SET_EMON_BUFFER_DRIVER_HELPER.");
            status = lwpmudrv_Set_Emon_Buffer_Driver_Helper(&local_args);
            break;

       /*
        * if none of the above, treat as unknown/illegal IOCTL command
        */

        default:
            SEP_DRV_LOG_ERROR("Unknown IOCTL group:%ld basecmd:%ld.",
                    IOCGROUP(cmd), IOCBASECMD(cmd));
            status = OS_ILLEGAL_IOCTL;
            break;
    }

    UTILITY_Driver_Set_Active_Ioctl(0);
    MTX_UNLOCK(&ioctl_lock);

    return status;
}

#if defined(HAVE_COMPAT_IOCTL) && defined(DRV_EM64T)
extern long
lwpmu_Device_Control_Compat (
    struct   file   *filp,
    unsigned int     cmd,
    unsigned long    arg
)
{
    int              status = OS_SUCCESS;
    IOCTL_ARGS_NODE  local_args;

    SEP_DRV_LOG_TRACE("Compat: type: %d, subcommand: %d.",_IOC_TYPE(cmd),_IOC_NR(cmd));

    if (_IOC_TYPE(cmd) != LWPMU_IOC_MAGIC) {
        return OS_ILLEGAL_IOCTL;
    }

    cmd = (U32)_IOC_NR(cmd);

    MTX_LOCK(&ioctl_lock);
    switch (cmd) {
        case DRV_OPERATION_VERSION:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_VERSION.");
            if (copy_from_user(&local_args, (IOCTL_ARGS)arg, sizeof(IOCTL_ARGS_NODE))) {
                status = OS_FAULT;
                goto cleanup;
            }
            status = lwpmudrv_Version(&local_args);
            break;

        case DRV_OPERATION_GET_DRIVER_STATE:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_DRIVER_STATE.");
            if (copy_from_user(&local_args, (IOCTL_ARGS)arg, sizeof(IOCTL_ARGS_NODE))) {
                status = OS_FAULT;
                goto cleanup;
            }
            status = lwpmudrv_Get_Driver_State(&local_args);
            break;

        case DRV_OPERATION_GET_NORMALIZED_TSC:
            SEP_DRV_LOG_TRACE("DRV_OPERATION_GET_NORMALIZED_TSC.");
            if (copy_from_user(&local_args, (IOCTL_ARGS)arg, sizeof(IOCTL_ARGS_NODE))) {
                status = OS_FAULT;
                goto cleanup;
            }
            status = lwpmudrv_Get_Normalized_TSC(&local_args);
            break;

    }
cleanup:
    MTX_UNLOCK(&ioctl_lock);

    return status;
}
#endif

/*
 * @fn        LWPMUDRV_Abnormal_Terminate(void)
 *
 * @brief     This routine is called from fbsd_os_Exit_Task_Notify if the user process has
 *            been killed by an uncatchable signal (example kill -9).  The state variable
 *            abormal_terminate is set to 1 and the clean up routines are called.  In this
 *            code path the OS notifier hooks should not be unloaded.
 *
 * @param     None
 *
 * @return    OS_STATUS
 *
 * <I>Special Notes:</I>
 *     <none>
 */
int
LWPMUDRV_Abnormal_Terminate (
    void
)
{
    int              status = OS_SUCCESS;
    SEP_DRV_LOG_FLOW_IN("");

    SEP_DRV_LOG_TRACE("Abnormal-Termination: Calling lwpmudrv_Prepare_Stop.");
    status = lwpmudrv_Prepare_Stop();
    SEP_DRV_LOG_TRACE("Abnormal-Termination: Calling lwpmudrv_Finish_Stop.");
    status = lwpmudrv_Finish_Stop();
    SEP_DRV_LOG_TRACE("Abnormal-Termination: Calling lwpmudrv_Terminate.");
    status = lwpmudrv_Terminate();

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}


/*****************************************************************************************
 *
 *   Driver Entry / Exit functions that will be called on when the driver is loaded and
 *   unloaded
 *
 ****************************************************************************************/

/*
 * Structure that declares the usual file access functions
 * First one is for lwpmu_c, the control functions
 */
static struct cdevsw lwpmu_Fops = {
    .d_version = D_VERSION,
    .d_flags =   0,
    .d_ioctl =   lwpmu_Device_Control,
};

/*
 * Second one is for lwpmu_m, the module notification functions
 */
static struct cdevsw lwmod_Fops = {
    .d_version = D_VERSION,
    .d_flags =   0,
    .d_read =    OUTPUT_Module_Read,
};

/*
 * Third one is for lwsamp_nn, the sampling functions
 */
static struct cdevsw lwsamp_Fops = {
    .d_version = D_VERSION,
    .d_flags =   0,
    .d_read =    OUTPUT_Sample_Read,
};

/*
 * Fourth one is for lwunc_nn, the uncore collection functions
 */
static struct cdevsw lwunc_Fops = {
    .d_version = D_VERSION,
    .d_flags =   0,
    .d_read =    OUTPUT_Uncore_Read,
};

/*
 * Fifth one is for lwpmu_e, the EMON collection functions
 */
static struct cdevsw lwemon_Fops = {
    .d_version = D_VERSION,
    .d_flags =   0,
    .d_read =    OUTPUT_Emon_Read,
};

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static int lwpmudrv_setup_cdev(dev, fops, dev_number)
 *
 * @param LWPMU_DEV               dev  - pointer to the device object
 * @param struct file_operations *fops - pointer to the file operations struct
 * @param dev_t                   dev_number - major/monor device number
 *
 * @return OS_STATUS
 *
 * @brief  Set up the device object.
 *
 * <I>Special Notes</I>
 */
static void
lwpmu_setup_cdev (
    LWPMU_DEV               dev,
    struct cdevsw          *cdevsw,
    char                   *dev_name
)
{
    int gid = GID_OPERATOR;
    int perm = 0660;        // default permission (requires prefix 0 for octal format)
    static int minor =  0;

    TUNABLE_INT_FETCH("hw.sep.gid", &gid);
    TUNABLE_INT_FETCH("hw.sep.perm", &perm);

    LWPMU_DEV_cdev(dev) = make_dev(cdevsw, minor++, UID_ROOT, gid, perm, "%s", dev_name);
}


/*
 * @fn        lwpmudrv_Abnormal_Handler(void *data)
 *
 * @brief     Main routine for the abnormal termination thread. Processes TERMINATING->UNINITIALIZED
 *            transitions, and checks whether the collector still exists on a periodic basis.
 *
 * @param     void* data: dummy parameter
 *
 * @return    none
 *
 * <I>Special Notes:</I>
 *     <none>
 */
static void
lwpmudrv_Abnormal_Handler(void *data)
{
    int        plus_one_iteration = 0;

    SEP_DRV_LOG_FLOW_IN("");

    while (!abnormal_end || !plus_one_iteration++) {

        if (!plus_one_iteration) {
            mtx_lock(&abnormal_mutex);
            cv_timedwait(&abnormal_event, &abnormal_mutex, 1000);
            mtx_unlock(&abnormal_mutex);
            SEP_DRV_LOG_TRACE("lwpmudrv_Abnormal_Handler: time out.");
        }

        if (GET_DRIVER_STATE() == DRV_STATE_TERMINATING) {
            SEP_DRV_LOG_WARNING("Processing abnormal termination...");
            MTX_LOCK(&ioctl_lock);
            SEP_DRV_LOG_TRACE("lwpmudrv_Abnormal_Handler: Locked ioctl_lock...");
            LWPMUDRV_Abnormal_Terminate();
            MTX_UNLOCK(&ioctl_lock);
            SEP_DRV_LOG_TRACE("lwpmudrv_Abnormal_Handler: Unlocked ioctl_lock!");
        }
        else {
            if (control_pid) {
                struct proc *process = pfind(control_pid);
                if (!process                        ||
                     process->p_state == PRS_ZOMBIE ||
                     P_SHOULDSTOP(process)          ||
                     P_KILLED(process)) {

                    SEP_DRV_LOG_ERROR("lwpmudrv_Abnormal_Handler: Detected abnormal collector termination.");
                    CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
                    mtx_lock(&abnormal_mutex);
                    cv_signal(&abnormal_event);
                    mtx_unlock(&abnormal_mutex);
                }
                else {
                    char* process_name = process->p_comm;
                    if (!((process_name[0] == 's' && process_name[1] == 'e' && process_name[2] == 'p' && !process_name[3])                        ||
                          (process_name[0] == 'e' && process_name[1] == 'm' && process_name[2] == 'o' && process_name[3] == 'n' && !process_name[4]))) {
                        SEP_DRV_LOG_ERROR("lwpmudrv_Abnormal_Handler: Detected abnormal collector (%s)!", process_name);
                        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
                        mtx_lock(&abnormal_mutex);
                        cv_signal(&abnormal_event);
                        mtx_unlock(&abnormal_mutex);
                    }
                }

                if (process) {
                    PROC_UNLOCK(process);
                }
            }
        }
    }

    SEP_DRV_LOG_FLOW_OUT("End of thread.");
    kthread_exit();
}


/* ------------------------------------------------------------------------- */
/*!
 * @fn  static int lwpmu_Load(void)
 *
 * @param none
 *
 * @return STATUS
 *
 * @brief  Load the driver module into the kernel.  Set up the driver object.
 * @brief  Set up the initial state of the driver and allocate the memory
 * @brief  needed to keep basic state information.
 */
static int
lwpmu_Load (
    VOID
)
{
    int        i, num_cpus;
    OS_STATUS  status      = OS_SUCCESS;

    SEP_DRV_LOG_LOAD("Driver loading...");
    if (UTILITY_Driver_Log_Init() != OS_SUCCESS) { // Do not use SEP_DRV_LOG_X (where X != LOAD) before this, or if this fails
        SEP_DRV_LOG_LOAD("Error: could not allocate log buffer.");
        return OS_NO_MEM;
    }
    SEP_DRV_LOG_FLOW_IN("Starting internal log monitoring.");

    status = lwpmudrv_Initialize_State();
    if (status < 0) {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Failed to initialize state!");
        return status;
    }
    num_cpus = GLOBAL_STATE_num_cpus(driver_state);
    SEP_DRV_LOG_DETECTION("Detected %d CPUs.", num_cpus);

    MTX_INIT(&ioctl_lock, "sep ioctl lock");

    mtx_init(&abnormal_mutex, "abnormal handler mutex", NULL, MTX_DEF);
    cv_init(&abnormal_event, "abnormal termination");
    kthread_add(lwpmudrv_Abnormal_Handler, NULL, NULL, &abnormal_handler, RFSTOPPED, 0, "my thread");
    if (abnormal_handler) {
        SEP_DRV_LOG_TRACE("Created abnormal handler thread!");
        thread_lock(abnormal_handler);
        TD_SET_CAN_RUN(abnormal_handler);
        sched_add(abnormal_handler, SRQ_BORING);
        // in FreeBSD 13 this thread_unlock was moved to inside sched_add as an
        // optimization in r355779, so it should no longer be called after
        // sched_add(). Doing this causes a kernel panic in FreeBSD 13
#if __FreeBSD_version < 1300068
        thread_unlock(abnormal_handler);
#endif
        SEP_DRV_LOG_TRACE("Scheduled abnormal handler thread!");
    }
    else {
        SEP_DRV_LOG_ERROR_FLOW_OUT("Failed to create abnormal termination thread!");
        return OS_NO_MEM;
    }

    /* Allocate memory for the control structures */
    lwpmu_control  = CONTROL_Allocate_Memory(sizeof(LWPMU_DEV_NODE));
    lwmod_control  = CONTROL_Allocate_Memory(sizeof(LWPMU_DEV_NODE));
    lwemon_control = CONTROL_Allocate_Memory(sizeof(LWPMU_DEV_NODE));
    lwsamp_control = CONTROL_Allocate_Memory(num_cpus*sizeof(LWPMU_DEV_NODE));

    if (!lwsamp_control || !lwpmu_control || !lwmod_control || !lwemon_control) {
        lwpmu_control  = CONTROL_Free_Memory(lwpmu_control);
        lwmod_control  = CONTROL_Free_Memory(lwmod_control);
        lwemon_control = CONTROL_Free_Memory(lwemon_control);
        lwsamp_control = CONTROL_Free_Memory(lwsamp_control);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for control structures!");
        return OS_NO_MEM;
    }

    /* Register the file operations with the OS */
    char buf[128];
    sprintf(buf, "%s/c", SEP_DRIVER_NAME);
    lwpmu_setup_cdev(lwpmu_control, &lwpmu_Fops, buf);

    sprintf(buf, "%s/m", SEP_DRIVER_NAME);
    lwpmu_setup_cdev(lwmod_control, &lwmod_Fops, buf);

    sprintf(buf, "%s/e", SEP_DRIVER_NAME);
    lwpmu_setup_cdev(lwemon_control, &lwemon_Fops, buf);

    /* Register the file operations with the OS */
    for (i = 0; i < num_cpus; i++) {
        sprintf(buf, "%s/s%d", SEP_DRIVER_NAME, i);
        lwpmu_setup_cdev(lwsamp_control+i, &lwsamp_Fops, buf);
        LWPMU_DEV_cdev(lwsamp_control+i)->si_drv1 = (void *)(uintptr_t)i;
        SEP_DRV_LOG_INIT("Added sampling device %d.", i);
    }

    cpu_tsc      = (U64 *)CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64));
    prev_cpu_tsc = (U64 *)CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64));
    diff_cpu_tsc = (U64 *)CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U64));
    atomic_set_int(&read_now, GLOBAL_STATE_num_cpus(driver_state));
    cv_init(&read_tsc_now_cv, "sep read tsc cv");
    mtx_init(&read_tsc_now_mtx, "sep tsc mtx", NULL, MTX_DEF);
    CONTROL_Invoke_Parallel(lwpmudrv_Fill_TSC_Info, (PVOID)(size_t)0);

    pcb_size              = GLOBAL_STATE_num_cpus(driver_state) * sizeof(CPU_STATE_NODE);
    pcb                   = CONTROL_Allocate_Memory(pcb_size);
    core_to_package_map   = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));
    core_to_module_map    = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));
    core_to_phys_core_map = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));
    core_to_thread_map    = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));
    threads_per_core      = CONTROL_Allocate_Memory(GLOBAL_STATE_num_cpus(driver_state) * sizeof(U32));

    SYS_INFO_Build();
    PCI_Initialize();

    if (total_ram <= OUTPUT_MEMORY_THRESHOLD) {
        output_buffer_size = OUTPUT_SMALL_BUFFER;
    }

    UNC_COMMON_Init();

    /* allocate one device per package (for uncore)*/
    lwunc_control= CONTROL_Allocate_Memory(num_packages*sizeof(LWPMU_DEV_NODE));
    if (!lwunc_control) {
        CONTROL_Free_Memory(lwunc_control);
        SEP_DRV_LOG_ERROR_FLOW_OUT("Memory allocation failure for uncore control structures!");
    }

    /* Register the file operations with the OS */
    for (i = 0; i < num_packages; i++) {
        sprintf(buf, "%s/u%d", SEP_DRIVER_NAME, i);
        lwpmu_setup_cdev(lwunc_control+i, &lwunc_Fops, buf);
        LWPMU_DEV_cdev(lwunc_control+i)->si_drv1 = (void *)(uintptr_t)i;
        SEP_DRV_LOG_INIT("Added uncore device %d.", i);
    }

    /*
     *  Initialize the SEP driver version (done once at driver load time)
     */
    SEP_VERSION_NODE_major(&drv_version) = SEP_MAJOR_VERSION;
    SEP_VERSION_NODE_minor(&drv_version) = SEP_MINOR_VERSION;
    SEP_VERSION_NODE_api(&drv_version)   = SEP_API_VERSION;

    //
    // Display driver version information
    //
    SEP_DRV_LOG_LOAD("PMU collection driver v%d.%d.%d has been loaded.",
                        SEP_VERSION_NODE_major(&drv_version),
                        SEP_VERSION_NODE_minor(&drv_version),
                        SEP_VERSION_NODE_api(&drv_version));

#if defined(DRV_IA32) || defined(DRV_EM64T)
    SEP_DRV_LOG_INIT("IDT vector 0x%x will be used for handling PMU interrupts.", CPU_PERF_VECTOR);
#endif

    PMU_LIST_Initialize(&allowlist_index);
    PMU_LIST_Build_MSR_List();
    PMU_LIST_Build_PCI_List();
    PMU_LIST_Build_MMIO_List();

    SEP_DRV_LOG_FLOW_OUT("Return value: %d.", status);
    return status;
}

/* ------------------------------------------------------------------------- */
/*!
 * @fn  static int lwpmu_Unload(void)
 *
 * @param none
 *
 * @return none
 *
 * @brief  Remove the driver module from the kernel.
 */
static VOID
lwpmu_Unload (
    VOID
)
{
    int        i = 0;
    int        num_cpus = GLOBAL_STATE_num_cpus(driver_state);
    struct mtx end_of_thread_mutex;
    PVOID      tmp_pcb;

    SEP_DRV_LOG_FLOW_IN("");

    SEP_DRV_LOG_LOAD("Driver unloading.");

    PMU_LIST_Clean_Up();

    if (GET_DRIVER_STATE() != DRV_STATE_UNINITIALIZED) {
        CHANGE_DRIVER_STATE(STATE_BIT_ANY, DRV_STATE_TERMINATING);
        mtx_lock(&abnormal_mutex);
        cv_signal(&abnormal_event);
        mtx_unlock(&abnormal_mutex);
    }

    mtx_init(&end_of_thread_mutex, "thread termination", NULL, MTX_DEF);
    mtx_lock(&end_of_thread_mutex);
    SEP_DRV_LOG_TRACE("Waiting for abnormal thread termination...");
    abnormal_end = 1;
    mtx_sleep(abnormal_handler, &end_of_thread_mutex, PWAIT, "unload: waiting for thread", 0);
    SEP_DRV_LOG_INIT("Abnormal thread terminated.");
    mtx_unlock(&end_of_thread_mutex);
    mtx_destroy(&end_of_thread_mutex);
    mtx_destroy(&abnormal_mutex);
    cv_destroy(&abnormal_event);

    MTX_DESTROY(&ioctl_lock);

    FBSD_OS_Uninstall_Hooks();
    SYS_INFO_Destroy();
    OUTPUT_Destroy();
    cpu_buf               = CONTROL_Free_Memory(cpu_buf);
    unc_buf               = CONTROL_Free_Memory(unc_buf);
    module_buf            = CONTROL_Free_Memory(module_buf);
    emon_buf              = CONTROL_Free_Memory(emon_buf);
    cpu_tsc               = CONTROL_Free_Memory(cpu_tsc);
    prev_cpu_tsc          = CONTROL_Free_Memory(prev_cpu_tsc);
    diff_cpu_tsc          = CONTROL_Free_Memory(diff_cpu_tsc);
    core_to_package_map   = CONTROL_Free_Memory(core_to_package_map);
    core_to_module_map    = CONTROL_Free_Memory(core_to_module_map);
    core_to_phys_core_map = CONTROL_Free_Memory(core_to_phys_core_map);
    core_to_thread_map    = CONTROL_Free_Memory(core_to_thread_map);
    module_id_index       = CONTROL_Free_Memory(module_id_index);
    cores_per_module      = CONTROL_Free_Memory(cores_per_module);
    threads_per_core      = CONTROL_Free_Memory(threads_per_core);

    tmp_pcb             = pcb;                          // Ensures there is no log message written (ERROR, ALLOC, ...)
    pcb                 = NULL;                         // between pcb being freed and pcb being NULL.
    tmp_pcb             = CONTROL_Free_Memory(tmp_pcb);
    pcb_size            = 0;

    UNC_COMMON_Clean_Up();

    if (lwpmu_control != NULL) {
        if (LWPMU_DEV_cdev(lwpmu_control) != NULL)
            destroy_dev(LWPMU_DEV_cdev(lwpmu_control));
        lwpmu_control  = CONTROL_Free_Memory(lwpmu_control);
    }

    if (lwmod_control != NULL) {
        if (LWPMU_DEV_cdev(lwmod_control) != NULL)
            destroy_dev(LWPMU_DEV_cdev(lwmod_control));
        lwmod_control  = CONTROL_Free_Memory(lwmod_control);
    }

    if (lwemon_control != NULL) {
        if (LWPMU_DEV_cdev(lwemon_control) != NULL)
            destroy_dev(LWPMU_DEV_cdev(lwemon_control));
        lwemon_control  = CONTROL_Free_Memory(lwemon_control);
    }

    if (lwsamp_control != NULL) {
        for (i = 0; i < num_cpus; i++) {
            if (LWPMU_DEV_cdev(&lwsamp_control[i]) != NULL)
                destroy_dev(LWPMU_DEV_cdev(&lwsamp_control[i]));
        }
        lwsamp_control = CONTROL_Free_Memory(lwsamp_control);
    }

    if (lwunc_control != NULL) {
        for (i = 0; i < num_packages; i++) {
            if (LWPMU_DEV_cdev(&lwunc_control[i]) != NULL) {
                destroy_dev(LWPMU_DEV_cdev(&lwunc_control[i]));
            }
        }
        lwunc_control = CONTROL_Free_Memory(lwunc_control);
    }

    SEP_DRV_LOG_FLOW_OUT("Log deallocation. Cannot track further in internal log.");
    UTILITY_Driver_Log_Free(); // Do not use SEP_DRV_LOG_X (where X != LOAD) after this

    SEP_DRV_LOG_LOAD("PMU collection driver v%d.%d.%d %s has been unloaded.",
              SEP_VERSION_NODE_major(&drv_version),
              SEP_VERSION_NODE_minor(&drv_version),
              SEP_VERSION_NODE_api(&drv_version),
              SEP_RELEASE_STRING);

    return;
}

