#ifndef COST_AUTO_FCN_H
#define COST_AUTO_FCN_H

#include <octave/oct.h>
#include <octave/oct-map.h>

#include "Cost_fcn.h"
#include "Permute_random_fcn.h"

class Cost_auto_fcn : public Cost_fcn, protected Permute_random_fcn
{
protected:
  double mean;
  double rms_ampl;
  //! Number of lags
  octave_idx_type no_lags;
  //! The type of the average used
  octave_idx_type average_type;
  //! Correlation matrix
  Matrix c0;
  Matrix c;

  //! Resizes c0 and c and calculates c0 - the basis for cost calculations
  void resize ()
  {
    c0.resize (no_lags,1);
    c.resize (no_lags,1);
  }

public:

  //! constructor
  Cost_auto_fcn (Matrix *input, octave_idx_type _no_lags = 1,
                 octave_idx_type _average_type = 0)
    : Cost_fcn (), Permute_random_fcn (input), no_lags (_no_lags),
      average_type (_average_type)
  {
    this->series = input;
    // resize c0 and c to fit new parameters
    this->resize ();
  }

  //! Sets no_lags and average_type from Octave struct that contains proper fields
  Cost_auto_fcn (Matrix *input, const octave_scalar_map &cost_params,
                 const octave_scalar_map &permute_params)
    :  Cost_fcn (), Permute_random_fcn (input, permute_params)
  {
    this->series = input;

    if (cost_params.isfield (std::string ("no_lags")) 
        && cost_params.isfield (std::string ("average")))
      {
        this->no_lags      = cost_params.getfield (std::string ("no_lags"))
                                         .idx_type_value ();
        this->average_type = cost_params.getfield (std::string ("average"))
                                         .idx_type_value ();

        // resize c0 and c to fit new parameers
        this->resize ();
      }
    else
      {
        error_with_id ("Octave:tisean", "Cost function constructor did not "
                                        "receive correct parameters");
      }
  }

  // "API" functions
  //! Normalizes @c x and assigns it to @c series
  /*!
      It also calls the recalculates base autocorelation c0.
   */
  void cost_transform (bool do_scramble = true)
  {
    double _mean, _rms_ampl;
    normal1 (*(this->series), _mean, _rms_ampl);

    this->mean = _mean;
    this->rms_ampl = _rms_ampl;

    autocorrelation (c0);

    if (do_scramble)
      this->scramble();
  }

  //! Returns the unnormalized version of series
  Matrix cost_inverse () const
  {
    Matrix y = (*series) * rms_ampl + mean;
    return y;
  }

  //! Calculates full cost using all of series (slower than cost_update())
  /*!
      Once the full cost is calculated it is assigned to member cost

      @return the current cost
   */
  double cost_full ()
  {
    autocorrelation (c);

    double cc = 0;
    for (octave_idx_type n = 0; n < no_lags; n++)
      cc = average (cc, c0(n) - c(n), n);
    this->cost = cc;

    return this->cost;
  }

  //! Performs a cost update
  /*! Generates two indexes to swap and calculates cost update
      @param cmax the maximum cost that is acceptable
      @param accept is set to true when the swap is acceptable (then the cost and
                    series were updated), otherwise false

      @return current cost

      @see cost_update (octave_idx_type nn1, octave_idx_type nn2, double cmax,
                        bool &accept)
   */
  double cost_update (double cmax, bool &accept)
  {
    // Generate indexes candidates for cost update
    octave_idx_type n1, n2;
    this->permute (n1, n2);
    // Check if the update is acceptable -- if so perform it
    this->cost_update (n1, n2, cmax, accept);
    // If the update was accepted, swap the elements in series.
    if (accept)
      this->exch (n1, n2);
    // Return current cost
    return this->cost;
  }

  //! Calculates an updated cost for swapped elements of series
  /*! @param nn1, nn2 the indexes of the elements for which an updated cost is
                      calculated when they are swapped
      @param cmax the maximum cost that is acceptable
      @param accept is set to true when the swap is acceptable (and the cost was
                    updated), otherwise false

      @return current cost
   */
  double
    cost_update (octave_idx_type nn1, octave_idx_type nn2, double cmax,
                 bool &accept);

  const Matrix & get_c () const { return this->c; }
  void set_c (const Matrix &other_c) { this->c = other_c; }

  //! Calculates the autocorrelation of series and saves it in cor_mat
  void autocorrelation (Matrix &cor_mat);
  //! Calculates average TODO:expand this
  double average (double comp, double dc, octave_idx_type n);

};

#endif
