"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const vscode_languageserver_1 = require("vscode-languageserver");
const completionProvider_1 = require("../../src/providers/completionProvider");
const helper_1 = require("../helper");
function testPlayKeywords(context, textDoc) {
    const tests = [
        {
            name: "name",
            position: { line: 0, character: 2 },
            triggerCharacter: "",
            completion: "name",
        },
        {
            name: "hosts",
            position: { line: 2, character: 5 },
            triggerCharacter: "hos",
            completion: "hosts",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.be.equal(completion);
                    }
                }
            });
        });
    });
}
function testRoleKeywords(context, textDoc) {
    const tests = [
        {
            name: "name",
            position: { line: 4, character: 6 },
            triggerCharacter: "",
            completion: "name",
        },
        {
            name: "when",
            position: { line: 5, character: 8 },
            triggerCharacter: "wh",
            completion: "when",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.be.equal(completion);
                    }
                }
            });
        });
    });
}
function testBlockKeywords(context, textDoc) {
    const tests = [
        {
            name: "become_user",
            position: { line: 8, character: 13 },
            triggerCharacter: "user",
            completion: "become_user",
        },
        {
            name: "become",
            position: { line: 7, character: 8 },
            triggerCharacter: "be",
            completion: "become",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.be.equal(completion);
                    }
                }
            });
        });
    });
}
function testTaskKeywords(context, textDoc) {
    const tests = [
        {
            name: "loop",
            position: { line: 10, character: 9 },
            triggerCharacter: "loop",
            completion: "loop",
        },
        {
            name: "debugger",
            position: { line: 13, character: 9 },
            triggerCharacter: "deb",
            completion: "debugger",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.be.equal(completion);
                    }
                }
            });
        });
    });
}
function testModuleNames(context, textDoc) {
    const tests = [
        {
            name: "with name as first option always",
            position: { line: 6, character: 6 },
            triggerCharacter: "",
            completion: "name",
        },
        {
            name: "for `ansible.builtin.ping` with `ping`",
            position: { line: 7, character: 8 },
            triggerCharacter: "ping",
            completion: "ansible.builtin.ping",
        },
        {
            name: "for `ansible.builtin.debug` with `debu`",
            position: { line: 7, character: 8 },
            triggerCharacter: "debu",
            completion: "ansible.builtin.debug",
        },
        {
            name: "list for all modules under ansible namespace with `ansible.`",
            position: { line: 7, character: 8 },
            triggerCharacter: "ansible.",
            completion: "ansible.",
        },
        {
            name: "list for all the modules under ansible.builtin with `ansible.builtin.`",
            position: { line: 7, character: 8 },
            triggerCharacter: "ansible.builtin.",
            completion: "ansible.builtin.",
        },
        {
            name: "list for all the collection modules starting with `c` under org_1 namespace with `org_1.c`",
            position: { line: 16, character: 13 },
            triggerCharacter: "org_1.c",
            completion: "org_1.c",
        },
        {
            name: "list for all the modules under coll_4 in org_1 with `org_1.coll_4.`",
            position: { line: 16, character: 19 },
            triggerCharacter: "org_1.coll_4.",
            completion: "org_1.coll_4.",
        },
        {
            name: "list for all the modules under coll_5 in org_1 with `org_1.coll_5.`",
            position: { line: 34, character: 19 },
            triggerCharacter: "org_1.coll_5.",
            completion: "org_1.coll_5.sub_coll_1.module_1",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).to.contain(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).to.contain(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.contain(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.contain(completion);
                    }
                }
            });
        });
    });
}
function testModuleOptions(context, textDoc) {
    const tests = [
        {
            name: "builtin module option (ansible.builtin.debug -> msg)",
            position: { line: 8, character: 9 },
            triggerCharacter: "m",
            completion: "msg",
        },
        {
            name: "collection module option (org_1.coll_4.module_1 -> opt_1)",
            position: { line: 17, character: 8 },
            triggerCharacter: "",
            completion: "opt_1",
        },
        {
            name: "collection module sub option (org_1.coll_4.module_1 -> opt_1 -> sub_opt_1)",
            position: { line: 21, character: 12 },
            triggerCharacter: "1",
            completion: "sub_opt_1",
        },
        {
            name: "collection module sub option (org_1.coll_4.module_1 -> opt_1 -> sub_opt_2 -> sub_sub_opt_3 -> sub_sub_sub_opt_2)",
            position: { line: 26, character: 20 },
            triggerCharacter: "2",
            completion: "sub_sub_sub_opt_2",
        },
        {
            name: "only non repeating options",
            position: { line: 9, character: 9 },
            triggerCharacter: "m",
            completion: "",
        },
        {
            name: "only non repeating suboptions",
            position: { line: 29, character: 20 },
            triggerCharacter: "1",
            completion: "",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).be.equal(completion);
                    }
                }
            });
        });
    });
}
function testModuleOptionsValues(context, textDoc) {
    const tests = [
        {
            name: "builtin module option (ansible.builtin.debug -> msg)",
            position: { line: 8, character: 13 },
            triggerCharacter: "",
            completion: ["Hello world!"],
        },
        {
            name: "collection module option (org_1.coll_4.module_1 -> opt_3)",
            position: { line: 30, character: 15 },
            triggerCharacter: "3",
            completion: ["choice_3"],
        },
        {
            name: "collection module sub option (org_1.coll_4.module_1 -> opt_1 -> sub_opt_1)",
            position: { line: 18, character: 23 },
            triggerCharacter: "1",
            completion: ["choice_1"],
        },
        {
            name: "default first",
            position: { line: 30, character: 15 },
            triggerCharacter: "",
            completion: ["choice_4", "choice_1", "choice_2", "choice_3"],
        },
        {
            name: "boolean values",
            position: { line: 31, character: 15 },
            triggerCharacter: "",
            completion: ["false", "true"],
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const labelCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter).map((completion) => {
                    if (!completion.item) {
                        return completion.label;
                    }
                    else {
                        return completion.item.label;
                    }
                });
                if (!completion) {
                    (0, chai_1.expect)(labelCompletion.length).be.equal(0);
                }
                else {
                    (0, chai_1.expect)(labelCompletion).be.deep.equal(completion);
                }
                const newTextCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter).map((completion) => {
                    if (!completion.item) {
                        return completion.textEdit.newText;
                    }
                    else {
                        return completion.item.textEdit.newText;
                    }
                });
                if (!completion) {
                    (0, chai_1.expect)(newTextCompletion.length).be.equal(0);
                }
                else {
                    (0, chai_1.expect)(newTextCompletion).be.deep.equal(completion);
                }
            });
        });
    });
}
function testModuleNamesWithoutFQCN(context, textDoc) {
    const tests = [
        {
            name: "`ping` with `pin` (ansible.builtin.ping)",
            position: { line: 7, character: 9 },
            triggerCharacter: "pin",
            completion: "ping",
        },
        {
            name: "module option for ping (ping -> data)",
            position: { line: 8, character: 8 },
            triggerCharacter: "",
            completion: "data",
        },
        {
            name: "`module_3` from `org_1.coll_3` with `module_3` (org_1.coll_3.module_3)",
            position: { line: 11, character: 14 },
            triggerCharacter: "module_3",
            completion: "module_3",
        },
        {
            name: "module sub option for module_3 (org_1.coll_3.module_3 -> opt_1 -> sub_opt_2)",
            position: { line: 13, character: 13 },
            triggerCharacter: "2",
            completion: "sub_opt_2",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                //   Update setting to stop using FQCN for module names
                const docSettings = context.documentSettings.get(textDoc.uri);
                const cachedDefaultSetting = (yield docSettings).validation.lint.enabled;
                (yield docSettings).ansible.useFullyQualifiedCollectionNames = false;
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                // Revert back the default settings
                (yield docSettings).ansible.useFullyQualifiedCollectionNames =
                    cachedDefaultSetting;
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).to.be.equal(completion);
                    }
                }
            });
        });
    });
}
function testPlaybookAdjacentCollection(context, textDoc) {
    const tests = [
        {
            name: "playbook adjacent module (adjacent_org.adjacent_coll.module_1)",
            position: { line: 5, character: 19 },
            triggerCharacter: "adjacent_org.",
            completion: "adjacent_org.adjacent_coll.module_1",
        },
        {
            name: "playbook adjacent module option (adjacent_org.adjacent_coll.module_1 -> opt_1)",
            position: { line: 6, character: 11 },
            triggerCharacter: "opt",
            completion: "opt_1",
        },
        {
            name: "playbook adjacent module sub option (adjacent_org.adjacent_coll.module_1 -> opt_1 -> sub_opt_1)",
            position: { line: 7, character: 19 },
            triggerCharacter: "sub_opt",
            completion: "sub_opt_1",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).be.equal(completion);
                    }
                }
            });
        });
    });
}
function testNonPlaybookAdjacentCollection(context, textDoc) {
    const tests = [
        {
            name: "non playbook adjacent module (adjacent_org.adjacent_coll.module_1)",
            position: { line: 5, character: 19 },
            triggerCharacter: "adjacent_org.",
            completion: "",
        },
        {
            name: "non playbook adjacent module option (adjacent_org.adjacent_coll.module_1 -> opt_1)",
            position: { line: 6, character: 11 },
            triggerCharacter: "opt",
            completion: "",
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should not provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter);
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    if (!filteredCompletion[0].item) {
                        (0, chai_1.expect)(filteredCompletion[0].label).be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].textEdit.newText).be.equal(completion);
                    }
                    else {
                        (0, chai_1.expect)(filteredCompletion[0].item.label).to.be.equal(completion);
                        (0, chai_1.expect)(filteredCompletion[0].item.textEdit.newText).be.equal(completion);
                    }
                }
            });
        });
    });
}
function testHostValues(context, textDoc) {
    const tests = [
        {
            name: "hello-worlds group",
            position: { line: 2, character: 9 },
            triggerCharacter: "hello",
            completion: ["hello-worlds", "hello.world.1", "hello.world.2"],
        },
        {
            name: "test-inventories group",
            position: { line: 2, character: 9 },
            triggerCharacter: "inventor",
            completion: ["test-inventories", "test.inventory.3", "test.inventory.4"],
        },
        {
            name: "localhost",
            position: { line: 2, character: 14 },
            triggerCharacter: "local",
            completion: ["localhost"],
        },
        {
            name: "all",
            position: { line: 2, character: 9 },
            triggerCharacter: "all",
            completion: ["all"],
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name} as hosts value`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter).map((completion) => {
                    if (!completion.item) {
                        return completion.label;
                    }
                    else {
                        return completion.item.label;
                    }
                });
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    (0, chai_1.expect)(filteredCompletion).be.deep.equal(completion);
                }
            });
        });
    });
}
function testVarsCompletionInsideJinja(context, textDoc) {
    const tests = [
        {
            name: "variables defined at task level",
            position: { line: 36, character: 17 },
            triggerCharacter: "",
            completion: ["url", "home", "os", "password", "username"],
        },
        {
            name: "variables defined at block level",
            position: { line: 30, character: 47 },
            triggerCharacter: "var",
            completion: ["task_var", "block_var_1", "block_var_2"],
        },
        {
            name: "variables defined at play level",
            position: { line: 14, character: 35 },
            triggerCharacter: "",
            completion: ["home", "os", "password", "username"],
        },
        {
            name: "variables defined inside a vars file",
            position: { line: 51, character: 33 },
            triggerCharacter: "",
            completion: [
                "filepath",
                "mode",
                "default_var_1",
                "default_var_2",
                "default_var_3",
            ],
        },
    ];
    tests.forEach(({ name, position, triggerCharacter, completion }) => {
        it(`should provide completion for ${name}`, function () {
            return __awaiter(this, void 0, void 0, function* () {
                const actualCompletion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
                const filteredCompletion = (0, helper_1.smartFilter)(actualCompletion, triggerCharacter).map((completion) => {
                    if (!completion.item) {
                        return completion.label;
                    }
                    else {
                        return completion.item.label;
                    }
                });
                if (!completion) {
                    (0, chai_1.expect)(filteredCompletion.length).be.equal(0);
                }
                else {
                    (0, chai_1.expect)(filteredCompletion).be.deep.equal(completion);
                }
            });
        });
    });
}
function testModuleKindAndDocumentation(context, textDoc) {
    const tests = [
        {
            moduleName: "org_1.coll_6.module_1",
            kind: vscode_languageserver_1.CompletionItemKind.Reference,
            documentation: "***Redirected to: org_1.coll_6.sub_coll_1.module_1***\n\n*Test module*\n\n**Description**\n\n- This is a test module 1\n\n**Notes**\n\n- This is a dummy module",
        },
        {
            moduleName: "org_1.coll_6.module_2",
            kind: vscode_languageserver_1.CompletionItemKind.Class,
            documentation: "*Test module*\n\n**Description**\n\n- This is a test module 2\n\n**Notes**\n\n- This is a dummy module",
        },
        {
            moduleName: "org_1.coll_6.sub_coll_1.module_1",
            kind: vscode_languageserver_1.CompletionItemKind.Class,
            documentation: "*Test module*\n\n**Description**\n\n- This is a test module 1\n\n**Notes**\n\n- This is a dummy module",
        },
        {
            moduleName: "org_1.coll_6.sub_coll_1.module_2",
            kind: vscode_languageserver_1.CompletionItemKind.Reference,
            documentation: "**DEPRECATED**\n\nUsage of org_1.coll_6.sub_coll_1.module_2 is deprecated.\n\nRemoval date: undefined, removal version: undefined\n\n***Redirected to: org_1.coll_6.module_2***\n\n*Test module*\n\n**Description**\n\n- This is a test module 2\n\n**Notes**\n\n- This is a dummy module",
        },
    ];
    tests.forEach(({ moduleName, kind, documentation }) => {
        let resolvedItem;
        const position = { line: 34, character: 19 };
        before(() => __awaiter(this, void 0, void 0, function* () {
            const completion = yield (0, completionProvider_1.doCompletion)(textDoc, position, context);
            const filteredCompletion = completion.filter((item) => item.label === moduleName);
            (0, chai_1.expect)(filteredCompletion.length).be.equal(1);
            (0, chai_1.expect)(filteredCompletion[0].label).be.equal(moduleName);
            resolvedItem = yield (0, completionProvider_1.doCompletionResolve)(filteredCompletion[0], context);
        }));
        it(`should provide completion kind ${kind} for ${moduleName}`, () => {
            (0, chai_1.expect)(resolvedItem.kind).be.equal(kind);
        });
        it(`should provide documentation for ${moduleName}`, () => {
            (0, chai_1.expect)(resolvedItem.documentation["value"]).be.equal(documentation);
        });
    });
}
describe("doCompletion()", () => {
    const workspaceManager = (0, helper_1.createTestWorkspaceManager)();
    let fixtureFilePath = "completion/simple_tasks.yml";
    let fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    let context = workspaceManager.getContext(fixtureFileUri);
    let textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    let docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for host values with static inventory file", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testHostValues(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                (0, helper_1.setAnsibleConfigEnv)();
            }));
            testHostValues(context, textDoc);
            after(() => {
                (0, helper_1.unsetAnsibleConfigEnv)();
            });
        });
    });
    describe("Completion for play keywords", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testPlayKeywords(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testPlayKeywords(context, textDoc);
        });
    });
    fixtureFilePath = "completion/with_roles.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for role keywords", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testRoleKeywords(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testRoleKeywords(context, textDoc);
        });
    });
    fixtureFilePath = "completion/with_blocks.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for block keywords", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testBlockKeywords(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testBlockKeywords(context, textDoc);
        });
    });
    fixtureFilePath = "completion/simple_tasks.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for task keywords", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testTaskKeywords(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testTaskKeywords(context, textDoc);
        });
    });
    describe("Completion for module names (with different trigger scenarios)", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleNames(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleNames(context, textDoc);
        });
    });
    describe("Check module kind and documentation of completion item", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleKindAndDocumentation(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleKindAndDocumentation(context, textDoc);
        });
    });
    describe("Completion for module options and suboptions", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleOptions(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleOptions(context, textDoc);
        });
    });
    describe("Completion for option and suboption values", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleOptionsValues(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleOptionsValues(context, textDoc);
        });
    });
    fixtureFilePath = "completion/tasks_without_fqcn.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for module name without FQCN", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleNamesWithoutFQCN(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testModuleNamesWithoutFQCN(context, textDoc);
        });
    });
    describe("Completion for variables inside jinja inline brackets", () => {
        fixtureFilePath = "completion/playbook_with_vars.yml";
        fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
        context = workspaceManager.getContext(fixtureFileUri);
        textDoc = (0, helper_1.getDoc)(fixtureFilePath);
        docSettings = context.documentSettings.get(textDoc.uri);
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testVarsCompletionInsideJinja(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testVarsCompletionInsideJinja(context, textDoc);
        });
    });
    fixtureFilePath = "playbook_adjacent_collection/playbook.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Completion for playbook adjacent collection", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testPlaybookAdjacentCollection(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testPlaybookAdjacentCollection(context, textDoc);
        });
    });
    fixtureFilePath =
        "playbook_adjacent_collection/non_adjacent_playbooks/playbook2.yml";
    fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    context = workspaceManager.getContext(fixtureFileUri);
    textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    docSettings = context.documentSettings.get(textDoc.uri);
    describe("Negate completion for non playbook adjacent collection", () => {
        describe("With EE enabled @ee", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)("/home/runner/.ansible/collections:/usr/share/ansible");
                yield (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }));
            testNonPlaybookAdjacentCollection(context, textDoc);
            after(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
        });
        describe("With EE disabled", () => {
            before(() => __awaiter(void 0, void 0, void 0, function* () {
                (0, helper_1.setFixtureAnsibleCollectionPathEnv)();
                yield (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
            }));
            testNonPlaybookAdjacentCollection(context, textDoc);
        });
    });
});
//# sourceMappingURL=completionProvider.test.js.map