# Copyright 2019-22 by Robert T. Miller.  All rights reserved.
# This file is part of the Biopython distribution and governed by your
# choice of the "Biopython License Agreement" or the "BSD 3-Clause License".
# Please see the LICENSE file that should have been included as part of this
# package.

"""Per residue backbone and sidechain hedra and dihedra definitions.

Find this file in <your biopython distribution>/Bio/PDB/ic_data.py

Listed in order of output for internal coordinates (.pic) output file.
Require sufficient overlap to link all defined dihedra.  Entries in these
tables without corresponding atom coordinates are ignored.

<http://www.imgt.org/IMGTeducation/Aide-memoire/_UK/aminoacids/formuleAA/>
for naming of individual atoms
"""

# Backbone hedra and dihedra - within residue, no next or prev (no psi, phi, omg).
ic_data_backbone = (
    ("N", "CA", "C", "O"),  # locate backbone O
    ("O", "C", "CA", "CB"),  # locate CB
    ("CA", "C", "O"),
    ("CB", "CA", "C"),
    ("CA", "C", "OXT"),  # OXT if present
    ("N", "CA", "C", "OXT"),
    ("H", "N", "CA"),  # amide proton if present
    ("C", "CA", "N", "H"),
    ("HA", "CA", "C"),  # CA proton
    ("O", "C", "CA", "HA"),
    ("HA2", "CA", "C"),  # gly CA proton
    ("O", "C", "CA", "HA2"),
    ("HA3", "CA", "C"),  # gly CA proton
    ("O", "C", "CA", "HA3"),
    ("N", "CA", "CB"),
    ("N", "CA", "CB", "HB"),  # CB protons
    ("N", "CA", "CB", "HB1"),
    ("N", "CA", "CB", "HB2"),
    ("N", "CA", "CB", "HB3"),
    ("CA", "CB", "HB"),
    ("CA", "CB", "HB1"),
    ("CA", "CB", "HB2"),
    ("CA", "CB", "HB3"),
    ("H1", "N", "CA"),  # chain start protons
    ("H2", "N", "CA"),
    ("H3", "N", "CA"),
    ("C", "CA", "N", "H1"),
    ("C", "CA", "N", "H2"),
    ("C", "CA", "N", "H3"),
)

# Sidechain hedra and dihedra.
# see http://www.mlb.co.jp/linux/science/garlic/doc/commands/dihedrals.html
# for official chi angles
ic_data_sidechains = {
    "V": (
        ("CA", "CB", "CG1"),
        ("N", "CA", "CB", "CG1", "chi1"),  # chi1
        ("CA", "CB", "CG2"),
        ("N", "CA", "CB", "CG2"),
        ("CB", "CG1", "HG11"),
        ("CB", "CG1", "HG12"),
        ("CB", "CG1", "HG13"),
        ("CB", "CG2", "HG21"),
        ("CB", "CG2", "HG22"),
        ("CB", "CG2", "HG23"),
        ("CA", "CB", "CG1", "HG11"),
        ("CA", "CB", "CG1", "HG12"),
        ("CA", "CB", "CG1", "HG13"),
        ("CA", "CB", "CG2", "HG21"),
        ("CA", "CB", "CG2", "HG22"),
        ("CA", "CB", "CG2", "HG23"),
    ),
    "L": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD1"),
        ("CA", "CB", "CG", "CD1", "chi2"),  # chi2
        ("CB", "CG", "CD2"),
        ("CA", "CB", "CG", "CD2"),
        ("CB", "CG", "HG"),
        ("CA", "CB", "CG", "HG"),
        ("CG", "CD1", "HD11"),
        ("CG", "CD1", "HD12"),
        ("CG", "CD1", "HD13"),
        ("CG", "CD2", "HD21"),
        ("CG", "CD2", "HD22"),
        ("CG", "CD2", "HD23"),
        ("CB", "CG", "CD1", "HD11"),
        ("CB", "CG", "CD1", "HD12"),
        ("CB", "CG", "CD1", "HD13"),
        ("CB", "CG", "CD2", "HD21"),
        ("CB", "CG", "CD2", "HD22"),
        ("CB", "CG", "CD2", "HD23"),
    ),
    "I": (
        ("CA", "CB", "CG1"),
        ("N", "CA", "CB", "CG1", "chi1"),  # chi1
        ("CB", "CG1", "CD1"),
        ("CA", "CB", "CG1", "CD1", "chi2"),  # chi2
        ("CA", "CB", "CG2"),
        ("N", "CA", "CB", "CG2"),
        ("CB", "CG1", "HG12"),
        ("CB", "CG1", "HG13"),
        ("CB", "CG2", "HG21"),
        ("CB", "CG2", "HG22"),
        ("CB", "CG2", "HG23"),
        ("CA", "CB", "CG1", "HG12"),
        ("CA", "CB", "CG1", "HG13"),
        ("CA", "CB", "CG2", "HG21"),
        ("CA", "CB", "CG2", "HG22"),
        ("CA", "CB", "CG2", "HG23"),
        ("CG1", "CD1", "HD11"),
        ("CG1", "CD1", "HD12"),
        ("CG1", "CD1", "HD13"),
        ("CB", "CG1", "CD1", "HD11"),
        ("CB", "CG1", "CD1", "HD12"),
        ("CB", "CG1", "CD1", "HD13"),
    ),
    "M": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "SD"),
        ("CA", "CB", "CG", "SD", "chi2"),  # chi2
        ("CG", "SD", "CE"),
        ("CB", "CG", "SD", "CE", "chi3"),  # chi3
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("SD", "CE", "HE1"),
        ("SD", "CE", "HE2"),
        ("SD", "CE", "HE3"),
        ("CG", "SD", "CE", "HE1"),
        ("CG", "SD", "CE", "HE2"),
        ("CG", "SD", "CE", "HE3"),
    ),
    "F": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD1"),
        ("CA", "CB", "CG", "CD1", "chi2"),  # chi2
        ("CG", "CD1", "CE1"),
        ("CB", "CG", "CD1", "CE1"),
        ("CD1", "CE1", "CZ"),
        ("CG", "CD1", "CE1", "CZ"),
        ("CB", "CG", "CD2"),
        ("CA", "CB", "CG", "CD2"),
        ("CG", "CD2", "CE2"),
        ("CB", "CG", "CD2", "CE2"),
        ("CG", "CD1", "HD1"),
        ("CB", "CG", "CD1", "HD1"),
        ("CG", "CD2", "HD2"),
        ("CB", "CG", "CD2", "HD2"),
        ("CD1", "CE1", "HE1"),
        ("CG", "CD1", "CE1", "HE1"),
        ("CD2", "CE2", "HE2"),
        ("CG", "CD2", "CE2", "HE2"),
        ("CE1", "CZ", "HZ"),
        ("CD1", "CE1", "CZ", "HZ"),
    ),
    "P": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD"),
        ("CA", "CB", "CG", "CD", "chi2"),  # chi2
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("CG", "CD", "HD2"),
        ("CG", "CD", "HD3"),
        ("CB", "CG", "CD", "HD2"),
        ("CB", "CG", "CD", "HD3"),
    ),
    "S": (
        ("CA", "CB", "OG"),
        ("N", "CA", "CB", "OG", "chi1"),  # chi1
        ("CB", "OG", "HG"),
        ("CA", "CB", "OG", "HG"),
    ),
    "T": (
        ("CA", "CB", "OG1"),
        ("N", "CA", "CB", "OG1", "chi1"),  # chi1
        ("CA", "CB", "CG2"),
        ("N", "CA", "CB", "CG2"),
        ("CB", "OG1", "HG1"),
        ("CA", "CB", "OG1", "HG1"),
        ("CB", "CG2", "HG21"),
        ("CB", "CG2", "HG22"),
        ("CB", "CG2", "HG23"),
        ("CA", "CB", "CG2", "HG21"),
        ("CA", "CB", "CG2", "HG22"),
        ("CA", "CB", "CG2", "HG23"),
    ),
    "C": (
        ("CA", "CB", "SG"),
        ("N", "CA", "CB", "SG", "chi1"),  # chi1
        ("CB", "SG", "HG"),
        ("CA", "CB", "SG", "HG"),
    ),
    "N": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "OD1"),
        ("CA", "CB", "CG", "OD1", "chi2"),  # chi2
        ("CB", "CG", "ND2"),
        ("CA", "CB", "CG", "ND2"),
        ("CG", "ND2", "HD21"),
        ("CG", "ND2", "HD22"),
        ("CB", "CG", "ND2", "HD21"),
        ("CB", "CG", "ND2", "HD22"),
    ),
    "Q": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD"),
        ("CA", "CB", "CG", "CD", "chi2"),  # chi2
        ("CG", "CD", "OE1"),
        ("CB", "CG", "CD", "OE1", "chi3"),  # chi3
        ("CG", "CD", "NE2"),
        ("CB", "CG", "CD", "NE2"),
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("CD", "NE2", "HE21"),
        ("CD", "NE2", "HE22"),
        ("CG", "CD", "NE2", "HE21"),
        ("CG", "CD", "NE2", "HE22"),
    ),
    "Y": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD1"),
        ("CA", "CB", "CG", "CD1", "chi2"),  # chi2
        ("CG", "CD1", "CE1"),
        ("CB", "CG", "CD1", "CE1"),
        ("CD1", "CE1", "CZ"),
        ("CG", "CD1", "CE1", "CZ"),
        ("CE1", "CZ", "OH"),
        ("CD1", "CE1", "CZ", "OH"),
        ("CB", "CG", "CD2"),
        ("CA", "CB", "CG", "CD2"),
        ("CG", "CD2", "CE2"),
        ("CB", "CG", "CD2", "CE2"),
        ("CG", "CD1", "HD1"),
        ("CB", "CG", "CD1", "HD1"),
        ("CG", "CD2", "HD2"),
        ("CB", "CG", "CD2", "HD2"),
        ("CD1", "CE1", "HE1"),
        ("CG", "CD1", "CE1", "HE1"),
        ("CD2", "CE2", "HE2"),
        ("CG", "CD2", "CE2", "HE2"),
        ("CZ", "OH", "HH"),
        ("CE1", "CZ", "OH", "HH"),
    ),
    "W": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD1"),
        ("CA", "CB", "CG", "CD1", "chi2"),  # chi2
        ("CG", "CD1", "NE1"),
        ("CB", "CG", "CD1", "NE1"),
        ("CB", "CG", "CD2"),
        ("CA", "CB", "CG", "CD2"),
        ("CG", "CD2", "CE2"),
        ("CB", "CG", "CD2", "CE2"),
        ("CD2", "CE2", "CZ2"),
        ("CG", "CD2", "CE2", "CZ2"),
        ("CE2", "CZ2", "CH2"),
        ("CD2", "CE2", "CZ2", "CH2"),
        ("CE2", "CZ2", "HZ2"),
        ("CD2", "CE2", "CZ2", "HZ2"),
        ("CG", "CD2", "CE3"),
        ("CB", "CG", "CD2", "CE3"),
        ("CZ2", "CH2", "CZ3"),
        ("CE2", "CZ2", "CH2", "CZ3"),
        ("CG", "CD1", "HD1"),
        ("CB", "CG", "CD1", "HD1"),
        ("CD1", "NE1", "HE1"),
        ("CG", "CD1", "NE1", "HE1"),
        ("CD2", "CE3", "HE3"),
        ("CG", "CD2", "CE3", "HE3"),
        ("CH2", "CZ3", "HZ3"),
        ("CZ2", "CH2", "CZ3", "HZ3"),
        ("CZ2", "CH2", "HH2"),
        ("CE2", "CZ2", "CH2", "HH2"),
    ),
    "D": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "OD1"),
        ("CA", "CB", "CG", "OD1", "chi2"),  # chi2
        ("CB", "CG", "OD2"),
        ("CA", "CB", "CG", "OD2"),
        ("CG", "OD2", "HD2"),
        ("CB", "CG", "OD2", "HD2"),
    ),
    "E": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD"),
        ("CA", "CB", "CG", "CD", "chi2"),  # chi2
        ("CG", "CD", "OE1"),
        ("CB", "CG", "CD", "OE1", "chi3"),  # chi3
        ("CG", "CD", "OE2"),
        ("CB", "CG", "CD", "OE2"),
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("CD", "OE2", "HE2"),
        ("CG", "CD", "OE2", "HE2"),
    ),
    "H": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "ND1"),
        ("CA", "CB", "CG", "ND1", "chi2"),  # chi2
        ("CG", "ND1", "CE1"),
        ("CB", "CG", "ND1", "CE1"),
        ("CB", "CG", "CD2"),
        ("CA", "CB", "CG", "CD2"),
        ("CG", "CD2", "NE2"),
        ("CB", "CG", "CD2", "NE2"),
        ("CG", "ND1", "HD1"),
        ("CB", "CG", "ND1", "HD1"),
        ("CG", "CD2", "HD2"),
        ("CB", "CG", "CD2", "HD2"),
        ("ND1", "CE1", "HE1"),
        ("CG", "ND1", "CE1", "HE1"),
        ("CD2", "NE2", "HE2"),
        ("CG", "CD2", "NE2", "HE2"),
    ),
    "K": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD"),
        ("CA", "CB", "CG", "CD", "chi2"),  # chi2
        ("CG", "CD", "CE"),
        ("CB", "CG", "CD", "CE", "chi3"),  # chi3
        ("CD", "CE", "NZ"),
        ("CG", "CD", "CE", "NZ", "chi4"),  # chi4
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("CG", "CD", "HD2"),
        ("CG", "CD", "HD3"),
        ("CB", "CG", "CD", "HD2"),
        ("CB", "CG", "CD", "HD3"),
        ("CD", "CE", "HE2"),
        ("CD", "CE", "HE3"),
        ("CG", "CD", "CE", "HE2"),
        ("CG", "CD", "CE", "HE3"),
        ("CE", "NZ", "HZ1"),
        ("CE", "NZ", "HZ2"),
        ("CE", "NZ", "HZ3"),
        ("CD", "CE", "NZ", "HZ1"),
        ("CD", "CE", "NZ", "HZ2"),
        ("CD", "CE", "NZ", "HZ3"),
    ),
    "R": (
        ("CA", "CB", "CG"),
        ("N", "CA", "CB", "CG", "chi1"),  # chi1
        ("CB", "CG", "CD"),
        ("CA", "CB", "CG", "CD", "chi2"),  # chi2
        ("CG", "CD", "NE"),
        ("CB", "CG", "CD", "NE", "chi3"),  # chi3
        ("CD", "NE", "CZ"),
        ("CG", "CD", "NE", "CZ", "chi4"),  # chi4
        ("NE", "CZ", "NH1"),
        ("CD", "NE", "CZ", "NH1", "chi5"),  # chi5
        ("NE", "CZ", "NH2"),
        ("CD", "NE", "CZ", "NH2"),
        ("CB", "CG", "HG2"),
        ("CB", "CG", "HG3"),
        ("CA", "CB", "CG", "HG2"),
        ("CA", "CB", "CG", "HG3"),
        ("CG", "CD", "HD2"),
        ("CG", "CD", "HD3"),
        ("CB", "CG", "CD", "HD2"),
        ("CB", "CG", "CD", "HD3"),
        ("CD", "NE", "HE"),
        ("CG", "CD", "NE", "HE"),
        ("CZ", "NH1", "HH11"),
        ("CZ", "NH1", "HH12"),
        ("NE", "CZ", "NH1", "HH11"),
        ("NE", "CZ", "NH1", "HH12"),
        ("CZ", "NH2", "HH21"),
        ("CZ", "NH2", "HH22"),
        ("NE", "CZ", "NH2", "HH21"),
        ("NE", "CZ", "NH2", "HH22"),
    ),
}

# Additional sidechain entries for explicit bonds.

# OpenSCAD output requires specification of bonds to be rendered as cylinders.
# These entries define hedra and dihedra to explicitly cover all bonds in
# rings, otherwise the entries above only capture atoms.

ic_data_sidechain_extras = {
    "F": (("CE1", "CZ", "CE2"), ("CD1", "CE1", "CZ", "CE2")),
    "P": (("CG", "CD", "N"), ("CB", "CG", "CD", "N")),
    "Y": (("CE1", "CZ", "CE2"), ("CD1", "CE1", "CZ", "CE2")),
    "W": (
        ("CD2", "CE3", "CZ3"),
        ("CG", "CD2", "CE3", "CZ3"),
        ("CD1", "NE1", "CE2"),
        ("CG", "CD1", "NE1", "CE2"),
    ),
    "H": (("ND1", "CE1", "NE2"), ("CG", "ND1", "CE1", "NE2")),
}

# sidechain primary angles dictionary

# """
primary_angles = {
    "CACBCG1CD1",
    "CACBCGCD",
    "CACBCGCD1",
    "CACBCGND1",
    "CACBCGOD1",
    "CACBCGSD",
    "CBCGCDCE",
    "CBCGCDNE",
    "CBCGCDOE1",
    "CBCGSDCE",
    "CDNECZNH1",
    "CGCDCENZ",
    "CGCDNECZ",
    "NCACBCG",
    "NCACBCG1",
    "NCACBOG",
    "NCACBOG1",
    "NCACBSG",
    "NCACOXT",
    # non-standard because considered constant
    "CBCGCD1CE1",  # phe, tyr
    "CGCD1CE1CZ",  # phe, tyr
    "CBCGCD2CE2",  # phe, tyr
    "CBCGND1CE1",  # his
    "CBCGCD2NE2",  # his
    "CD1CE1CZOH",  # tyr
    "CBCGCD1NE1",  # trp
    "CGCD2CE2CZ2",  # trp
    "CD2CE2CZ2CH2",  # trp
    "CE2CZ2CH2CZ3",  # trp
}
# """

# Covalent radii for OpenSCAD output.

# Covalent radii from Heyrovska, Raji : 'Atomic Structures of all the Twenty
# Essential Amino Acids and a Tripeptide, with Bond Lengths as Sums of Atomic
# Covalent Radii' <https://arxiv.org/pdf/0804.2488.pdf>
# Adding Ores between Osb and Odb for Asp and Glu, Nres between Nsb and Ndb
# for Arg, as PDB does not specify

covalent_radii = {
    "Csb": 0.77,
    "Cres": 0.72,
    "Cdb": 0.67,
    "Osb": 0.67,
    "Ores": 0.635,
    "Odb": 0.60,
    "Nsb": 0.70,
    "Nres": 0.66,
    "Ndb": 0.62,
    "Ssb": 1.04,
    "Hsb": 0.37,
}

# Atom classes based on Heyrovska, Raji covalent radii paper.

residue_atom_bond_state = {
    "X": {
        "N": "Nsb",
        "CA": "Csb",
        "C": "Cdb",
        "O": "Odb",
        "OXT": "Osb",
        "CB": "Csb",
        "H": "Hsb",
    },
    "V": {"CG1": "Csb", "CG2": "Csb"},
    "L": {"CG": "Csb", "CD1": "Csb", "CD2": "Csb"},
    "I": {"CG1": "Csb", "CG2": "Csb", "CD1": "Csb"},
    "M": {"CG": "Csb", "SD": "Ssb", "CE": "Csb"},
    "F": {
        "CG": "Cdb",
        "CD1": "Cres",
        "CD2": "Cres",
        "CE1": "Cdb",
        "CE2": "Cdb",
        "CZ": "Cres",
    },
    "P": {"CG": "Csb", "CD": "Csb"},
    "S": {"OG": "Osb"},
    "T": {"OG1": "Osb", "CG2": "Csb"},
    "C": {"SG": "Ssb"},
    "N": {"CG": "Csb", "OD1": "Odb", "ND2": "Ndb"},
    "Q": {"CG": "Csb", "CD": "Csb", "OE1": "Odb", "NE2": "Ndb"},
    "Y": {
        "CG": "Cdb",
        "CD1": "Cres",
        "CD2": "Cres",
        "CE1": "Cdb",
        "CE2": "Cdb",
        "CZ": "Cres",
        "OH": "Osb",
    },
    "W": {
        "CG": "Cdb",
        "CD1": "Cdb",
        "CD2": "Cres",
        "NE1": "Nsb",
        "CE2": "Cdb",
        "CE3": "Cdb",
        "CZ2": "Cres",
        "CZ3": "Cres",
        "CH2": "Cdb",
    },
    "D": {"CG": "Csb", "OD1": "Ores", "OD2": "Ores"},
    "E": {"CG": "Csb", "CD": "Csb", "OE1": "Ores", "OE2": "Ores"},
    "H": {"CG": "Cdb", "CD2": "Cdb", "ND1": "Nsb", "CE1": "Cdb", "NE2": "Ndb"},
    "K": {"CG": "Csb", "CD": "Csb", "CE": "Csb", "NZ": "Nsb"},
    "R": {
        "CG": "Csb",
        "CD": "Csb",
        "NE": "Nsb",
        "CZ": "Cdb",
        "NH1": "Nres",
        "NH2": "Nres",
    },
}


# atomic weights of C,O,N,H,S
atomic_weight = {
    "C": 12.0107,
    "O": 15.9994,
    "N": 14.0067,
    "H": 1.0079,
    "S": 32.065,
}

# electronegativity values for C,O,N,H,S
electronegativity = {"C": 2.55, "O": 3.44, "N": 3.04, "H": 2.20, "S": 2.58}

# hedra and dihedra default values reference database:
#
# generated from Pisces cullpdb cullpdb_pc20_res2.2_R1.0_d190812_chains8767.gz
# (Aug 2019) https://dunbrack.fccc.edu/pisces
# G. Wang and R. L. Dunbrack, Jr. PISCES: a protein sequence culling server.
# Bioinformatics, 19:1589-1591, 2003.
#
# hedra defaults based on residue-atom classes from reference database
#
# entry format is
# [(len1, angle, len3 averages), angle_sd]  # len1, len3 std dev, [total count]
#
# angle standard deviation is used by  :func:`ic_data.write_PIC`
#
# N.B. these are just intended as reasonable starting values.  Despite the low
# standard deviations, structures degrade quickly with any variation from true
# measurements.

# 226 total entries
hedra_defaults = {
    # [(averages), angle_sd]  # len1, len3 std dev, [total count]
    "ACAACAO": [
        (1.52579, 120.49833, 1.23331),
        0.91020,
    ],  # 0.01155  0.01086 [37394]
    "ACAACAOXT": [
        (1.52535, 119.74925, 1.24174),
        2.62360,
    ],  # 0.01142  0.01116 [81]
    "ACAACXN": [
        (1.52577, 116.69299, 1.33094),
        1.11553,
    ],  # 0.01152  0.00834 [36750]
    "ACBACAAC": [
        (1.52608, 110.28430, 1.52579),
        0.94952,
    ],  # 0.01629  0.01155 [37388]
    "AHANACA": [
        (0.86805, 114.48737, 1.45832),
        1.27049,
    ],  # 0.03005  0.00756 [2765]
    "ANACAAC": [
        (1.46120, 111.05839, 1.52579),
        2.18854,
    ],  # 0.01140  0.01155 [37398]
    "ANACAACB": [
        (1.46120, 110.40921, 1.52608),
        0.88125,
    ],  # 0.01140  0.01629 [37388]
    "CCACCBCSG": [
        (1.52946, 113.86203, 1.80753),
        2.04695,
    ],  # 0.01183  0.02318 [5998]
    "CCACCCO": [
        (1.52388, 120.45381, 1.23272),
        0.96573,
    ],  # 0.01181  0.01089 [6007]
    "CCACCCOXT": [
        (1.52380, 118.37897, 1.24397),
        1.65356,
    ],  # 0.01273  0.01088 [14]
    "CCACCXN": [
        (1.52384, 116.58489, 1.33066),
        1.24455,
    ],  # 0.01182  0.00862 [5926]
    "CCBCCACC": [
        (1.52946, 109.94838, 1.52388),
        1.64395,
    ],  # 0.01183  0.01181 [6003]
    "CHCNCCA": [
        (0.87646, 114.49849, 1.45766),
        1.61226,
    ],  # 0.04363  0.00890 [520]
    "CNCCACC": [
        (1.45964, 110.93113, 1.52388),
        2.48827,
    ],  # 0.01157  0.01181 [6007]
    "CNCCACCB": [
        (1.45963, 110.53598, 1.52946),
        1.28524,
    ],  # 0.01158  0.01183 [6003]
    "DCADCBDCG": [
        (1.53233, 113.01339, 1.51951),
        1.09007,
    ],  # 0.01090  0.01684 [27365]
    "DCADCDO": [
        (1.52681, 120.45665, 1.23263),
        0.97342,
    ],  # 0.01191  0.01077 [27641]
    "DCADCDOXT": [
        (1.52718, 120.50013, 1.24705),
        4.69659,
    ],  # 0.01149  0.05767 [39]
    "DCADCXN": [
        (1.52680, 116.74172, 1.33107),
        1.21325,
    ],  # 0.01191  0.00855 [27228]
    "DCBDCADC": [
        (1.53235, 110.14545, 1.52681),
        1.77412,
    ],  # 0.01090  0.01191 [27628]
    "DCBDCGDOD1": [
        (1.51951, 119.15528, 1.25091),
        1.65280,
    ],  # 0.01685  0.02470 [27345]
    "DCBDCGDOD2": [
        (1.51951, 118.08816, 1.25102),
        1.70244,
    ],  # 0.01685  0.02107 [27345]
    "DHDNDCA": [
        (0.86939, 114.26667, 1.45930),
        1.22719,
    ],  # 0.03380  0.00813 [1934]
    "DNDCADC": [
        (1.46141, 111.03743, 1.52681),
        2.63643,
    ],  # 0.01184  0.01191 [27642]
    "DNDCADCB": [
        (1.46141, 110.56192, 1.53235),
        1.31965,
    ],  # 0.01184  0.01090 [27628]
    "ECAECBECG": [
        (1.53167, 113.93062, 1.52189),
        1.86367,
    ],  # 0.01056  0.01867 [31977]
    "ECAECEO": [
        (1.52593, 120.44996, 1.23262),
        0.95085,
    ],  # 0.01185  0.01072 [32752]
    "ECAECEOXT": [
        (1.52528, 119.67583, 1.24355),
        2.10431,
    ],  # 0.01091  0.02096 [73]
    "ECAECXN": [
        (1.52591, 116.74841, 1.33073),
        1.16616,
    ],  # 0.01183  0.00811 [32144]
    "ECBECAEC": [
        (1.53172, 110.13602, 1.52593),
        1.58401,
    ],  # 0.01052  0.01184 [32734]
    "ECBECGECD": [
        (1.52187, 113.25685, 1.52165),
        1.74528,
    ],  # 0.01868  0.02069 [31790]
    "ECGECDEOE1": [
        (1.52164, 118.96293, 1.25216),
        1.47937,
    ],  # 0.02070  0.01122 [31755]
    "ECGECDEOE2": [
        (1.52164, 118.11536, 1.25165),
        1.45495,
    ],  # 0.02070  0.01301 [31752]
    "EHENECA": [
        (0.86888, 114.41246, 1.45880),
        1.49683,
    ],  # 0.03581  0.01978 [2274]
    "ENECAEC": [
        (1.46058, 111.15787, 1.52593),
        2.27719,
    ],  # 0.01226  0.01185 [32753]
    "ENECAECB": [
        (1.46059, 110.55046, 1.53172),
        1.23650,
    ],  # 0.01226  0.01052 [32734]
    "FCAFCBFCG": [
        (1.53345, 113.82114, 1.50431),
        1.13141,
    ],  # 0.01162  0.01303 [18987]
    "FCAFCFO": [
        (1.52456, 120.49095, 1.23299),
        0.92876,
    ],  # 0.01188  0.01129 [19038]
    "FCAFCFOXT": [
        (1.52401, 119.63248, 1.24271),
        1.73167,
    ],  # 0.01429  0.01193 [42]
    "FCAFCXN": [
        (1.52453, 116.58552, 1.33064),
        1.20900,
    ],  # 0.01187  0.00855 [18775]
    "FCBFCAFC": [
        (1.53345, 110.10780, 1.52457),
        1.76722,
    ],  # 0.01161  0.01188 [19036]
    "FCBFCGFCD1": [
        (1.50431, 120.70752, 1.38869),
        0.74543,
    ],  # 0.01303  0.01110 [18983]
    "FCBFCGFCD2": [
        (1.50431, 120.54385, 1.38836),
        0.68888,
    ],  # 0.01303  0.01104 [18983]
    "FCD1FCE1FCZ": [
        (1.39121, 119.89822, 1.38441),
        0.58792,
    ],  # 0.01538  0.01516 [18982]
    "FCGFCD1FCE1": [
        (1.38869, 120.78060, 1.39121),
        0.56650,
    ],  # 0.01110  0.01538 [18982]
    "FCGFCD2FCE2": [
        (1.38836, 120.79762, 1.39086),
        0.57719,
    ],  # 0.01104  0.01495 [18982]
    "FHFNFCA": [
        (0.86889, 114.28542, 1.45775),
        1.24716,
    ],  # 0.03367  0.00836 [1374]
    "FNFCAFC": [
        (1.46000, 110.81237, 1.52457),
        2.47224,
    ],  # 0.01149  0.01188 [19039]
    "FNFCAFCB": [
        (1.46000, 110.60289, 1.53345),
        1.35478,
    ],  # 0.01150  0.01161 [19037]
    "GCAGCGO": [
        (1.51677, 120.48820, 1.23313),
        1.09788,
    ],  # 0.00984  0.01079 [31845]
    "GCAGCGOXT": [
        (1.51836, 120.01750, 1.24928),
        4.04221,
    ],  # 0.01201  0.03755 [57]
    "GCAGCXN": [
        (1.51675, 116.65649, 1.33072),
        1.35033,
    ],  # 0.00983  0.00853 [31254]
    "GHGNGCA": [
        (0.86934, 114.35896, 1.45001),
        1.33528,
    ],  # 0.03309  0.00781 [2219]
    "GNGCAGC": [
        (1.45534, 113.13469, 1.51677),
        2.73621,
    ],  # 0.01227  0.00984 [31847]
    "HCAHCBHCG": [
        (1.53191, 113.63847, 1.49588),
        1.08749,
    ],  # 0.01218  0.00823 [10862]
    "HCAHCHO": [
        (1.52435, 120.44439, 1.23267),
        0.98586,
    ],  # 0.01258  0.01107 [10943]
    "HCAHCHOXT": [
        (1.51933, 119.63321, 1.24306),
        1.32706,
    ],  # 0.01541  0.01146 [30]
    "HCAHCXN": [
        (1.52433, 116.66524, 1.33078),
        1.24379,
    ],  # 0.01261  0.00914 [10708]
    "HCBHCAHC": [
        (1.53191, 110.12661, 1.52435),
        1.69525,
    ],  # 0.01215  0.01258 [10944]
    "HCBHCGHCD2": [
        (1.49588, 131.02926, 1.35512),
        0.88822,
    ],  # 0.00823  0.00773 [10855]
    "HCBHCGHND1": [
        (1.49588, 122.63934, 1.37822),
        0.85608,
    ],  # 0.00823  0.00799 [10855]
    "HCGHCD2HNE2": [
        (1.35512, 107.18603, 1.37374),
        0.46636,
    ],  # 0.00773  0.00737 [10854]
    "HCGHND1HCE1": [
        (1.37822, 109.16038, 1.32333),
        0.81076,
    ],  # 0.00799  0.00842 [10854]
    "HHHNHCA": [
        (0.86918, 114.34031, 1.45782),
        1.46340,
    ],  # 0.03246  0.00853 [815]
    "HNHCAHC": [
        (1.46046, 111.09335, 1.52435),
        2.58066,
    ],  # 0.01208  0.01258 [10946]
    "HNHCAHCB": [
        (1.46046, 110.46512, 1.53191),
        1.38925,
    ],  # 0.01208  0.01215 [10944]
    "ICAICBICG1": [
        (1.54567, 110.42270, 1.53337),
        1.28403,
    ],  # 0.01493  0.01361 [27238]
    "ICAICBICG2": [
        (1.54567, 110.72098, 1.53061),
        1.04463,
    ],  # 0.01493  0.01830 [27237]
    "ICAICIO": [
        (1.52636, 120.50872, 1.23343),
        0.88888,
    ],  # 0.01177  0.01089 [27316]
    "ICAICIOXT": [
        (1.52902, 119.47607, 1.24279),
        3.18851,
    ],  # 0.01090  0.01223 [46]
    "ICAICXN": [
        (1.52634, 116.66184, 1.33089),
        1.13349,
    ],  # 0.01179  0.00937 [26993]
    "ICBICAIC": [
        (1.54566, 110.43886, 1.52636),
        1.73720,
    ],  # 0.01492  0.01177 [27314]
    "ICBICG1ICD1": [
        (1.53337, 113.96450, 1.51990),
        1.43141,
    ],  # 0.01361  0.02677 [27229]
    "IHINICA": [
        (0.86749, 114.20034, 1.45804),
        1.26213,
    ],  # 0.03137  0.00805 [1899]
    "INICAIC": [
        (1.46029, 109.82832, 1.52636),
        2.41611,
    ],  # 0.01165  0.01177 [27316]
    "INICAICB": [
        (1.46029, 111.38983, 1.54566),
        1.29388,
    ],  # 0.01165  0.01492 [27314]
    "KCAKCBKCG": [
        (1.53126, 114.04683, 1.52169),
        1.74391,
    ],  # 0.01054  0.01758 [26244]
    "KCAKCKO": [
        (1.52595, 120.47261, 1.23256),
        0.95867,
    ],  # 0.01341  0.01075 [27000]
    "KCAKCKOXT": [
        (1.52688, 119.77013, 1.24180),
        2.43662,
    ],  # 0.01084  0.01157 [93]
    "KCAKCXN": [
        (1.52593, 116.70069, 1.33064),
        1.17674,
    ],  # 0.01345  0.00816 [26509]
    "KCBKCAKC": [
        (1.53130, 110.09100, 1.52595),
        1.49272,
    ],  # 0.01049  0.01342 [26987]
    "KCBKCGKCD": [
        (1.52167, 111.57829, 1.52379),
        1.94916,
    ],  # 0.01759  0.02433 [25968]
    "KCDKCEKNZ": [
        (1.52230, 111.75306, 1.49305),
        2.32602,
    ],  # 0.01893  0.01971 [25629]
    "KCGKCDKCE": [
        (1.52369, 111.62887, 1.52231),
        1.75651,
    ],  # 0.02411  0.01892 [25714]
    "KHKNKCA": [
        (0.86934, 114.33829, 1.45890),
        1.23454,
    ],  # 0.03511  0.02392 [1822]
    "KNKCAKC": [
        (1.46080, 111.08063, 1.52595),
        2.33744,
    ],  # 0.01278  0.01341 [27001]
    "KNKCAKCB": [
        (1.46080, 110.54772, 1.53130),
        1.17805,
    ],  # 0.01278  0.01049 [26987]
    "LCALCBLCG": [
        (1.53140, 116.33839, 1.53041),
        2.65234,
    ],  # 0.01072  0.01173 [43982]
    "LCALCLO": [
        (1.52503, 120.41783, 1.23287),
        0.90118,
    ],  # 0.01180  0.01101 [44095]
    "LCALCLOXT": [
        (1.52511, 119.77807, 1.24171),
        2.96371,
    ],  # 0.01649  0.01938 [97]
    "LCALCXN": [
        (1.52501, 116.73371, 1.33088),
        1.11117,
    ],  # 0.01162  0.01016 [43450]
    "LCBLCALC": [
        (1.53140, 109.99081, 1.52503),
        1.43419,
    ],  # 0.01072  0.01180 [44095]
    "LCBLCGLCD1": [
        (1.53041, 110.43155, 1.52397),
        1.76987,
    ],  # 0.01173  0.01785 [43971]
    "LCBLCGLCD2": [
        (1.53041, 110.63050, 1.52450),
        1.73888,
    ],  # 0.01173  0.01808 [43969]
    "LHLNLCA": [
        (0.87043, 114.38647, 1.45799),
        1.36194,
    ],  # 0.03526  0.00840 [3044]
    "LNLCALC": [
        (1.46035, 110.90523, 1.52503),
        2.36226,
    ],  # 0.01171  0.01180 [44096]
    "LNLCALCB": [
        (1.46035, 110.50322, 1.53140),
        1.14993,
    ],  # 0.01171  0.01072 [44095]
    "MCAMCBMCG": [
        (1.53120, 113.86413, 1.52052),
        2.01777,
    ],  # 0.01199  0.02222 [7201]
    "MCAMCMO": [
        (1.52499, 120.43685, 1.23294),
        0.98426,
    ],  # 0.01184  0.01131 [7251]
    "MCAMCMOXT": [
        (1.52632, 119.62203, 1.24398),
        1.70106,
    ],  # 0.01036  0.00862 [7]
    "MCAMCXN": [
        (1.52498, 116.72073, 1.33086),
        1.22277,
    ],  # 0.01186  0.00891 [7191]
    "MCBMCAMC": [
        (1.53120, 110.04147, 1.52500),
        1.60511,
    ],  # 0.01199  0.01184 [7248]
    "MCBMCGMSD": [
        (1.52051, 112.65759, 1.80619),
        2.73841,
    ],  # 0.02223  0.02344 [7198]
    "MCGMSDMCE": [
        (1.80620, 100.62190, 1.78579),
        2.41850,
    ],  # 0.02344  0.03908 [7196]
    "MHMNMCA": [
        (0.86915, 114.41773, 1.45852),
        1.45978,
    ],  # 0.03366  0.00880 [595]
    "MNMCAMC": [
        (1.46091, 110.97184, 1.52499),
        2.26945,
    ],  # 0.01368  0.01184 [7251]
    "MNMCAMCB": [
        (1.46091, 110.56896, 1.53120),
        1.27435,
    ],  # 0.01368  0.01199 [7248]
    "NCANCBNCG": [
        (1.53197, 112.64738, 1.51608),
        1.12454,
    ],  # 0.01524  0.01660 [19907]
    "NCANCNO": [
        (1.52569, 120.44938, 1.23255),
        0.99273,
    ],  # 0.01205  0.01104 [20064]
    "NCANCNOXT": [
        (1.52885, 119.91220, 1.24238),
        2.46382,
    ],  # 0.01806  0.01795 [40]
    "NCANCXN": [
        (1.52568, 116.69676, 1.33090),
        1.23077,
    ],  # 0.01206  0.00847 [19740]
    "NCBNCANC": [
        (1.53197, 110.30885, 1.52570),
        1.73927,
    ],  # 0.01521  0.01205 [20053]
    "NCBNCGNND2": [
        (1.51608, 116.47117, 1.32781),
        1.19956,
    ],  # 0.01661  0.02243 [19898]
    "NCBNCGNOD1": [
        (1.51608, 120.89146, 1.23316),
        1.20758,
    ],  # 0.01660  0.02656 [19899]
    "NHNNNCA": [
        (0.86984, 114.38896, 1.45812),
        1.48714,
    ],  # 0.03532  0.00825 [1331]
    "NNNCANC": [
        (1.46048, 111.52947, 1.52569),
        2.76559,
    ],  # 0.01161  0.01205 [20066]
    "NNNCANCB": [
        (1.46048, 110.53625, 1.53197),
        1.33219,
    ],  # 0.01161  0.01521 [20054]
    "PCAPCBPCG": [
        (1.53273, 104.39244, 1.49426),
        1.38197,
    ],  # 0.00986  0.02559 [20886]
    "PCAPCPO": [
        (1.52595, 120.19725, 1.23340),
        1.18822,
    ],  # 0.01193  0.01091 [20902]
    "PCAPCPOXT": [
        (1.52734, 119.54010, 1.24292),
        2.65747,
    ],  # 0.00925  0.01366 [37]
    "PCAPCXN": [
        (1.52593, 116.77550, 1.33171),
        1.40651,
    ],  # 0.01183  0.23591 [20372]
    "PCBPCAPC": [
        (1.53273, 110.90929, 1.52595),
        1.33990,
    ],  # 0.00986  0.01193 [20903]
    "PCBPCGPCD": [
        (1.49426, 105.28138, 1.50552),
        2.19879,
    ],  # 0.02559  0.01875 [20886]
    "PHPNPCA": [
        (0.96061, 109.46806, 1.47178),
        0.00000,
    ],  # 0.00000  0.00000 [1]
    "PNPCAPC": [
        (1.46668, 112.64045, 1.52595),
        2.33215,
    ],  # 0.01057  0.01193 [20906]
    "PNPCAPCB": [
        (1.46669, 103.26041, 1.53273),
        0.69882,
    ],  # 0.01057  0.00986 [20904]
    "QCAQCBQCG": [
        (1.53106, 113.89466, 1.52254),
        1.83279,
    ],  # 0.01201  0.08019 [18045]
    "QCAQCQO": [
        (1.52562, 120.46310, 1.23271),
        0.91801,
    ],  # 0.01217  0.01129 [18318]
    "QCAQCQOXT": [
        (1.52576, 120.40130, 1.24054),
        2.17575,
    ],  # 0.01525  0.01248 [39]
    "QCAQCXN": [
        (1.52559, 116.71033, 1.33069),
        1.15544,
    ],  # 0.01217  0.00840 [17980]
    "QCBQCAQC": [
        (1.53109, 110.07668, 1.52562),
        1.64027,
    ],  # 0.01196  0.01217 [18315]
    "QCBQCGQCD": [
        (1.52256, 112.72750, 1.51803),
        1.63202,
    ],  # 0.08033  0.02059 [17972]
    "QCGQCDQNE2": [
        (1.51802, 116.43791, 1.32835),
        1.06786,
    ],  # 0.02060  0.01347 [17952]
    "QCGQCDQOE1": [
        (1.51802, 120.93630, 1.23433),
        1.02076,
    ],  # 0.02059  0.01374 [17953]
    "QHQNQCA": [
        (0.87040, 114.46502, 1.45862),
        1.42022,
    ],  # 0.03559  0.00758 [1282]
    "QNQCAQC": [
        (1.46078, 111.10228, 1.52562),
        2.34770,
    ],  # 0.01404  0.01217 [18320]
    "QNQCAQCB": [
        (1.46078, 110.56616, 1.53109),
        1.41193,
    ],  # 0.01404  0.01196 [18315]
    "RCARCBRCG": [
        (1.53122, 113.88552, 1.52077),
        1.85845,
    ],  # 0.01119  0.02076 [23691]
    "RCARCRO": [
        (1.52541, 120.47935, 1.23270),
        0.91912,
    ],  # 0.01164  0.01058 [24114]
    "RCARCROXT": [
        (1.52634, 120.17987, 1.24104),
        2.50707,
    ],  # 0.01129  0.01042 [61]
    "RCARCXN": [
        (1.52538, 116.68310, 1.33064),
        1.13002,
    ],  # 0.01164  0.00810 [23688]
    "RCBRCARC": [
        (1.53126, 110.10652, 1.52542),
        1.56579,
    ],  # 0.01116  0.01165 [24103]
    "RCBRCGRCD": [
        (1.52076, 111.73051, 1.52227),
        2.04838,
    ],  # 0.02076  0.01978 [23622]
    "RCDRNERCZ": [
        (1.46073, 124.72448, 1.33067),
        1.50694,
    ],  # 0.01228  0.00806 [23496]
    "RCGRCDRNE": [
        (1.52225, 111.67959, 1.46073),
        2.20249,
    ],  # 0.01979  0.01227 [23556]
    "RHRNRCA": [
        (0.86985, 114.36013, 1.45821),
        1.27468,
    ],  # 0.03443  0.00757 [1811]
    "RNERCZRNH1": [
        (1.33067, 120.61365, 1.32922),
        1.16135,
    ],  # 0.00806  0.00988 [23489]
    "RNERCZRNH2": [
        (1.33067, 119.72946, 1.32726),
        1.06991,
    ],  # 0.00806  0.00957 [23487]
    "RNRCARC": [
        (1.46051, 111.01141, 1.52541),
        2.33294,
    ],  # 0.01136  0.01164 [24114]
    "RNRCARCB": [
        (1.46051, 110.57356, 1.53126),
        1.25022,
    ],  # 0.01135  0.01116 [24103]
    "SCASCBSOG": [
        (1.53013, 110.77341, 1.41685),
        1.52308,
    ],  # 0.01158  0.01304 [28300]
    "SCASCSO": [
        (1.52539, 120.46650, 1.23290),
        1.01182,
    ],  # 0.01529  0.01088 [28433]
    "SCASCSOXT": [
        (1.52807, 120.91469, 1.24202),
        2.64919,
    ],  # 0.01641  0.01578 [51]
    "SCASCXN": [
        (1.52530, 116.64870, 1.33098),
        1.28030,
    ],  # 0.01214  0.01498 [27939]
    "SCBSCASC": [
        (1.53014, 109.87197, 1.52539),
        1.52276,
    ],  # 0.01158  0.01529 [28425]
    "SHSNSCA": [
        (0.86951, 114.30447, 1.45799),
        1.36681,
    ],  # 0.03443  0.02922 [1952]
    "SNSCASC": [
        (1.46027, 111.24019, 1.52539),
        2.45592,
    ],  # 0.01472  0.01529 [28434]
    "SNSCASCB": [
        (1.46027, 110.50165, 1.53014),
        1.23020,
    ],  # 0.01472  0.01158 [28425]
    "TCATCBTCG2": [
        (1.54072, 111.14321, 1.52457),
        1.13578,
    ],  # 0.01532  0.02075 [25482]
    "TCATCBTOG1": [
        (1.54072, 109.25291, 1.43241),
        1.13095,
    ],  # 0.01532  0.00917 [25489]
    "TCATCTO": [
        (1.52573, 120.48233, 1.23318),
        0.98286,
    ],  # 0.01190  0.01096 [25576]
    "TCATCTOXT": [
        (1.52755, 119.66033, 1.23887),
        1.40536,
    ],  # 0.01034  0.00954 [30]
    "TCATCXN": [
        (1.52571, 116.64426, 1.33079),
        1.22228,
    ],  # 0.01189  0.00878 [25220]
    "TCBTCATC": [
        (1.54072, 109.68682, 1.52573),
        1.87880,
    ],  # 0.01537  0.01190 [25569]
    "THTNTCA": [
        (0.86832, 114.26435, 1.45680),
        1.25833,
    ],  # 0.03213  0.00785 [1853]
    "TNTCATC": [
        (1.45979, 110.71436, 1.52573),
        2.46117,
    ],  # 0.01171  0.01190 [25576]
    "TNTCATCB": [
        (1.45979, 111.20333, 1.54072),
        1.44118,
    ],  # 0.01171  0.01537 [25569]
    "VCAVCBVCG1": [
        (1.54600, 110.76896, 1.52671),
        1.05246,
    ],  # 0.01458  0.01806 [31953]
    "VCAVCBVCG2": [
        (1.54600, 110.48323, 1.52593),
        1.13183,
    ],  # 0.01458  0.01841 [31953]
    "VCAVCVO": [
        (1.52633, 120.53418, 1.23364),
        0.88058,
    ],  # 0.01147  0.01072 [32001]
    "VCAVCVOXT": [
        (1.52365, 119.91629, 1.24178),
        2.30810,
    ],  # 0.01742  0.01279 [58]
    "VCAVCXN": [
        (1.52632, 116.61840, 1.33078),
        1.12023,
    ],  # 0.01146  0.00830 [31561]
    "VCBVCAVC": [
        (1.54599, 110.37127, 1.52633),
        1.67136,
    ],  # 0.01458  0.01147 [32000]
    "VHVNVCA": [
        (0.86865, 114.22727, 1.45822),
        1.30080,
    ],  # 0.03224  0.00769 [2372]
    "VNVCAVC": [
        (1.46011, 109.79444, 1.52633),
        2.41989,
    ],  # 0.01146  0.01147 [32003]
    "VNVCAVCB": [
        (1.46011, 111.50512, 1.54599),
        1.33066,
    ],  # 0.01146  0.01458 [32000]
    "WCAWCBWCG": [
        (1.53282, 113.97168, 1.49904),
        2.24736,
    ],  # 0.01277  0.01963 [6670]
    "WCAWCWO": [
        (1.52458, 120.45502, 1.23347),
        1.04358,
    ],  # 0.01632  0.01110 [6686]
    "WCAWCWOXT": [
        (1.52011, 119.00257, 1.24155),
        1.69651,
    ],  # 0.01311  0.01121 [13]
    "WCAWCXN": [
        (1.52456, 116.66104, 1.33076),
        1.27754,
    ],  # 0.01636  0.00901 [6603]
    "WCBWCAWC": [
        (1.53282, 110.03494, 1.52458),
        1.68844,
    ],  # 0.01277  0.01632 [6686]
    "WCBWCGWCD1": [
        (1.49905, 127.04161, 1.36776),
        0.82872,
    ],  # 0.01961  0.01209 [6667]
    "WCBWCGWCD2": [
        (1.49905, 126.62363, 1.43264),
        0.78472,
    ],  # 0.01961  0.01052 [6667]
    "WCD2WCE2WCZ2": [
        (1.41285, 122.37162, 1.39683),
        0.39598,
    ],  # 0.01076  0.01110 [6667]
    "WCE2WCZ2WCH2": [
        (1.39683, 117.46338, 1.37181),
        0.48283,
    ],  # 0.01110  0.01002 [6667]
    "WCGWCD1WNE1": [
        (1.36776, 110.09367, 1.37511),
        0.60391,
    ],  # 0.01209  0.01180 [6667]
    "WCGWCD2WCE2": [
        (1.43264, 107.20412, 1.41285),
        0.45148,
    ],  # 0.01052  0.01076 [6667]
    "WCGWCD2WCE3": [
        (1.43264, 133.89457, 1.40061),
        0.46905,
    ],  # 0.01052  0.00787 [6667]
    "WCZ2WCH2WCZ3": [
        (1.37181, 121.52060, 1.40362),
        0.48184,
    ],  # 0.01002  0.01255 [6667]
    "WHWNWCA": [
        (0.87227, 114.45217, 1.45809),
        1.56347,
    ],  # 0.03567  0.00824 [502]
    "WNWCAWC": [
        (1.46029, 110.90325, 1.52458),
        2.40138,
    ],  # 0.01173  0.01632 [6686]
    "WNWCAWCB": [
        (1.46029, 110.55141, 1.53282),
        1.35224,
    ],  # 0.01173  0.01277 [6686]
    "XCANACA": [
        (1.33090, 121.43998, 1.46106),
        1.42147,
    ],  # 0.00858  0.01120 [36738]
    "XCCNCCA": [
        (1.33032, 121.51876, 1.45957),
        1.56672,
    ],  # 0.00808  0.01146 [5949]
    "XCDNDCA": [
        (1.33068, 121.56274, 1.46130),
        1.61419,
    ],  # 0.00823  0.01171 [27198]
    "XCENECA": [
        (1.33029, 121.45065, 1.46048),
        1.51425,
    ],  # 0.00786  0.01216 [32234]
    "XCFNFCA": [
        (1.33047, 121.57766, 1.45994),
        1.60046,
    ],  # 0.00787  0.01139 [18837]
    "XCGNGCA": [
        (1.32948, 121.38416, 1.45504),
        1.48756,
    ],  # 0.01404  0.01181 [30946]
    "XCHNHCA": [
        (1.33052, 121.56887, 1.46026),
        1.70772,
    ],  # 0.00842  0.01180 [10739]
    "XCINICA": [
        (1.32998, 121.58807, 1.46022),
        1.51501,
    ],  # 0.00789  0.01155 [26996]
    "XCKNKCA": [
        (1.33029, 121.55854, 1.46068),
        1.49435,
    ],  # 0.00786  0.01264 [26578]
    "XCLNLCA": [
        (1.33033, 121.48377, 1.46027),
        1.44799,
    ],  # 0.00783  0.01161 [43582]
    "XCMNMCA": [
        (1.33037, 121.38091, 1.46012),
        1.61882,
    ],  # 0.00840  0.01278 [6955]
    "XCNNNCA": [
        (1.33062, 121.61726, 1.46036),
        1.69745,
    ],  # 0.00820  0.01139 [19720]
    "XCPNPCA": [
        (1.34101, 120.33021, 1.46667),
        2.55702,
    ],  # 0.01068  0.01051 [20520]
    "XCQNQCA": [
        (1.33224, 121.45675, 1.46058),
        1.77761,
    ],  # 0.25095  0.01136 [17998]
    "XCRNRCA": [
        (1.33033, 121.45120, 1.46045),
        1.54197,
    ],  # 0.00792  0.01125 [23753]
    "XCSNSCA": [
        (1.33027, 121.49507, 1.45999),
        1.58653,
    ],  # 0.00803  0.01432 [27790]
    "XCTNTCA": [
        (1.32996, 121.53022, 1.45965),
        1.53181,
    ],  # 0.00828  0.01146 [25119]
    "XCVNVCA": [
        (1.32997, 121.61608, 1.46006),
        1.51000,
    ],  # 0.01029  0.01136 [31661]
    "XCWNWCA": [
        (1.33060, 121.52771, 1.46022),
        1.60459,
    ],  # 0.00834  0.01156 [6627]
    "XCYNYCA": [
        (1.33039, 121.64133, 1.45990),
        1.67930,
    ],  # 0.00870  0.01180 [16676]
    "YCAYCBYCG": [
        (1.53316, 113.73017, 1.51069),
        2.10543,
    ],  # 0.01194  0.01218 [16797]
    "YCAYCXN": [
        (1.52434, 116.57590, 1.33050),
        1.23191,
    ],  # 0.01245  0.00901 [16620]
    "YCAYCYO": [
        (1.52434, 120.49480, 1.23293),
        0.93410,
    ],  # 0.01247  0.01112 [16834]
    "YCAYCYOXT": [
        (1.52321, 119.49565, 1.24463),
        2.34968,
    ],  # 0.01606  0.01854 [36]
    "YCBYCAYC": [
        (1.53316, 110.02329, 1.52434),
        1.69072,
    ],  # 0.01193  0.01247 [16834]
    "YCBYCGYCD1": [
        (1.51069, 120.93847, 1.39171),
        0.68236,
    ],  # 0.01218  0.01060 [16794]
    "YCBYCGYCD2": [
        (1.51069, 120.85238, 1.39117),
        0.65315,
    ],  # 0.01218  0.01080 [16794]
    "YCD1YCE1YCZ": [
        (1.38947, 119.53062, 1.38197),
        0.62808,
    ],  # 0.01503  0.01268 [16793]
    "YCE1YCZYOH": [
        (1.38198, 119.78227, 1.37715),
        1.31664,
    ],  # 0.01268  0.01162 [16792]
    "YCGYCD1YCE1": [
        (1.39171, 121.13695, 1.38947),
        0.53265,
    ],  # 0.01060  0.01503 [16794]
    "YCGYCD2YCE2": [
        (1.39117, 121.14721, 1.38918),
        0.55038,
    ],  # 0.01080  0.01471 [16793]
    "YHYNYCA": [
        (0.86913, 114.22549, 1.45737),
        1.30753,
    ],  # 0.03360  0.00806 [1204]
    "YNYCAYC": [
        (1.45996, 110.93364, 1.52434),
        2.48714,
    ],  # 0.01187  0.01247 [16834]
    "YNYCAYCB": [
        (1.45996, 110.58072, 1.53316),
        1.29512,
    ],  # 0.01187  0.01193 [16835]
}


# primary dihedra defaults based on residue-atom classes from reference
# database.
#
# primary dihedra are those not feasible to measure as a rotation from another
# dihedral angle: psi, omega, phi, sidechain chi angles, and a few more for
# aromatic sidechains as required by the algorithm.  These are the ones you
# really need to specify.
#
# each entry is the most common rounded int value observed over the class,
# followed by standard deviation (used by :func:`ic_data.write_PIC`)
#
# numbers in comments after each entry are count of dihedra in class and average
# angle
#
# N.B. these are just intended as reasonable starting values, in practice they
# generate a helical backbone with a kink at proline residues

# 134 total primary entries
dihedra_primary_defaults = {
    # [most common int(value), std_dev] # count average
    "ANACAACAOXT": [169, 95.25474],  # 81 -165.46932
    "ANACAACXN": [-42, 98.15673],  # 36750 -29.47243
    "CNCCACCBCSG": [-70, 81.01579],  # 5998 -85.23394
    "CNCCACCCOXT": [-176, 80.00406],  # 14 140.50216
    "CNCCACCXN": [-44, 91.42604],  # 5926  94.93048
    "DCADCBDCGDOD1": [-13, 52.88900],  # 27345 -11.84937
    "DNDCADCBDCG": [-70, 79.12777],  # 27365 -96.25186
    "DNDCADCDOXT": [-7, 95.85319],  # 39 158.65904
    "DNDCADCXN": [-41, 83.80110],  # 27228  14.67151
    "ECAECBECGECD": [178, 65.34682],  # 31790 -176.30888
    "ECBECGECDEOE1": [-10, 68.20002],  # 31755  -5.06398
    "ENECAECBECG": [-68, 70.35756],  # 31977 -98.71636
    "ENECAECEOXT": [-25, 99.97143],  # 73 -168.10401
    "ENECAECXN": [-42, 92.43985],  # 32144 -24.04536
    "FCAFCBFCGFCD1": [85, 79.85765],  # 18983  25.61167
    "FCBFCGFCD1FCE1": [179, 1.83591],  # 18982 -179.99254
    "FCBFCGFCD2FCE2": [-179, 1.84368],  # 18982 179.99540
    "FCGFCD1FCE1FCZ": [0, 1.47424],  # 18982  -0.02249
    "FNFCAFCBFCG": [-65, 76.22754],  # 18987 -102.70403
    "FNFCAFCFOXT": [-6, 99.25114],  # 42 158.48470
    "FNFCAFCXN": [-47, 87.47370],  # 18775  63.18666
    "GNGCAGCGOXT": [171, 104.01396],  # 57 -179.22384
    "GNGCAGCXN": [-42, 101.30511],  # 31254 -26.19519
    "HCAHCBHCGHND1": [-73, 105.57152],  # 10855 -80.79397
    "HCBHCGHCD2HNE2": [-179, 2.10579],  # 10854 -179.94222
    "HCBHCGHND1HCE1": [179, 1.85979],  # 10854 179.94389
    "HNHCAHCBHCG": [-65, 76.36890],  # 10862 -96.99583
    "HNHCAHCHOXT": [178, 89.67199],  # 30 -141.00653
    "HNHCAHCXN": [-42, 84.52974],  # 10708  37.08787
    "ICAICBICG1ICD1": [169, 56.74755],  # 27229 176.16728
    "INICAICBICG1": [-63, 55.70911],  # 27238 -61.62701
    "INICAICIOXT": [-70, 89.44826],  # 46  51.25035
    "INICAICXN": [-46, 88.35710],  # 26993  73.43770
    "KCAKCBKCGKCD": [178, 53.85008],  # 25968 -175.38550
    "KCBKCGKCDKCE": [179, 56.41606],  # 25714 -179.94658
    "KCGKCDKCEKNZ": [-179, 67.53216],  # 25629 -179.61758
    "KNKCAKCBKCG": [-69, 68.26710],  # 26244 -100.50122
    "KNKCAKCKOXT": [-47, 103.08025],  # 93 161.30451
    "KNKCAKCXN": [-42, 95.81378],  # 26509 -16.93019
    "LCALCBLCGLCD1": [174, 58.10966],  # 43971 141.93425
    "LNLCALCBLCG": [-67, 56.26252],  # 43982 -96.34969
    "LNLCALCLOXT": [156, 98.73531],  # 97 178.42308
    "LNLCALCXN": [-41, 95.14039],  # 43450 -14.58895
    "MCAMCBMCGMSD": [178, 70.69856],  # 7198 -161.34475
    "MCBMCGMSDMCE": [-73, 97.48432],  # 7196 -30.14847
    "MNMCAMCBMCG": [-66, 66.26181],  # 7201 -91.32003
    "MNMCAMCMOXT": [-178, 72.38494],  # 7 -102.91898
    "MNMCAMCXN": [-42, 98.49678],  # 7191 -11.35059
    "NCANCBNCGNOD1": [-16, 67.43697],  # 19899 -18.72551
    "NNNCANCBNCG": [-68, 73.98113],  # 19907 -92.41878
    "NNNCANCNOXT": [147, 89.25031],  # 40 152.04805
    "NNNCANCXN": [-40, 77.32702],  # 19740  27.97970
    "PCAPCBPCGPCD": [36, 34.61280],  # 20886   1.80409
    "PNPCAPCBPCG": [-28, 25.95226],  # 20886   0.27287
    "PNPCAPCPOXT": [-56, 102.31786],  # 37 160.48791
    "PNPCAPCXN": [148, 99.98739],  # 20372 127.77237
    "QCAQCBQCGQCD": [176, 68.87188],  # 17972 -175.20642
    "QCBQCGQCDQOE1": [-51, 71.82863],  # 17953  -4.14997
    "QNQCAQCBQCG": [-67, 66.94384],  # 18045 -94.68057
    "QNQCAQCQOXT": [-179, 95.53036],  # 39 -177.40254
    "QNQCAQCXN": [-42, 93.49684],  # 17980 -18.74463
    "RCARCBRCGRCD": [177, 49.67635],  # 23622 -176.16102
    "RCBRCGRCDRNE": [179, 84.50289],  # 23556 -173.50404
    "RCDRNERCZRNH1": [0, 54.62565],  # 23489   0.04129
    "RCGRCDRNERCZ": [-88, 69.24645],  # 23496 -175.95812
    "RNRCARCBRCG": [-68, 71.72865],  # 23691 -102.63698
    "RNRCARCROXT": [171, 99.50938],  # 61 -177.32361
    "RNRCARCXN": [-41, 96.71025],  # 23688 -14.55186
    "SNSCASCBSOG": [66, 89.69619],  # 28300  63.90162
    "SNSCASCSOXT": [-30, 89.77697],  # 51 -140.58823
    "SNSCASCXN": [-42, 92.34618],  # 27939  34.90087
    "TNTCATCBTOG1": [-61, 75.97906],  # 25489   0.35355
    "TNTCATCTOXT": [-4, 85.80840],  # 30 116.33387
    "TNTCATCXN": [-43, 97.81897],  # 25220 103.78844
    "VNVCAVCBVCG1": [177, 60.41895],  # 31953 -176.53074
    "VNVCAVCVOXT": [162, 100.62799],  # 58 139.89586
    "VNVCAVCXN": [-44, 97.87826],  # 31561 111.78211
    "WCAWCBWCGWCD1": [90, 96.85361],  # 6667  65.40881
    "WCBWCGWCD1WNE1": [-179, 2.16184],  # 6667 -179.84187
    "WCBWCGWCD2WCE2": [179, 2.19723],  # 6667 179.84352
    "WCD2WCE2WCZ2WCH2": [0, 1.44032],  # 6667   0.01803
    "WCE2WCZ2WCH2WCZ3": [0, 1.38967],  # 6667  -0.01231
    "WCGWCD2WCE2WCZ2": [180, 1.02636],  # 6667 179.96302
    "WNWCAWCBWCG": [-71, 83.39480],  # 6670 -103.58680
    "WNWCAWCWOXT": [-178, 93.40045],  # 13 131.22360
    "WNWCAWCXN": [-43, 94.36376],  # 6603  12.91505
    "XCANACAAC": [-63, 34.11665],  # 36738 -77.24348
    "XCAXCANACA": [179, 5.71694],  # 36738 179.41421
    "XCAXCCNCCA": [179, 7.65580],  # 5949 179.50458
    "XCAXCDNDCA": [179, 7.69439],  # 27198 179.07623
    "XCAXCENECA": [179, 6.37868],  # 32234 179.34942
    "XCAXCFNFCA": [179, 6.19707],  # 18837 179.35525
    "XCAXCGNGCA": [179, 8.86313],  # 30946 178.98640
    "XCAXCHNHCA": [179, 7.15842],  # 10739 179.50565
    "XCAXCINICA": [179, 5.72861],  # 26996 178.85749
    "XCAXCKNKCA": [179, 6.23316],  # 26578 179.16930
    "XCAXCLNLCA": [179, 5.44198],  # 43582 179.13937
    "XCAXCMNMCA": [179, 6.15780],  # 6955 179.29453
    "XCAXCNNNCA": [179, 7.94004],  # 19720 179.28144
    "XCAXCPNPCA": [-180, 38.10480],  # 20520 179.72853
    "XCAXCQNQCA": [179, 5.98589],  # 17998 179.17709
    "XCAXCRNRCA": [179, 6.13625],  # 23753 179.08276
    "XCAXCSNSCA": [179, 7.81457],  # 27790 179.45259
    "XCAXCTNTCA": [179, 6.63019],  # 25119 179.08898
    "XCAXCVNVCA": [179, 6.28535],  # 31661 178.92630
    "XCAXCWNWCA": [179, 8.05982],  # 6627 179.45312
    "XCAXCYNYCA": [179, 6.42966],  # 16676 179.66147
    "XCCNCCACC": [-63, 39.85404],  # 5949 -93.17092
    "XCDNDCADC": [-63, 42.66497],  # 27198 -81.36742
    "XCENECAEC": [-64, 33.27969],  # 32234 -78.04899
    "XCFNFCAFC": [-64, 37.53881],  # 18836 -92.58604
    "XCGNGCAGC": [-62, 105.46218],  # 30945  84.19701
    "XCHNHCAHC": [-62, 44.74280],  # 10739 -90.93838
    "XCINICAIC": [-63, 28.64344],  # 26996 -90.03017
    "XCKNKCAKC": [-63, 37.33533],  # 26577 -80.74953
    "XCLNLCALC": [-63, 28.04783],  # 43580 -81.71257
    "XCMNMCAMC": [-63, 33.98800],  # 6955 -83.88604
    "XCNNNCANC": [-64, 56.30326],  # 19719 -85.16140
    "XCPNPCAPC": [-60, 10.60764],  # 20519 -65.25196
    "XCQNQCAQC": [-64, 36.43753],  # 17998 -81.03040
    "XCRNRCARC": [-64, 37.52840],  # 23753 -83.03807
    "XCSNSCASC": [-63, 39.99619],  # 27790 -88.27000
    "XCTNTCATC": [-62, 31.66804],  # 25119 -95.01154
    "XCVNVCAVC": [-62, 29.95014],  # 31661 -94.87235
    "XCWNWCAWC": [-64, 35.41390],  # 6627 -87.28116
    "XCYNYCAYC": [-63, 37.42074],  # 16675 -93.61795
    "YCAYCBYCGYCD1": [83, 79.95637],  # 16794  21.14006
    "YCBYCGYCD1YCE1": [-179, 1.95641],  # 16794 -179.91379
    "YCBYCGYCD2YCE2": [179, 1.97956],  # 16793 179.92677
    "YCD1YCE1YCZYOH": [180, 1.22077],  # 16792 179.99504
    "YCGYCD1YCE1YCZ": [0, 1.43861],  # 16793  -0.02863
    "YNYCAYCBYCG": [-67, 76.72110],  # 16797 -102.49923
    "YNYCAYCXN": [-46, 88.47760],  # 16620  72.07880
    "YNYCAYCYOXT": [132, 87.47211],  # 36 155.93068
}


# secondary dihedra defaults based on residue-atom classes from reference
# database.
#
# secondary dihedra are those with relatively constant rotations from other
# (primary) dihedra, such as the planar orientation of the amide O atom
# (N-Ca-C-O) relative to the second N atom in the psi backbone dihedral
# (N-CA-C-N).
#
# each entry is the primary angle to rotate from and the average rotation
# observed over the class
#
# numbers in comments after each entry are count of dihedra in class and
# standard deviation.
#
# N.B. these are just intended as reasonable starting values, but sufficient to
# achieve <1.0 angstrom RMSD from the original structure in limited testing
# using ChimeraX matchmaker

dihedra_secondary_defaults = {
    # primary angle to rotate from, average rotation # count std dev
    "ANACAACAO": [("N", "CA", "C", "N"), 179.92568],  # 36749   1.98809
    "AOACACAACB": [("N", "CA", "C", "N"), -57.30099],  # 36740   2.54018
    "CNCCACCCO": [("N", "CA", "C", "N"), 179.94158],  # 5926   2.20422
    "COCCCCACCB": [("N", "CA", "C", "N"), -57.44056],  # 5923   2.99793
    "DCADCBDCGDOD2": [
        ("CA", "CB", "CG", "OD1"),
        -179.86172,
    ],  # 27344   2.38528
    "DNDCADCDO": [("N", "CA", "C", "N"), -179.97109],  # 27227   2.11112
    "DODCDCADCB": [("N", "CA", "C", "N"), -57.07933],  # 27214   3.07908
    "ECBECGECDEOE2": [
        ("CB", "CG", "CD", "OE1"),
        -179.97281,
    ],  # 31751   2.90866
    "ENECAECEO": [("N", "CA", "C", "N"), 179.92920],  # 32144   2.23044
    "EOECECAECB": [("N", "CA", "C", "N"), -57.13533],  # 32127   3.00382
    "FCAFCBFCGFCD2": [("CA", "CB", "CG", "CD1"), 179.97232],  # 18983   2.66092
    "FNFCAFCFO": [("N", "CA", "C", "N"), 179.86787],  # 18775   2.24567
    "FOFCFCAFCB": [("N", "CA", "C", "N"), -57.36288],  # 18773   3.19500
    "GNGCAGCGO": [("N", "CA", "C", "N"), 179.87864],  # 31254   1.93653
    "HCAHCBHCGHCD2": [("CA", "CB", "CG", "ND1"), 179.92892],  # 10855   3.13324
    "HNHCAHCHO": [("N", "CA", "C", "N"), 179.88083],  # 10707   2.13861
    "HOHCHCAHCB": [("N", "CA", "C", "N"), -57.32474],  # 10705   3.15196
    "INICAICBICG2": [("N", "CA", "CB", "CG1"), -123.20171],  # 27237   2.42092
    "INICAICIO": [("N", "CA", "C", "N"), -179.99219],  # 26993   2.09900
    "IOICICAICB": [("N", "CA", "C", "N"), -56.67074],  # 26992   2.87033
    "KNKCAKCKO": [("N", "CA", "C", "N"), 179.91063],  # 26508   1.98813
    "KOKCKCAKCB": [("N", "CA", "C", "N"), -57.24410],  # 26495   2.74225
    "LCALCBLCGLCD2": [("CA", "CB", "CG", "CD1"), 122.93111],  # 43968   2.16725
    "LNLCALCLO": [("N", "CA", "C", "N"), 179.91619],  # 43450   2.02277
    "LOLCLCALCB": [("N", "CA", "C", "N"), -57.47486],  # 43450   2.83489
    "MNMCAMCMO": [("N", "CA", "C", "N"), 179.89452],  # 7191   2.16025
    "MOMCMCAMCB": [("N", "CA", "C", "N"), -57.33882],  # 7188   2.97285
    "NCANCBNCGNND2": [("CA", "CB", "CG", "OD1"), 179.94543],  # 19898   2.12515
    "NNNCANCNO": [("N", "CA", "C", "N"), 179.92115],  # 19739   2.13981
    "NONCNCANCB": [("N", "CA", "C", "N"), -56.76402],  # 19726   3.06823
    "PNPCAPCPO": [("N", "CA", "C", "N"), 179.99101],  # 20371   2.09744
    "POPCPCAPCB": [("N", "CA", "C", "N"), -64.77747],  # 20369   2.66024
    "QCBQCGQCDQNE2": [
        ("CB", "CG", "CD", "OE1"),
        -179.98865,
    ],  # 17951   2.77403
    "QNQCAQCQO": [("N", "CA", "C", "N"), 179.92706],  # 17980   2.08396
    "QOQCQCAQCB": [("N", "CA", "C", "N"), -57.18151],  # 17976   3.18161
    "RCDRNERCZRNH2": [("CD", "NE", "CZ", "NH1"), 179.99965],  # 23487   3.02813
    "RNRCARCRO": [("N", "CA", "C", "N"), 179.92085],  # 23688   2.06352
    "RORCRCARCB": [("N", "CA", "C", "N"), -57.21965],  # 23677   3.05380
    "SNSCASCSO": [("N", "CA", "C", "N"), 179.87018],  # 27939   2.11127
    "SOSCSCASCB": [("N", "CA", "C", "N"), -57.37555],  # 27932   3.02066
    "TNTCATCBTCG2": [("N", "CA", "CB", "OG1"), -120.62370],  # 25482   2.72546
    "TNTCATCTO": [("N", "CA", "C", "N"), 179.93709],  # 25220   2.08437
    "TOTCTCATCB": [("N", "CA", "C", "N"), -56.91727],  # 25214   2.98795
    "VNVCAVCBVCG2": [("N", "CA", "CB", "CG1"), 122.92482],  # 31953   1.99439
    "VNVCAVCVO": [("N", "CA", "C", "N"), -179.91262],  # 31561   2.06055
    "VOVCVCAVCB": [("N", "CA", "C", "N"), -56.53189],  # 31558   2.84749
    "WCAWCBWCGWCD2": [("CA", "CB", "CG", "CD1"), -179.77249],  # 6667   3.15519
    "WCBWCGWCD2WCE3": [
        ("CB", "CG", "CD2", "CE2"),
        179.97400,
    ],  # 6667   1.70848
    "WNWCAWCWO": [("N", "CA", "C", "N"), 179.79740],  # 6603   2.20883
    "WOWCWCAWCB": [("N", "CA", "C", "N"), -57.49156],  # 6603   3.27149
    "YCAYCBYCGYCD2": [
        ("CA", "CB", "CG", "CD1"),
        -179.94185,
    ],  # 16794   2.64007
    "YNYCAYCYO": [("N", "CA", "C", "N"), 179.83537],  # 16620   2.14952
    "YOYCYCAYCB": [("N", "CA", "C", "N"), -57.40953],  # 16620   3.10134
}  # 53 total secondary default entries

# secondary dihedra defaults based on residue-atom classes from reference
# database.
#
# as above, but this table is for limited set of secondary dihedra which are
# defined by rotation from chain terminal OXT or use the alternate CB locator
# dihedral C(i-1)-C-N-CA-CB when the amide O is missing

dihedra_secondary_xoxt_defaults = {
    # primary angle to rotate from, average rotation # count std dev
    "ANACAACAO": [("N", "CA", "C", "OXT"), 179.94094],  # 81    2.11763
    "AOACACAACB": [("N", "CA", "C", "OXT"), -57.60123],  # 81    2.80240
    "CNCCACCCO": [("N", "CA", "C", "OXT"), -179.80335],  # 14    1.44006
    "COCCCCACCB": [("N", "CA", "C", "OXT"), -56.72357],  # 14    2.10630
    "DNDCADCDO": [("N", "CA", "C", "OXT"), 179.49564],  # 39    2.90408
    "DODCDCADCB": [("N", "CA", "C", "OXT"), -57.21308],  # 39    3.78367
    "ENECAECEO": [("N", "CA", "C", "OXT"), 179.92513],  # 73    5.63173
    "EOECECAECB": [("N", "CA", "C", "OXT"), -57.23113],  # 73    6.62931
    "FNFCAFCFO": [("N", "CA", "C", "OXT"), -179.59346],  # 42    3.26581
    "FOFCFCAFCB": [("N", "CA", "C", "OXT"), -57.03927],  # 42    4.29847
    "GNGCAGCGO": [("N", "CA", "C", "OXT"), 179.69001],  # 57    1.64200
    "HNHCAHCHO": [("N", "CA", "C", "OXT"), -179.75032],  # 30    1.44791
    "HOHCHCAHCB": [("N", "CA", "C", "OXT"), -56.67577],  # 30    2.65028
    "INICAICIO": [("N", "CA", "C", "OXT"), 179.93893],  # 46    1.77873
    "IOICICAICB": [("N", "CA", "C", "OXT"), -56.65160],  # 45    3.62735
    "KNKCAKCKO": [("N", "CA", "C", "OXT"), 179.71490],  # 93   18.78315
    "KOKCKCAKCB": [("N", "CA", "C", "OXT"), -57.20240],  # 93   19.02926
    "LNLCALCLO": [("N", "CA", "C", "OXT"), 179.77678],  # 97    1.98355
    "LOLCLCALCB": [("N", "CA", "C", "OXT"), -57.59144],  # 97    2.75152
    "MNMCAMCMO": [("N", "CA", "C", "OXT"), -179.97795],  # 7    1.96286
    "MOMCMCAMCB": [("N", "CA", "C", "OXT"), -55.77939],  # 7    4.32745
    "NNNCANCNO": [("N", "CA", "C", "OXT"), 179.71084],  # 40    1.95590
    "NONCNCANCB": [("N", "CA", "C", "OXT"), -57.09828],  # 40    3.22053
    "PNPCAPCPO": [("N", "CA", "C", "OXT"), -179.53729],  # 37    1.61467
    "POPCPCAPCB": [("N", "CA", "C", "OXT"), -64.53147],  # 37    2.37847
    "QNQCAQCQO": [("N", "CA", "C", "OXT"), 179.58623],  # 37    3.13328
    "QOQCQCAQCB": [("N", "CA", "C", "OXT"), -57.50893],  # 36    4.43485
    "RNRCARCRO": [("N", "CA", "C", "OXT"), 179.93106],  # 61    4.38718
    "RORCRCARCB": [("N", "CA", "C", "OXT"), -57.72993],  # 61    5.16366
    "SNSCASCSO": [("N", "CA", "C", "OXT"), 179.56523],  # 51    2.65719
    "SOSCSCASCB": [("N", "CA", "C", "OXT"), -57.18238],  # 51    3.57789
    "TNTCATCTO": [("N", "CA", "C", "OXT"), -179.79984],  # 30    1.33044
    "TOTCTCATCB": [("N", "CA", "C", "OXT"), -56.58614],  # 30    2.06022
    "VNVCAVCVO": [("N", "CA", "C", "OXT"), -179.70350],  # 57    3.55209
    "VOVCVCAVCB": [("N", "CA", "C", "OXT"), -56.23549],  # 57    4.14683
    "WNWCAWCWO": [("N", "CA", "C", "OXT"), -179.82426],  # 13    1.83266
    "WOWCWCAWCB": [("N", "CA", "C", "OXT"), -57.66758],  # 13    2.36066
    "XCANACAACB": [("C", "N", "CA", "C"), -122.64404],  # 4    0.71089
    "XCDNDCADCB": [("C", "N", "CA", "C"), -129.79903],  # 1    0.00000
    "XCENECAECB": [("C", "N", "CA", "C"), -130.15620],  # 1    0.00000
    "XCFNFCAFCB": [("C", "N", "CA", "C"), -121.28140],  # 1    0.00000
    "XCHNHCAHCB": [("C", "N", "CA", "C"), -123.67335],  # 3    1.41030
    "XCKNKCAKCB": [("C", "N", "CA", "C"), -123.09741],  # 1    0.00000
    "XCLNLCALCB": [("C", "N", "CA", "C"), -122.14996],  # 1    0.00000
    "XCNNNCANCB": [("C", "N", "CA", "C"), -123.45323],  # 2    1.53987
    "XCPNPCAPCB": [("C", "N", "CA", "C"), -122.07120],  # 4    3.14794
    "XCQNQCAQCB": [("C", "N", "CA", "C"), -124.41558],  # 2    1.44826
    "XCSNSCASCB": [("C", "N", "CA", "C"), -121.29809],  # 1    0.00000
    "XCVNVCAVCB": [("C", "N", "CA", "C"), -122.15680],  # 2    0.50441
    "YNYCAYCYO": [("N", "CA", "C", "OXT"), 179.55952],  # 36   13.58661
    "YOYCYCAYCB": [("N", "CA", "C", "OXT"), -57.28932],  # 36   13.86761
}  # 51 total secondary X / OXT default entries
