"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToExcel = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
var _excelBuilder = require("./excelBuilder");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  let sorting = report._source.sorting;

  // We expect a list of [field, order] pairs for sorting. In some migration paths, though it's not
  // clear why, this list can get unnested in the case of one sort, [["field", "asc"]] becomes
  // ["field", "asc"]. The true root cause remains a mystery, so we work around it.
  // See: https://github.com/opensearch-project/dashboards-reporting/issues/371
  if (sorting.length > 0 && typeof sorting[0] === 'string') {
    sorting = [sorting];
  }
  if (sorting.length > 0) {
    const sorts = sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sorts);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields === null || fields === void 0 ? void 0 : fields[dateField];
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if ((dateValue === null || dateValue === void 0 ? void 0 : dateValue.length) !== 0 && dateValue instanceof Array) {
              fieldDateValue === null || fieldDateValue === void 0 || fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (fieldDateValue && typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if ((dateValue === null || dateValue === void 0 ? void 0 : dateValue.length) !== 0 && dateValue instanceof Array) {
              let tempArray = [];
              fieldDateValue === null || fieldDateValue === void 0 || fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        let result = flattenHits(data);
        hits.push(params.excel ? sanitize(result) : result);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

// Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(input, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(input)) {
    const newPrefix = `${prefix}${key}.`;
    if (value === null || typeof value !== 'object' || value instanceof Date) {
      result[prefix.replace(/^_source\./, '') + key] = value;
    } else if (Array.isArray(value)) {
      if (value.every(v => typeof v === 'object' && v !== null && !Array.isArray(v))) {
        const grouped = {};
        for (const obj of value) {
          const flat = flattenHits(obj, {}, '');
          for (const [subKey, subVal] of Object.entries(flat)) {
            if (!grouped[`${key}.${subKey}`]) {
              grouped[`${key}.${subKey}`] = [];
            }
            grouped[`${key}.${subKey}`].push(subVal);
          }
        }
        for (const [flatKey, flatVals] of Object.entries(grouped)) {
          result[prefix.replace(/^_source\./, '') + flatKey] = flatVals.join(',');
        }
      } else {
        result[prefix.replace(/^_source\./, '') + key] = value.join(',');
      }
    } else {
      flattenHits(value, result, newPrefix);
    }
  }
  return result;
}
function flattenObject(obj = {}, parentKey = '', result = {}) {
  for (const [key, value] of Object.entries(obj)) {
    const newKey = parentKey ? `${parentKey}.${key}` : key;
    if (typeof value == 'object' && value !== null && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenObject(value, newKey, result);
    } else if (Array.isArray(value)) {
      result[newKey] = JSON.stringify(value);
    } else {
      result[newKey] = value;
    }
  }
  return result;
}
function flattenArray(array = []) {
  return array.map(item => flattenObject(item));
}
const convertToExcel = async dataset => {
  const flatDataset = flattenArray(dataset[0]);
  const excelBuilder = new _excelBuilder.ExcelBuilder();
  const base64 = await excelBuilder.addHeaders(flatDataset).addRows(flatDataset).updateColumnWidths().applyHeaderStyles().applyAutoFilter().applyFreeze().getWorkbookAsBase64();
  return 'data:application/vnd.openxmlformats-officedocument.spreadsheetml.sheet;base64,' + base64;
};

//Return only the selected fields
exports.convertToExcel = convertToExcel;
function traverse(data, keys, result = {}) {
  const flatData = flattenHits(data);
  for (const key of keys) {
    if (flatData[key] !== undefined) {
      result[key] = flatData[key];
      continue;
    }
    for (const flatKey of Object.keys(flatData)) {
      if (flatKey === key || flatKey.startsWith(key + '.')) {
        result[flatKey] = flatData[flatKey];
      }
    }
  }
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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