"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GenericMLRouter = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Generic ML client detector with caching for performance
 * Finds any client in the request context that has a request() method
 */
let cachedClientKey = null;
let detectionAttempted = false;
function findMLClient(context) {
  // Use cached result if available
  if (detectionAttempted) {
    if (cachedClientKey) {
      const contextItem = context[cachedClientKey];
      return contextItem === null || contextItem === void 0 ? void 0 : contextItem.client;
    }
    return undefined;
  }

  // First-time detection - cache the result
  for (const key of Object.keys(context)) {
    var _contextItem$client;
    const contextItem = context[key];
    if (contextItem !== null && contextItem !== void 0 && (_contextItem$client = contextItem.client) !== null && _contextItem$client !== void 0 && _contextItem$client.request && typeof contextItem.client.request === 'function') {
      cachedClientKey = key; // Cache the key name
      detectionAttempted = true;
      return contextItem.client;
    }
  }
  detectionAttempted = true; // Mark as attempted even if not found
  return undefined;
}

// ============================================================================
// GENERIC ML CLIENT TYPES - To avoid compile-time dependencies
// ============================================================================

/**
 * Type guard to check if response is streaming
 */
function isStreamResponse(response) {
  return response && typeof response.body === 'object' && 'pipe' in response.body;
}

/**
 * Generic ML Commons agent router
 * Uses generic ML client detection to communicate with ML Commons agents
 * Works with any ML client provider that has a request() method
 */
class GenericMLRouter {
  async forward(context, request, response, logger, configuredAgentId, dataSourceId) {
    if (!configuredAgentId) {
      return response.customError({
        statusCode: 503,
        body: {
          message: 'ML Commons agent ID not configured'
        }
      });
    }

    // Validate request body
    if (!request.body || typeof request.body !== 'object') {
      return response.customError({
        statusCode: 400,
        body: {
          message: 'Invalid request body for ML Commons agent'
        }
      });
    }
    const mlClient = findMLClient(context);
    if (!mlClient) {
      return response.customError({
        statusCode: 503,
        body: {
          message: 'ML client not available in request context'
        }
      });
    }
    try {
      logger.info('Forwarding request to ML Commons agent', {
        agentId: configuredAgentId,
        dataSourceId
      });

      // Use detected ML client from request context
      const mlResponse = await mlClient.request({
        method: 'POST',
        path: `/_plugins/_ml/agents/${configuredAgentId}/_execute/stream`,
        body: JSON.stringify(request.body),
        datasourceId: dataSourceId,
        // Use actual dataSourceId from request
        stream: true,
        timeout: 300000
      }, request, context);

      // Handle streaming response properly using type guard
      if (isStreamResponse(mlResponse)) {
        return response.custom({
          statusCode: mlResponse.status,
          headers: {
            'Content-Type': 'text/event-stream',
            'Content-Encoding': 'identity',
            Connection: 'keep-alive'
          },
          body: mlResponse.body
        });
      } else {
        return response.custom({
          statusCode: mlResponse.status,
          headers: {
            'Content-Type': 'application/json',
            ...mlResponse.headers
          },
          body: typeof mlResponse.body === 'string' ? JSON.parse(mlResponse.body) : mlResponse.body
        });
      }
    } catch (error) {
      logger.error(`Error forwarding to ML Commons agent: ${error}`);
      if (error instanceof Error && error.message.includes('404')) {
        return response.customError({
          statusCode: 404,
          body: {
            message: `ML Commons agent "${configuredAgentId}" not found`
          }
        });
      }
      return response.customError({
        statusCode: 500,
        body: {
          message: `ML Commons agent error: ${error instanceof Error ? error.message : 'Unknown error'}`
        }
      });
    }
  }
  getRouterName() {
    return 'GenericMLRouter';
  }
}
exports.GenericMLRouter = GenericMLRouter;