/*
    SPDX-FileCopyrightText: 2008 Nicola Gigante <nicola.gigante@gmail.com>
    SPDX-FileCopyrightText: 2020 Harald Sitter <sitter@kde.org>

    SPDX-License-Identifier: LGPL-2.1-or-later
*/

#ifndef KAUTH_HELPER_SUPPORT_H
#define KAUTH_HELPER_SUPPORT_H

#include <QObject>
#include <QVariant>

#include "kauthcore_export.h"

/**
 * @relates KAuth
 *
 * The main macro for writing a helper tool.
 * @param ID The helper ID as passed to the macro
 * @param HelperClass The class name of the responder object for the helper
 *
 * @see @ref kauth_helper
 */
#define KAUTH_HELPER_MAIN(ID, HelperClass)                                                                                                                     \
    int main(int argc, char **argv)                                                                                                                            \
    {                                                                                                                                                          \
        return KAuth::HelperSupport::helperMain(argc, argv, ID, new HelperClass());                                                                            \
    }

namespace KAuth
{
/**
 * @brief Support class with some KAUTHCORE_EXPORT methods useful to the helper's code
 *
 * This class provides the API to write the helper tool that executes your actions.
 * You don't create instances of HelperSupport. Instead, you use its KAUTHCORE_EXPORT methods.
 *
 * This them you can notify the application of progress in your action's execution
 * and you can check if the application asked you to terminate it.
 *
 * @since 4.4
 */
namespace HelperSupport
{
/**
 * @brief Send a progressStep signal to the caller application
 *
 * You can use this method to notify progress information about the
 * action execution. When you call this method, the KAuth::ExecuteJob
 * object associated with the current action will emit the KJob::percent(KJob*, unsigned long)
 * signal. The meaning of the integer passed here is totally application dependent,
 * but you'll want to use it as a sort of percentage.
 * If you need to be more expressive, use the other overload which takes a QVariantMap
 *
 * @param step The progress indicator
 */
KAUTHCORE_EXPORT void progressStep(int step);

/**
 * @brief Send a progressStep signal to the caller application
 *
 * You can use this method to notify progress information about the
 * action execution. When you call this method, the KAuth::ExecuteJob
 * object associated with the current action will emit the progressStep(QVariantMap)
 * signal. The meaning of the data passed here is totally application dependent.
 *
 * If you only need a simple percentage value, use the other overload which takes an int.
 *
 * @param data The progress data
 */
KAUTHCORE_EXPORT void progressStep(const QVariantMap &data);

/**
 * @brief Check if the caller asked the helper to stop the execution
 *
 * This method will return @c true if the helper has been asked to stop the
 * execution of the current action. If this happens, your helper should
 * return (NOT exit). The meaning of the data you return in this case is
 * application-dependent.
 *
 * It's good practice to check it regularly if you have a long-running action
 *
 * @return @c true if the helper has been asked to stop, @c false otherwise
 *
 * @see ExecuteJob::kill
 */
KAUTHCORE_EXPORT bool isStopped();

/**
 * @brief Method that implements the main function of the helper tool. Do not call directly
 *
 * This method is called in the proper way by the code generated by the KAUTH_HELPER_MAIN() macro,
 * which creates a main() function for the helper tool.
 * You shouldn't call this method directly.
 *
 * @param argc The argc parameter from the main() function.
 * @param argv The argv parameter from the main() function.
 * @param id The helper ID as passed to the macro
 * @param responder The responder object for the helper. The macro passes a default-constructed,
 *                  heap-allocated object of the class specified as the last macro parameter
 */
KAUTHCORE_EXPORT int helperMain(int argc, char **argv, const char *id, QObject *responder);

/**
 * @brief Obtains the caller user id if available.
 *
 * This method offers a way to obtain the unprivileged client's UID if possible.
 * For example when a D-Bus-based backend is used this will resolve the caller
 * of the D-Bus method and obtain its process' UID.
 *
 * @since 5.74
 *
 * @return caller UID or -1 when not available
 */
KAUTHCORE_EXPORT int callerUid();
} // namespace HelperSupport

} // namespace Auth

#endif
