/**
 * WinPR: Windows Portable Runtime
 * Error Handling Functions
 *
 * Copyright 2012 Marc-Andre Moreau <marcandre.moreau@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef WINPR_ERROR_H
#define WINPR_ERROR_H

#include <winpr/cast.h>
#include <winpr/wtypes.h>

#ifdef _WIN32

#include <winerror.h>

/* mingw is possibly missing some definitions */
#ifndef RPC_S_PROXY_ACCESS_DENIED
#define RPC_S_PROXY_ACCESS_DENIED 0x000006C1
#endif

#ifndef RPC_S_COOKIE_AUTH_FAILED
#define RPC_S_COOKIE_AUTH_FAILED 0x00000729
#endif

#ifndef ERROR_OPERATION_IN_PROGRESS
#define ERROR_OPERATION_IN_PROGRESS 0x00000149
#endif

#else

#ifndef NO_ERROR
#define NO_ERROR 0
#endif

#define E_UNEXPECTED -2147418113l   // 0x8000FFFFL
#define E_ACCESSDENIED -2147024891l // 0x80070005L
#define E_HANDLE -2147024890l       // 0x80070006L
#define E_OUTOFMEMORY -2147024882l  // 0x8007000EL

#define E_INVALIDARG -2147024809l  // 0x80070057L
#define E_NOTIMPL -2147467263l     // 0x80004001L
#define E_NOINTERFACE -2147467262l // 0x80004002L
#define E_POINTER -2147467261l     // 0x80004003L
#define E_ABORT -2147467260l       // 0x80004004L
#define E_FAIL -2147467259l        // 0x80004005L

#define CO_E_INIT_TLS -2147467258l                                 // 0x80004006l
#define CO_E_INIT_SHARED_ALLOCATOR -2147467257l                    // 0x80004007l
#define CO_E_INIT_MEMORY_ALLOCATOR -2147467256l                    // 0x80004008l
#define CO_E_INIT_CLASS_CACHE -2147467255l                         // 0x80004009l
#define CO_E_INIT_RPC_CHANNEL -2147467254l                         // 0x8000400Al
#define CO_E_INIT_TLS_SET_CHANNEL_CONTROL -2147467253l             // 0x8000400Bl
#define CO_E_INIT_TLS_CHANNEL_CONTROL -2147467252l                 // 0x8000400Cl
#define CO_E_INIT_UNACCEPTED_USER_ALLOCATOR -2147467251l           // 0x8000400Dl
#define CO_E_INIT_SCM_MUTEX_EXISTS -2147467250l                    // 0x8000400El
#define CO_E_INIT_SCM_FILE_MAPPING_EXISTS -2147467249l             // 0x8000400Fl
#define CO_E_INIT_SCM_MAP_VIEW_OF_FILE -2147467248l                // 0x80004010l
#define CO_E_INIT_SCM_EXEC_FAILURE -2147467247l                    // 0x80004011l
#define CO_E_INIT_ONLY_SINGLE_THREADED -2147467246l                // 0x80004012l
#define CO_E_CANT_REMOTE -2147467245l                              // 0x80004013l
#define CO_E_BAD_SERVER_NAME -2147467244l                          // 0x80004014l
#define CO_E_WRONG_SERVER_IDENTITY -2147467243l                    // 0x80004015l
#define CO_E_OLE1DDE_DISABLED -2147467242l                         // 0x80004016l
#define CO_E_RUNAS_SYNTAX -2147467241l                             // 0x80004017l
#define CO_E_CREATEPROCESS_FAILURE -2147467240l                    // 0x80004018l
#define CO_E_RUNAS_CREATEPROCESS_FAILURE -2147467239l              // 0x80004019l
#define CO_E_RUNAS_LOGON_FAILURE -2147467238l                      // 0x8000401Al
#define CO_E_LAUNCH_PERMSSION_DENIED -2147467237l                  // 0x8000401Bl
#define CO_E_START_SERVICE_FAILURE -2147467236l                    // 0x8000401Cl
#define CO_E_REMOTE_COMMUNICATION_FAILURE -2147467235l             // 0x8000401Dl
#define CO_E_SERVER_START_TIMEOUT -2147467234l                     // 0x8000401El
#define CO_E_CLSREG_INCONSISTENT -2147467233l                      // 0x8000401Fl
#define CO_E_IIDREG_INCONSISTENT -2147467232l                      // 0x80004020l
#define CO_E_NOT_SUPPORTED -2147467231l                            // 0x80004021l
#define CO_E_RELOAD_DLL -2147467230l                               // 0x80004022l
#define CO_E_MSI_ERROR -2147467229l                                // 0x80004023l
#define CO_E_ATTEMPT_TO_CREATE_OUTSIDE_CLIENT_CONTEXT -2147467228l // 0x80004024l
#define CO_E_SERVER_PAUSED -2147467227l                            // 0x80004025l
#define CO_E_SERVER_NOT_PAUSED -2147467226l                        // 0x80004026l
#define CO_E_CLASS_DISABLED -2147467225l                           // 0x80004027l
#define CO_E_CLRNOTAVAILABLE -2147467224l                          // 0x80004028l
#define CO_E_ASYNC_WORK_REJECTED -2147467223l                      // 0x80004029l
#define CO_E_SERVER_INIT_TIMEOUT -2147467222l                      // 0x8000402Al
#define CO_E_NO_SECCTX_IN_ACTIVATE -2147467221l                    // 0x8000402Bl
#define CO_E_TRACKER_CONFIG -2147467216l                           // 0x80004030l
#define CO_E_THREADPOOL_CONFIG -2147467215l                        // 0x80004031l
#define CO_E_SXS_CONFIG -2147467214l                               // 0x80004032l
#define CO_E_MALFORMED_SPN -2147467213l                            // 0x80004033l

#define FACILITY_WINDOWSUPDATE 36
#define FACILITY_WINDOWS_CE 24
#define FACILITY_WINDOWS 8
#define FACILITY_URT 19
#define FACILITY_UMI 22
#define FACILITY_SXS 23
#define FACILITY_STORAGE 3
#define FACILITY_STATE_MANAGEMENT 34
#define FACILITY_SSPI 9
#define FACILITY_SCARD 16
#define FACILITY_SETUPAPI 15
#define FACILITY_SECURITY 9
#define FACILITY_RPC 1
#define FACILITY_WIN32 7
#define FACILITY_CONTROL 10
#define FACILITY_NULL 0
#define FACILITY_METADIRECTORY 35
#define FACILITY_MSMQ 14
#define FACILITY_MEDIASERVER 13
#define FACILITY_INTERNET 12
#define FACILITY_ITF 4
#define FACILITY_HTTP 25
#define FACILITY_DPLAY 21
#define FACILITY_DISPATCH 2
#define FACILITY_DIRECTORYSERVICE 37
#define FACILITY_CONFIGURATION 33
#define FACILITY_COMPLUS 17
#define FACILITY_CERT 11
#define FACILITY_BACKGROUNDCOPY 32
#define FACILITY_ACS 20
#define FACILITY_AAF 18

#define FACILITY_NT_BIT 0x10000000

#define SEVERITY_SUCCESS 0
#define SEVERITY_ERROR 1

#define HRESULT_CODE(hr) ((hr)&0xFFFF)
#define HRESULT_FACILITY(hr) (((hr) >> 16) & 0x1FFF)

#define HRESULT_FROM_NT(x) (((x) | FACILITY_NT_BIT))

WINPR_PRAGMA_DIAG_PUSH
WINPR_PRAGMA_DIAG_IGNORED_RESERVED_ID_MACRO

#define ERROR_CAST(t, val) WINPR_CXX_COMPAT_CAST(t, val)

static INLINE HRESULT HRESULT_FROM_WIN32(unsigned long x)
{
	HRESULT hx = ERROR_CAST(HRESULT, x);
	if (hx <= 0)
		return hx;
	return ERROR_CAST(HRESULT, (((x)&0x0000FFFF) | (FACILITY_WIN32 << 16) | 0x80000000));
}

WINPR_PRAGMA_DIAG_POP

#define HRESULT_SEVERITY(hr) (((hr) >> 31) & 0x1)

#define SUCCEEDED(hr) (((hr)) >= 0)
#define FAILED(hr) (((hr)) < 0)
#define IS_ERROR(Status) ((ERROR_CAST(unsigned long, Status)) >> 31 == SEVERITY_ERROR)

#define MAKE_HRESULT(sev, fac, code)                                                             \
	((HRESULT)((ERROR_CAST(unsigned long, sev) << 31) | (ERROR_CAST(unsigned long, fac) << 16) | \
	           (ERROR_CAST(unsigned long, code))))

#define SCODE_CODE(sc) ((sc)&0xFFFF)
#define SCODE_FACILITY(sc) (((sc) >> 16) & 0x1FFF)
#define SCODE_SEVERITY(sc) (((sc) >> 31) & 0x1)

#define MAKE_SCODE(sev, fac, code)                                                             \
	((SCODE)((ERROR_CAST(unsigned long, sev) << 31) | (ERROR_CAST(unsigned long, fac) << 16) | \
	         (ERROR_CAST(unsigned long, code))))

#define S_OK (0L)
#define S_FALSE (1L)

/* System Error Codes (0-499) */

#define ERROR_SUCCESS 0x00000000
#define ERROR_INVALID_FUNCTION 0x00000001
#define ERROR_FILE_NOT_FOUND 0x00000002
#define ERROR_PATH_NOT_FOUND 0x00000003
#define ERROR_TOO_MANY_OPEN_FILES 0x00000004
#define ERROR_ACCESS_DENIED 0x00000005
#define ERROR_INVALID_HANDLE 0x00000006
#define ERROR_ARENA_TRASHED 0x00000007
#define ERROR_NOT_ENOUGH_MEMORY 0x00000008
#define ERROR_INVALID_BLOCK 0x00000009
#define ERROR_BAD_ENVIRONMENT 0x0000000A
#define ERROR_BAD_FORMAT 0x0000000B
#define ERROR_INVALID_ACCESS 0x0000000C
#define ERROR_INVALID_DATA 0x0000000D
#define ERROR_OUTOFMEMORY 0x0000000E
#define ERROR_INVALID_DRIVE 0x0000000F
#define ERROR_CURRENT_DIRECTORY 0x00000010
#define ERROR_NOT_SAME_DEVICE 0x00000011
#define ERROR_NO_MORE_FILES 0x00000012
#define ERROR_WRITE_PROTECT 0x00000013
#define ERROR_BAD_UNIT 0x00000014
#define ERROR_NOT_READY 0x00000015
#define ERROR_BAD_COMMAND 0x00000016
#define ERROR_CRC 0x00000017
#define ERROR_BAD_LENGTH 0x00000018
#define ERROR_SEEK 0x00000019
#define ERROR_NOT_DOS_DISK 0x0000001A
#define ERROR_SECTOR_NOT_FOUND 0x0000001B
#define ERROR_OUT_OF_PAPER 0x0000001C
#define ERROR_WRITE_FAULT 0x0000001D
#define ERROR_READ_FAULT 0x0000001E
#define ERROR_GEN_FAILURE 0x0000001F
#define ERROR_SHARING_VIOLATION 0x00000020
#define ERROR_LOCK_VIOLATION 0x00000021
#define ERROR_WRONG_DISK 0x00000022
#define ERROR_SHARING_BUFFER_EXCEEDED 0x00000024
#define ERROR_HANDLE_EOF 0x00000026
#define ERROR_HANDLE_DISK_FULL 0x00000027
#define ERROR_NOT_SUPPORTED 0x00000032
#define ERROR_REM_NOT_LIST 0x00000033
#define ERROR_DUP_NAME 0x00000034
#define ERROR_BAD_NETPATH 0x00000035
#define ERROR_NETWORK_BUSY 0x00000036
#define ERROR_DEV_NOT_EXIST 0x00000037
#define ERROR_TOO_MANY_CMDS 0x00000038
#define ERROR_ADAP_HDW_ERR 0x00000039
#define ERROR_BAD_NET_RESP 0x0000003A
#define ERROR_UNEXP_NET_ERR 0x0000003B
#define ERROR_BAD_REM_ADAP 0x0000003C
#define ERROR_PRINTQ_FULL 0x0000003D
#define ERROR_NO_SPOOL_SPACE 0x0000003E
#define ERROR_PRINT_CANCELLED 0x0000003F
#define ERROR_NETNAME_DELETED 0x00000040
#define ERROR_NETWORK_ACCESS_DENIED 0x00000041
#define ERROR_BAD_DEV_TYPE 0x00000042
#define ERROR_BAD_NET_NAME 0x00000043
#define ERROR_TOO_MANY_NAMES 0x00000044
#define ERROR_TOO_MANY_SESS 0x00000045
#define ERROR_SHARING_PAUSED 0x00000046
#define ERROR_REQ_NOT_ACCEP 0x00000047
#define ERROR_REDIR_PAUSED 0x00000048
#define ERROR_FILE_EXISTS 0x00000050
#define ERROR_CANNOT_MAKE 0x00000052
#define ERROR_FAIL_I24 0x00000053
#define ERROR_OUT_OF_STRUCTURES 0x00000054
#define ERROR_ALREADY_ASSIGNED 0x00000055
#define ERROR_INVALID_PASSWORD 0x00000056
#define ERROR_INVALID_PARAMETER 0x00000057
#define ERROR_NET_WRITE_FAULT 0x00000058
#define ERROR_NO_PROC_SLOTS 0x00000059
#define ERROR_TOO_MANY_SEMAPHORES 0x00000064
#define ERROR_EXCL_SEM_ALREADY_OWNED 0x00000065
#define ERROR_SEM_IS_SET 0x00000066
#define ERROR_TOO_MANY_SEM_REQUESTS 0x00000067
#define ERROR_INVALID_AT_INTERRUPT_TIME 0x00000068
#define ERROR_SEM_OWNER_DIED 0x00000069
#define ERROR_SEM_USER_LIMIT 0x0000006A
#define ERROR_DISK_CHANGE 0x0000006B
#define ERROR_DRIVE_LOCKED 0x0000006C
#define ERROR_BROKEN_PIPE 0x0000006D
#define ERROR_OPEN_FAILED 0x0000006E
#define ERROR_BUFFER_OVERFLOW 0x0000006F
#define ERROR_DISK_FULL 0x00000070
#define ERROR_NO_MORE_SEARCH_HANDLES 0x00000071
#define ERROR_INVALID_TARGET_HANDLE 0x00000072
#define ERROR_INVALID_CATEGORY 0x00000075
#define ERROR_INVALID_VERIFY_SWITCH 0x00000076
#define ERROR_BAD_DRIVER_LEVEL 0x00000077
#define ERROR_CALL_NOT_IMPLEMENTED 0x00000078
#define ERROR_SEM_TIMEOUT 0x00000079
#define ERROR_INSUFFICIENT_BUFFER 0x0000007A
#define ERROR_INVALID_NAME 0x0000007B
#define ERROR_INVALID_LEVEL 0x0000007C
#define ERROR_NO_VOLUME_LABEL 0x0000007D
#define ERROR_MOD_NOT_FOUND 0x0000007E
#define ERROR_PROC_NOT_FOUND 0x0000007F
#define ERROR_WAIT_NO_CHILDREN 0x00000080
#define ERROR_CHILD_NOT_COMPLETE 0x00000081
#define ERROR_DIRECT_ACCESS_HANDLE 0x00000082
#define ERROR_NEGATIVE_SEEK 0x00000083
#define ERROR_SEEK_ON_DEVICE 0x00000084
#define ERROR_IS_JOIN_TARGET 0x00000085
#define ERROR_IS_JOINED 0x00000086
#define ERROR_IS_SUBSTED 0x00000087
#define ERROR_NOT_JOINED 0x00000088
#define ERROR_NOT_SUBSTED 0x00000089
#define ERROR_JOIN_TO_JOIN 0x0000008A
#define ERROR_SUBST_TO_SUBST 0x0000008B
#define ERROR_JOIN_TO_SUBST 0x0000008C
#define ERROR_SUBST_TO_JOIN 0x0000008D
#define ERROR_BUSY_DRIVE 0x0000008E
#define ERROR_SAME_DRIVE 0x0000008F
#define ERROR_DIR_NOT_ROOT 0x00000090
#define ERROR_DIR_NOT_EMPTY 0x00000091
#define ERROR_IS_SUBST_PATH 0x00000092
#define ERROR_IS_JOIN_PATH 0x00000093
#define ERROR_PATH_BUSY 0x00000094
#define ERROR_IS_SUBST_TARGET 0x00000095
#define ERROR_SYSTEM_TRACE 0x00000096
#define ERROR_INVALID_EVENT_COUNT 0x00000097
#define ERROR_TOO_MANY_MUXWAITERS 0x00000098
#define ERROR_INVALID_LIST_FORMAT 0x00000099
#define ERROR_LABEL_TOO_LONG 0x0000009A
#define ERROR_TOO_MANY_TCBS 0x0000009B
#define ERROR_SIGNAL_REFUSED 0x0000009C
#define ERROR_DISCARDED 0x0000009D
#define ERROR_NOT_LOCKED 0x0000009E
#define ERROR_BAD_THREADID_ADDR 0x0000009F
#define ERROR_BAD_ARGUMENTS 0x000000A0
#define ERROR_BAD_PATHNAME 0x000000A1
#define ERROR_SIGNAL_PENDING 0x000000A2
#define ERROR_MAX_THRDS_REACHED 0x000000A4
#define ERROR_LOCK_FAILED 0x000000A7
#define ERROR_BUSY 0x000000AA
#define ERROR_DEVICE_SUPPORT_IN_PROGRESS 0x000000AB
#define ERROR_CANCEL_VIOLATION 0x000000AD
#define ERROR_ATOMIC_LOCKS_NOT_SUPPORTED 0x000000AE
#define ERROR_INVALID_SEGMENT_NUMBER 0x000000B4
#define ERROR_INVALID_ORDINAL 0x000000B6
#define ERROR_ALREADY_EXISTS 0x000000B7
#define ERROR_INVALID_FLAG_NUMBER 0x000000BA
#define ERROR_SEM_NOT_FOUND 0x000000BB
#define ERROR_INVALID_STARTING_CODESEG 0x000000BC
#define ERROR_INVALID_STACKSEG 0x000000BD
#define ERROR_INVALID_MODULETYPE 0x000000BE
#define ERROR_INVALID_EXE_SIGNATURE 0x000000BF
#define ERROR_EXE_MARKED_INVALID 0x000000C0
#define ERROR_BAD_EXE_FORMAT 0x000000C1
#define ERROR_ITERATED_DATA_EXCEEDS_64k 0x000000C2
#define ERROR_INVALID_MINALLOCSIZE 0x000000C3
#define ERROR_DYNLINK_FROM_INVALID_RING 0x000000C4
#define ERROR_IOPL_NOT_ENABLED 0x000000C5
#define ERROR_INVALID_SEGDPL 0x000000C6
#define ERROR_AUTODATASEG_EXCEEDS_64k 0x000000C7
#define ERROR_RING2SEG_MUST_BE_MOVABLE 0x000000C8
#define ERROR_RELOC_CHAIN_XEEDS_SEGLIM 0x000000C9
#define ERROR_INFLOOP_IN_RELOC_CHAIN 0x000000CA
#define ERROR_ENVVAR_NOT_FOUND 0x000000CB
#define ERROR_NO_SIGNAL_SENT 0x000000CD
#define ERROR_FILENAME_EXCED_RANGE 0x000000CE
#define ERROR_RING2_STACK_IN_USE 0x000000CF
#define ERROR_META_EXPANSION_TOO_LONG 0x000000D0
#define ERROR_INVALID_SIGNAL_NUMBER 0x000000D1
#define ERROR_THREAD_1_INACTIVE 0x000000D2
#define ERROR_LOCKED 0x000000D4
#define ERROR_TOO_MANY_MODULES 0x000000D6
#define ERROR_NESTING_NOT_ALLOWED 0x000000D7
#define ERROR_EXE_MACHINE_TYPE_MISMATCH 0x000000D8
#define ERROR_EXE_CANNOT_MODIFY_SIGNED_BINARY 0x000000D9
#define ERROR_EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY 0x000000DA
#define ERROR_FILE_CHECKED_OUT 0x000000DC
#define ERROR_CHECKOUT_REQUIRED 0x000000DD
#define ERROR_BAD_FILE_TYPE 0x000000DE
#define ERROR_FILE_TOO_LARGE 0x000000DF
#define ERROR_FORMS_AUTH_REQUIRED 0x000000E0
#define ERROR_VIRUS_INFECTED 0x000000E1
#define ERROR_VIRUS_DELETED 0x000000E2
#define ERROR_PIPE_LOCAL 0x000000E5
#define ERROR_BAD_PIPE 0x000000E6
#define ERROR_PIPE_BUSY 0x000000E7
#define ERROR_NO_DATA 0x000000E8
#define ERROR_PIPE_NOT_CONNECTED 0x000000E9
#define ERROR_MORE_DATA 0x000000EA
#define ERROR_VC_DISCONNECTED 0x000000F0
#define ERROR_INVALID_EA_NAME 0x000000FE
#define ERROR_EA_LIST_INCONSISTENT 0x000000FF
#define WAIT_TIMEOUT 0x00000102
#define ERROR_NO_MORE_ITEMS 0x00000103
#define ERROR_CANNOT_COPY 0x0000010A
#define ERROR_DIRECTORY 0x0000010B
#define ERROR_EAS_DIDNT_FIT 0x00000113
#define ERROR_EA_FILE_CORRUPT 0x00000114
#define ERROR_EA_TABLE_FULL 0x00000115
#define ERROR_INVALID_EA_HANDLE 0x00000116
#define ERROR_EAS_NOT_SUPPORTED 0x0000011A
#define ERROR_NOT_OWNER 0x00000120
#define ERROR_TOO_MANY_POSTS 0x0000012A
#define ERROR_PARTIAL_COPY 0x0000012B
#define ERROR_OPLOCK_NOT_GRANTED 0x0000012C
#define ERROR_INVALID_OPLOCK_PROTOCOL 0x0000012D
#define ERROR_DISK_TOO_FRAGMENTED 0x0000012E
#define ERROR_DELETE_PENDING 0x0000012F
#define ERROR_INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING 0x00000130
#define ERROR_SHORT_NAMES_NOT_ENABLED_ON_VOLUME 0x00000131
#define ERROR_SECURITY_STREAM_IS_INCONSISTENT 0x00000132
#define ERROR_INVALID_LOCK_RANGE 0x00000133
#define ERROR_IMAGE_SUBSYSTEM_NOT_PRESENT 0x00000134
#define ERROR_NOTIFICATION_GUID_ALREADY_DEFINED 0x00000135
#define ERROR_INVALID_EXCEPTION_HANDLER 0x00000136
#define ERROR_DUPLICATE_PRIVILEGES 0x00000137
#define ERROR_NO_RANGES_PROCESSED 0x00000138
#define ERROR_NOT_ALLOWED_ON_SYSTEM_FILE 0x00000139
#define ERROR_DISK_RESOURCES_EXHAUSTED 0x0000013A
#define ERROR_INVALID_TOKEN 0x0000013B
#define ERROR_DEVICE_FEATURE_NOT_SUPPORTED 0x0000013C
#define ERROR_MR_MID_NOT_FOUND 0x0000013D
#define ERROR_SCOPE_NOT_FOUND 0x0000013E
#define ERROR_UNDEFINED_SCOPE 0x0000013F
#define ERROR_INVALID_CAP 0x00000140
#define ERROR_DEVICE_UNREACHABLE 0x00000141
#define ERROR_DEVICE_NO_RESOURCES 0x00000142
#define ERROR_DATA_CHECKSUM_ERROR 0x00000143
#define ERROR_INTERMIXED_KERNEL_EA_OPERATION 0x00000144
#define ERROR_FILE_LEVEL_TRIM_NOT_SUPPORTED 0x00000146
#define ERROR_OFFSET_ALIGNMENT_VIOLATION 0x00000147
#define ERROR_INVALID_FIELD_IN_PARAMETER_LIST 0x00000148
#define ERROR_OPERATION_IN_PROGRESS 0x00000149
#define ERROR_BAD_DEVICE_PATH 0x0000014A
#define ERROR_TOO_MANY_DESCRIPTORS 0x0000014B
#define ERROR_SCRUB_DATA_DISABLED 0x0000014C
#define ERROR_NOT_REDUNDANT_STORAGE 0x0000014D
#define ERROR_RESIDENT_FILE_NOT_SUPPORTED 0x0000014E
#define ERROR_COMPRESSED_FILE_NOT_SUPPORTED 0x0000014F
#define ERROR_DIRECTORY_NOT_SUPPORTED 0x00000150
#define ERROR_NOT_READ_FROM_COPY 0x00000151
#define ERROR_FAIL_NOACTION_REBOOT 0x0000015E
#define ERROR_FAIL_SHUTDOWN 0x0000015F
#define ERROR_FAIL_RESTART 0x00000160
#define ERROR_MAX_SESSIONS_REACHED 0x00000161
#define ERROR_THREAD_MODE_ALREADY_BACKGROUND 0x00000190
#define ERROR_THREAD_MODE_NOT_BACKGROUND 0x00000191
#define ERROR_PROCESS_MODE_ALREADY_BACKGROUND 0x00000192
#define ERROR_PROCESS_MODE_NOT_BACKGROUND 0x00000193
#define ERROR_INVALID_ADDRESS 0x000001E7

/* System Error Codes (500-999) */

#define ERROR_USER_PROFILE_LOAD 0x000001F4
#define ERROR_ARITHMETIC_OVERFLOW 0x00000216
#define ERROR_PIPE_CONNECTED 0x00000217
#define ERROR_PIPE_LISTENING 0x00000218
#define ERROR_VERIFIER_STOP 0x00000219
#define ERROR_ABIOS_ERROR 0x0000021A
#define ERROR_WX86_WARNING 0x0000021B
#define ERROR_WX86_ERROR 0x0000021C
#define ERROR_TIMER_NOT_CANCELED 0x0000021D
#define ERROR_UNWIND 0x0000021E
#define ERROR_BAD_STACK 0x0000021F
#define ERROR_INVALID_UNWIND_TARGET 0x00000220
#define ERROR_INVALID_PORT_ATTRIBUTES 0x00000221
#define ERROR_PORT_MESSAGE_TOO_LONG 0x00000222
#define ERROR_INVALID_QUOTA_LOWER 0x00000223
#define ERROR_DEVICE_ALREADY_ATTACHED 0x00000224
#define ERROR_INSTRUCTION_MISALIGNMENT 0x00000225
#define ERROR_PROFILING_NOT_STARTED 0x00000226
#define ERROR_PROFILING_NOT_STOPPED 0x00000227
#define ERROR_COULD_NOT_INTERPRET 0x00000228
#define ERROR_PROFILING_AT_LIMIT 0x00000229
#define ERROR_CANT_WAIT 0x0000022A
#define ERROR_CANT_TERMINATE_SELF 0x0000022B
#define ERROR_UNEXPECTED_MM_CREATE_ERR 0x0000022C
#define ERROR_UNEXPECTED_MM_MAP_ERROR 0x0000022D
#define ERROR_UNEXPECTED_MM_EXTEND_ERR 0x0000022E
#define ERROR_BAD_FUNCTION_TABLE 0x0000022F
#define ERROR_NO_GUID_TRANSLATION 0x00000230
#define ERROR_INVALID_LDT_SIZE 0x00000231
#define ERROR_INVALID_LDT_OFFSET 0x00000233
#define ERROR_INVALID_LDT_DESCRIPTOR 0x00000234
#define ERROR_TOO_MANY_THREADS 0x00000235
#define ERROR_THREAD_NOT_IN_PROCESS 0x00000236
#define ERROR_PAGEFILE_QUOTA_EXCEEDED 0x00000237
#define ERROR_LOGON_SERVER_CONFLICT 0x00000238
#define ERROR_SYNCHRONIZATION_REQUIRED 0x00000239
#define ERROR_NET_OPEN_FAILED 0x0000023A
#define ERROR_IO_PRIVILEGE_FAILED 0x0000023B
#define ERROR_CONTROL_C_EXIT 0x0000023C
#define ERROR_MISSING_SYSTEMFILE 0x0000023D
#define ERROR_UNHANDLED_EXCEPTION 0x0000023E
#define ERROR_APP_INIT_FAILURE 0x0000023F
#define ERROR_PAGEFILE_CREATE_FAILED 0x00000240
#define ERROR_INVALID_IMAGE_HASH 0x00000241
#define ERROR_NO_PAGEFILE 0x00000242
#define ERROR_ILLEGAL_FLOAT_CONTEXT 0x00000243
#define ERROR_NO_EVENT_PAIR 0x00000244
#define ERROR_DOMAIN_CTRLR_CONFIG_ERROR 0x00000245
#define ERROR_ILLEGAL_CHARACTER 0x00000246
#define ERROR_UNDEFINED_CHARACTER 0x00000247
#define ERROR_FLOPPY_VOLUME 0x00000248
#define ERROR_BIOS_FAILED_TO_CONNECT_INTERRUPT 0x00000249
#define ERROR_BACKUP_CONTROLLER 0x0000024A
#define ERROR_MUTANT_LIMIT_EXCEEDED 0x0000024B
#define ERROR_FS_DRIVER_REQUIRED 0x0000024C
#define ERROR_CANNOT_LOAD_REGISTRY_FILE 0x0000024D
#define ERROR_DEBUG_ATTACH_FAILED 0x0000024E
#define ERROR_SYSTEM_PROCESS_TERMINATED 0x0000024F
#define ERROR_DATA_NOT_ACCEPTED 0x00000250
#define ERROR_VDM_HARD_ERROR 0x00000251
#define ERROR_DRIVER_CANCEL_TIMEOUT 0x00000252
#define ERROR_REPLY_MESSAGE_MISMATCH 0x00000253
#define ERROR_LOST_WRITEBEHIND_DATA 0x00000254
#define ERROR_CLIENT_SERVER_PARAMETERS_INVALID 0x00000255
#define ERROR_NOT_TINY_STREAM 0x00000256
#define ERROR_STACK_OVERFLOW_READ 0x00000257
#define ERROR_CONVERT_TO_LARGE 0x00000258
#define ERROR_FOUND_OUT_OF_SCOPE 0x00000259
#define ERROR_ALLOCATE_BUCKET 0x0000025A
#define ERROR_MARSHALL_OVERFLOW 0x0000025B
#define ERROR_INVALID_VARIANT 0x0000025C
#define ERROR_BAD_COMPRESSION_BUFFER 0x0000025D
#define ERROR_AUDIT_FAILED 0x0000025E
#define ERROR_TIMER_RESOLUTION_NOT_SET 0x0000025F
#define ERROR_INSUFFICIENT_LOGON_INFO 0x00000260
#define ERROR_BAD_DLL_ENTRYPOINT 0x00000261
#define ERROR_BAD_SERVICE_ENTRYPOINT 0x00000262
#define ERROR_IP_ADDRESS_CONFLICT1 0x00000263
#define ERROR_IP_ADDRESS_CONFLICT2 0x00000264
#define ERROR_REGISTRY_QUOTA_LIMIT 0x00000265
#define ERROR_NO_CALLBACK_ACTIVE 0x00000266
#define ERROR_PWD_TOO_SHORT 0x00000267
#define ERROR_PWD_TOO_RECENT 0x00000268
#define ERROR_PWD_HISTORY_CONFLICT 0x00000269
#define ERROR_UNSUPPORTED_COMPRESSION 0x0000026A
#define ERROR_INVALID_HW_PROFILE 0x0000026B
#define ERROR_INVALID_PLUGPLAY_DEVICE_PATH 0x0000026C
#define ERROR_QUOTA_LIST_INCONSISTENT 0x0000026D
#define ERROR_EVALUATION_EXPIRATION 0x0000026E
#define ERROR_ILLEGAL_DLL_RELOCATION 0x0000026F
#define ERROR_DLL_INIT_FAILED_LOGOFF 0x00000270
#define ERROR_VALIDATE_CONTINUE 0x00000271
#define ERROR_NO_MORE_MATCHES 0x00000272
#define ERROR_RANGE_LIST_CONFLICT 0x00000273
#define ERROR_SERVER_SID_MISMATCH 0x00000274
#define ERROR_CANT_ENABLE_DENY_ONLY 0x00000275
#define ERROR_FLOAT_MULTIPLE_FAULTS 0x00000276
#define ERROR_FLOAT_MULTIPLE_TRAPS 0x00000277
#define ERROR_NOINTERFACE 0x00000278
#define ERROR_DRIVER_FAILED_SLEEP 0x00000279
#define ERROR_CORRUPT_SYSTEM_FILE 0x0000027A
#define ERROR_COMMITMENT_MINIMUM 0x0000027B
#define ERROR_PNP_RESTART_ENUMERATION 0x0000027C
#define ERROR_SYSTEM_IMAGE_BAD_SIGNATURE 0x0000027D
#define ERROR_PNP_REBOOT_REQUIRED 0x0000027E
#define ERROR_INSUFFICIENT_POWER 0x0000027F
#define ERROR_MULTIPLE_FAULT_VIOLATION 0x00000280
#define ERROR_SYSTEM_SHUTDOWN 0x00000281
#define ERROR_PORT_NOT_SET 0x00000282
#define ERROR_DS_VERSION_CHECK_FAILURE 0x00000283
#define ERROR_RANGE_NOT_FOUND 0x00000284
#define ERROR_NOT_SAFE_MODE_DRIVER 0x00000286
#define ERROR_FAILED_DRIVER_ENTRY 0x00000287
#define ERROR_DEVICE_ENUMERATION_ERROR 0x00000288
#define ERROR_MOUNT_POINT_NOT_RESOLVED 0x00000289
#define ERROR_INVALID_DEVICE_OBJECT_PARAMETER 0x0000028A
/* The following is not a typo. It's the same spelling as in the Microsoft headers */
#define ERROR_MCA_OCCURED 0x0000028B
#define ERROR_DRIVER_DATABASE_ERROR 0x0000028C
#define ERROR_SYSTEM_HIVE_TOO_LARGE 0x0000028D
#define ERROR_DRIVER_FAILED_PRIOR_UNLOAD 0x0000028E
#define ERROR_VOLSNAP_PREPARE_HIBERNATE 0x0000028F
#define ERROR_HIBERNATION_FAILURE 0x00000290
#define ERROR_PWD_TOO_LONG 0x00000291
#define ERROR_FILE_SYSTEM_LIMITATION 0x00000299
#define ERROR_ASSERTION_FAILURE 0x0000029C
#define ERROR_ACPI_ERROR 0x0000029D
#define ERROR_WOW_ASSERTION 0x0000029E
#define ERROR_PNP_BAD_MPS_TABLE 0x0000029F
#define ERROR_PNP_TRANSLATION_FAILED 0x000002A0
#define ERROR_PNP_IRQ_TRANSLATION_FAILED 0x000002A1
#define ERROR_PNP_INVALID_ID 0x000002A2
#define ERROR_WAKE_SYSTEM_DEBUGGER 0x000002A3
#define ERROR_HANDLES_CLOSED 0x000002A4
#define ERROR_EXTRANEOUS_INFORMATION 0x000002A5
#define ERROR_RXACT_COMMIT_NECESSARY 0x000002A6
#define ERROR_MEDIA_CHECK 0x000002A7
#define ERROR_GUID_SUBSTITUTION_MADE 0x000002A8
#define ERROR_STOPPED_ON_SYMLINK 0x000002A9
#define ERROR_LONGJUMP 0x000002AA
#define ERROR_PLUGPLAY_QUERY_VETOED 0x000002AB
#define ERROR_UNWIND_CONSOLIDATE 0x000002AC
#define ERROR_REGISTRY_HIVE_RECOVERED 0x000002AD
#define ERROR_DLL_MIGHT_BE_INSECURE 0x000002AE
#define ERROR_DLL_MIGHT_BE_INCOMPATIBLE 0x000002AF
#define ERROR_DBG_EXCEPTION_NOT_HANDLED 0x000002B0
#define ERROR_DBG_REPLY_LATER 0x000002B1
#define ERROR_DBG_UNABLE_TO_PROVIDE_HANDLE 0x000002B2
#define ERROR_DBG_TERMINATE_THREAD 0x000002B3
#define ERROR_DBG_TERMINATE_PROCESS 0x000002B4
#define ERROR_DBG_CONTROL_C 0x000002B5
#define ERROR_DBG_PRINTEXCEPTION_C 0x000002B6
#define ERROR_DBG_RIPEXCEPTION 0x000002B7
#define ERROR_DBG_CONTROL_BREAK 0x000002B8
#define ERROR_DBG_COMMAND_EXCEPTION 0x000002B9
#define ERROR_OBJECT_NAME_EXISTS 0x000002BA
#define ERROR_THREAD_WAS_SUSPENDED 0x000002BB
#define ERROR_IMAGE_NOT_AT_BASE 0x000002BC
#define ERROR_RXACT_STATE_CREATED 0x000002BD
#define ERROR_SEGMENT_NOTIFICATION 0x000002BE
#define ERROR_BAD_CURRENT_DIRECTORY 0x000002BF
#define ERROR_FT_READ_RECOVERY_FROM_BACKUP 0x000002C0
#define ERROR_FT_WRITE_RECOVERY 0x000002C1
#define ERROR_IMAGE_MACHINE_TYPE_MISMATCH 0x000002C2
#define ERROR_RECEIVE_PARTIAL 0x000002C3
#define ERROR_RECEIVE_EXPEDITED 0x000002C4
#define ERROR_RECEIVE_PARTIAL_EXPEDITED 0x000002C5
#define ERROR_EVENT_DONE 0x000002C6
#define ERROR_EVENT_PENDING 0x000002C7
#define ERROR_CHECKING_FILE_SYSTEM 0x000002C8
#define ERROR_FATAL_APP_EXIT 0x000002C9
#define ERROR_PREDEFINED_HANDLE 0x000002CA
#define ERROR_WAS_UNLOCKED 0x000002CB
#define ERROR_SERVICE_NOTIFICATION 0x000002CC
#define ERROR_WAS_LOCKED 0x000002CD
#define ERROR_LOG_HARD_ERROR 0x000002CE
#define ERROR_ALREADY_WIN32 0x000002CF
#define ERROR_IMAGE_MACHINE_TYPE_MISMATCH_EXE 0x000002D0
#define ERROR_NO_YIELD_PERFORMED 0x000002D1
#define ERROR_TIMER_RESUME_IGNORED 0x000002D2
#define ERROR_ARBITRATION_UNHANDLED 0x000002D3
#define ERROR_CARDBUS_NOT_SUPPORTED 0x000002D4
#define ERROR_MP_PROCESSOR_MISMATCH 0x000002D5
#define ERROR_HIBERNATED 0x000002D6
#define ERROR_RESUME_HIBERNATION 0x000002D7
#define ERROR_FIRMWARE_UPDATED 0x000002D8
#define ERROR_DRIVERS_LEAKING_LOCKED_PAGES 0x000002D9
#define ERROR_WAKE_SYSTEM 0x000002DA
#define ERROR_WAIT_1 0x000002DB
#define ERROR_WAIT_2 0x000002DC
#define ERROR_WAIT_3 0x000002DD
#define ERROR_WAIT_63 0x000002DE
#define ERROR_ABANDONED_WAIT_0 0x000002DF
#define ERROR_ABANDONED_WAIT_63 0x000002E0
#define ERROR_USER_APC 0x000002E1
#define ERROR_KERNEL_APC 0x000002E2
#define ERROR_ALERTED 0x000002E3
#define ERROR_ELEVATION_REQUIRED 0x000002E4
#define ERROR_REPARSE 0x000002E5
#define ERROR_OPLOCK_BREAK_IN_PROGRESS 0x000002E6
#define ERROR_VOLUME_MOUNTED 0x000002E7
#define ERROR_RXACT_COMMITTED 0x000002E8
#define ERROR_NOTIFY_CLEANUP 0x000002E9
#define ERROR_PRIMARY_TRANSPORT_CONNECT_FAILED 0x000002EA
#define ERROR_PAGE_FAULT_TRANSITION 0x000002EB
#define ERROR_PAGE_FAULT_DEMAND_ZERO 0x000002EC
#define ERROR_PAGE_FAULT_COPY_ON_WRITE 0x000002ED
#define ERROR_PAGE_FAULT_GUARD_PAGE 0x000002EE
#define ERROR_PAGE_FAULT_PAGING_FILE 0x000002EF
#define ERROR_CACHE_PAGE_LOCKED 0x000002F0
#define ERROR_CRASH_DUMP 0x000002F1
#define ERROR_BUFFER_ALL_ZEROS 0x000002F2
#define ERROR_REPARSE_OBJECT 0x000002F3
#define ERROR_RESOURCE_REQUIREMENTS_CHANGED 0x000002F4
#define ERROR_TRANSLATION_COMPLETE 0x000002F5
#define ERROR_NOTHING_TO_TERMINATE 0x000002F6
#define ERROR_PROCESS_NOT_IN_JOB 0x000002F7
#define ERROR_PROCESS_IN_JOB 0x000002F8
#define ERROR_VOLSNAP_HIBERNATE_READY 0x000002F9
#define ERROR_FSFILTER_OP_COMPLETED_SUCCESSFULLY 0x000002FA
#define ERROR_INTERRUPT_VECTOR_ALREADY_CONNECTED 0x000002FB
#define ERROR_INTERRUPT_STILL_CONNECTED 0x000002FC
#define ERROR_WAIT_FOR_OPLOCK 0x000002FD
#define ERROR_DBG_EXCEPTION_HANDLED 0x000002FE
#define ERROR_DBG_CONTINUE 0x000002FF
#define ERROR_CALLBACK_POP_STACK 0x00000300
#define ERROR_COMPRESSION_DISABLED 0x00000301
#define ERROR_CANTFETCHBACKWARDS 0x00000302
#define ERROR_CANTSCROLLBACKWARDS 0x00000303
#define ERROR_ROWSNOTRELEASED 0x00000304
#define ERROR_BAD_ACCESSOR_FLAGS 0x00000305
#define ERROR_ERRORS_ENCOUNTERED 0x00000306
#define ERROR_NOT_CAPABLE 0x00000307
#define ERROR_REQUEST_OUT_OF_SEQUENCE 0x00000308
#define ERROR_VERSION_PARSE_ERROR 0x00000309
#define ERROR_BADSTARTPOSITION 0x0000030A
#define ERROR_MEMORY_HARDWARE 0x0000030B
#define ERROR_DISK_REPAIR_DISABLED 0x0000030C
#define ERROR_INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE 0x0000030D
#define ERROR_SYSTEM_POWERSTATE_TRANSITION 0x0000030E
#define ERROR_SYSTEM_POWERSTATE_COMPLEX_TRANSITION 0x0000030F
#define ERROR_MCA_EXCEPTION 0x00000310
#define ERROR_ACCESS_AUDIT_BY_POLICY 0x00000311
#define ERROR_ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY 0x00000312
#define ERROR_ABANDON_HIBERFILE 0x00000313
#define ERROR_LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED 0x00000314
#define ERROR_LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR 0x00000315
#define ERROR_LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR 0x00000316
#define ERROR_BAD_MCFG_TABLE 0x00000317
#define ERROR_DISK_REPAIR_REDIRECTED 0x00000318
#define ERROR_DISK_REPAIR_UNSUCCESSFUL 0x00000319
#define ERROR_CORRUPT_LOG_OVERFULL 0x0000031A
#define ERROR_CORRUPT_LOG_CORRUPTED 0x0000031B
#define ERROR_CORRUPT_LOG_UNAVAILABLE 0x0000031C
#define ERROR_CORRUPT_LOG_DELETED_FULL 0x0000031D
#define ERROR_CORRUPT_LOG_CLEARED 0x0000031E
#define ERROR_ORPHAN_NAME_EXHAUSTED 0x0000031F
#define ERROR_OPLOCK_SWITCHED_TO_NEW_HANDLE 0x00000320
#define ERROR_CANNOT_GRANT_REQUESTED_OPLOCK 0x00000321
#define ERROR_CANNOT_BREAK_OPLOCK 0x00000322
#define ERROR_OPLOCK_HANDLE_CLOSED 0x00000323
#define ERROR_NO_ACE_CONDITION 0x00000324
#define ERROR_INVALID_ACE_CONDITION 0x00000325
#define ERROR_FILE_HANDLE_REVOKED 0x00000326
#define ERROR_IMAGE_AT_DIFFERENT_BASE 0x00000327
#define ERROR_EA_ACCESS_DENIED 0x000003E2
#define ERROR_OPERATION_ABORTED 0x000003E3
#define ERROR_IO_INCOMPLETE 0x000003E4
#define ERROR_IO_PENDING 0x000003E5
#define ERROR_NOACCESS 0x000003E6
#define ERROR_SWAPERROR 0x000003E7

/* System Error Codes (1000-1299) */

#define ERROR_STACK_OVERFLOW 0x000003E9
#define ERROR_INVALID_MESSAGE 0x000003EA
#define ERROR_CAN_NOT_COMPLETE 0x000003EB
#define ERROR_INVALID_FLAGS 0x000003EC
#define ERROR_UNRECOGNIZED_VOLUME 0x000003ED
#define ERROR_FILE_INVALID 0x000003EE
#define ERROR_FULLSCREEN_MODE 0x000003EF
#define ERROR_NO_TOKEN 0x000003F0
#define ERROR_BADDB 0x000003F1
#define ERROR_BADKEY 0x000003F2
#define ERROR_CANTOPEN 0x000003F3
#define ERROR_CANTREAD 0x000003F4
#define ERROR_CANTWRITE 0x000003F5
#define ERROR_REGISTRY_RECOVERED 0x000003F6
#define ERROR_REGISTRY_CORRUPT 0x000003F7
#define ERROR_REGISTRY_IO_FAILED 0x000003F8
#define ERROR_NOT_REGISTRY_FILE 0x000003F9
#define ERROR_KEY_DELETED 0x000003FA
#define ERROR_NO_LOG_SPACE 0x000003FB
#define ERROR_KEY_HAS_CHILDREN 0x000003FC
#define ERROR_CHILD_MUST_BE_VOLATILE 0x000003FD
#define ERROR_NOTIFY_ENUM_DIR 0x000003FE
#define ERROR_DEPENDENT_SERVICES_RUNNING 0x0000041B
#define ERROR_INVALID_SERVICE_CONTROL 0x0000041C
#define ERROR_SERVICE_REQUEST_TIMEOUT 0x0000041D
#define ERROR_SERVICE_NO_THREAD 0x0000041E
#define ERROR_SERVICE_DATABASE_LOCKED 0x0000041F
#define ERROR_SERVICE_ALREADY_RUNNING 0x00000420
#define ERROR_INVALID_SERVICE_ACCOUNT 0x00000421
#define ERROR_SERVICE_DISABLED 0x00000422
#define ERROR_CIRCULAR_DEPENDENCY 0x00000423
#define ERROR_SERVICE_DOES_NOT_EXIST 0x00000424
#define ERROR_SERVICE_CANNOT_ACCEPT_CTRL 0x00000425
#define ERROR_SERVICE_NOT_ACTIVE 0x00000426
#define ERROR_FAILED_SERVICE_CONTROLLER_CONNECT 0x00000427
#define ERROR_EXCEPTION_IN_SERVICE 0x00000428
#define ERROR_DATABASE_DOES_NOT_EXIST 0x00000429
#define ERROR_SERVICE_SPECIFIC_ERROR 0x0000042A
#define ERROR_PROCESS_ABORTED 0x0000042B
#define ERROR_SERVICE_DEPENDENCY_FAIL 0x0000042C
#define ERROR_SERVICE_LOGON_FAILED 0x0000042D
#define ERROR_SERVICE_START_HANG 0x0000042E
#define ERROR_INVALID_SERVICE_LOCK 0x0000042F
#define ERROR_SERVICE_MARKED_FOR_DELETE 0x00000430
#define ERROR_SERVICE_EXISTS 0x00000431
#define ERROR_ALREADY_RUNNING_LKG 0x00000432
#define ERROR_SERVICE_DEPENDENCY_DELETED 0x00000433
#define ERROR_BOOT_ALREADY_ACCEPTED 0x00000434
#define ERROR_SERVICE_NEVER_STARTED 0x00000435
#define ERROR_DUPLICATE_SERVICE_NAME 0x00000436
#define ERROR_DIFFERENT_SERVICE_ACCOUNT 0x00000437
#define ERROR_CANNOT_DETECT_DRIVER_FAILURE 0x00000438
#define ERROR_CANNOT_DETECT_PROCESS_ABORT 0x00000439
#define ERROR_NO_RECOVERY_PROGRAM 0x0000043A
#define ERROR_SERVICE_NOT_IN_EXE 0x0000043B
#define ERROR_NOT_SAFEBOOT_SERVICE 0x0000043C
#define ERROR_END_OF_MEDIA 0x0000044C
#define ERROR_FILEMARK_DETECTED 0x0000044D
#define ERROR_BEGINNING_OF_MEDIA 0x0000044E
#define ERROR_SETMARK_DETECTED 0x0000044F
#define ERROR_NO_DATA_DETECTED 0x00000450
#define ERROR_PARTITION_FAILURE 0x00000451
#define ERROR_INVALID_BLOCK_LENGTH 0x00000452
#define ERROR_DEVICE_NOT_PARTITIONED 0x00000453
#define ERROR_UNABLE_TO_LOCK_MEDIA 0x00000454
#define ERROR_UNABLE_TO_UNLOAD_MEDIA 0x00000455
#define ERROR_MEDIA_CHANGED 0x00000456
#define ERROR_BUS_RESET 0x00000457
#define ERROR_NO_MEDIA_IN_DRIVE 0x00000458
#define ERROR_NO_UNICODE_TRANSLATION 0x00000459
#define ERROR_DLL_INIT_FAILED 0x0000045A
#define ERROR_SHUTDOWN_IN_PROGRESS 0x0000045B
#define ERROR_NO_SHUTDOWN_IN_PROGRESS 0x0000045C
#define ERROR_IO_DEVICE 0x0000045D
#define ERROR_SERIAL_NO_DEVICE 0x0000045E
#define ERROR_IRQ_BUSY 0x0000045F
#define ERROR_MORE_WRITES 0x00000460
#define ERROR_COUNTER_TIMEOUT 0x00000461
#define ERROR_FLOPPY_ID_MARK_NOT_FOUND 0x00000462
#define ERROR_FLOPPY_WRONG_CYLINDER 0x00000463
#define ERROR_FLOPPY_UNKNOWN_ERROR 0x00000464
#define ERROR_FLOPPY_BAD_REGISTERS 0x00000465
#define ERROR_DISK_RECALIBRATE_FAILED 0x00000466
#define ERROR_DISK_OPERATION_FAILED 0x00000467
#define ERROR_DISK_RESET_FAILED 0x00000468
#define ERROR_EOM_OVERFLOW 0x00000469
#define ERROR_NOT_ENOUGH_SERVER_MEMORY 0x0000046A
#define ERROR_POSSIBLE_DEADLOCK 0x0000046B
#define ERROR_MAPPED_ALIGNMENT 0x0000046C
#define ERROR_SET_POWER_STATE_VETOED 0x00000474
#define ERROR_SET_POWER_STATE_FAILED 0x00000475
#define ERROR_TOO_MANY_LINKS 0x00000476
#define ERROR_OLD_WIN_VERSION 0x0000047E
#define ERROR_APP_WRONG_OS 0x0000047F
#define ERROR_SINGLE_INSTANCE_APP 0x00000480
#define ERROR_RMODE_APP 0x00000481
#define ERROR_INVALID_DLL 0x00000482
#define ERROR_NO_ASSOCIATION 0x00000483
#define ERROR_DDE_FAIL 0x00000484
#define ERROR_DLL_NOT_FOUND 0x00000485
#define ERROR_NO_MORE_USER_HANDLES 0x00000486
#define ERROR_MESSAGE_SYNC_ONLY 0x00000487
#define ERROR_SOURCE_ELEMENT_EMPTY 0x00000488
#define ERROR_DESTINATION_ELEMENT_FULL 0x00000489
#define ERROR_ILLEGAL_ELEMENT_ADDRESS 0x0000048A
#define ERROR_MAGAZINE_NOT_PRESENT 0x0000048B
#define ERROR_DEVICE_REINITIALIZATION_NEEDED 0x0000048C
#define ERROR_DEVICE_REQUIRES_CLEANING 0x0000048D
#define ERROR_DEVICE_DOOR_OPEN 0x0000048E
#define ERROR_DEVICE_NOT_CONNECTED 0x0000048F
#define ERROR_NOT_FOUND 0x00000490
#define ERROR_NO_MATCH 0x00000491
#define ERROR_SET_NOT_FOUND 0x00000492
#define ERROR_POINT_NOT_FOUND 0x00000493
#define ERROR_NO_TRACKING_SERVICE 0x00000494
#define ERROR_NO_VOLUME_ID 0x00000495
#define ERROR_UNABLE_TO_REMOVE_REPLACED 0x00000497
#define ERROR_UNABLE_TO_MOVE_REPLACEMENT 0x00000498
#define ERROR_UNABLE_TO_MOVE_REPLACEMENT_2 0x00000499
#define ERROR_JOURNAL_DELETE_IN_PROGRESS 0x0000049A
#define ERROR_JOURNAL_NOT_ACTIVE 0x0000049B
#define ERROR_POTENTIAL_FILE_FOUND 0x0000049C
#define ERROR_JOURNAL_ENTRY_DELETED 0x0000049D
#define ERROR_SHUTDOWN_IS_SCHEDULED 0x000004A6
#define ERROR_SHUTDOWN_USERS_LOGGED_ON 0x000004A7
#define ERROR_BAD_DEVICE 0x000004B0
#define ERROR_CONNECTION_UNAVAIL 0x000004B1
#define ERROR_DEVICE_ALREADY_REMEMBERED 0x000004B2
#define ERROR_NO_NET_OR_BAD_PATH 0x000004B3
#define ERROR_BAD_PROVIDER 0x000004B4
#define ERROR_CANNOT_OPEN_PROFILE 0x000004B5
#define ERROR_BAD_PROFILE 0x000004B6
#define ERROR_NOT_CONTAINER 0x000004B7
#define ERROR_EXTENDED_ERROR 0x000004B8
#define ERROR_INVALID_GROUPNAME 0x000004B9
#define ERROR_INVALID_COMPUTERNAME 0x000004BA
#define ERROR_INVALID_EVENTNAME 0x000004BB
#define ERROR_INVALID_DOMAINNAME 0x000004BC
#define ERROR_INVALID_SERVICENAME 0x000004BD
#define ERROR_INVALID_NETNAME 0x000004BE
#define ERROR_INVALID_SHARENAME 0x000004BF
#define ERROR_INVALID_PASSWORDNAME 0x000004C0
#define ERROR_INVALID_MESSAGENAME 0x000004C1
#define ERROR_INVALID_MESSAGEDEST 0x000004C2
#define ERROR_SESSION_CREDENTIAL_CONFLICT 0x000004C3
#define ERROR_REMOTE_SESSION_LIMIT_EXCEEDED 0x000004C4
#define ERROR_DUP_DOMAINNAME 0x000004C5
#define ERROR_NO_NETWORK 0x000004C6
#define ERROR_CANCELLED 0x000004C7
#define ERROR_USER_MAPPED_FILE 0x000004C8
#define ERROR_CONNECTION_REFUSED 0x000004C9
#define ERROR_GRACEFUL_DISCONNECT 0x000004CA
#define ERROR_ADDRESS_ALREADY_ASSOCIATED 0x000004CB
#define ERROR_ADDRESS_NOT_ASSOCIATED 0x000004CC
#define ERROR_CONNECTION_INVALID 0x000004CD
#define ERROR_CONNECTION_ACTIVE 0x000004CE
#define ERROR_NETWORK_UNREACHABLE 0x000004CF
#define ERROR_HOST_UNREACHABLE 0x000004D0
#define ERROR_PROTOCOL_UNREACHABLE 0x000004D1
#define ERROR_PORT_UNREACHABLE 0x000004D2
#define ERROR_REQUEST_ABORTED 0x000004D3
#define ERROR_CONNECTION_ABORTED 0x000004D4
#define ERROR_RETRY 0x000004D5
#define ERROR_CONNECTION_COUNT_LIMIT 0x000004D6
#define ERROR_LOGIN_TIME_RESTRICTION 0x000004D7
#define ERROR_LOGIN_WKSTA_RESTRICTION 0x000004D8
#define ERROR_INCORRECT_ADDRESS 0x000004D9
#define ERROR_ALREADY_REGISTERED 0x000004DA
#define ERROR_SERVICE_NOT_FOUND 0x000004DB
#define ERROR_NOT_AUTHENTICATED 0x000004DC
#define ERROR_NOT_LOGGED_ON 0x000004DD
#define ERROR_CONTINUE 0x000004DE
#define ERROR_ALREADY_INITIALIZED 0x000004DF
#define ERROR_NO_MORE_DEVICES 0x000004E0
#define ERROR_NO_SUCH_SITE 0x000004E1
#define ERROR_DOMAIN_CONTROLLER_EXISTS 0x000004E2
#define ERROR_ONLY_IF_CONNECTED 0x000004E3
#define ERROR_OVERRIDE_NOCHANGES 0x000004E4
#define ERROR_BAD_USER_PROFILE 0x000004E5
#define ERROR_NOT_SUPPORTED_ON_SBS 0x000004E6
#define ERROR_SERVER_SHUTDOWN_IN_PROGRESS 0x000004E7
#define ERROR_HOST_DOWN 0x000004E8
#define ERROR_NON_ACCOUNT_SID 0x000004E9
#define ERROR_NON_DOMAIN_SID 0x000004EA
#define ERROR_APPHELP_BLOCK 0x000004EB
#define ERROR_ACCESS_DISABLED_BY_POLICY 0x000004EC
#define ERROR_REG_NAT_CONSUMPTION 0x000004ED
#define ERROR_CSCSHARE_OFFLINE 0x000004EE
#define ERROR_PKINIT_FAILURE 0x000004EF
#define ERROR_SMARTCARD_SUBSYSTEM_FAILURE 0x000004F0
#define ERROR_DOWNGRADE_DETECTED 0x000004F1
#define ERROR_MACHINE_LOCKED 0x000004F7
#define ERROR_CALLBACK_SUPPLIED_INVALID_DATA 0x000004F9
#define ERROR_SYNC_FOREGROUND_REFRESH_REQUIRED 0x000004FA
#define ERROR_DRIVER_BLOCKED 0x000004FB
#define ERROR_INVALID_IMPORT_OF_NON_DLL 0x000004FC
#define ERROR_ACCESS_DISABLED_WEBBLADE 0x000004FD
#define ERROR_ACCESS_DISABLED_WEBBLADE_TAMPER 0x000004FE
#define ERROR_RECOVERY_FAILURE 0x000004FF
#define ERROR_ALREADY_FIBER 0x00000500
#define ERROR_ALREADY_THREAD 0x00000501
#define ERROR_STACK_BUFFER_OVERRUN 0x00000502
#define ERROR_PARAMETER_QUOTA_EXCEEDED 0x00000503
#define ERROR_DEBUGGER_INACTIVE 0x00000504
#define ERROR_DELAY_LOAD_FAILED 0x00000505
#define ERROR_VDM_DISALLOWED 0x00000506
#define ERROR_UNIDENTIFIED_ERROR 0x00000507
#define ERROR_INVALID_CRUNTIME_PARAMETER 0x00000508
#define ERROR_BEYOND_VDL 0x00000509
#define ERROR_INCOMPATIBLE_SERVICE_SID_TYPE 0x0000050A
#define ERROR_DRIVER_PROCESS_TERMINATED 0x0000050B
#define ERROR_IMPLEMENTATION_LIMIT 0x0000050C
#define ERROR_PROCESS_IS_PROTECTED 0x0000050D
#define ERROR_SERVICE_NOTIFY_CLIENT_LAGGING 0x0000050E
#define ERROR_DISK_QUOTA_EXCEEDED 0x0000050F
#define ERROR_CONTENT_BLOCKED 0x00000510
#define ERROR_INCOMPATIBLE_SERVICE_PRIVILEGE 0x00000511
#define ERROR_APP_HANG 0x00000512
#define ERROR_INVALID_LABEL 0x00000513

/* System Error Codes (1300-1699) */
#define ERROR_NOT_ALL_ASSIGNED 0x00000514
#define ERROR_SOME_NOT_MAPPED 0x00000515
#define ERROR_NO_QUOTAS_FOR_ACCOUNT 0x00000516
#define ERROR_LOCAL_USER_SESSION_KEY 0x00000517
#define ERROR_NULL_LM_PASSWORD 0x00000518
#define ERROR_UNKNOWN_REVISION 0x00000519
#define ERROR_REVISION_MISMATCH 0x0000051A
#define ERROR_INVALID_OWNER 0x0000051B
#define ERROR_INVALID_PRIMARY_GROUP 0x0000051C
#define ERROR_NO_IMPERSONATION_TOKEN 0x0000051D
#define ERROR_CANT_DISABLE_MANDATORY 0x0000051E
#define ERROR_NO_LOGON_SERVERS 0x0000051F
#define ERROR_NO_SUCH_LOGON_SESSION 0x00000520
#define ERROR_NO_SUCH_PRIVILEGE 0x00000521
#define ERROR_PRIVILEGE_NOT_HELD 0x00000522
#define ERROR_INVALID_ACCOUNT_NAME 0x00000523
#define ERROR_USER_EXISTS 0x00000524
#define ERROR_NO_SUCH_USER 0x00000525
#define ERROR_GROUP_EXISTS 0x00000526
#define ERROR_NO_SUCH_GROUP 0x00000527
#define ERROR_MEMBER_IN_GROUP 0x00000528
#define ERROR_MEMBER_NOT_IN_GROUP 0x00000529
#define ERROR_LAST_ADMIN 0x0000052A
#define ERROR_WRONG_PASSWORD 0x0000052B
#define ERROR_ILL_FORMED_PASSWORD 0x0000052C
#define ERROR_PASSWORD_RESTRICTION 0x0000052D
#define ERROR_LOGON_FAILURE 0x0000052E
#define ERROR_ACCOUNT_RESTRICTION 0x0000052F
#define ERROR_INVALID_LOGON_HOURS 0x00000530
#define ERROR_INVALID_WORKSTATION 0x00000531
#define ERROR_PASSWORD_EXPIRED 0x00000532
#define ERROR_ACCOUNT_DISABLED 0x00000533
#define ERROR_NONE_MAPPED 0x00000534
#define ERROR_TOO_MANY_LUIDS_REQUESTED 0x00000535
#define ERROR_LUIDS_EXHAUSTED 0x00000536
#define ERROR_INVALID_SUB_AUTHORITY 0x00000537
#define ERROR_INVALID_ACL 0x00000538
#define ERROR_INVALID_SID 0x00000539
#define ERROR_INVALID_SECURITY_DESCR 0x0000053A
#define ERROR_BAD_INHERITANCE_ACL 0x0000053C
#define ERROR_SERVER_DISABLED 0x0000053D
#define ERROR_SERVER_NOT_DISABLED 0x0000053E
#define ERROR_INVALID_ID_AUTHORITY 0x0000053F
#define ERROR_ALLOTTED_SPACE_EXCEEDED 0x00000540
#define ERROR_INVALID_GROUP_ATTRIBUTES 0x00000541
#define ERROR_BAD_IMPERSONATION_LEVEL 0x00000542
#define ERROR_CANT_OPEN_ANONYMOUS 0x00000543
#define ERROR_BAD_VALIDATION_CLASS 0x00000544
#define ERROR_BAD_TOKEN_TYPE 0x00000545
#define ERROR_NO_SECURITY_ON_OBJECT 0x00000546
#define ERROR_CANT_ACCESS_DOMAIN_INFO 0x00000547
#define ERROR_INVALID_SERVER_STATE 0x00000548
#define ERROR_INVALID_DOMAIN_STATE 0x00000549
#define ERROR_INVALID_DOMAIN_ROLE 0x0000054A
#define ERROR_NO_SUCH_DOMAIN 0x0000054B
#define ERROR_DOMAIN_EXISTS 0x0000054C
#define ERROR_DOMAIN_LIMIT_EXCEEDED 0x0000054D
#define ERROR_INTERNAL_DB_CORRUPTION 0x0000054E
#define ERROR_INTERNAL_ERROR 0x0000054F
#define ERROR_GENERIC_NOT_MAPPED 0x00000550
#define ERROR_BAD_DESCRIPTOR_FORMAT 0x00000551
#define ERROR_NOT_LOGON_PROCESS 0x00000552
#define ERROR_LOGON_SESSION_EXISTS 0x00000553
#define ERROR_NO_SUCH_PACKAGE 0x00000554
#define ERROR_BAD_LOGON_SESSION_STATE 0x00000555
#define ERROR_LOGON_SESSION_COLLISION 0x00000556
#define ERROR_INVALID_LOGON_TYPE 0x00000557
#define ERROR_CANNOT_IMPERSONATE 0x00000558
#define ERROR_RXACT_INVALID_STATE 0x00000559
#define ERROR_RXACT_COMMIT_FAILURE 0x0000055A
#define ERROR_SPECIAL_ACCOUNT 0x0000055B
#define ERROR_SPECIAL_GROUP 0x0000055C
#define ERROR_SPECIAL_USER 0x0000055D
#define ERROR_MEMBERS_PRIMARY_GROUP 0x0000055E
#define ERROR_TOKEN_ALREADY_IN_USE 0x0000055F
#define ERROR_NO_SUCH_ALIAS 0x00000560
#define ERROR_MEMBER_NOT_IN_ALIAS 0x00000561
#define ERROR_MEMBER_IN_ALIAS 0x00000562
#define ERROR_ALIAS_EXISTS 0x00000563
#define ERROR_LOGON_NOT_GRANTED 0x00000564
#define ERROR_TOO_MANY_SECRETS 0x00000565
#define ERROR_SECRET_TOO_LONG 0x00000566
#define ERROR_INTERNAL_DB_ERROR 0x00000567
#define ERROR_TOO_MANY_CONTEXT_IDS 0x00000568
#define ERROR_LOGON_TYPE_NOT_GRANTED 0x00000569
#define ERROR_NT_CROSS_ENCRYPTION_REQUIRED 0x0000056A
#define ERROR_NO_SUCH_MEMBER 0x0000056B
#define ERROR_INVALID_MEMBER 0x0000056C
#define ERROR_TOO_MANY_SIDS 0x0000056D
#define ERROR_LM_CROSS_ENCRYPTION_REQUIRED 0x0000056E
#define ERROR_NO_INHERITANCE 0x0000056F
#define ERROR_FILE_CORRUPT 0x00000570
#define ERROR_DISK_CORRUPT 0x00000571
#define ERROR_NO_USER_SESSION_KEY 0x00000572
#define ERROR_LICENSE_QUOTA_EXCEEDED 0x00000573
#define ERROR_WRONG_TARGET_NAME 0x00000574
#define ERROR_MUTUAL_AUTH_FAILED 0x00000575
#define ERROR_TIME_SKEW 0x00000576
#define ERROR_CURRENT_DOMAIN_NOT_ALLOWED 0x00000577
#define ERROR_INVALID_WINDOW_HANDLE 0x00000578
#define ERROR_INVALID_MENU_HANDLE 0x00000579
#define ERROR_INVALID_CURSOR_HANDLE 0x0000057A
#define ERROR_INVALID_ACCEL_HANDLE 0x0000057B
#define ERROR_INVALID_HOOK_HANDLE 0x0000057C
#define ERROR_INVALID_DWP_HANDLE 0x0000057D
#define ERROR_TLW_WITH_WSCHILD 0x0000057E
#define ERROR_CANNOT_FIND_WND_CLASS 0x0000057F
#define ERROR_WINDOW_OF_OTHER_THREAD 0x00000580
#define ERROR_HOTKEY_ALREADY_REGISTERED 0x00000581
#define ERROR_CLASS_ALREADY_EXISTS 0x00000582
#define ERROR_CLASS_DOES_NOT_EXIST 0x00000583
#define ERROR_CLASS_HAS_WINDOWS 0x00000584
#define ERROR_INVALID_INDEX 0x00000585
#define ERROR_INVALID_ICON_HANDLE 0x00000586
#define ERROR_PRIVATE_DIALOG_INDEX 0x00000587
#define ERROR_LISTBOX_ID_NOT_FOUND 0x00000588
#define ERROR_NO_WILDCARD_CHARACTERS 0x00000589
#define ERROR_CLIPBOARD_NOT_OPEN 0x0000058A
#define ERROR_HOTKEY_NOT_REGISTERED 0x0000058B
#define ERROR_WINDOW_NOT_DIALOG 0x0000058C
#define ERROR_CONTROL_ID_NOT_FOUND 0x0000058D
#define ERROR_INVALID_COMBOBOX_MESSAGE 0x0000058E
#define ERROR_WINDOW_NOT_COMBOBOX 0x0000058F
#define ERROR_INVALID_EDIT_HEIGHT 0x00000590
#define ERROR_DC_NOT_FOUND 0x00000591
#define ERROR_INVALID_HOOK_FILTER 0x00000592
#define ERROR_INVALID_FILTER_PROC 0x00000593
#define ERROR_HOOK_NEEDS_HMOD 0x00000594
#define ERROR_GLOBAL_ONLY_HOOK 0x00000595
#define ERROR_JOURNAL_HOOK_SET 0x00000596
#define ERROR_HOOK_NOT_INSTALLED 0x00000597
#define ERROR_INVALID_LB_MESSAGE 0x00000598
#define ERROR_SETCOUNT_ON_BAD_LB 0x00000599
#define ERROR_LB_WITHOUT_TABSTOPS 0x0000059A
#define ERROR_DESTROY_OBJECT_OF_OTHER_THREAD 0x0000059B
#define ERROR_CHILD_WINDOW_MENU 0x0000059C
#define ERROR_NO_SYSTEM_MENU 0x0000059D
#define ERROR_INVALID_MSGBOX_STYLE 0x0000059E
#define ERROR_INVALID_SPI_VALUE 0x0000059F
#define ERROR_SCREEN_ALREADY_LOCKED 0x000005A0
#define ERROR_HWNDS_HAVE_DIFF_PARENT 0x000005A1
#define ERROR_NOT_CHILD_WINDOW 0x000005A2
#define ERROR_INVALID_GW_COMMAND 0x000005A3
#define ERROR_INVALID_THREAD_ID 0x000005A4
#define ERROR_NON_MDICHILD_WINDOW 0x000005A5
#define ERROR_POPUP_ALREADY_ACTIVE 0x000005A6
#define ERROR_NO_SCROLLBARS 0x000005A7
#define ERROR_INVALID_SCROLLBAR_RANGE 0x000005A8
#define ERROR_INVALID_SHOWWIN_COMMAND 0x000005A9
#define ERROR_NO_SYSTEM_RESOURCES 0x000005AA
#define ERROR_NONPAGED_SYSTEM_RESOURCES 0x000005AB
#define ERROR_PAGED_SYSTEM_RESOURCES 0x000005AC
#define ERROR_WORKING_SET_QUOTA 0x000005AD
#define ERROR_PAGEFILE_QUOTA 0x000005AE
#define ERROR_COMMITMENT_LIMIT 0x000005AF
#define ERROR_MENU_ITEM_NOT_FOUND 0x000005B0
#define ERROR_INVALID_KEYBOARD_HANDLE 0x000005B1
#define ERROR_HOOK_TYPE_NOT_ALLOWED 0x000005B2
#define ERROR_REQUIRES_INTERACTIVE_WINDOWSTATION 0x000005B3
#define ERROR_TIMEOUT 0x000005B4
#define ERROR_INVALID_MONITOR_HANDLE 0x000005B5
#define ERROR_INCORRECT_SIZE 0x000005B6
#define ERROR_SYMLINK_CLASS_DISABLED 0x000005B7
#define ERROR_SYMLINK_NOT_SUPPORTED 0x000005B8
#define ERROR_XML_PARSE_ERROR 0x000005B9
#define ERROR_XMLDSIG_ERROR 0x000005BA
#define ERROR_RESTART_APPLICATION 0x000005BB
#define ERROR_WRONG_COMPARTMENT 0x000005BC
#define ERROR_AUTHIP_FAILURE 0x000005BD
#define ERROR_NO_NVRAM_RESOURCES 0x000005BE
#define ERROR_NOT_GUI_PROCESS 0x000005BF
#define ERROR_EVENTLOG_FILE_CORRUPT 0x000005DC
#define ERROR_EVENTLOG_CANT_START 0x000005DD
#define ERROR_LOG_FILE_FULL 0x000005DE
#define ERROR_EVENTLOG_FILE_CHANGED 0x000005DF
#define ERROR_INVALID_TASK_NAME 0x0000060E
#define ERROR_INVALID_TASK_INDEX 0x0000060F
#define ERROR_THREAD_ALREADY_IN_TASK 0x00000610
#define ERROR_INSTALL_SERVICE_FAILURE 0x00000641
#define ERROR_INSTALL_USEREXIT 0x00000642
#define ERROR_INSTALL_FAILURE 0x00000643
#define ERROR_INSTALL_SUSPEND 0x00000644
#define ERROR_UNKNOWN_PRODUCT 0x00000645
#define ERROR_UNKNOWN_FEATURE 0x00000646
#define ERROR_UNKNOWN_COMPONENT 0x00000647
#define ERROR_UNKNOWN_PROPERTY 0x00000648
#define ERROR_INVALID_HANDLE_STATE 0x00000649
#define ERROR_BAD_CONFIGURATION 0x0000064A
#define ERROR_INDEX_ABSENT 0x0000064B
#define ERROR_INSTALL_SOURCE_ABSENT 0x0000064C
#define ERROR_INSTALL_PACKAGE_VERSION 0x0000064D
#define ERROR_PRODUCT_UNINSTALLED 0x0000064E
#define ERROR_BAD_QUERY_SYNTAX 0x0000064F
#define ERROR_INVALID_FIELD 0x00000650
#define ERROR_DEVICE_REMOVED 0x00000651
#define ERROR_INSTALL_ALREADY_RUNNING 0x00000652
#define ERROR_INSTALL_PACKAGE_OPEN_FAILED 0x00000653
#define ERROR_INSTALL_PACKAGE_INVALID 0x00000654
#define ERROR_INSTALL_UI_FAILURE 0x00000655
#define ERROR_INSTALL_LOG_FAILURE 0x00000656
#define ERROR_INSTALL_LANGUAGE_UNSUPPORTED 0x00000657
#define ERROR_INSTALL_TRANSFORM_FAILURE 0x00000658
#define ERROR_INSTALL_PACKAGE_REJECTED 0x00000659
#define ERROR_FUNCTION_NOT_CALLED 0x0000065A
#define ERROR_FUNCTION_FAILED 0x0000065B
#define ERROR_INVALID_TABLE 0x0000065C
#define ERROR_DATATYPE_MISMATCH 0x0000065D
#define ERROR_UNSUPPORTED_TYPE 0x0000065E
#define ERROR_CREATE_FAILED 0x0000065F
#define ERROR_INSTALL_TEMP_UNWRITABLE 0x00000660
#define ERROR_INSTALL_PLATFORM_UNSUPPORTED 0x00000661
#define ERROR_INSTALL_NOTUSED 0x00000662
#define ERROR_PATCH_PACKAGE_OPEN_FAILED 0x00000663
#define ERROR_PATCH_PACKAGE_INVALID 0x00000664
#define ERROR_PATCH_PACKAGE_UNSUPPORTED 0x00000665
#define ERROR_PRODUCT_VERSION 0x00000666
#define ERROR_INVALID_COMMAND_LINE 0x00000667
#define ERROR_INSTALL_REMOTE_DISALLOWED 0x00000668
#define ERROR_SUCCESS_REBOOT_INITIATED 0x00000669
#define ERROR_PATCH_TARGET_NOT_FOUND 0x0000066A
#define ERROR_PATCH_PACKAGE_REJECTED 0x0000066B
#define ERROR_INSTALL_TRANSFORM_REJECTED 0x0000066C
#define ERROR_INSTALL_REMOTE_PROHIBITED 0x0000066D
#define ERROR_PATCH_REMOVAL_UNSUPPORTED 0x0000066E
#define ERROR_UNKNOWN_PATCH 0x0000066F
#define ERROR_PATCH_NO_SEQUENCE 0x00000670
#define ERROR_PATCH_REMOVAL_DISALLOWED 0x00000671
#define ERROR_INVALID_PATCH_XML 0x00000672
#define ERROR_PATCH_MANAGED_ADVERTISED_PRODUCT 0x00000673
#define ERROR_INSTALL_SERVICE_SAFEBOOT 0x00000674
#define ERROR_FAIL_FAST_EXCEPTION 0x00000675
#define ERROR_INSTALL_REJECTED 0x00000676

/* System Error Codes (1700-3999) */

#define RPC_S_INVALID_STRING_BINDING 0x000006A4
#define RPC_S_WRONG_KIND_OF_BINDING 0x000006A5
#define RPC_S_INVALID_BINDING 0x000006A6
#define RPC_S_PROTSEQ_NOT_SUPPORTED 0x000006A7
#define RPC_S_INVALID_RPC_PROTSEQ 0x000006A8
#define RPC_S_INVALID_STRING_UUID 0x000006A9
#define RPC_S_INVALID_ENDPOINT_FORMAT 0x000006AA
#define RPC_S_INVALID_NET_ADDR 0x000006AB
#define RPC_S_NO_ENDPOINT_FOUND 0x000006AC
#define RPC_S_INVALID_TIMEOUT 0x000006AD
#define RPC_S_OBJECT_NOT_FOUND 0x000006AE
#define RPC_S_ALREADY_REGISTERED 0x000006AF
#define RPC_S_TYPE_ALREADY_REGISTERED 0x000006B0
#define RPC_S_ALREADY_LISTENING 0x000006B1
#define RPC_S_NO_PROTSEQS_REGISTERED 0x000006B2
#define RPC_S_NOT_LISTENING 0x000006B3
#define RPC_S_UNKNOWN_MGR_TYPE 0x000006B4
#define RPC_S_UNKNOWN_IF 0x000006B5
#define RPC_S_NO_BINDINGS 0x000006B6
#define RPC_S_NO_PROTSEQS 0x000006B7
#define RPC_S_CANT_CREATE_ENDPOINT 0x000006B8
#define RPC_S_OUT_OF_RESOURCES 0x000006B9
#define RPC_S_SERVER_UNAVAILABLE 0x000006BA
#define RPC_S_SERVER_TOO_BUSY 0x000006BB
#define RPC_S_INVALID_NETWORK_OPTIONS 0x000006BC
#define RPC_S_NO_CALL_ACTIVE 0x000006BD
#define RPC_S_CALL_FAILED 0x000006BE
#define RPC_S_CALL_FAILED_DNE 0x000006BF
#define RPC_S_PROTOCOL_ERROR 0x000006C0
#define RPC_S_PROXY_ACCESS_DENIED 0x000006C1
#define RPC_S_UNSUPPORTED_TRANS_SYN 0x000006C2
#define RPC_S_UNSUPPORTED_TYPE 0x000006C4
#define RPC_S_INVALID_TAG 0x000006C5
#define RPC_S_INVALID_BOUND 0x000006C6
#define RPC_S_NO_ENTRY_NAME 0x000006C7
#define RPC_S_INVALID_NAME_SYNTAX 0x000006C8
#define RPC_S_UNSUPPORTED_NAME_SYNTAX 0x000006C9
#define RPC_S_UUID_NO_ADDRESS 0x000006CB
#define RPC_S_DUPLICATE_ENDPOINT 0x000006CC
#define RPC_S_UNKNOWN_AUTHN_TYPE 0x000006CD
#define RPC_S_MAX_CALLS_TOO_SMALL 0x000006CE
#define RPC_S_STRING_TOO_LONG 0x000006CF
#define RPC_S_PROTSEQ_NOT_FOUND 0x000006D0
#define RPC_S_PROCNUM_OUT_OF_RANGE 0x000006D1
#define RPC_S_BINDING_HAS_NO_AUTH 0x000006D2
#define RPC_S_UNKNOWN_AUTHN_SERVICE 0x000006D3
#define RPC_S_UNKNOWN_AUTHN_LEVEL 0x000006D4
#define RPC_S_INVALID_AUTH_IDENTITY 0x000006D5
#define RPC_S_UNKNOWN_AUTHZ_SERVICE 0x000006D6
#define EPT_S_INVALID_ENTRY 0x000006D7
#define EPT_S_CANT_PERFORM_OP 0x000006D8
#define EPT_S_NOT_REGISTERED 0x000006D9
#define RPC_S_NOTHING_TO_EXPORT 0x000006DA
#define RPC_S_INCOMPLETE_NAME 0x000006DB
#define RPC_S_INVALID_VERS_OPTION 0x000006DC
#define RPC_S_NO_MORE_MEMBERS 0x000006DD
#define RPC_S_NOT_ALL_OBJS_UNEXPORTED 0x000006DE
#define RPC_S_INTERFACE_NOT_FOUND 0x000006DF
#define RPC_S_ENTRY_ALREADY_EXISTS 0x000006E0
#define RPC_S_ENTRY_NOT_FOUND 0x000006E1
#define RPC_S_NAME_SERVICE_UNAVAILABLE 0x000006E2
#define RPC_S_INVALID_NAF_ID 0x000006E3
#define RPC_S_CANNOT_SUPPORT 0x000006E4
#define RPC_S_NO_CONTEXT_AVAILABLE 0x000006E5
#define RPC_S_INTERNAL_ERROR 0x000006E6
#define RPC_S_ZERO_DIVIDE 0x000006E7
#define RPC_S_ADDRESS_ERROR 0x000006E8
#define RPC_S_FP_DIV_ZERO 0x000006E9
#define RPC_S_FP_UNDERFLOW 0x000006EA
#define RPC_S_FP_OVERFLOW 0x000006EB
#define RPC_X_NO_MORE_ENTRIES 0x000006EC
#define RPC_X_SS_CHAR_TRANS_OPEN_FAIL 0x000006ED
#define RPC_X_SS_CHAR_TRANS_SHORT_FILE 0x000006EE
#define RPC_X_SS_IN_NULL_CONTEXT 0x000006EF
#define RPC_X_SS_CONTEXT_DAMAGED 0x000006F1
#define RPC_X_SS_HANDLES_MISMATCH 0x000006F2
#define RPC_X_SS_CANNOT_GET_CALL_HANDLE 0x000006F3
#define RPC_X_NULL_REF_POINTER 0x000006F4
#define RPC_X_ENUM_VALUE_OUT_OF_RANGE 0x000006F5
#define RPC_X_BYTE_COUNT_TOO_SMALL 0x000006F6
#define RPC_X_BAD_STUB_DATA 0x000006F7
#define ERROR_INVALID_USER_BUFFER 0x000006F8
#define ERROR_UNRECOGNIZED_MEDIA 0x000006F9
#define ERROR_NO_TRUST_LSA_SECRET 0x000006FA
#define ERROR_NO_TRUST_SAM_ACCOUNT 0x000006FB
#define ERROR_TRUSTED_DOMAIN_FAILURE 0x000006FC
#define ERROR_TRUSTED_RELATIONSHIP_FAILURE 0x000006FD
#define ERROR_TRUST_FAILURE 0x000006FE
#define RPC_S_CALL_IN_PROGRESS 0x000006FF
#define ERROR_NETLOGON_NOT_STARTED 0x00000700
#define ERROR_ACCOUNT_EXPIRED 0x00000701
#define ERROR_REDIRECTOR_HAS_OPEN_HANDLES 0x00000702
#define ERROR_PRINTER_DRIVER_ALREADY_INSTALLED 0x00000703
#define ERROR_UNKNOWN_PORT 0x00000704
#define ERROR_UNKNOWN_PRINTER_DRIVER 0x00000705
#define ERROR_UNKNOWN_PRINTPROCESSOR 0x00000706
#define ERROR_INVALID_SEPARATOR_FILE 0x00000707
#define ERROR_INVALID_PRIORITY 0x00000708
#define ERROR_INVALID_PRINTER_NAME 0x00000709
#define ERROR_PRINTER_ALREADY_EXISTS 0x0000070A
#define ERROR_INVALID_PRINTER_COMMAND 0x0000070B
#define ERROR_INVALID_DATATYPE 0x0000070C
#define ERROR_INVALID_ENVIRONMENT 0x0000070D
#define RPC_S_NO_MORE_BINDINGS 0x0000070E
#define ERROR_NOLOGON_INTERDOMAIN_TRUST_ACCOUNT 0x0000070F
#define ERROR_NOLOGON_WORKSTATION_TRUST_ACCOUNT 0x00000710
#define ERROR_NOLOGON_SERVER_TRUST_ACCOUNT 0x00000711
#define ERROR_DOMAIN_TRUST_INCONSISTENT 0x00000712
#define ERROR_SERVER_HAS_OPEN_HANDLES 0x00000713
#define ERROR_RESOURCE_DATA_NOT_FOUND 0x00000714
#define ERROR_RESOURCE_TYPE_NOT_FOUND 0x00000715
#define ERROR_RESOURCE_NAME_NOT_FOUND 0x00000716
#define ERROR_RESOURCE_LANG_NOT_FOUND 0x00000717
#define ERROR_NOT_ENOUGH_QUOTA 0x00000718
#define RPC_S_NO_INTERFACES 0x00000719
#define RPC_S_CALL_CANCELLED 0x0000071A
#define RPC_S_BINDING_INCOMPLETE 0x0000071B
#define RPC_S_COMM_FAILURE 0x0000071C
#define RPC_S_UNSUPPORTED_AUTHN_LEVEL 0x0000071D
#define RPC_S_NO_PRINC_NAME 0x0000071E
#define RPC_S_NOT_RPC_ERROR 0x0000071F
#define RPC_S_UUID_LOCAL_ONLY 0x00000720
#define RPC_S_SEC_PKG_ERROR 0x00000721
#define RPC_S_NOT_CANCELLED 0x00000722
#define RPC_X_INVALID_ES_ACTION 0x00000723
#define RPC_X_WRONG_ES_VERSION 0x00000724
#define RPC_X_WRONG_STUB_VERSION 0x00000725
#define RPC_X_INVALID_PIPE_OBJECT 0x00000726
#define RPC_X_WRONG_PIPE_ORDER 0x00000727
#define RPC_X_WRONG_PIPE_VERSION 0x00000728
#define RPC_S_COOKIE_AUTH_FAILED 0x00000729
#define RPC_S_GROUP_MEMBER_NOT_FOUND 0x0000076A
#define EPT_S_CANT_CREATE 0x0000076B
#define RPC_S_INVALID_OBJECT 0x0000076C
#define ERROR_INVALID_TIME 0x0000076D
#define ERROR_INVALID_FORM_NAME 0x0000076E
#define ERROR_INVALID_FORM_SIZE 0x0000076F
#define ERROR_ALREADY_WAITING 0x00000770
#define ERROR_PRINTER_DELETED 0x00000771
#define ERROR_INVALID_PRINTER_STATE 0x00000772
#define ERROR_PASSWORD_MUST_CHANGE 0x00000773
#define ERROR_DOMAIN_CONTROLLER_NOT_FOUND 0x00000774
#define ERROR_ACCOUNT_LOCKED_OUT 0x00000775
#define OR_INVALID_OXID 0x00000776
#define OR_INVALID_OID 0x00000777
#define OR_INVALID_SET 0x00000778
#define RPC_S_SEND_INCOMPLETE 0x00000779
#define RPC_S_INVALID_ASYNC_HANDLE 0x0000077A
#define RPC_S_INVALID_ASYNC_CALL 0x0000077B
#define RPC_X_PIPE_CLOSED 0x0000077C
#define RPC_X_PIPE_DISCIPLINE_ERROR 0x0000077D
#define RPC_X_PIPE_EMPTY 0x0000077E
#define ERROR_NO_SITENAME 0x0000077F
#define ERROR_CANT_ACCESS_FILE 0x00000780
#define ERROR_CANT_RESOLVE_FILENAME 0x00000781
#define RPC_S_ENTRY_TYPE_MISMATCH 0x00000782
#define RPC_S_NOT_ALL_OBJS_EXPORTED 0x00000783
#define RPC_S_INTERFACE_NOT_EXPORTED 0x00000784
#define RPC_S_PROFILE_NOT_ADDED 0x00000785
#define RPC_S_PRF_ELT_NOT_ADDED 0x00000786
#define RPC_S_PRF_ELT_NOT_REMOVED 0x00000787
#define RPC_S_GRP_ELT_NOT_ADDED 0x00000788
#define RPC_S_GRP_ELT_NOT_REMOVED 0x00000789
#define ERROR_KM_DRIVER_BLOCKED 0x0000078A
#define ERROR_CONTEXT_EXPIRED 0x0000078B
#define ERROR_PER_USER_TRUST_QUOTA_EXCEEDED 0x0000078C
#define ERROR_ALL_USER_TRUST_QUOTA_EXCEEDED 0x0000078D
#define ERROR_USER_DELETE_TRUST_QUOTA_EXCEEDED 0x0000078E
#define ERROR_AUTHENTICATION_FIREWALL_FAILED 0x0000078F
#define ERROR_REMOTE_PRINT_CONNECTIONS_BLOCKED 0x00000790
#define ERROR_NTLM_BLOCKED 0x00000791
#define ERROR_PASSWORD_CHANGE_REQUIRED 0x00000792
#define ERROR_INVALID_PIXEL_FORMAT 0x000007D0
#define ERROR_BAD_DRIVER 0x000007D1
#define ERROR_INVALID_WINDOW_STYLE 0x000007D2
#define ERROR_METAFILE_NOT_SUPPORTED 0x000007D3
#define ERROR_TRANSFORM_NOT_SUPPORTED 0x000007D4
#define ERROR_CLIPPING_NOT_SUPPORTED 0x000007D5
#define ERROR_INVALID_CMM 0x000007DA
#define ERROR_INVALID_PROFILE 0x000007DB
#define ERROR_TAG_NOT_FOUND 0x000007DC
#define ERROR_TAG_NOT_PRESENT 0x000007DD
#define ERROR_DUPLICATE_TAG 0x000007DE
#define ERROR_PROFILE_NOT_ASSOCIATED_WITH_DEVICE 0x000007DF
#define ERROR_PROFILE_NOT_FOUND 0x000007E0
#define ERROR_INVALID_COLORSPACE 0x000007E1
#define ERROR_ICM_NOT_ENABLED 0x000007E2
#define ERROR_DELETING_ICM_XFORM 0x000007E3
#define ERROR_INVALID_TRANSFORM 0x000007E4
#define ERROR_COLORSPACE_MISMATCH 0x000007E5
#define ERROR_INVALID_COLORINDEX 0x000007E6
#define ERROR_PROFILE_DOES_NOT_MATCH_DEVICE 0x000007E7
#define ERROR_CONNECTED_OTHER_PASSWORD 0x0000083C
#define ERROR_CONNECTED_OTHER_PASSWORD_DEFAULT 0x0000083D
#define ERROR_BAD_USERNAME 0x0000089A
#define ERROR_NOT_CONNECTED 0x000008CA
#define ERROR_OPEN_FILES 0x00000961
#define ERROR_ACTIVE_CONNECTIONS 0x00000962
#define ERROR_DEVICE_IN_USE 0x00000964
#define ERROR_UNKNOWN_PRINT_MONITOR 0x00000BB8
#define ERROR_PRINTER_DRIVER_IN_USE 0x00000BB9
#define ERROR_SPOOL_FILE_NOT_FOUND 0x00000BBA
#define ERROR_SPL_NO_STARTDOC 0x00000BBB
#define ERROR_SPL_NO_ADDJOB 0x00000BBC
#define ERROR_PRINT_PROCESSOR_ALREADY_INSTALLED 0x00000BBD
#define ERROR_PRINT_MONITOR_ALREADY_INSTALLED 0x00000BBE
#define ERROR_INVALID_PRINT_MONITOR 0x00000BBF
#define ERROR_PRINT_MONITOR_IN_USE 0x00000BC0
#define ERROR_PRINTER_HAS_JOBS_QUEUED 0x00000BC1
#define ERROR_SUCCESS_REBOOT_REQUIRED 0x00000BC2
#define ERROR_SUCCESS_RESTART_REQUIRED 0x00000BC3
#define ERROR_PRINTER_NOT_FOUND 0x00000BC4
#define ERROR_PRINTER_DRIVER_WARNED 0x00000BC5
#define ERROR_PRINTER_DRIVER_BLOCKED 0x00000BC6
#define ERROR_PRINTER_DRIVER_PACKAGE_IN_USE 0x00000BC7
#define ERROR_CORE_DRIVER_PACKAGE_NOT_FOUND 0x00000BC8
#define ERROR_FAIL_REBOOT_REQUIRED 0x00000BC9
#define ERROR_FAIL_REBOOT_INITIATED 0x00000BCA
#define ERROR_PRINTER_DRIVER_DOWNLOAD_NEEDED 0x00000BCB
#define ERROR_PRINT_JOB_RESTART_REQUIRED 0x00000BCC
#define ERROR_INVALID_PRINTER_DRIVER_MANIFEST 0x00000BCD
#define ERROR_PRINTER_NOT_SHAREABLE 0x00000BCE
#define ERROR_REQUEST_PAUSED 0x00000BEA
#define ERROR_IO_REISSUE_AS_CACHED 0x00000F6E

/* System Error Codes (4000-5999) */

#define ERROR_WINS_INTERNAL 0x00000FA0
#define ERROR_CAN_NOT_DEL_LOCAL_WINS 0x00000FA1
#define ERROR_STATIC_INIT 0x00000FA2
#define ERROR_INC_BACKUP 0x00000FA3
#define ERROR_FULL_BACKUP 0x00000FA4
#define ERROR_REC_NON_EXISTENT 0x00000FA5
#define ERROR_RPL_NOT_ALLOWED 0x00000FA6
#define PEERDIST_ERROR_CONTENTINFO_VERSION_UNSUPPORTED 0x00000FD2
#define PEERDIST_ERROR_CANNOT_PARSE_CONTENTINFO 0x00000FD3
#define PEERDIST_ERROR_MISSING_DATA 0x00000FD4
#define PEERDIST_ERROR_NO_MORE 0x00000FD5
#define PEERDIST_ERROR_NOT_INITIALIZED 0x00000FD6
#define PEERDIST_ERROR_ALREADY_INITIALIZED 0x00000FD7
#define PEERDIST_ERROR_SHUTDOWN_IN_PROGRESS 0x00000FD8
#define PEERDIST_ERROR_INVALIDATED 0x00000FD9
#define PEERDIST_ERROR_ALREADY_EXISTS 0x00000FDA
#define PEERDIST_ERROR_OPERATION_NOTFOUND 0x00000FDB
#define PEERDIST_ERROR_ALREADY_COMPLETED 0x00000FDC
#define PEERDIST_ERROR_OUT_OF_BOUNDS 0x00000FDD
#define PEERDIST_ERROR_VERSION_UNSUPPORTED 0x00000FDE
#define PEERDIST_ERROR_INVALID_CONFIGURATION 0x00000FDF
#define PEERDIST_ERROR_NOT_LICENSED 0x00000FE0
#define PEERDIST_ERROR_SERVICE_UNAVAILABLE 0x00000FE1
#define PEERDIST_ERROR_TRUST_FAILURE 0x00000FE2
#define ERROR_DHCP_ADDRESS_CONFLICT 0x00001004
#define ERROR_WMI_GUID_NOT_FOUND 0x00001068
#define ERROR_WMI_INSTANCE_NOT_FOUND 0x00001069
#define ERROR_WMI_ITEMID_NOT_FOUND 0x0000106A
#define ERROR_WMI_TRY_AGAIN 0x0000106B
#define ERROR_WMI_DP_NOT_FOUND 0x0000106C
#define ERROR_WMI_UNRESOLVED_INSTANCE_REF 0x0000106D
#define ERROR_WMI_ALREADY_ENABLED 0x0000106E
#define ERROR_WMI_GUID_DISCONNECTED 0x0000106F
#define ERROR_WMI_SERVER_UNAVAILABLE 0x00001070
#define ERROR_WMI_DP_FAILED 0x00001071
#define ERROR_WMI_INVALID_MOF 0x00001072
#define ERROR_WMI_INVALID_REGINFO 0x00001073
#define ERROR_WMI_ALREADY_DISABLED 0x00001074
#define ERROR_WMI_READ_ONLY 0x00001075
#define ERROR_WMI_SET_FAILURE 0x00001076
#define ERROR_NOT_APPCONTAINER 0x0000109A
#define ERROR_APPCONTAINER_REQUIRED 0x0000109B
#define ERROR_NOT_SUPPORTED_IN_APPCONTAINER 0x0000109C
#define ERROR_INVALID_PACKAGE_SID_LENGTH 0x0000109D
#define ERROR_INVALID_MEDIA 0x000010CC
#define ERROR_INVALID_LIBRARY 0x000010CD
#define ERROR_INVALID_MEDIA_POOL 0x000010CE
#define ERROR_DRIVE_MEDIA_MISMATCH 0x000010CF
#define ERROR_MEDIA_OFFLINE 0x000010D0
#define ERROR_LIBRARY_OFFLINE 0x000010D1
#define ERROR_EMPTY 0x000010D2
#define ERROR_NOT_EMPTY 0x000010D3
#define ERROR_MEDIA_UNAVAILABLE 0x000010D4
#define ERROR_RESOURCE_DISABLED 0x000010D5
#define ERROR_INVALID_CLEANER 0x000010D6
#define ERROR_UNABLE_TO_CLEAN 0x000010D7
#define ERROR_OBJECT_NOT_FOUND 0x000010D8
#define ERROR_DATABASE_FAILURE 0x000010D9
#define ERROR_DATABASE_FULL 0x000010DA
#define ERROR_MEDIA_INCOMPATIBLE 0x000010DB
#define ERROR_RESOURCE_NOT_PRESENT 0x000010DC
#define ERROR_INVALID_OPERATION 0x000010DD
#define ERROR_MEDIA_NOT_AVAILABLE 0x000010DE
#define ERROR_DEVICE_NOT_AVAILABLE 0x000010DF
#define ERROR_REQUEST_REFUSED 0x000010E0
#define ERROR_INVALID_DRIVE_OBJECT 0x000010E1
#define ERROR_LIBRARY_FULL 0x000010E2
#define ERROR_MEDIUM_NOT_ACCESSIBLE 0x000010E3
#define ERROR_UNABLE_TO_LOAD_MEDIUM 0x000010E4
#define ERROR_UNABLE_TO_INVENTORY_DRIVE 0x000010E5
#define ERROR_UNABLE_TO_INVENTORY_SLOT 0x000010E6
#define ERROR_UNABLE_TO_INVENTORY_TRANSPORT 0x000010E7
#define ERROR_TRANSPORT_FULL 0x000010E8
#define ERROR_CONTROLLING_IEPORT 0x000010E9
#define ERROR_UNABLE_TO_EJECT_MOUNTED_MEDIA 0x000010EA
#define ERROR_CLEANER_SLOT_SET 0x000010EB
#define ERROR_CLEANER_SLOT_NOT_SET 0x000010EC
#define ERROR_CLEANER_CARTRIDGE_SPENT 0x000010ED
#define ERROR_UNEXPECTED_OMID 0x000010EE
#define ERROR_CANT_DELETE_LAST_ITEM 0x000010EF
#define ERROR_MESSAGE_EXCEEDS_MAX_SIZE 0x000010F0
#define ERROR_VOLUME_CONTAINS_SYS_FILES 0x000010F1
#define ERROR_INDIGENOUS_TYPE 0x000010F2
#define ERROR_NO_SUPPORTING_DRIVES 0x000010F3
#define ERROR_CLEANER_CARTRIDGE_INSTALLED 0x000010F4
#define ERROR_IEPORT_FULL 0x000010F5
#define ERROR_FILE_OFFLINE 0x000010FE
#define ERROR_REMOTE_STORAGE_NOT_ACTIVE 0x000010FF
#define ERROR_REMOTE_STORAGE_MEDIA_ERROR 0x00001100
#define ERROR_NOT_A_REPARSE_POINT 0x00001126
#define ERROR_REPARSE_ATTRIBUTE_CONFLICT 0x00001127
#define ERROR_INVALID_REPARSE_DATA 0x00001128
#define ERROR_REPARSE_TAG_INVALID 0x00001129
#define ERROR_REPARSE_TAG_MISMATCH 0x0000112A
#define ERROR_APP_DATA_NOT_FOUND 0x00001130
#define ERROR_APP_DATA_EXPIRED 0x00001131
#define ERROR_APP_DATA_CORRUPT 0x00001132
#define ERROR_APP_DATA_LIMIT_EXCEEDED 0x00001133
#define ERROR_APP_DATA_REBOOT_REQUIRED 0x00001134
#define ERROR_SECUREBOOT_ROLLBACK_DETECTED 0x00001144
#define ERROR_SECUREBOOT_POLICY_VIOLATION 0x00001145
#define ERROR_SECUREBOOT_INVALID_POLICY 0x00001146
#define ERROR_SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND 0x00001147
#define ERROR_SECUREBOOT_POLICY_NOT_SIGNED 0x00001148
#define ERROR_SECUREBOOT_NOT_ENABLED 0x00001149
#define ERROR_SECUREBOOT_FILE_REPLACED 0x0000114A
#define ERROR_OFFLOAD_READ_FLT_NOT_SUPPORTED 0x00001158
#define ERROR_OFFLOAD_WRITE_FLT_NOT_SUPPORTED 0x00001159
#define ERROR_OFFLOAD_READ_FILE_NOT_SUPPORTED 0x0000115A
#define ERROR_OFFLOAD_WRITE_FILE_NOT_SUPPORTED 0x0000115B
#define ERROR_VOLUME_NOT_SIS_ENABLED 0x00001194
#define ERROR_DEPENDENT_RESOURCE_EXISTS 0x00001389
#define ERROR_DEPENDENCY_NOT_FOUND 0x0000138A
#define ERROR_DEPENDENCY_ALREADY_EXISTS 0x0000138B
#define ERROR_RESOURCE_NOT_ONLINE 0x0000138C
#define ERROR_HOST_NODE_NOT_AVAILABLE 0x0000138D
#define ERROR_RESOURCE_NOT_AVAILABLE 0x0000138E
#define ERROR_RESOURCE_NOT_FOUND 0x0000138F
#define ERROR_SHUTDOWN_CLUSTER 0x00001390
#define ERROR_CANT_EVICT_ACTIVE_NODE 0x00001391
#define ERROR_OBJECT_ALREADY_EXISTS 0x00001392
#define ERROR_OBJECT_IN_LIST 0x00001393
#define ERROR_GROUP_NOT_AVAILABLE 0x00001394
#define ERROR_GROUP_NOT_FOUND 0x00001395
#define ERROR_GROUP_NOT_ONLINE 0x00001396
#define ERROR_HOST_NODE_NOT_RESOURCE_OWNER 0x00001397
#define ERROR_HOST_NODE_NOT_GROUP_OWNER 0x00001398
#define ERROR_RESMON_CREATE_FAILED 0x00001399
#define ERROR_RESMON_ONLINE_FAILED 0x0000139A
#define ERROR_RESOURCE_ONLINE 0x0000139B
#define ERROR_QUORUM_RESOURCE 0x0000139C
#define ERROR_NOT_QUORUM_CAPABLE 0x0000139D
#define ERROR_CLUSTER_SHUTTING_DOWN 0x0000139E
#define ERROR_INVALID_STATE 0x0000139F
#define ERROR_RESOURCE_PROPERTIES_STORED 0x000013A0
#define ERROR_NOT_QUORUM_CLASS 0x000013A1
#define ERROR_CORE_RESOURCE 0x000013A2
#define ERROR_QUORUM_RESOURCE_ONLINE_FAILED 0x000013A3
#define ERROR_QUORUMLOG_OPEN_FAILED 0x000013A4
#define ERROR_CLUSTERLOG_CORRUPT 0x000013A5
#define ERROR_CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE 0x000013A6
#define ERROR_CLUSTERLOG_EXCEEDS_MAXSIZE 0x000013A7
#define ERROR_CLUSTERLOG_CHKPOINT_NOT_FOUND 0x000013A8
#define ERROR_CLUSTERLOG_NOT_ENOUGH_SPACE 0x000013A9
#define ERROR_QUORUM_OWNER_ALIVE 0x000013AA
#define ERROR_NETWORK_NOT_AVAILABLE 0x000013AB
#define ERROR_NODE_NOT_AVAILABLE 0x000013AC
#define ERROR_ALL_NODES_NOT_AVAILABLE 0x000013AD
#define ERROR_RESOURCE_FAILED 0x000013AE
#define ERROR_CLUSTER_INVALID_NODE 0x000013AF
#define ERROR_CLUSTER_NODE_EXISTS 0x000013B0
#define ERROR_CLUSTER_JOIN_IN_PROGRESS 0x000013B1
#define ERROR_CLUSTER_NODE_NOT_FOUND 0x000013B2
#define ERROR_CLUSTER_LOCAL_NODE_NOT_FOUND 0x000013B3
#define ERROR_CLUSTER_NETWORK_EXISTS 0x000013B4
#define ERROR_CLUSTER_NETWORK_NOT_FOUND 0x000013B5
#define ERROR_CLUSTER_NETINTERFACE_EXISTS 0x000013B6
#define ERROR_CLUSTER_NETINTERFACE_NOT_FOUND 0x000013B7
#define ERROR_CLUSTER_INVALID_REQUEST 0x000013B8
#define ERROR_CLUSTER_INVALID_NETWORK_PROVIDER 0x000013B9
#define ERROR_CLUSTER_NODE_DOWN 0x000013BA
#define ERROR_CLUSTER_NODE_UNREACHABLE 0x000013BB
#define ERROR_CLUSTER_NODE_NOT_MEMBER 0x000013BC
#define ERROR_CLUSTER_JOIN_NOT_IN_PROGRESS 0x000013BD
#define ERROR_CLUSTER_INVALID_NETWORK 0x000013BE
#define ERROR_CLUSTER_NODE_UP 0x000013C0
#define ERROR_CLUSTER_IPADDR_IN_USE 0x000013C1
#define ERROR_CLUSTER_NODE_NOT_PAUSED 0x000013C2
#define ERROR_CLUSTER_NO_SECURITY_CONTEXT 0x000013C3
#define ERROR_CLUSTER_NETWORK_NOT_INTERNAL 0x000013C4
#define ERROR_CLUSTER_NODE_ALREADY_UP 0x000013C5
#define ERROR_CLUSTER_NODE_ALREADY_DOWN 0x000013C6
#define ERROR_CLUSTER_NETWORK_ALREADY_ONLINE 0x000013C7
#define ERROR_CLUSTER_NETWORK_ALREADY_OFFLINE 0x000013C8
#define ERROR_CLUSTER_NODE_ALREADY_MEMBER 0x000013C9
#define ERROR_CLUSTER_LAST_INTERNAL_NETWORK 0x000013CA
#define ERROR_CLUSTER_NETWORK_HAS_DEPENDENTS 0x000013CB
#define ERROR_INVALID_OPERATION_ON_QUORUM 0x000013CC
#define ERROR_DEPENDENCY_NOT_ALLOWED 0x000013CD
#define ERROR_CLUSTER_NODE_PAUSED 0x000013CE
#define ERROR_NODE_CANT_HOST_RESOURCE 0x000013CF
#define ERROR_CLUSTER_NODE_NOT_READY 0x000013D0
#define ERROR_CLUSTER_NODE_SHUTTING_DOWN 0x000013D1
#define ERROR_CLUSTER_JOIN_ABORTED 0x000013D2
#define ERROR_CLUSTER_INCOMPATIBLE_VERSIONS 0x000013D3
#define ERROR_CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED 0x000013D4
#define ERROR_CLUSTER_SYSTEM_CONFIG_CHANGED 0x000013D5
#define ERROR_CLUSTER_RESOURCE_TYPE_NOT_FOUND 0x000013D6
#define ERROR_CLUSTER_RESTYPE_NOT_SUPPORTED 0x000013D7
#define ERROR_CLUSTER_RESNAME_NOT_FOUND 0x000013D8
#define ERROR_CLUSTER_NO_RPC_PACKAGES_REGISTERED 0x000013D9
#define ERROR_CLUSTER_OWNER_NOT_IN_PREFLIST 0x000013DA
#define ERROR_CLUSTER_DATABASE_SEQMISMATCH 0x000013DB
#define ERROR_RESMON_INVALID_STATE 0x000013DC
#define ERROR_CLUSTER_GUM_NOT_LOCKER 0x000013DD
#define ERROR_QUORUM_DISK_NOT_FOUND 0x000013DE
#define ERROR_DATABASE_BACKUP_CORRUPT 0x000013DF
#define ERROR_CLUSTER_NODE_ALREADY_HAS_DFS_ROOT 0x000013E0
#define ERROR_RESOURCE_PROPERTY_UNCHANGEABLE 0x000013E1
#define ERROR_CLUSTER_MEMBERSHIP_INVALID_STATE 0x00001702
#define ERROR_CLUSTER_QUORUMLOG_NOT_FOUND 0x00001703
#define ERROR_CLUSTER_MEMBERSHIP_HALT 0x00001704
#define ERROR_CLUSTER_INSTANCE_ID_MISMATCH 0x00001705
#define ERROR_CLUSTER_NETWORK_NOT_FOUND_FOR_IP 0x00001706
#define ERROR_CLUSTER_PROPERTY_DATA_TYPE_MISMATCH 0x00001707
#define ERROR_CLUSTER_EVICT_WITHOUT_CLEANUP 0x00001708
#define ERROR_CLUSTER_PARAMETER_MISMATCH 0x00001709
#define ERROR_NODE_CANNOT_BE_CLUSTERED 0x0000170A
#define ERROR_CLUSTER_WRONG_OS_VERSION 0x0000170B
#define ERROR_CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME 0x0000170C
#define ERROR_CLUSCFG_ALREADY_COMMITTED 0x0000170D
#define ERROR_CLUSCFG_ROLLBACK_FAILED 0x0000170E
#define ERROR_CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT 0x0000170F
#define ERROR_CLUSTER_OLD_VERSION 0x00001710
#define ERROR_CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME 0x00001711
#define ERROR_CLUSTER_NO_NET_ADAPTERS 0x00001712
#define ERROR_CLUSTER_POISONED 0x00001713
#define ERROR_CLUSTER_GROUP_MOVING 0x00001714
#define ERROR_CLUSTER_RESOURCE_TYPE_BUSY 0x00001715
#define ERROR_RESOURCE_CALL_TIMED_OUT 0x00001716
#define ERROR_INVALID_CLUSTER_IPV6_ADDRESS 0x00001717
#define ERROR_CLUSTER_INTERNAL_INVALID_FUNCTION 0x00001718
#define ERROR_CLUSTER_PARAMETER_OUT_OF_BOUNDS 0x00001719
#define ERROR_CLUSTER_PARTIAL_SEND 0x0000171A
#define ERROR_CLUSTER_REGISTRY_INVALID_FUNCTION 0x0000171B
#define ERROR_CLUSTER_INVALID_STRING_TERMINATION 0x0000171C
#define ERROR_CLUSTER_INVALID_STRING_FORMAT 0x0000171D
#define ERROR_CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS 0x0000171E
#define ERROR_CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS 0x0000171F
#define ERROR_CLUSTER_NULL_DATA 0x00001720
#define ERROR_CLUSTER_PARTIAL_READ 0x00001721
#define ERROR_CLUSTER_PARTIAL_WRITE 0x00001722
#define ERROR_CLUSTER_CANT_DESERIALIZE_DATA 0x00001723
#define ERROR_DEPENDENT_RESOURCE_PROPERTY_CONFLICT 0x00001724
#define ERROR_CLUSTER_NO_QUORUM 0x00001725
#define ERROR_CLUSTER_INVALID_IPV6_NETWORK 0x00001726
#define ERROR_CLUSTER_INVALID_IPV6_TUNNEL_NETWORK 0x00001727
#define ERROR_QUORUM_NOT_ALLOWED_IN_THIS_GROUP 0x00001728
#define ERROR_DEPENDENCY_TREE_TOO_COMPLEX 0x00001729
#define ERROR_EXCEPTION_IN_RESOURCE_CALL 0x0000172A
#define ERROR_CLUSTER_RHS_FAILED_INITIALIZATION 0x0000172B
#define ERROR_CLUSTER_NOT_INSTALLED 0x0000172C
#define ERROR_CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE 0x0000172D
#define ERROR_CLUSTER_MAX_NODES_IN_CLUSTER 0x0000172E
#define ERROR_CLUSTER_TOO_MANY_NODES 0x0000172F
#define ERROR_CLUSTER_OBJECT_ALREADY_USED 0x00001730
#define ERROR_NONCORE_GROUPS_FOUND 0x00001731
#define ERROR_FILE_SHARE_RESOURCE_CONFLICT 0x00001732
#define ERROR_CLUSTER_EVICT_INVALID_REQUEST 0x00001733
#define ERROR_CLUSTER_SINGLETON_RESOURCE 0x00001734
#define ERROR_CLUSTER_GROUP_SINGLETON_RESOURCE 0x00001735
#define ERROR_CLUSTER_RESOURCE_PROVIDER_FAILED 0x00001736
#define ERROR_CLUSTER_RESOURCE_CONFIGURATION_ERROR 0x00001737
#define ERROR_CLUSTER_GROUP_BUSY 0x00001738
#define ERROR_CLUSTER_NOT_SHARED_VOLUME 0x00001739
#define ERROR_CLUSTER_INVALID_SECURITY_DESCRIPTOR 0x0000173A
#define ERROR_CLUSTER_SHARED_VOLUMES_IN_USE 0x0000173B
#define ERROR_CLUSTER_USE_SHARED_VOLUMES_API 0x0000173C
#define ERROR_CLUSTER_BACKUP_IN_PROGRESS 0x0000173D
#define ERROR_NON_CSV_PATH 0x0000173E
#define ERROR_CSV_VOLUME_NOT_LOCAL 0x0000173F
#define ERROR_CLUSTER_WATCHDOG_TERMINATING 0x00001740
#define ERROR_CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES 0x00001741
#define ERROR_CLUSTER_INVALID_NODE_WEIGHT 0x00001742
#define ERROR_CLUSTER_RESOURCE_VETOED_CALL 0x00001743
#define ERROR_RESMON_SYSTEM_RESOURCES_LACKING 0x00001744
#define ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION 0x00001745
#define ERROR_CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE 0x00001746
#define ERROR_CLUSTER_GROUP_QUEUED 0x00001747
#define ERROR_CLUSTER_RESOURCE_LOCKED_STATUS 0x00001748
#define ERROR_CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED 0x00001749
#define ERROR_CLUSTER_NODE_DRAIN_IN_PROGRESS 0x0000174A
#define ERROR_CLUSTER_DISK_NOT_CONNECTED 0x0000174B
#define ERROR_DISK_NOT_CSV_CAPABLE 0x0000174C
#define ERROR_RESOURCE_NOT_IN_AVAILABLE_STORAGE 0x0000174D
#define ERROR_CLUSTER_SHARED_VOLUME_REDIRECTED 0x0000174E
#define ERROR_CLUSTER_SHARED_VOLUME_NOT_REDIRECTED 0x0000174F
#define ERROR_CLUSTER_CANNOT_RETURN_PROPERTIES 0x00001750
#define ERROR_CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES 0x00001751
#define ERROR_CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE 0x00001752
#define ERROR_CLUSTER_AFFINITY_CONFLICT 0x00001753
#define ERROR_CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE 0x00001754

/* System Error Codes (6000-8199) */

#define ERROR_ENCRYPTION_FAILED 0x00001770
#define ERROR_DECRYPTION_FAILED 0x00001771
#define ERROR_FILE_ENCRYPTED 0x00001772
#define ERROR_NO_RECOVERY_POLICY 0x00001773
#define ERROR_NO_EFS 0x00001774
#define ERROR_WRONG_EFS 0x00001775
#define ERROR_NO_USER_KEYS 0x00001776
#define ERROR_FILE_NOT_ENCRYPTED 0x00001777
#define ERROR_NOT_EXPORT_FORMAT 0x00001778
#define ERROR_FILE_READ_ONLY 0x00001779
#define ERROR_DIR_EFS_DISALLOWED 0x0000177A
#define ERROR_EFS_SERVER_NOT_TRUSTED 0x0000177B
#define ERROR_BAD_RECOVERY_POLICY 0x0000177C
#define ERROR_EFS_ALG_BLOB_TOO_BIG 0x0000177D
#define ERROR_VOLUME_NOT_SUPPORT_EFS 0x0000177E
#define ERROR_EFS_DISABLED 0x0000177F
#define ERROR_EFS_VERSION_NOT_SUPPORT 0x00001780
#define ERROR_CS_ENCRYPTION_INVALID_SERVER_RESPONSE 0x00001781
#define ERROR_CS_ENCRYPTION_UNSUPPORTED_SERVER 0x00001782
#define ERROR_CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE 0x00001783
#define ERROR_CS_ENCRYPTION_NEW_ENCRYPTED_FILE 0x00001784
#define ERROR_CS_ENCRYPTION_FILE_NOT_CSE 0x00001785
#define ERROR_ENCRYPTION_POLICY_DENIES_OPERATION 0x00001786
#define ERROR_NO_BROWSER_SERVERS_FOUND 0x000017E6
#define SCHED_E_SERVICE_NOT_LOCALSYSTEM 0x00001838
#define ERROR_LOG_SECTOR_INVALID 0x000019C8
#define ERROR_LOG_SECTOR_PARITY_INVALID 0x000019C9
#define ERROR_LOG_SECTOR_REMAPPED 0x000019CA
#define ERROR_LOG_BLOCK_INCOMPLETE 0x000019CB
#define ERROR_LOG_INVALID_RANGE 0x000019CC
#define ERROR_LOG_BLOCKS_EXHAUSTED 0x000019CD
#define ERROR_LOG_READ_CONTEXT_INVALID 0x000019CE
#define ERROR_LOG_RESTART_INVALID 0x000019CF
#define ERROR_LOG_BLOCK_VERSION 0x000019D0
#define ERROR_LOG_BLOCK_INVALID 0x000019D1
#define ERROR_LOG_READ_MODE_INVALID 0x000019D2
#define ERROR_LOG_NO_RESTART 0x000019D3
#define ERROR_LOG_METADATA_CORRUPT 0x000019D4
#define ERROR_LOG_METADATA_INVALID 0x000019D5
#define ERROR_LOG_METADATA_INCONSISTENT 0x000019D6
#define ERROR_LOG_RESERVATION_INVALID 0x000019D7
#define ERROR_LOG_CANT_DELETE 0x000019D8
#define ERROR_LOG_CONTAINER_LIMIT_EXCEEDED 0x000019D9
#define ERROR_LOG_START_OF_LOG 0x000019DA
#define ERROR_LOG_POLICY_ALREADY_INSTALLED 0x000019DB
#define ERROR_LOG_POLICY_NOT_INSTALLED 0x000019DC
#define ERROR_LOG_POLICY_INVALID 0x000019DD
#define ERROR_LOG_POLICY_CONFLICT 0x000019DE
#define ERROR_LOG_PINNED_ARCHIVE_TAIL 0x000019DF
#define ERROR_LOG_RECORD_NONEXISTENT 0x000019E0
#define ERROR_LOG_RECORDS_RESERVED_INVALID 0x000019E1
#define ERROR_LOG_SPACE_RESERVED_INVALID 0x000019E2
#define ERROR_LOG_TAIL_INVALID 0x000019E3
#define ERROR_LOG_FULL 0x000019E4
#define ERROR_COULD_NOT_RESIZE_LOG 0x000019E5
#define ERROR_LOG_MULTIPLEXED 0x000019E6
#define ERROR_LOG_DEDICATED 0x000019E7
#define ERROR_LOG_ARCHIVE_NOT_IN_PROGRESS 0x000019E8
#define ERROR_LOG_ARCHIVE_IN_PROGRESS 0x000019E9
#define ERROR_LOG_EPHEMERAL 0x000019EA
#define ERROR_LOG_NOT_ENOUGH_CONTAINERS 0x000019EB
#define ERROR_LOG_CLIENT_ALREADY_REGISTERED 0x000019EC
#define ERROR_LOG_CLIENT_NOT_REGISTERED 0x000019ED
#define ERROR_LOG_FULL_HANDLER_IN_PROGRESS 0x000019EE
#define ERROR_LOG_CONTAINER_READ_FAILED 0x000019EF
#define ERROR_LOG_CONTAINER_WRITE_FAILED 0x000019F0
#define ERROR_LOG_CONTAINER_OPEN_FAILED 0x000019F1
#define ERROR_LOG_CONTAINER_STATE_INVALID 0x000019F2
#define ERROR_LOG_STATE_INVALID 0x000019F3
#define ERROR_LOG_PINNED 0x000019F4
#define ERROR_LOG_METADATA_FLUSH_FAILED 0x000019F5
#define ERROR_LOG_INCONSISTENT_SECURITY 0x000019F6
#define ERROR_LOG_APPENDED_FLUSH_FAILED 0x000019F7
#define ERROR_LOG_PINNED_RESERVATION 0x000019F8
#define ERROR_INVALID_TRANSACTION 0x00001A2C
#define ERROR_TRANSACTION_NOT_ACTIVE 0x00001A2D
#define ERROR_TRANSACTION_REQUEST_NOT_VALID 0x00001A2E
#define ERROR_TRANSACTION_NOT_REQUESTED 0x00001A2F
#define ERROR_TRANSACTION_ALREADY_ABORTED 0x00001A30
#define ERROR_TRANSACTION_ALREADY_COMMITTED 0x00001A31
#define ERROR_TM_INITIALIZATION_FAILED 0x00001A32
#define ERROR_RESOURCEMANAGER_READ_ONLY 0x00001A33
#define ERROR_TRANSACTION_NOT_JOINED 0x00001A34
#define ERROR_TRANSACTION_SUPERIOR_EXISTS 0x00001A35
#define ERROR_CRM_PROTOCOL_ALREADY_EXISTS 0x00001A36
#define ERROR_TRANSACTION_PROPAGATION_FAILED 0x00001A37
#define ERROR_CRM_PROTOCOL_NOT_FOUND 0x00001A38
#define ERROR_TRANSACTION_INVALID_MARSHALL_BUFFER 0x00001A39
#define ERROR_CURRENT_TRANSACTION_NOT_VALID 0x00001A3A
#define ERROR_TRANSACTION_NOT_FOUND 0x00001A3B
#define ERROR_RESOURCEMANAGER_NOT_FOUND 0x00001A3C
#define ERROR_ENLISTMENT_NOT_FOUND 0x00001A3D
#define ERROR_TRANSACTIONMANAGER_NOT_FOUND 0x00001A3E
#define ERROR_TRANSACTIONMANAGER_NOT_ONLINE 0x00001A3F
#define ERROR_TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION 0x00001A40
#define ERROR_TRANSACTION_NOT_ROOT 0x00001A41
#define ERROR_TRANSACTION_OBJECT_EXPIRED 0x00001A42
#define ERROR_TRANSACTION_RESPONSE_NOT_ENLISTED 0x00001A43
#define ERROR_TRANSACTION_RECORD_TOO_LONG 0x00001A44
#define ERROR_IMPLICIT_TRANSACTION_NOT_SUPPORTED 0x00001A45
#define ERROR_TRANSACTION_INTEGRITY_VIOLATED 0x00001A46
#define ERROR_TRANSACTIONMANAGER_IDENTITY_MISMATCH 0x00001A47
#define ERROR_RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT 0x00001A48
#define ERROR_TRANSACTION_MUST_WRITETHROUGH 0x00001A49
#define ERROR_TRANSACTION_NO_SUPERIOR 0x00001A4A
#define ERROR_HEURISTIC_DAMAGE_POSSIBLE 0x00001A4B
#define ERROR_TRANSACTIONAL_CONFLICT 0x00001A90
#define ERROR_RM_NOT_ACTIVE 0x00001A91
#define ERROR_RM_METADATA_CORRUPT 0x00001A92
#define ERROR_DIRECTORY_NOT_RM 0x00001A93
#define ERROR_TRANSACTIONS_UNSUPPORTED_REMOTE 0x00001A95
#define ERROR_LOG_RESIZE_INVALID_SIZE 0x00001A96
#define ERROR_OBJECT_NO_LONGER_EXISTS 0x00001A97
#define ERROR_STREAM_MINIVERSION_NOT_FOUND 0x00001A98
#define ERROR_STREAM_MINIVERSION_NOT_VALID 0x00001A99
#define ERROR_MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION 0x00001A9A
#define ERROR_CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT 0x00001A9B
#define ERROR_CANT_CREATE_MORE_STREAM_MINIVERSIONS 0x00001A9C
#define ERROR_REMOTE_FILE_VERSION_MISMATCH 0x00001A9E
#define ERROR_HANDLE_NO_LONGER_VALID 0x00001A9F
#define ERROR_NO_TXF_METADATA 0x00001AA0
#define ERROR_LOG_CORRUPTION_DETECTED 0x00001AA1
#define ERROR_CANT_RECOVER_WITH_HANDLE_OPEN 0x00001AA2
#define ERROR_RM_DISCONNECTED 0x00001AA3
#define ERROR_ENLISTMENT_NOT_SUPERIOR 0x00001AA4
#define ERROR_RECOVERY_NOT_NEEDED 0x00001AA5
#define ERROR_RM_ALREADY_STARTED 0x00001AA6
#define ERROR_FILE_IDENTITY_NOT_PERSISTENT 0x00001AA7
#define ERROR_CANT_BREAK_TRANSACTIONAL_DEPENDENCY 0x00001AA8
#define ERROR_CANT_CROSS_RM_BOUNDARY 0x00001AA9
#define ERROR_TXF_DIR_NOT_EMPTY 0x00001AAA
#define ERROR_INDOUBT_TRANSACTIONS_EXIST 0x00001AAB
#define ERROR_TM_VOLATILE 0x00001AAC
#define ERROR_ROLLBACK_TIMER_EXPIRED 0x00001AAD
#define ERROR_TXF_ATTRIBUTE_CORRUPT 0x00001AAE
#define ERROR_EFS_NOT_ALLOWED_IN_TRANSACTION 0x00001AAF
#define ERROR_TRANSACTIONAL_OPEN_NOT_ALLOWED 0x00001AB0
#define ERROR_LOG_GROWTH_FAILED 0x00001AB1
#define ERROR_TRANSACTED_MAPPING_UNSUPPORTED_REMOTE 0x00001AB2
#define ERROR_TXF_METADATA_ALREADY_PRESENT 0x00001AB3
#define ERROR_TRANSACTION_SCOPE_CALLBACKS_NOT_SET 0x00001AB4
#define ERROR_TRANSACTION_REQUIRED_PROMOTION 0x00001AB5
#define ERROR_CANNOT_EXECUTE_FILE_IN_TRANSACTION 0x00001AB6
#define ERROR_TRANSACTIONS_NOT_FROZEN 0x00001AB7
#define ERROR_TRANSACTION_FREEZE_IN_PROGRESS 0x00001AB8
#define ERROR_NOT_SNAPSHOT_VOLUME 0x00001AB9
#define ERROR_NO_SAVEPOINT_WITH_OPEN_FILES 0x00001ABA
#define ERROR_DATA_LOST_REPAIR 0x00001ABB
#define ERROR_SPARSE_NOT_ALLOWED_IN_TRANSACTION 0x00001ABC
#define ERROR_TM_IDENTITY_MISMATCH 0x00001ABD
#define ERROR_FLOATED_SECTION 0x00001ABE
#define ERROR_CANNOT_ACCEPT_TRANSACTED_WORK 0x00001ABF
#define ERROR_CANNOT_ABORT_TRANSACTIONS 0x00001AC0
#define ERROR_BAD_CLUSTERS 0x00001AC1
#define ERROR_COMPRESSION_NOT_ALLOWED_IN_TRANSACTION 0x00001AC2
#define ERROR_VOLUME_DIRTY 0x00001AC3
#define ERROR_NO_LINK_TRACKING_IN_TRANSACTION 0x00001AC4
#define ERROR_OPERATION_NOT_SUPPORTED_IN_TRANSACTION 0x00001AC5
#define ERROR_EXPIRED_HANDLE 0x00001AC6
#define ERROR_TRANSACTION_NOT_ENLISTED 0x00001AC7
#define ERROR_CTX_WINSTATION_NAME_INVALID 0x00001B59
#define ERROR_CTX_INVALID_PD 0x00001B5A
#define ERROR_CTX_PD_NOT_FOUND 0x00001B5B
#define ERROR_CTX_WD_NOT_FOUND 0x00001B5C
#define ERROR_CTX_CANNOT_MAKE_EVENTLOG_ENTRY 0x00001B5D
#define ERROR_CTX_SERVICE_NAME_COLLISION 0x00001B5E
#define ERROR_CTX_CLOSE_PENDING 0x00001B5F
#define ERROR_CTX_NO_OUTBUF 0x00001B60
#define ERROR_CTX_MODEM_INF_NOT_FOUND 0x00001B61
#define ERROR_CTX_INVALID_MODEMNAME 0x00001B62
#define ERROR_CTX_MODEM_RESPONSE_ERROR 0x00001B63
#define ERROR_CTX_MODEM_RESPONSE_TIMEOUT 0x00001B64
#define ERROR_CTX_MODEM_RESPONSE_NO_CARRIER 0x00001B65
#define ERROR_CTX_MODEM_RESPONSE_NO_DIALTONE 0x00001B66
#define ERROR_CTX_MODEM_RESPONSE_BUSY 0x00001B67
#define ERROR_CTX_MODEM_RESPONSE_VOICE 0x00001B68
#define ERROR_CTX_TD_ERROR 0x00001B69
#define ERROR_CTX_WINSTATION_NOT_FOUND 0x00001B6E
#define ERROR_CTX_WINSTATION_ALREADY_EXISTS 0x00001B6F
#define ERROR_CTX_WINSTATION_BUSY 0x00001B70
#define ERROR_CTX_BAD_VIDEO_MODE 0x00001B71
#define ERROR_CTX_GRAPHICS_INVALID 0x00001B7B
#define ERROR_CTX_LOGON_DISABLED 0x00001B7D
#define ERROR_CTX_NOT_CONSOLE 0x00001B7E
#define ERROR_CTX_CLIENT_QUERY_TIMEOUT 0x00001B80
#define ERROR_CTX_CONSOLE_DISCONNECT 0x00001B81
#define ERROR_CTX_CONSOLE_CONNECT 0x00001B82
#define ERROR_CTX_SHADOW_DENIED 0x00001B84
#define ERROR_CTX_WINSTATION_ACCESS_DENIED 0x00001B85
#define ERROR_CTX_INVALID_WD 0x00001B89
#define ERROR_CTX_SHADOW_INVALID 0x00001B8A
#define ERROR_CTX_SHADOW_DISABLED 0x00001B8B
#define ERROR_CTX_CLIENT_LICENSE_IN_USE 0x00001B8C
#define ERROR_CTX_CLIENT_LICENSE_NOT_SET 0x00001B8D
#define ERROR_CTX_LICENSE_NOT_AVAILABLE 0x00001B8E
#define ERROR_CTX_LICENSE_CLIENT_INVALID 0x00001B8F
#define ERROR_CTX_LICENSE_EXPIRED 0x00001B90
#define ERROR_CTX_SHADOW_NOT_RUNNING 0x00001B91
#define ERROR_CTX_SHADOW_ENDED_BY_MODE_CHANGE 0x00001B92
#define ERROR_ACTIVATION_COUNT_EXCEEDED 0x00001B93
#define ERROR_CTX_WINSTATIONS_DISABLED 0x00001B94
#define ERROR_CTX_ENCRYPTION_LEVEL_REQUIRED 0x00001B95
#define ERROR_CTX_SESSION_IN_USE 0x00001B96
#define ERROR_CTX_NO_FORCE_LOGOFF 0x00001B97
#define ERROR_CTX_ACCOUNT_RESTRICTION 0x00001B98
#define ERROR_RDP_PROTOCOL_ERROR 0x00001B99
#define ERROR_CTX_CDM_CONNECT 0x00001B9A
#define ERROR_CTX_CDM_DISCONNECT 0x00001B9B
#define ERROR_CTX_SECURITY_LAYER_ERROR 0x00001B9C
#define ERROR_TS_INCOMPATIBLE_SESSIONS 0x00001B9D
#define ERROR_TS_VIDEO_SUBSYSTEM_ERROR 0x00001B9E
#define FRS_ERR_INVALID_API_SEQUENCE 0x00001F41
#define FRS_ERR_STARTING_SERVICE 0x00001F42
#define FRS_ERR_STOPPING_SERVICE 0x00001F43
#define FRS_ERR_INTERNAL_API 0x00001F44
#define FRS_ERR_INTERNAL 0x00001F45
#define FRS_ERR_SERVICE_COMM 0x00001F46
#define FRS_ERR_INSUFFICIENT_PRIV 0x00001F47
#define FRS_ERR_AUTHENTICATION 0x00001F48
#define FRS_ERR_PARENT_INSUFFICIENT_PRIV 0x00001F49
#define FRS_ERR_PARENT_AUTHENTICATION 0x00001F4A
#define FRS_ERR_CHILD_TO_PARENT_COMM 0x00001F4B
#define FRS_ERR_PARENT_TO_CHILD_COMM 0x00001F4C
#define FRS_ERR_SYSVOL_POPULATE 0x00001F4D
#define FRS_ERR_SYSVOL_POPULATE_TIMEOUT 0x00001F4E
#define FRS_ERR_SYSVOL_IS_BUSY 0x00001F4F
#define FRS_ERR_SYSVOL_DEMOTE 0x00001F50
#define FRS_ERR_INVALID_SERVICE_PARAMETER 0x00001F51

/* System Error Codes (8200-8999) */

#define ERROR_DS_NOT_INSTALLED 0x00002008
#define ERROR_DS_MEMBERSHIP_EVALUATED_LOCALLY 0x00002009
#define ERROR_DS_NO_ATTRIBUTE_OR_VALUE 0x0000200A
#define ERROR_DS_INVALID_ATTRIBUTE_SYNTAX 0x0000200B
#define ERROR_DS_ATTRIBUTE_TYPE_UNDEFINED 0x0000200C
#define ERROR_DS_ATTRIBUTE_OR_VALUE_EXISTS 0x0000200D
#define ERROR_DS_BUSY 0x0000200E
#define ERROR_DS_UNAVAILABLE 0x0000200F
#define ERROR_DS_NO_RIDS_ALLOCATED 0x00002010
#define ERROR_DS_NO_MORE_RIDS 0x00002011
#define ERROR_DS_INCORRECT_ROLE_OWNER 0x00002012
#define ERROR_DS_RIDMGR_INIT_ERROR 0x00002013
#define ERROR_DS_OBJ_CLASS_VIOLATION 0x00002014
#define ERROR_DS_CANT_ON_NON_LEAF 0x00002015
#define ERROR_DS_CANT_ON_RDN 0x00002016
#define ERROR_DS_CANT_MOD_OBJ_CLASS 0x00002017
#define ERROR_DS_CROSS_DOM_MOVE_ERROR 0x00002018
#define ERROR_DS_GC_NOT_AVAILABLE 0x00002019
#define ERROR_SHARED_POLICY 0x0000201A
#define ERROR_POLICY_OBJECT_NOT_FOUND 0x0000201B
#define ERROR_POLICY_ONLY_IN_DS 0x0000201C
#define ERROR_PROMOTION_ACTIVE 0x0000201D
#define ERROR_NO_PROMOTION_ACTIVE 0x0000201E
#define ERROR_DS_OPERATIONS_ERROR 0x00002020
#define ERROR_DS_PROTOCOL_ERROR 0x00002021
#define ERROR_DS_TIMELIMIT_EXCEEDED 0x00002022
#define ERROR_DS_SIZELIMIT_EXCEEDED 0x00002023
#define ERROR_DS_ADMIN_LIMIT_EXCEEDED 0x00002024
#define ERROR_DS_COMPARE_FALSE 0x00002025
#define ERROR_DS_COMPARE_TRUE 0x00002026
#define ERROR_DS_AUTH_METHOD_NOT_SUPPORTED 0x00002027
#define ERROR_DS_STRONG_AUTH_REQUIRED 0x00002028
#define ERROR_DS_INAPPROPRIATE_AUTH 0x00002029
#define ERROR_DS_AUTH_UNKNOWN 0x0000202A
#define ERROR_DS_REFERRAL 0x0000202B
#define ERROR_DS_UNAVAILABLE_CRIT_EXTENSION 0x0000202C
#define ERROR_DS_CONFIDENTIALITY_REQUIRED 0x0000202D
#define ERROR_DS_INAPPROPRIATE_MATCHING 0x0000202E
#define ERROR_DS_CONSTRAINT_VIOLATION 0x0000202F
#define ERROR_DS_NO_SUCH_OBJECT 0x00002030
#define ERROR_DS_ALIAS_PROBLEM 0x00002031
#define ERROR_DS_INVALID_DN_SYNTAX 0x00002032
#define ERROR_DS_IS_LEAF 0x00002033
#define ERROR_DS_ALIAS_DEREF_PROBLEM 0x00002034
#define ERROR_DS_UNWILLING_TO_PERFORM 0x00002035
#define ERROR_DS_LOOP_DETECT 0x00002036
#define ERROR_DS_NAMING_VIOLATION 0x00002037
#define ERROR_DS_OBJECT_RESULTS_TOO_LARGE 0x00002038
#define ERROR_DS_AFFECTS_MULTIPLE_DSAS 0x00002039
#define ERROR_DS_SERVER_DOWN 0x0000203A
#define ERROR_DS_LOCAL_ERROR 0x0000203B
#define ERROR_DS_ENCODING_ERROR 0x0000203C
#define ERROR_DS_DECODING_ERROR 0x0000203D
#define ERROR_DS_FILTER_UNKNOWN 0x0000203E
#define ERROR_DS_PARAM_ERROR 0x0000203F
#define ERROR_DS_NOT_SUPPORTED 0x00002040
#define ERROR_DS_NO_RESULTS_RETURNED 0x00002041
#define ERROR_DS_CONTROL_NOT_FOUND 0x00002042
#define ERROR_DS_CLIENT_LOOP 0x00002043
#define ERROR_DS_REFERRAL_LIMIT_EXCEEDED 0x00002044
#define ERROR_DS_SORT_CONTROL_MISSING 0x00002045
#define ERROR_DS_OFFSET_RANGE_ERROR 0x00002046
#define ERROR_DS_RIDMGR_DISABLED 0x00002047
#define ERROR_DS_ROOT_MUST_BE_NC 0x0000206D
#define ERROR_DS_ADD_REPLICA_INHIBITED 0x0000206E
#define ERROR_DS_ATT_NOT_DEF_IN_SCHEMA 0x0000206F
#define ERROR_DS_MAX_OBJ_SIZE_EXCEEDED 0x00002070
#define ERROR_DS_OBJ_STRING_NAME_EXISTS 0x00002071
#define ERROR_DS_NO_RDN_DEFINED_IN_SCHEMA 0x00002072
#define ERROR_DS_RDN_DOESNT_MATCH_SCHEMA 0x00002073
#define ERROR_DS_NO_REQUESTED_ATTS_FOUND 0x00002074
#define ERROR_DS_USER_BUFFER_TO_SMALL 0x00002075
#define ERROR_DS_ATT_IS_NOT_ON_OBJ 0x00002076
#define ERROR_DS_ILLEGAL_MOD_OPERATION 0x00002077
#define ERROR_DS_OBJ_TOO_LARGE 0x00002078
#define ERROR_DS_BAD_INSTANCE_TYPE 0x00002079
#define ERROR_DS_MASTERDSA_REQUIRED 0x0000207A
#define ERROR_DS_OBJECT_CLASS_REQUIRED 0x0000207B
#define ERROR_DS_MISSING_REQUIRED_ATT 0x0000207C
#define ERROR_DS_ATT_NOT_DEF_FOR_CLASS 0x0000207D
#define ERROR_DS_ATT_ALREADY_EXISTS 0x0000207E
#define ERROR_DS_CANT_ADD_ATT_VALUES 0x00002080
#define ERROR_DS_SINGLE_VALUE_CONSTRAINT 0x00002081
#define ERROR_DS_RANGE_CONSTRAINT 0x00002082
#define ERROR_DS_ATT_VAL_ALREADY_EXISTS 0x00002083
#define ERROR_DS_CANT_REM_MISSING_ATT 0x00002084
#define ERROR_DS_CANT_REM_MISSING_ATT_VAL 0x00002085
#define ERROR_DS_ROOT_CANT_BE_SUBREF 0x00002086
#define ERROR_DS_NO_CHAINING 0x00002087
#define ERROR_DS_NO_CHAINED_EVAL 0x00002088
#define ERROR_DS_NO_PARENT_OBJECT 0x00002089
#define ERROR_DS_PARENT_IS_AN_ALIAS 0x0000208A
#define ERROR_DS_CANT_MIX_MASTER_AND_REPS 0x0000208B
#define ERROR_DS_CHILDREN_EXIST 0x0000208C
#define ERROR_DS_OBJ_NOT_FOUND 0x0000208D
#define ERROR_DS_ALIASED_OBJ_MISSING 0x0000208E
#define ERROR_DS_BAD_NAME_SYNTAX 0x0000208F
#define ERROR_DS_ALIAS_POINTS_TO_ALIAS 0x00002090
#define ERROR_DS_CANT_DEREF_ALIAS 0x00002091
#define ERROR_DS_OUT_OF_SCOPE 0x00002092
#define ERROR_DS_OBJECT_BEING_REMOVED 0x00002093
#define ERROR_DS_CANT_DELETE_DSA_OBJ 0x00002094
#define ERROR_DS_GENERIC_ERROR 0x00002095
#define ERROR_DS_DSA_MUST_BE_INT_MASTER 0x00002096
#define ERROR_DS_CLASS_NOT_DSA 0x00002097
#define ERROR_DS_INSUFF_ACCESS_RIGHTS 0x00002098
#define ERROR_DS_ILLEGAL_SUPERIOR 0x00002099
#define ERROR_DS_ATTRIBUTE_OWNED_BY_SAM 0x0000209A
#define ERROR_DS_NAME_TOO_MANY_PARTS 0x0000209B
#define ERROR_DS_NAME_TOO_LONG 0x0000209C
#define ERROR_DS_NAME_VALUE_TOO_LONG 0x0000209D
#define ERROR_DS_NAME_UNPARSEABLE 0x0000209E
#define ERROR_DS_NAME_TYPE_UNKNOWN 0x0000209F
#define ERROR_DS_NOT_AN_OBJECT 0x000020A0
#define ERROR_DS_SEC_DESC_TOO_SHORT 0x000020A1
#define ERROR_DS_SEC_DESC_INVALID 0x000020A2
#define ERROR_DS_NO_DELETED_NAME 0x000020A3
#define ERROR_DS_SUBREF_MUST_HAVE_PARENT 0x000020A4
#define ERROR_DS_NCNAME_MUST_BE_NC 0x000020A5
#define ERROR_DS_CANT_ADD_SYSTEM_ONLY 0x000020A6
#define ERROR_DS_CLASS_MUST_BE_CONCRETE 0x000020A7
#define ERROR_DS_INVALID_DMD 0x000020A8
#define ERROR_DS_OBJ_GUID_EXISTS 0x000020A9
#define ERROR_DS_NOT_ON_BACKLINK 0x000020AA
#define ERROR_DS_NO_CROSSREF_FOR_NC 0x000020AB
#define ERROR_DS_SHUTTING_DOWN 0x000020AC
#define ERROR_DS_UNKNOWN_OPERATION 0x000020AD
#define ERROR_DS_INVALID_ROLE_OWNER 0x000020AE
#define ERROR_DS_COULDNT_CONTACT_FSMO 0x000020AF
#define ERROR_DS_CROSS_NC_DN_RENAME 0x000020B0
#define ERROR_DS_CANT_MOD_SYSTEM_ONLY 0x000020B1
#define ERROR_DS_REPLICATOR_ONLY 0x000020B2
#define ERROR_DS_OBJ_CLASS_NOT_DEFINED 0x000020B3
#define ERROR_DS_OBJ_CLASS_NOT_SUBCLASS 0x000020B4
#define ERROR_DS_NAME_REFERENCE_INVALID 0x000020B5
#define ERROR_DS_CROSS_REF_EXISTS 0x000020B6
#define ERROR_DS_CANT_DEL_MASTER_CROSSREF 0x000020B7
#define ERROR_DS_SUBTREE_NOTIFY_NOT_NC_HEAD 0x000020B8
#define ERROR_DS_NOTIFY_FILTER_TOO_COMPLEX 0x000020B9
#define ERROR_DS_DUP_RDN 0x000020BA
#define ERROR_DS_DUP_OID 0x000020BB
#define ERROR_DS_DUP_MAPI_ID 0x000020BC
#define ERROR_DS_DUP_SCHEMA_ID_GUID 0x000020BD
#define ERROR_DS_DUP_LDAP_DISPLAY_NAME 0x000020BE
#define ERROR_DS_SEMANTIC_ATT_TEST 0x000020BF
#define ERROR_DS_SYNTAX_MISMATCH 0x000020C0
#define ERROR_DS_EXISTS_IN_MUST_HAVE 0x000020C1
#define ERROR_DS_EXISTS_IN_MAY_HAVE 0x000020C2
#define ERROR_DS_NONEXISTENT_MAY_HAVE 0x000020C3
#define ERROR_DS_NONEXISTENT_MUST_HAVE 0x000020C4
#define ERROR_DS_AUX_CLS_TEST_FAIL 0x000020C5
#define ERROR_DS_NONEXISTENT_POSS_SUP 0x000020C6
#define ERROR_DS_SUB_CLS_TEST_FAIL 0x000020C7
#define ERROR_DS_BAD_RDN_ATT_ID_SYNTAX 0x000020C8
#define ERROR_DS_EXISTS_IN_AUX_CLS 0x000020C9
#define ERROR_DS_EXISTS_IN_SUB_CLS 0x000020CA
#define ERROR_DS_EXISTS_IN_POSS_SUP 0x000020CB
#define ERROR_DS_RECALCSCHEMA_FAILED 0x000020CC
#define ERROR_DS_TREE_DELETE_NOT_FINISHED 0x000020CD
#define ERROR_DS_CANT_DELETE 0x000020CE
#define ERROR_DS_ATT_SCHEMA_REQ_ID 0x000020CF
#define ERROR_DS_BAD_ATT_SCHEMA_SYNTAX 0x000020D0
#define ERROR_DS_CANT_CACHE_ATT 0x000020D1
#define ERROR_DS_CANT_CACHE_CLASS 0x000020D2
#define ERROR_DS_CANT_REMOVE_ATT_CACHE 0x000020D3
#define ERROR_DS_CANT_REMOVE_CLASS_CACHE 0x000020D4
#define ERROR_DS_CANT_RETRIEVE_DN 0x000020D5
#define ERROR_DS_MISSING_SUPREF 0x000020D6
#define ERROR_DS_CANT_RETRIEVE_INSTANCE 0x000020D7
#define ERROR_DS_CODE_INCONSISTENCY 0x000020D8
#define ERROR_DS_DATABASE_ERROR 0x000020D9
#define ERROR_DS_GOVERNSID_MISSING 0x000020DA
#define ERROR_DS_MISSING_EXPECTED_ATT 0x000020DB
#define ERROR_DS_NCNAME_MISSING_CR_REF 0x000020DC
#define ERROR_DS_SECURITY_CHECKING_ERROR 0x000020DD
#define ERROR_DS_SCHEMA_NOT_LOADED 0x000020DE
#define ERROR_DS_SCHEMA_ALLOC_FAILED 0x000020DF
#define ERROR_DS_ATT_SCHEMA_REQ_SYNTAX 0x000020E0
#define ERROR_DS_GCVERIFY_ERROR 0x000020E1
#define ERROR_DS_DRA_SCHEMA_MISMATCH 0x000020E2
#define ERROR_DS_CANT_FIND_DSA_OBJ 0x000020E3
#define ERROR_DS_CANT_FIND_EXPECTED_NC 0x000020E4
#define ERROR_DS_CANT_FIND_NC_IN_CACHE 0x000020E5
#define ERROR_DS_CANT_RETRIEVE_CHILD 0x000020E6
#define ERROR_DS_SECURITY_ILLEGAL_MODIFY 0x000020E7
#define ERROR_DS_CANT_REPLACE_HIDDEN_REC 0x000020E8
#define ERROR_DS_BAD_HIERARCHY_FILE 0x000020E9
#define ERROR_DS_BUILD_HIERARCHY_TABLE_FAILED 0x000020EA
#define ERROR_DS_CONFIG_PARAM_MISSING 0x000020EB
#define ERROR_DS_COUNTING_AB_INDICES_FAILED 0x000020EC
#define ERROR_DS_HIERARCHY_TABLE_MALLOC_FAILED 0x000020ED
#define ERROR_DS_INTERNAL_FAILURE 0x000020EE
#define ERROR_DS_UNKNOWN_ERROR 0x000020EF
#define ERROR_DS_ROOT_REQUIRES_CLASS_TOP 0x000020F0
#define ERROR_DS_REFUSING_FSMO_ROLES 0x000020F1
#define ERROR_DS_MISSING_FSMO_SETTINGS 0x000020F2
#define ERROR_DS_UNABLE_TO_SURRENDER_ROLES 0x000020F3
#define ERROR_DS_DRA_GENERIC 0x000020F4
#define ERROR_DS_DRA_INVALID_PARAMETER 0x000020F5
#define ERROR_DS_DRA_BUSY 0x000020F6
#define ERROR_DS_DRA_BAD_DN 0x000020F7
#define ERROR_DS_DRA_BAD_NC 0x000020F8
#define ERROR_DS_DRA_DN_EXISTS 0x000020F9
#define ERROR_DS_DRA_INTERNAL_ERROR 0x000020FA
#define ERROR_DS_DRA_INCONSISTENT_DIT 0x000020FB
#define ERROR_DS_DRA_CONNECTION_FAILED 0x000020FC
#define ERROR_DS_DRA_BAD_INSTANCE_TYPE 0x000020FD
#define ERROR_DS_DRA_OUT_OF_MEM 0x000020FE
#define ERROR_DS_DRA_MAIL_PROBLEM 0x000020FF
#define ERROR_DS_DRA_REF_ALREADY_EXISTS 0x00002100
#define ERROR_DS_DRA_REF_NOT_FOUND 0x00002101
#define ERROR_DS_DRA_OBJ_IS_REP_SOURCE 0x00002102
#define ERROR_DS_DRA_DB_ERROR 0x00002103
#define ERROR_DS_DRA_NO_REPLICA 0x00002104
#define ERROR_DS_DRA_ACCESS_DENIED 0x00002105
#define ERROR_DS_DRA_NOT_SUPPORTED 0x00002106
#define ERROR_DS_DRA_RPC_CANCELLED 0x00002107
#define ERROR_DS_DRA_SOURCE_DISABLED 0x00002108
#define ERROR_DS_DRA_SINK_DISABLED 0x00002109
#define ERROR_DS_DRA_NAME_COLLISION 0x0000210A
#define ERROR_DS_DRA_SOURCE_REINSTALLED 0x0000210B
#define ERROR_DS_DRA_MISSING_PARENT 0x0000210C
#define ERROR_DS_DRA_PREEMPTED 0x0000210D
#define ERROR_DS_DRA_ABANDON_SYNC 0x0000210E
#define ERROR_DS_DRA_SHUTDOWN 0x0000210F
#define ERROR_DS_DRA_INCOMPATIBLE_PARTIAL_SET 0x00002110
#define ERROR_DS_DRA_SOURCE_IS_PARTIAL_REPLICA 0x00002111
#define ERROR_DS_DRA_EXTN_CONNECTION_FAILED 0x00002112
#define ERROR_DS_INSTALL_SCHEMA_MISMATCH 0x00002113
#define ERROR_DS_DUP_LINK_ID 0x00002114
#define ERROR_DS_NAME_ERROR_RESOLVING 0x00002115
#define ERROR_DS_NAME_ERROR_NOT_FOUND 0x00002116
#define ERROR_DS_NAME_ERROR_NOT_UNIQUE 0x00002117
#define ERROR_DS_NAME_ERROR_NO_MAPPING 0x00002118
#define ERROR_DS_NAME_ERROR_DOMAIN_ONLY 0x00002119
#define ERROR_DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING 0x0000211A
#define ERROR_DS_CONSTRUCTED_ATT_MOD 0x0000211B
#define ERROR_DS_WRONG_OM_OBJ_CLASS 0x0000211C
#define ERROR_DS_DRA_REPL_PENDING 0x0000211D
#define ERROR_DS_DS_REQUIRED 0x0000211E
#define ERROR_DS_INVALID_LDAP_DISPLAY_NAME 0x0000211F
#define ERROR_DS_NON_BASE_SEARCH 0x00002120
#define ERROR_DS_CANT_RETRIEVE_ATTS 0x00002121
#define ERROR_DS_BACKLINK_WITHOUT_LINK 0x00002122
#define ERROR_DS_EPOCH_MISMATCH 0x00002123
#define ERROR_DS_SRC_NAME_MISMATCH 0x00002124
#define ERROR_DS_SRC_AND_DST_NC_IDENTICAL 0x00002125
#define ERROR_DS_DST_NC_MISMATCH 0x00002126
#define ERROR_DS_NOT_AUTHORITIVE_FOR_DST_NC 0x00002127
#define ERROR_DS_SRC_GUID_MISMATCH 0x00002128
#define ERROR_DS_CANT_MOVE_DELETED_OBJECT 0x00002129
#define ERROR_DS_PDC_OPERATION_IN_PROGRESS 0x0000212A
#define ERROR_DS_CROSS_DOMAIN_CLEANUP_REQD 0x0000212B
#define ERROR_DS_ILLEGAL_XDOM_MOVE_OPERATION 0x0000212C
#define ERROR_DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS 0x0000212D
#define ERROR_DS_NC_MUST_HAVE_NC_PARENT 0x0000212E
#define ERROR_DS_CR_IMPOSSIBLE_TO_VALIDATE 0x0000212F
#define ERROR_DS_DST_DOMAIN_NOT_NATIVE 0x00002130
#define ERROR_DS_MISSING_INFRASTRUCTURE_CONTAINER 0x00002131
#define ERROR_DS_CANT_MOVE_ACCOUNT_GROUP 0x00002132
#define ERROR_DS_CANT_MOVE_RESOURCE_GROUP 0x00002133
#define ERROR_DS_INVALID_SEARCH_FLAG 0x00002134
#define ERROR_DS_NO_TREE_DELETE_ABOVE_NC 0x00002135
#define ERROR_DS_COULDNT_LOCK_TREE_FOR_DELETE 0x00002136
#define ERROR_DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE 0x00002137
#define ERROR_DS_SAM_INIT_FAILURE 0x00002138
#define ERROR_DS_SENSITIVE_GROUP_VIOLATION 0x00002139
#define ERROR_DS_CANT_MOD_PRIMARYGROUPID 0x0000213A
#define ERROR_DS_ILLEGAL_BASE_SCHEMA_MOD 0x0000213B
#define ERROR_DS_NONSAFE_SCHEMA_CHANGE 0x0000213C
#define ERROR_DS_SCHEMA_UPDATE_DISALLOWED 0x0000213D
#define ERROR_DS_CANT_CREATE_UNDER_SCHEMA 0x0000213E
#define ERROR_DS_INSTALL_NO_SRC_SCH_VERSION 0x0000213F
#define ERROR_DS_INSTALL_NO_SCH_VERSION_IN_INIFILE 0x00002140
#define ERROR_DS_INVALID_GROUP_TYPE 0x00002141
#define ERROR_DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN 0x00002142
#define ERROR_DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN 0x00002143
#define ERROR_DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER 0x00002144
#define ERROR_DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER 0x00002145
#define ERROR_DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER 0x00002146
#define ERROR_DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER 0x00002147
#define ERROR_DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER 0x00002148
#define ERROR_DS_HAVE_PRIMARY_MEMBERS 0x00002149
#define ERROR_DS_STRING_SD_CONVERSION_FAILED 0x0000214A
#define ERROR_DS_NAMING_MASTER_GC 0x0000214B
#define ERROR_DS_DNS_LOOKUP_FAILURE 0x0000214C
#define ERROR_DS_COULDNT_UPDATE_SPNS 0x0000214D
#define ERROR_DS_CANT_RETRIEVE_SD 0x0000214E
#define ERROR_DS_KEY_NOT_UNIQUE 0x0000214F
#define ERROR_DS_WRONG_LINKED_ATT_SYNTAX 0x00002150
#define ERROR_DS_SAM_NEED_BOOTKEY_PASSWORD 0x00002151
#define ERROR_DS_SAM_NEED_BOOTKEY_FLOPPY 0x00002152
#define ERROR_DS_CANT_START 0x00002153
#define ERROR_DS_INIT_FAILURE 0x00002154
#define ERROR_DS_NO_PKT_PRIVACY_ON_CONNECTION 0x00002155
#define ERROR_DS_SOURCE_DOMAIN_IN_FOREST 0x00002156
#define ERROR_DS_DESTINATION_DOMAIN_NOT_IN_FOREST 0x00002157
#define ERROR_DS_DESTINATION_AUDITING_NOT_ENABLED 0x00002158
#define ERROR_DS_CANT_FIND_DC_FOR_SRC_DOMAIN 0x00002159
#define ERROR_DS_SRC_OBJ_NOT_GROUP_OR_USER 0x0000215A
#define ERROR_DS_SRC_SID_EXISTS_IN_FOREST 0x0000215B
#define ERROR_DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH 0x0000215C
#define ERROR_SAM_INIT_FAILURE 0x0000215D
#define ERROR_DS_DRA_SCHEMA_INFO_SHIP 0x0000215E
#define ERROR_DS_DRA_SCHEMA_CONFLICT 0x0000215F
#define ERROR_DS_DRA_EARLIER_SCHEMA_CONFLICT 0x00002160
#define ERROR_DS_DRA_OBJ_NC_MISMATCH 0x00002161
#define ERROR_DS_NC_STILL_HAS_DSAS 0x00002162
#define ERROR_DS_GC_REQUIRED 0x00002163
#define ERROR_DS_LOCAL_MEMBER_OF_LOCAL_ONLY 0x00002164
#define ERROR_DS_NO_FPO_IN_UNIVERSAL_GROUPS 0x00002165
#define ERROR_DS_CANT_ADD_TO_GC 0x00002166
#define ERROR_DS_NO_CHECKPOINT_WITH_PDC 0x00002167
#define ERROR_DS_SOURCE_AUDITING_NOT_ENABLED 0x00002168
#define ERROR_DS_CANT_CREATE_IN_NONDOMAIN_NC 0x00002169
#define ERROR_DS_INVALID_NAME_FOR_SPN 0x0000216A
#define ERROR_DS_FILTER_USES_CONTRUCTED_ATTRS 0x0000216B
#define ERROR_DS_UNICODEPWD_NOT_IN_QUOTES 0x0000216C
#define ERROR_DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED 0x0000216D
#define ERROR_DS_MUST_BE_RUN_ON_DST_DC 0x0000216E
#define ERROR_DS_SRC_DC_MUST_BE_SP4_OR_GREATER 0x0000216F
#define ERROR_DS_CANT_TREE_DELETE_CRITICAL_OBJ 0x00002170
#define ERROR_DS_INIT_FAILURE_CONSOLE 0x00002171
#define ERROR_DS_SAM_INIT_FAILURE_CONSOLE 0x00002172
#define ERROR_DS_FOREST_VERSION_TOO_HIGH 0x00002173
#define ERROR_DS_DOMAIN_VERSION_TOO_HIGH 0x00002174
#define ERROR_DS_FOREST_VERSION_TOO_LOW 0x00002175
#define ERROR_DS_DOMAIN_VERSION_TOO_LOW 0x00002176
#define ERROR_DS_INCOMPATIBLE_VERSION 0x00002177
#define ERROR_DS_LOW_DSA_VERSION 0x00002178
#define ERROR_DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN 0x00002179
#define ERROR_DS_NOT_SUPPORTED_SORT_ORDER 0x0000217A
#define ERROR_DS_NAME_NOT_UNIQUE 0x0000217B
#define ERROR_DS_MACHINE_ACCOUNT_CREATED_PRENT4 0x0000217C
#define ERROR_DS_OUT_OF_VERSION_STORE 0x0000217D
#define ERROR_DS_INCOMPATIBLE_CONTROLS_USED 0x0000217E
#define ERROR_DS_NO_REF_DOMAIN 0x0000217F
#define ERROR_DS_RESERVED_LINK_ID 0x00002180
#define ERROR_DS_LINK_ID_NOT_AVAILABLE 0x00002181
#define ERROR_DS_AG_CANT_HAVE_UNIVERSAL_MEMBER 0x00002182
#define ERROR_DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE 0x00002183
#define ERROR_DS_NO_OBJECT_MOVE_IN_SCHEMA_NC 0x00002184
#define ERROR_DS_MODIFYDN_DISALLOWED_BY_FLAG 0x00002185
#define ERROR_DS_MODIFYDN_WRONG_GRANDPARENT 0x00002186
#define ERROR_DS_NAME_ERROR_TRUST_REFERRAL 0x00002187
#define ERROR_NOT_SUPPORTED_ON_STANDARD_SERVER 0x00002188
#define ERROR_DS_CANT_ACCESS_REMOTE_PART_OF_AD 0x00002189
#define ERROR_DS_CR_IMPOSSIBLE_TO_VALIDATE_V2 0x0000218A
#define ERROR_DS_THREAD_LIMIT_EXCEEDED 0x0000218B
#define ERROR_DS_NOT_CLOSEST 0x0000218C
#define ERROR_DS_CANT_DERIVE_SPN_WITHOUT_SERVER_REF 0x0000218D
#define ERROR_DS_SINGLE_USER_MODE_FAILED 0x0000218E
#define ERROR_DS_NTDSCRIPT_SYNTAX_ERROR 0x0000218F
#define ERROR_DS_NTDSCRIPT_PROCESS_ERROR 0x00002190
#define ERROR_DS_DIFFERENT_REPL_EPOCHS 0x00002191
#define ERROR_DS_DRS_EXTENSIONS_CHANGED 0x00002192
#define ERROR_DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR 0x00002193
#define ERROR_DS_NO_MSDS_INTID 0x00002194
#define ERROR_DS_DUP_MSDS_INTID 0x00002195
#define ERROR_DS_EXISTS_IN_RDNATTID 0x00002196
#define ERROR_DS_AUTHORIZATION_FAILED 0x00002197
#define ERROR_DS_INVALID_SCRIPT 0x00002198
#define ERROR_DS_REMOTE_CROSSREF_OP_FAILED 0x00002199
#define ERROR_DS_CROSS_REF_BUSY 0x0000219A
#define ERROR_DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN 0x0000219B
#define ERROR_DS_CANT_DEMOTE_WITH_WRITEABLE_NC 0x0000219C
#define ERROR_DS_DUPLICATE_ID_FOUND 0x0000219D
#define ERROR_DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT 0x0000219E
#define ERROR_DS_GROUP_CONVERSION_ERROR 0x0000219F
#define ERROR_DS_CANT_MOVE_APP_BASIC_GROUP 0x000021A0
#define ERROR_DS_CANT_MOVE_APP_QUERY_GROUP 0x000021A1
#define ERROR_DS_ROLE_NOT_VERIFIED 0x000021A2
#define ERROR_DS_WKO_CONTAINER_CANNOT_BE_SPECIAL 0x000021A3
#define ERROR_DS_DOMAIN_RENAME_IN_PROGRESS 0x000021A4
#define ERROR_DS_EXISTING_AD_CHILD_NC 0x000021A5
#define ERROR_DS_REPL_LIFETIME_EXCEEDED 0x000021A6
#define ERROR_DS_DISALLOWED_IN_SYSTEM_CONTAINER 0x000021A7
#define ERROR_DS_LDAP_SEND_QUEUE_FULL 0x000021A8
#define ERROR_DS_DRA_OUT_SCHEDULE_WINDOW 0x000021A9
#define ERROR_DS_POLICY_NOT_KNOWN 0x000021AA
#define ERROR_NO_SITE_SETTINGS_OBJECT 0x000021AB
#define ERROR_NO_SECRETS 0x000021AC
#define ERROR_NO_WRITABLE_DC_FOUND 0x000021AD
#define ERROR_DS_NO_SERVER_OBJECT 0x000021AE
#define ERROR_DS_NO_NTDSA_OBJECT 0x000021AF
#define ERROR_DS_NON_ASQ_SEARCH 0x000021B0
#define ERROR_DS_AUDIT_FAILURE 0x000021B1
#define ERROR_DS_INVALID_SEARCH_FLAG_SUBTREE 0x000021B2
#define ERROR_DS_INVALID_SEARCH_FLAG_TUPLE 0x000021B3
#define ERROR_DS_HIERARCHY_TABLE_TOO_DEEP 0x000021B4
#define ERROR_DS_DRA_CORRUPT_UTD_VECTOR 0x000021B5
#define ERROR_DS_DRA_SECRETS_DENIED 0x000021B6
#define ERROR_DS_RESERVED_MAPI_ID 0x000021B7
#define ERROR_DS_MAPI_ID_NOT_AVAILABLE 0x000021B8
#define ERROR_DS_DRA_MISSING_KRBTGT_SECRET 0x000021B9
#define ERROR_DS_DOMAIN_NAME_EXISTS_IN_FOREST 0x000021BA
#define ERROR_DS_FLAT_NAME_EXISTS_IN_FOREST 0x000021BB
#define ERROR_INVALID_USER_PRINCIPAL_NAME 0x000021BC
#define ERROR_DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS 0x000021BD
#define ERROR_DS_OID_NOT_FOUND 0x000021BE
#define ERROR_DS_DRA_RECYCLED_TARGET 0x000021BF
#define ERROR_DS_DISALLOWED_NC_REDIRECT 0x000021C0
#define ERROR_DS_HIGH_ADLDS_FFL 0x000021C1
#define ERROR_DS_HIGH_DSA_VERSION 0x000021C2
#define ERROR_DS_LOW_ADLDS_FFL 0x000021C3
#define ERROR_DOMAIN_SID_SAME_AS_LOCAL_WORKSTATION 0x000021C4
#define ERROR_DS_UNDELETE_SAM_VALIDATION_FAILED 0x000021C5
#define ERROR_INCORRECT_ACCOUNT_TYPE 0x000021C6

/* System Error Codes (9000-11999) */

#define DNS_ERROR_RCODE_FORMAT_ERROR 0x00002329
#define DNS_ERROR_RCODE_SERVER_FAILURE 0x0000232A
#define DNS_ERROR_RCODE_NAME_ERROR 0x0000232B
#define DNS_ERROR_RCODE_NOT_IMPLEMENTED 0x0000232C
#define DNS_ERROR_RCODE_REFUSED 0x0000232D
#define DNS_ERROR_RCODE_YXDOMAIN 0x0000232E
#define DNS_ERROR_RCODE_YXRRSET 0x0000232F
#define DNS_ERROR_RCODE_NXRRSET 0x00002330
#define DNS_ERROR_RCODE_NOTAUTH 0x00002331
#define DNS_ERROR_RCODE_NOTZONE 0x00002332
#define DNS_ERROR_RCODE_BADSIG 0x00002338
#define DNS_ERROR_RCODE_BADKEY 0x00002339
#define DNS_ERROR_RCODE_BADTIME 0x0000233A
#define DNS_ERROR_KEYMASTER_REQUIRED 0x0000238D
#define DNS_ERROR_NOT_ALLOWED_ON_SIGNED_ZONE 0x0000238E
#define DNS_ERROR_NSEC3_INCOMPATIBLE_WITH_RSA_SHA1 0x0000238F
#define DNS_ERROR_NOT_ENOUGH_SIGNING_KEY_DESCRIPTORS 0x00002390
#define DNS_ERROR_UNSUPPORTED_ALGORITHM 0x00002391
#define DNS_ERROR_INVALID_KEY_SIZE 0x00002392
#define DNS_ERROR_SIGNING_KEY_NOT_ACCESSIBLE 0x00002393
#define DNS_ERROR_KSP_DOES_NOT_SUPPORT_PROTECTION 0x00002394
#define DNS_ERROR_UNEXPECTED_DATA_PROTECTION_ERROR 0x00002395
#define DNS_ERROR_UNEXPECTED_CNG_ERROR 0x00002396
#define DNS_ERROR_UNKNOWN_SIGNING_PARAMETER_VERSION 0x00002397
#define DNS_ERROR_KSP_NOT_ACCESSIBLE 0x00002398
#define DNS_ERROR_TOO_MANY_SKDS 0x00002399
#define DNS_ERROR_INVALID_ROLLOVER_PERIOD 0x0000239A
#define DNS_ERROR_INVALID_INITIAL_ROLLOVER_OFFSET 0x0000239B
#define DNS_ERROR_ROLLOVER_IN_PROGRESS 0x0000239C
#define DNS_ERROR_STANDBY_KEY_NOT_PRESENT 0x0000239D
#define DNS_ERROR_NOT_ALLOWED_ON_ZSK 0x0000239E
#define DNS_ERROR_NOT_ALLOWED_ON_ACTIVE_SKD 0x0000239F
#define DNS_ERROR_ROLLOVER_ALREADY_QUEUED 0x000023A0
#define DNS_ERROR_NOT_ALLOWED_ON_UNSIGNED_ZONE 0x000023A1
#define DNS_ERROR_BAD_KEYMASTER 0x000023A2
#define DNS_ERROR_INVALID_SIGNATURE_VALIDITY_PERIOD 0x000023A3
#define DNS_ERROR_INVALID_NSEC3_ITERATION_COUNT 0x000023A4
#define DNS_ERROR_DNSSEC_IS_DISABLED 0x000023A5
#define DNS_ERROR_INVALID_XML 0x000023A6
#define DNS_ERROR_NO_VALID_TRUST_ANCHORS 0x000023A7
#define DNS_ERROR_ROLLOVER_NOT_POKEABLE 0x000023A8
#define DNS_ERROR_NSEC3_NAME_COLLISION 0x000023A9
#define DNS_ERROR_NSEC_INCOMPATIBLE_WITH_NSEC3_RSA_SHA1 0x000023AA
#define DNS_INFO_NO_RECORDS 0x0000251D
#define DNS_ERROR_BAD_PACKET 0x0000251E
#define DNS_ERROR_NO_PACKET 0x0000251F
#define DNS_ERROR_RCODE 0x00002520
#define DNS_ERROR_UNSECURE_PACKET 0x00002521
#define DNS_REQUEST_PENDING 0x00002522
#define DNS_ERROR_INVALID_TYPE 0x0000254F
#define DNS_ERROR_INVALID_IP_ADDRESS 0x00002550
#define DNS_ERROR_INVALID_PROPERTY 0x00002551
#define DNS_ERROR_TRY_AGAIN_LATER 0x00002552
#define DNS_ERROR_NOT_UNIQUE 0x00002553
#define DNS_ERROR_NON_RFC_NAME 0x00002554
#define DNS_STATUS_FQDN 0x00002555
#define DNS_STATUS_DOTTED_NAME 0x00002556
#define DNS_STATUS_SINGLE_PART_NAME 0x00002557
#define DNS_ERROR_INVALID_NAME_CHAR 0x00002558
#define DNS_ERROR_NUMERIC_NAME 0x00002559
#define DNS_ERROR_NOT_ALLOWED_ON_ROOT_SERVER 0x0000255A
#define DNS_ERROR_NOT_ALLOWED_UNDER_DELEGATION 0x0000255B
#define DNS_ERROR_CANNOT_FIND_ROOT_HINTS 0x0000255C
#define DNS_ERROR_INCONSISTENT_ROOT_HINTS 0x0000255D
#define DNS_ERROR_DWORD_VALUE_TOO_SMALL 0x0000255E
#define DNS_ERROR_DWORD_VALUE_TOO_LARGE 0x0000255F
#define DNS_ERROR_BACKGROUND_LOADING 0x00002560
#define DNS_ERROR_NOT_ALLOWED_ON_RODC 0x00002561
#define DNS_ERROR_NOT_ALLOWED_UNDER_DNAME 0x00002562
#define DNS_ERROR_DELEGATION_REQUIRED 0x00002563
#define DNS_ERROR_INVALID_POLICY_TABLE 0x00002564
#define DNS_ERROR_ZONE_DOES_NOT_EXIST 0x00002581
#define DNS_ERROR_NO_ZONE_INFO 0x00002582
#define DNS_ERROR_INVALID_ZONE_OPERATION 0x00002583
#define DNS_ERROR_ZONE_CONFIGURATION_ERROR 0x00002584
#define DNS_ERROR_ZONE_HAS_NO_SOA_RECORD 0x00002585
#define DNS_ERROR_ZONE_HAS_NO_NS_RECORDS 0x00002586
#define DNS_ERROR_ZONE_LOCKED 0x00002587
#define DNS_ERROR_ZONE_CREATION_FAILED 0x00002588
#define DNS_ERROR_ZONE_ALREADY_EXISTS 0x00002589
#define DNS_ERROR_AUTOZONE_ALREADY_EXISTS 0x0000258A
#define DNS_ERROR_INVALID_ZONE_TYPE 0x0000258B
#define DNS_ERROR_SECONDARY_REQUIRES_MASTER_IP 0x0000258C
#define DNS_ERROR_ZONE_NOT_SECONDARY 0x0000258D
#define DNS_ERROR_NEED_SECONDARY_ADDRESSES 0x0000258E
#define DNS_ERROR_WINS_INIT_FAILED 0x0000258F
#define DNS_ERROR_NEED_WINS_SERVERS 0x00002590
#define DNS_ERROR_NBSTAT_INIT_FAILED 0x00002591
#define DNS_ERROR_SOA_DELETE_INVALID 0x00002592
#define DNS_ERROR_FORWARDER_ALREADY_EXISTS 0x00002593
#define DNS_ERROR_ZONE_REQUIRES_MASTER_IP 0x00002594
#define DNS_ERROR_ZONE_IS_SHUTDOWN 0x00002595
#define DNS_ERROR_ZONE_LOCKED_FOR_SIGNING 0x00002596
#define DNS_ERROR_PRIMARY_REQUIRES_DATAFILE 0x000025B3
#define DNS_ERROR_INVALID_DATAFILE_NAME 0x000025B4
#define DNS_ERROR_DATAFILE_OPEN_FAILURE 0x000025B5
#define DNS_ERROR_FILE_WRITEBACK_FAILED 0x000025B6
#define DNS_ERROR_DATAFILE_PARSING 0x000025B7
#define DNS_ERROR_RECORD_DOES_NOT_EXIST 0x000025E5
#define DNS_ERROR_RECORD_FORMAT 0x000025E6
#define DNS_ERROR_NODE_CREATION_FAILED 0x000025E7
#define DNS_ERROR_UNKNOWN_RECORD_TYPE 0x000025E8
#define DNS_ERROR_RECORD_TIMED_OUT 0x000025E9
#define DNS_ERROR_NAME_NOT_IN_ZONE 0x000025EA
#define DNS_ERROR_CNAME_LOOP 0x000025EB
#define DNS_ERROR_NODE_IS_CNAME 0x000025EC
#define DNS_ERROR_CNAME_COLLISION 0x000025ED
#define DNS_ERROR_RECORD_ONLY_AT_ZONE_ROOT 0x000025EE
#define DNS_ERROR_RECORD_ALREADY_EXISTS 0x000025EF
#define DNS_ERROR_SECONDARY_DATA 0x000025F0
#define DNS_ERROR_NO_CREATE_CACHE_DATA 0x000025F1
#define DNS_ERROR_NAME_DOES_NOT_EXIST 0x000025F2
#define DNS_WARNING_PTR_CREATE_FAILED 0x000025F3
#define DNS_WARNING_DOMAIN_UNDELETED 0x000025F4
#define DNS_ERROR_DS_UNAVAILABLE 0x000025F5
#define DNS_ERROR_DS_ZONE_ALREADY_EXISTS 0x000025F6
#define DNS_ERROR_NO_BOOTFILE_IF_DS_ZONE 0x000025F7
#define DNS_ERROR_NODE_IS_DNAME 0x000025F8
#define DNS_ERROR_DNAME_COLLISION 0x000025F9
#define DNS_ERROR_ALIAS_LOOP 0x000025FA
#define DNS_INFO_AXFR_COMPLETE 0x00002617
#define DNS_ERROR_AXFR 0x00002618
#define DNS_INFO_ADDED_LOCAL_WINS 0x00002619
#define DNS_STATUS_CONTINUE_NEEDED 0x00002649
#define DNS_ERROR_NO_TCPIP 0x0000267B
#define DNS_ERROR_NO_DNS_SERVERS 0x0000267C
#define DNS_ERROR_DP_DOES_NOT_EXIST 0x000026AD
#define DNS_ERROR_DP_ALREADY_EXISTS 0x000026AE
#define DNS_ERROR_DP_NOT_ENLISTED 0x000026AF
#define DNS_ERROR_DP_ALREADY_ENLISTED 0x000026B0
#define DNS_ERROR_DP_NOT_AVAILABLE 0x000026B1
#define DNS_ERROR_DP_FSMO_ERROR 0x000026B2
#define WSAEINTR 0x00002714
#define WSAEBADF 0x00002719
#define WSAEACCES 0x0000271D
#define WSAEFAULT 0x0000271E
#define WSAEINVAL 0x00002726
#define WSAEMFILE 0x00002728
#define WSAEWOULDBLOCK 0x00002733
#define WSAEINPROGRESS 0x00002734
#define WSAEALREADY 0x00002735
#define WSAENOTSOCK 0x00002736
#define WSAEDESTADDRREQ 0x00002737
#define WSAEMSGSIZE 0x00002738
#define WSAEPROTOTYPE 0x00002739
#define WSAENOPROTOOPT 0x0000273A
#define WSAEPROTONOSUPPORT 0x0000273B
#define WSAESOCKTNOSUPPORT 0x0000273C
#define WSAEOPNOTSUPP 0x0000273D
#define WSAEPFNOSUPPORT 0x0000273E
#define WSAEAFNOSUPPORT 0x0000273F
#define WSAEADDRINUSE 0x00002740
#define WSAEADDRNOTAVAIL 0x00002741
#define WSAENETDOWN 0x00002742
#define WSAENETUNREACH 0x00002743
#define WSAENETRESET 0x00002744
#define WSAECONNABORTED 0x00002745
#define WSAECONNRESET 0x00002746
#define WSAENOBUFS 0x00002747
#define WSAEISCONN 0x00002748
#define WSAENOTCONN 0x00002749
#define WSAESHUTDOWN 0x0000274A
#define WSAETOOMANYREFS 0x0000274B
#define WSAETIMEDOUT 0x0000274C
#define WSAECONNREFUSED 0x0000274D
#define WSAELOOP 0x0000274E
#define WSAENAMETOOLONG 0x0000274F
#define WSAEHOSTDOWN 0x00002750
#define WSAEHOSTUNREACH 0x00002751
#define WSAENOTEMPTY 0x00002752
#define WSAEPROCLIM 0x00002753
#define WSAEUSERS 0x00002754
#define WSAEDQUOT 0x00002755
#define WSAESTALE 0x00002756
#define WSAEREMOTE 0x00002757
#define WSASYSNOTREADY 0x0000276B
#define WSAVERNOTSUPPORTED 0x0000276C
#define WSANOTINITIALISED 0x0000276D
#define WSAEDISCON 0x00002775
#define WSAENOMORE 0x00002776
#define WSAECANCELLED 0x00002777
#define WSAEINVALIDPROCTABLE 0x00002778
#define WSAEINVALIDPROVIDER 0x00002779
#define WSAEPROVIDERFAILEDINIT 0x0000277A
#define WSASYSCALLFAILURE 0x0000277B
#define WSASERVICE_NOT_FOUND 0x0000277C
#define WSATYPE_NOT_FOUND 0x0000277D
#define WSA_E_NO_MORE 0x0000277E
#define WSA_E_CANCELLED 0x0000277F
#define WSAEREFUSED 0x00002780
#define WSAHOST_NOT_FOUND 0x00002AF9
#define WSATRY_AGAIN 0x00002AFA
#define WSANO_RECOVERY 0x00002AFB
#define WSANO_DATA 0x00002AFC
#define WSA_QOS_RECEIVERS 0x00002AFD
#define WSA_QOS_SENDERS 0x00002AFE
#define WSA_QOS_NO_SENDERS 0x00002AFF
#define WSA_QOS_NO_RECEIVERS 0x00002B00
#define WSA_QOS_REQUEST_CONFIRMED 0x00002B01
#define WSA_QOS_ADMISSION_FAILURE 0x00002B02
#define WSA_QOS_POLICY_FAILURE 0x00002B03
#define WSA_QOS_BAD_STYLE 0x00002B04
#define WSA_QOS_BAD_OBJECT 0x00002B05
#define WSA_QOS_TRAFFIC_CTRL_ERROR 0x00002B06
#define WSA_QOS_GENERIC_ERROR 0x00002B07
#define WSA_QOS_ESERVICETYPE 0x00002B08
#define WSA_QOS_EFLOWSPEC 0x00002B09
#define WSA_QOS_EPROVSPECBUF 0x00002B0A
#define WSA_QOS_EFILTERSTYLE 0x00002B0B
#define WSA_QOS_EFILTERTYPE 0x00002B0C
#define WSA_QOS_EFILTERCOUNT 0x00002B0D
#define WSA_QOS_EOBJLENGTH 0x00002B0E
#define WSA_QOS_EFLOWCOUNT 0x00002B0F
#define WSA_QOS_EUNKOWNPSOBJ 0x00002B10
#define WSA_QOS_EPOLICYOBJ 0x00002B11
#define WSA_QOS_EFLOWDESC 0x00002B12
#define WSA_QOS_EPSFLOWSPEC 0x00002B13
#define WSA_QOS_EPSFILTERSPEC 0x00002B14
#define WSA_QOS_ESDMODEOBJ 0x00002B15
#define WSA_QOS_ESHAPERATEOBJ 0x00002B16
#define WSA_QOS_RESERVED_PETYPE 0x00002B17
#define WSA_SECURE_HOST_NOT_FOUND 0x00002B18
#define WSA_IPSEC_NAME_POLICY_ERROR 0x00002B19

/* System Error Codes (12000-15999) */

/* ERROR_INTERNET_* : (12000 - 12175) defined in WinInet.h */

#define ERROR_IPSEC_QM_POLICY_EXISTS 0x000032C8
#define ERROR_IPSEC_QM_POLICY_NOT_FOUND 0x000032C9
#define ERROR_IPSEC_QM_POLICY_IN_USE 0x000032CA
#define ERROR_IPSEC_MM_POLICY_EXISTS 0x000032CB
#define ERROR_IPSEC_MM_POLICY_NOT_FOUND 0x000032CC
#define ERROR_IPSEC_MM_POLICY_IN_USE 0x000032CD
#define ERROR_IPSEC_MM_FILTER_EXISTS 0x000032CE
#define ERROR_IPSEC_MM_FILTER_NOT_FOUND 0x000032CF
#define ERROR_IPSEC_TRANSPORT_FILTER_EXISTS 0x000032D0
#define ERROR_IPSEC_TRANSPORT_FILTER_NOT_FOUND 0x000032D1
#define ERROR_IPSEC_MM_AUTH_EXISTS 0x000032D2
#define ERROR_IPSEC_MM_AUTH_NOT_FOUND 0x000032D3
#define ERROR_IPSEC_MM_AUTH_IN_USE 0x000032D4
#define ERROR_IPSEC_DEFAULT_MM_POLICY_NOT_FOUND 0x000032D5
#define ERROR_IPSEC_DEFAULT_MM_AUTH_NOT_FOUND 0x000032D6
#define ERROR_IPSEC_DEFAULT_QM_POLICY_NOT_FOUND 0x000032D7
#define ERROR_IPSEC_TUNNEL_FILTER_EXISTS 0x000032D8
#define ERROR_IPSEC_TUNNEL_FILTER_NOT_FOUND 0x000032D9
#define ERROR_IPSEC_MM_FILTER_PENDING_DELETION 0x000032DA
#define ERROR_IPSEC_TRANSPORT_FILTER_PENDING_DELETION 0x000032DB
#define ERROR_IPSEC_TUNNEL_FILTER_PENDING_DELETION 0x000032DC
#define ERROR_IPSEC_MM_POLICY_PENDING_DELETION 0x000032DD
#define ERROR_IPSEC_MM_AUTH_PENDING_DELETION 0x000032DE
#define ERROR_IPSEC_QM_POLICY_PENDING_DELETION 0x000032DF
#define WARNING_IPSEC_MM_POLICY_PRUNED 0x000032E0
#define WARNING_IPSEC_QM_POLICY_PRUNED 0x000032E1
#define ERROR_IPSEC_IKE_NEG_STATUS_BEGIN 0x000035E8
#define ERROR_IPSEC_IKE_AUTH_FAIL 0x000035E9
#define ERROR_IPSEC_IKE_ATTRIB_FAIL 0x000035EA
#define ERROR_IPSEC_IKE_NEGOTIATION_PENDING 0x000035EB
#define ERROR_IPSEC_IKE_GENERAL_PROCESSING_ERROR 0x000035EC
#define ERROR_IPSEC_IKE_TIMED_OUT 0x000035ED
#define ERROR_IPSEC_IKE_NO_CERT 0x000035EE
#define ERROR_IPSEC_IKE_SA_DELETED 0x000035EF
#define ERROR_IPSEC_IKE_SA_REAPED 0x000035F0
#define ERROR_IPSEC_IKE_MM_ACQUIRE_DROP 0x000035F1
#define ERROR_IPSEC_IKE_QM_ACQUIRE_DROP 0x000035F2
#define ERROR_IPSEC_IKE_QUEUE_DROP_MM 0x000035F3
#define ERROR_IPSEC_IKE_QUEUE_DROP_NO_MM 0x000035F4
#define ERROR_IPSEC_IKE_DROP_NO_RESPONSE 0x000035F5
#define ERROR_IPSEC_IKE_MM_DELAY_DROP 0x000035F6
#define ERROR_IPSEC_IKE_QM_DELAY_DROP 0x000035F7
#define ERROR_IPSEC_IKE_ERROR 0x000035F8
#define ERROR_IPSEC_IKE_CRL_FAILED 0x000035F9
#define ERROR_IPSEC_IKE_INVALID_KEY_USAGE 0x000035FA
#define ERROR_IPSEC_IKE_INVALID_CERT_TYPE 0x000035FB
#define ERROR_IPSEC_IKE_NO_PRIVATE_KEY 0x000035FC
#define ERROR_IPSEC_IKE_SIMULTANEOUS_REKEY 0x000035FD
#define ERROR_IPSEC_IKE_DH_FAIL 0x000035FE
#define ERROR_IPSEC_IKE_CRITICAL_PAYLOAD_NOT_RECOGNIZED 0x000035FF
#define ERROR_IPSEC_IKE_INVALID_HEADER 0x00003600
#define ERROR_IPSEC_IKE_NO_POLICY 0x00003601
#define ERROR_IPSEC_IKE_INVALID_SIGNATURE 0x00003602
#define ERROR_IPSEC_IKE_KERBEROS_ERROR 0x00003603
#define ERROR_IPSEC_IKE_NO_PUBLIC_KEY 0x00003604
#define ERROR_IPSEC_IKE_PROCESS_ERR 0x00003605
#define ERROR_IPSEC_IKE_PROCESS_ERR_SA 0x00003606
#define ERROR_IPSEC_IKE_PROCESS_ERR_PROP 0x00003607
#define ERROR_IPSEC_IKE_PROCESS_ERR_TRANS 0x00003608
#define ERROR_IPSEC_IKE_PROCESS_ERR_KE 0x00003609
#define ERROR_IPSEC_IKE_PROCESS_ERR_ID 0x0000360A
#define ERROR_IPSEC_IKE_PROCESS_ERR_CERT 0x0000360B
#define ERROR_IPSEC_IKE_PROCESS_ERR_CERT_REQ 0x0000360C
#define ERROR_IPSEC_IKE_PROCESS_ERR_HASH 0x0000360D
#define ERROR_IPSEC_IKE_PROCESS_ERR_SIG 0x0000360E
#define ERROR_IPSEC_IKE_PROCESS_ERR_NONCE 0x0000360F
#define ERROR_IPSEC_IKE_PROCESS_ERR_NOTIFY 0x00003610
#define ERROR_IPSEC_IKE_PROCESS_ERR_DELETE 0x00003611
#define ERROR_IPSEC_IKE_PROCESS_ERR_VENDOR 0x00003612
#define ERROR_IPSEC_IKE_INVALID_PAYLOAD 0x00003613
#define ERROR_IPSEC_IKE_LOAD_SOFT_SA 0x00003614
#define ERROR_IPSEC_IKE_SOFT_SA_TORN_DOWN 0x00003615
#define ERROR_IPSEC_IKE_INVALID_COOKIE 0x00003616
#define ERROR_IPSEC_IKE_NO_PEER_CERT 0x00003617
#define ERROR_IPSEC_IKE_PEER_CRL_FAILED 0x00003618
#define ERROR_IPSEC_IKE_POLICY_CHANGE 0x00003619
#define ERROR_IPSEC_IKE_NO_MM_POLICY 0x0000361A
#define ERROR_IPSEC_IKE_NOTCBPRIV 0x0000361B
#define ERROR_IPSEC_IKE_SECLOADFAIL 0x0000361C
#define ERROR_IPSEC_IKE_FAILSSPINIT 0x0000361D
#define ERROR_IPSEC_IKE_FAILQUERYSSP 0x0000361E
#define ERROR_IPSEC_IKE_SRVACQFAIL 0x0000361F
#define ERROR_IPSEC_IKE_SRVQUERYCRED 0x00003620
#define ERROR_IPSEC_IKE_GETSPIFAIL 0x00003621
#define ERROR_IPSEC_IKE_INVALID_FILTER 0x00003622
#define ERROR_IPSEC_IKE_OUT_OF_MEMORY 0x00003623
#define ERROR_IPSEC_IKE_ADD_UPDATE_KEY_FAILED 0x00003624
#define ERROR_IPSEC_IKE_INVALID_POLICY 0x00003625
#define ERROR_IPSEC_IKE_UNKNOWN_DOI 0x00003626
#define ERROR_IPSEC_IKE_INVALID_SITUATION 0x00003627
#define ERROR_IPSEC_IKE_DH_FAILURE 0x00003628
#define ERROR_IPSEC_IKE_INVALID_GROUP 0x00003629
#define ERROR_IPSEC_IKE_ENCRYPT 0x0000362A
#define ERROR_IPSEC_IKE_DECRYPT 0x0000362B
#define ERROR_IPSEC_IKE_POLICY_MATCH 0x0000362C
#define ERROR_IPSEC_IKE_UNSUPPORTED_ID 0x0000362D
#define ERROR_IPSEC_IKE_INVALID_HASH 0x0000362E
#define ERROR_IPSEC_IKE_INVALID_HASH_ALG 0x0000362F
#define ERROR_IPSEC_IKE_INVALID_HASH_SIZE 0x00003630
#define ERROR_IPSEC_IKE_INVALID_ENCRYPT_ALG 0x00003631
#define ERROR_IPSEC_IKE_INVALID_AUTH_ALG 0x00003632
#define ERROR_IPSEC_IKE_INVALID_SIG 0x00003633
#define ERROR_IPSEC_IKE_LOAD_FAILED 0x00003634
#define ERROR_IPSEC_IKE_RPC_DELETE 0x00003635
#define ERROR_IPSEC_IKE_BENIGN_REINIT 0x00003636
#define ERROR_IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY 0x00003637
#define ERROR_IPSEC_IKE_INVALID_MAJOR_VERSION 0x00003638
#define ERROR_IPSEC_IKE_INVALID_CERT_KEYLEN 0x00003639
#define ERROR_IPSEC_IKE_MM_LIMIT 0x0000363A
#define ERROR_IPSEC_IKE_NEGOTIATION_DISABLED 0x0000363B
#define ERROR_IPSEC_IKE_QM_LIMIT 0x0000363C
#define ERROR_IPSEC_IKE_MM_EXPIRED 0x0000363D
#define ERROR_IPSEC_IKE_PEER_MM_ASSUMED_INVALID 0x0000363E
#define ERROR_IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH 0x0000363F
#define ERROR_IPSEC_IKE_UNEXPECTED_MESSAGE_ID 0x00003640
#define ERROR_IPSEC_IKE_INVALID_AUTH_PAYLOAD 0x00003641
#define ERROR_IPSEC_IKE_DOS_COOKIE_SENT 0x00003642
#define ERROR_IPSEC_IKE_SHUTTING_DOWN 0x00003643
#define ERROR_IPSEC_IKE_CGA_AUTH_FAILED 0x00003644
#define ERROR_IPSEC_IKE_PROCESS_ERR_NATOA 0x00003645
#define ERROR_IPSEC_IKE_INVALID_MM_FOR_QM 0x00003646
#define ERROR_IPSEC_IKE_QM_EXPIRED 0x00003647
#define ERROR_IPSEC_IKE_TOO_MANY_FILTERS 0x00003648
#define ERROR_IPSEC_IKE_NEG_STATUS_END 0x00003649
#define ERROR_IPSEC_IKE_KILL_DUMMY_NAP_TUNNEL 0x0000364A
#define ERROR_IPSEC_IKE_INNER_IP_ASSIGNMENT_FAILURE 0x0000364B
#define ERROR_IPSEC_IKE_REQUIRE_CP_PAYLOAD_MISSING 0x0000364C
#define ERROR_IPSEC_KEY_MODULE_IMPERSONATION_NEGOTIATION_PENDING 0x0000364D
#define ERROR_IPSEC_IKE_COEXISTENCE_SUPPRESS 0x0000364E
#define ERROR_IPSEC_IKE_RATELIMIT_DROP 0x0000364F
#define ERROR_IPSEC_IKE_PEER_DOESNT_SUPPORT_MOBIKE 0x00003650
#define ERROR_IPSEC_IKE_AUTHORIZATION_FAILURE 0x00003651
#define ERROR_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_FAILURE 0x00003652
#define ERROR_IPSEC_IKE_AUTHORIZATION_FAILURE_WITH_OPTIONAL_RETRY 0x00003653
#define ERROR_IPSEC_IKE_STRONG_CRED_AUTHORIZATION_AND_CERTMAP_FAILURE 0x00003654
#define ERROR_IPSEC_IKE_NEG_STATUS_EXTENDED_END 0x00003655
#define ERROR_IPSEC_BAD_SPI 0x00003656
#define ERROR_IPSEC_SA_LIFETIME_EXPIRED 0x00003657
#define ERROR_IPSEC_WRONG_SA 0x00003658
#define ERROR_IPSEC_REPLAY_CHECK_FAILED 0x00003659
#define ERROR_IPSEC_INVALID_PACKET 0x0000365A
#define ERROR_IPSEC_INTEGRITY_CHECK_FAILED 0x0000365B
#define ERROR_IPSEC_CLEAR_TEXT_DROP 0x0000365C
#define ERROR_IPSEC_AUTH_FIREWALL_DROP 0x0000365D
#define ERROR_IPSEC_THROTTLE_DROP 0x0000365E
#define ERROR_IPSEC_DOSP_BLOCK 0x00003665
#define ERROR_IPSEC_DOSP_RECEIVED_MULTICAST 0x00003666
#define ERROR_IPSEC_DOSP_INVALID_PACKET 0x00003667
#define ERROR_IPSEC_DOSP_STATE_LOOKUP_FAILED 0x00003668
#define ERROR_IPSEC_DOSP_MAX_ENTRIES 0x00003669
#define ERROR_IPSEC_DOSP_KEYMOD_NOT_ALLOWED 0x0000366A
#define ERROR_IPSEC_DOSP_NOT_INSTALLED 0x0000366B
#define ERROR_IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES 0x0000366C
#define ERROR_SXS_SECTION_NOT_FOUND 0x000036B0
#define ERROR_SXS_CANT_GEN_ACTCTX 0x000036B1
#define ERROR_SXS_INVALID_ACTCTXDATA_FORMAT 0x000036B2
#define ERROR_SXS_ASSEMBLY_NOT_FOUND 0x000036B3
#define ERROR_SXS_MANIFEST_FORMAT_ERROR 0x000036B4
#define ERROR_SXS_MANIFEST_PARSE_ERROR 0x000036B5
#define ERROR_SXS_ACTIVATION_CONTEXT_DISABLED 0x000036B6
#define ERROR_SXS_KEY_NOT_FOUND 0x000036B7
#define ERROR_SXS_VERSION_CONFLICT 0x000036B8
#define ERROR_SXS_WRONG_SECTION_TYPE 0x000036B9
#define ERROR_SXS_THREAD_QUERIES_DISABLED 0x000036BA
#define ERROR_SXS_PROCESS_DEFAULT_ALREADY_SET 0x000036BB
#define ERROR_SXS_UNKNOWN_ENCODING_GROUP 0x000036BC
#define ERROR_SXS_UNKNOWN_ENCODING 0x000036BD
#define ERROR_SXS_INVALID_XML_NAMESPACE_URI 0x000036BE
#define ERROR_SXS_ROOT_MANIFEST_DEPENDENCY_NOT_INSTALLED 0x000036BF
#define ERROR_SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED 0x000036C0
#define ERROR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE 0x000036C1
#define ERROR_SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE 0x000036C2
#define ERROR_SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE 0x000036C3
#define ERROR_SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT 0x000036C4
#define ERROR_SXS_DUPLICATE_DLL_NAME 0x000036C5
#define ERROR_SXS_DUPLICATE_WINDOWCLASS_NAME 0x000036C6
#define ERROR_SXS_DUPLICATE_CLSID 0x000036C7
#define ERROR_SXS_DUPLICATE_IID 0x000036C8
#define ERROR_SXS_DUPLICATE_TLBID 0x000036C9
#define ERROR_SXS_DUPLICATE_PROGID 0x000036CA
#define ERROR_SXS_DUPLICATE_ASSEMBLY_NAME 0x000036CB
#define ERROR_SXS_FILE_HASH_MISMATCH 0x000036CC
#define ERROR_SXS_POLICY_PARSE_ERROR 0x000036CD
#define ERROR_SXS_XML_E_MISSINGQUOTE 0x000036CE
#define ERROR_SXS_XML_E_COMMENTSYNTAX 0x000036CF
#define ERROR_SXS_XML_E_BADSTARTNAMECHAR 0x000036D0
#define ERROR_SXS_XML_E_BADNAMECHAR 0x000036D1
#define ERROR_SXS_XML_E_BADCHARINSTRING 0x000036D2
#define ERROR_SXS_XML_E_XMLDECLSYNTAX 0x000036D3
#define ERROR_SXS_XML_E_BADCHARDATA 0x000036D4
#define ERROR_SXS_XML_E_MISSINGWHITESPACE 0x000036D5
#define ERROR_SXS_XML_E_EXPECTINGTAGEND 0x000036D6
#define ERROR_SXS_XML_E_MISSINGSEMICOLON 0x000036D7
#define ERROR_SXS_XML_E_UNBALANCEDPAREN 0x000036D8
#define ERROR_SXS_XML_E_INTERNALERROR 0x000036D9
#define ERROR_SXS_XML_E_UNEXPECTED_WHITESPACE 0x000036DA
#define ERROR_SXS_XML_E_INCOMPLETE_ENCODING 0x000036DB
#define ERROR_SXS_XML_E_MISSING_PAREN 0x000036DC
#define ERROR_SXS_XML_E_EXPECTINGCLOSEQUOTE 0x000036DD
#define ERROR_SXS_XML_E_MULTIPLE_COLONS 0x000036DE
#define ERROR_SXS_XML_E_INVALID_DECIMAL 0x000036DF
#define ERROR_SXS_XML_E_INVALID_HEXIDECIMAL 0x000036E0
#define ERROR_SXS_XML_E_INVALID_UNICODE 0x000036E1
#define ERROR_SXS_XML_E_WHITESPACEORQUESTIONMARK 0x000036E2
#define ERROR_SXS_XML_E_UNEXPECTEDENDTAG 0x000036E3
#define ERROR_SXS_XML_E_UNCLOSEDTAG 0x000036E4
#define ERROR_SXS_XML_E_DUPLICATEATTRIBUTE 0x000036E5
#define ERROR_SXS_XML_E_MULTIPLEROOTS 0x000036E6
#define ERROR_SXS_XML_E_INVALIDATROOTLEVEL 0x000036E7
#define ERROR_SXS_XML_E_BADXMLDECL 0x000036E8
#define ERROR_SXS_XML_E_MISSINGROOT 0x000036E9
#define ERROR_SXS_XML_E_UNEXPECTEDEOF 0x000036EA
#define ERROR_SXS_XML_E_BADPEREFINSUBSET 0x000036EB
#define ERROR_SXS_XML_E_UNCLOSEDSTARTTAG 0x000036EC
#define ERROR_SXS_XML_E_UNCLOSEDENDTAG 0x000036ED
#define ERROR_SXS_XML_E_UNCLOSEDSTRING 0x000036EE
#define ERROR_SXS_XML_E_UNCLOSEDCOMMENT 0x000036EF
#define ERROR_SXS_XML_E_UNCLOSEDDECL 0x000036F0
#define ERROR_SXS_XML_E_UNCLOSEDCDATA 0x000036F1
#define ERROR_SXS_XML_E_RESERVEDNAMESPACE 0x000036F2
#define ERROR_SXS_XML_E_INVALIDENCODING 0x000036F3
#define ERROR_SXS_XML_E_INVALIDSWITCH 0x000036F4
#define ERROR_SXS_XML_E_BADXMLCASE 0x000036F5
#define ERROR_SXS_XML_E_INVALID_STANDALONE 0x000036F6
#define ERROR_SXS_XML_E_UNEXPECTED_STANDALONE 0x000036F7
#define ERROR_SXS_XML_E_INVALID_VERSION 0x000036F8
#define ERROR_SXS_XML_E_MISSINGEQUALS 0x000036F9
#define ERROR_SXS_PROTECTION_RECOVERY_FAILED 0x000036FA
#define ERROR_SXS_PROTECTION_PUBLIC_KEY_TOO_SHORT 0x000036FB
#define ERROR_SXS_PROTECTION_CATALOG_NOT_VALID 0x000036FC
#define ERROR_SXS_UNTRANSLATABLE_HRESULT 0x000036FD
#define ERROR_SXS_PROTECTION_CATALOG_FILE_MISSING 0x000036FE
#define ERROR_SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE 0x000036FF
#define ERROR_SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME 0x00003700
#define ERROR_SXS_ASSEMBLY_MISSING 0x00003701
#define ERROR_SXS_CORRUPT_ACTIVATION_STACK 0x00003702
#define ERROR_SXS_CORRUPTION 0x00003703
#define ERROR_SXS_EARLY_DEACTIVATION 0x00003704
#define ERROR_SXS_INVALID_DEACTIVATION 0x00003705
#define ERROR_SXS_MULTIPLE_DEACTIVATION 0x00003706
#define ERROR_SXS_PROCESS_TERMINATION_REQUESTED 0x00003707
#define ERROR_SXS_RELEASE_ACTIVATION_CONTEXT 0x00003708
#define ERROR_SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY 0x00003709
#define ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE 0x0000370A
#define ERROR_SXS_INVALID_IDENTITY_ATTRIBUTE_NAME 0x0000370B
#define ERROR_SXS_IDENTITY_DUPLICATE_ATTRIBUTE 0x0000370C
#define ERROR_SXS_IDENTITY_PARSE_ERROR 0x0000370D
#define ERROR_MALFORMED_SUBSTITUTION_STRING 0x0000370E
#define ERROR_SXS_INCORRECT_PUBLIC_KEY_TOKEN 0x0000370F
#define ERROR_UNMAPPED_SUBSTITUTION_STRING 0x00003710
#define ERROR_SXS_ASSEMBLY_NOT_LOCKED 0x00003711
#define ERROR_SXS_COMPONENT_STORE_CORRUPT 0x00003712
#define ERROR_ADVANCED_INSTALLER_FAILED 0x00003713
#define ERROR_XML_ENCODING_MISMATCH 0x00003714
#define ERROR_SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT 0x00003715
#define ERROR_SXS_IDENTITIES_DIFFERENT 0x00003716
#define ERROR_SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT 0x00003717
#define ERROR_SXS_FILE_NOT_PART_OF_ASSEMBLY 0x00003718
#define ERROR_SXS_MANIFEST_TOO_BIG 0x00003719
#define ERROR_SXS_SETTING_NOT_REGISTERED 0x0000371A
#define ERROR_SXS_TRANSACTION_CLOSURE_INCOMPLETE 0x0000371B
#define ERROR_SMI_PRIMITIVE_INSTALLER_FAILED 0x0000371C
#define ERROR_GENERIC_COMMAND_FAILED 0x0000371D
#define ERROR_SXS_FILE_HASH_MISSING 0x0000371E
#define ERROR_EVT_INVALID_CHANNEL_PATH 0x00003A98
#define ERROR_EVT_INVALID_QUERY 0x00003A99
#define ERROR_EVT_PUBLISHER_METADATA_NOT_FOUND 0x00003A9A
#define ERROR_EVT_EVENT_TEMPLATE_NOT_FOUND 0x00003A9B
#define ERROR_EVT_INVALID_PUBLISHER_NAME 0x00003A9C
#define ERROR_EVT_INVALID_EVENT_DATA 0x00003A9D
#define ERROR_EVT_CHANNEL_NOT_FOUND 0x00003A9F
#define ERROR_EVT_MALFORMED_XML_TEXT 0x00003AA0
#define ERROR_EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL 0x00003AA1
#define ERROR_EVT_CONFIGURATION_ERROR 0x00003AA2
#define ERROR_EVT_QUERY_RESULT_STALE 0x00003AA3
#define ERROR_EVT_QUERY_RESULT_INVALID_POSITION 0x00003AA4
#define ERROR_EVT_NON_VALIDATING_MSXML 0x00003AA5
#define ERROR_EVT_FILTER_ALREADYSCOPED 0x00003AA6
#define ERROR_EVT_FILTER_NOTELTSET 0x00003AA7
#define ERROR_EVT_FILTER_INVARG 0x00003AA8
#define ERROR_EVT_FILTER_INVTEST 0x00003AA9
#define ERROR_EVT_FILTER_INVTYPE 0x00003AAA
#define ERROR_EVT_FILTER_PARSEERR 0x00003AAB
#define ERROR_EVT_FILTER_UNSUPPORTEDOP 0x00003AAC
#define ERROR_EVT_FILTER_UNEXPECTEDTOKEN 0x00003AAD
#define ERROR_EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL 0x00003AAE
#define ERROR_EVT_INVALID_CHANNEL_PROPERTY_VALUE 0x00003AAF
#define ERROR_EVT_INVALID_PUBLISHER_PROPERTY_VALUE 0x00003AB0
#define ERROR_EVT_CHANNEL_CANNOT_ACTIVATE 0x00003AB1
#define ERROR_EVT_FILTER_TOO_COMPLEX 0x00003AB2
#define ERROR_EVT_MESSAGE_NOT_FOUND 0x00003AB3
#define ERROR_EVT_MESSAGE_ID_NOT_FOUND 0x00003AB4
#define ERROR_EVT_UNRESOLVED_VALUE_INSERT 0x00003AB5
#define ERROR_EVT_UNRESOLVED_PARAMETER_INSERT 0x00003AB6
#define ERROR_EVT_MAX_INSERTS_REACHED 0x00003AB7
#define ERROR_EVT_EVENT_DEFINITION_NOT_FOUND 0x00003AB8
#define ERROR_EVT_MESSAGE_LOCALE_NOT_FOUND 0x00003AB9
#define ERROR_EVT_VERSION_TOO_OLD 0x00003ABA
#define ERROR_EVT_VERSION_TOO_NEW 0x00003ABB
#define ERROR_EVT_CANNOT_OPEN_CHANNEL_OF_QUERY 0x00003ABC
#define ERROR_EVT_PUBLISHER_DISABLED 0x00003ABD
#define ERROR_EVT_FILTER_OUT_OF_RANGE 0x00003ABE
#define ERROR_EC_SUBSCRIPTION_CANNOT_ACTIVATE 0x00003AE8
#define ERROR_EC_LOG_DISABLED 0x00003AE9
#define ERROR_EC_CIRCULAR_FORWARDING 0x00003AEA
#define ERROR_EC_CREDSTORE_FULL 0x00003AEB
#define ERROR_EC_CRED_NOT_FOUND 0x00003AEC
#define ERROR_EC_NO_ACTIVE_CHANNEL 0x00003AED
#define ERROR_MUI_FILE_NOT_FOUND 0x00003AFC
#define ERROR_MUI_INVALID_FILE 0x00003AFD
#define ERROR_MUI_INVALID_RC_CONFIG 0x00003AFE
#define ERROR_MUI_INVALID_LOCALE_NAME 0x00003AFF
#define ERROR_MUI_INVALID_ULTIMATEFALLBACK_NAME 0x00003B00
#define ERROR_MUI_FILE_NOT_LOADED 0x00003B01
#define ERROR_RESOURCE_ENUM_USER_STOP 0x00003B02
#define ERROR_MUI_INTLSETTINGS_UILANG_NOT_INSTALLED 0x00003B03
#define ERROR_MUI_INTLSETTINGS_INVALID_LOCALE_NAME 0x00003B04
#define ERROR_MRM_RUNTIME_NO_DEFAULT_OR_NEUTRAL_RESOURCE 0x00003B06
#define ERROR_MRM_INVALID_PRICONFIG 0x00003B07
#define ERROR_MRM_INVALID_FILE_TYPE 0x00003B08
#define ERROR_MRM_UNKNOWN_QUALIFIER 0x00003B09
#define ERROR_MRM_INVALID_QUALIFIER_VALUE 0x00003B0A
#define ERROR_MRM_NO_CANDIDATE 0x00003B0B
#define ERROR_MRM_NO_MATCH_OR_DEFAULT_CANDIDATE 0x00003B0C
#define ERROR_MRM_RESOURCE_TYPE_MISMATCH 0x00003B0D
#define ERROR_MRM_DUPLICATE_MAP_NAME 0x00003B0E
#define ERROR_MRM_DUPLICATE_ENTRY 0x00003B0F
#define ERROR_MRM_INVALID_RESOURCE_IDENTIFIER 0x00003B10
#define ERROR_MRM_FILEPATH_TOO_LONG 0x00003B11
#define ERROR_MRM_UNSUPPORTED_DIRECTORY_TYPE 0x00003B12
#define ERROR_MRM_INVALID_PRI_FILE 0x00003B16
#define ERROR_MRM_NAMED_RESOURCE_NOT_FOUND 0x00003B17
#define ERROR_MRM_MAP_NOT_FOUND 0x00003B1F
#define ERROR_MRM_UNSUPPORTED_PROFILE_TYPE 0x00003B20
#define ERROR_MRM_INVALID_QUALIFIER_OPERATOR 0x00003B21
#define ERROR_MRM_INDETERMINATE_QUALIFIER_VALUE 0x00003B22
#define ERROR_MRM_AUTOMERGE_ENABLED 0x00003B23
#define ERROR_MRM_TOO_MANY_RESOURCES 0x00003B24
#define ERROR_MCA_INVALID_CAPABILITIES_STRING 0x00003B60
#define ERROR_MCA_INVALID_VCP_VERSION 0x00003B61
#define ERROR_MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION 0x00003B62
#define ERROR_MCA_MCCS_VERSION_MISMATCH 0x00003B63
#define ERROR_MCA_UNSUPPORTED_MCCS_VERSION 0x00003B64
#define ERROR_MCA_INTERNAL_ERROR 0x00003B65
#define ERROR_MCA_INVALID_TECHNOLOGY_TYPE_RETURNED 0x00003B66
#define ERROR_MCA_UNSUPPORTED_COLOR_TEMPERATURE 0x00003B67
#define ERROR_AMBIGUOUS_SYSTEM_DEVICE 0x00003B92
#define ERROR_SYSTEM_DEVICE_NOT_FOUND 0x00003BC3
#define ERROR_HASH_NOT_SUPPORTED 0x00003BC4
#define ERROR_HASH_NOT_PRESENT 0x00003BC5
#define ERROR_SECONDARY_IC_PROVIDER_NOT_REGISTERED 0x00003BD9
#define ERROR_GPIO_CLIENT_INFORMATION_INVALID 0x00003BDA
#define ERROR_GPIO_VERSION_NOT_SUPPORTED 0x00003BDB
#define ERROR_GPIO_INVALID_REGISTRATION_PACKET 0x00003BDC
#define ERROR_GPIO_OPERATION_DENIED 0x00003BDD
#define ERROR_GPIO_INCOMPATIBLE_CONNECT_MODE 0x00003BDE
#define ERROR_GPIO_INTERRUPT_ALREADY_UNMASKED 0x00003BDF
#define ERROR_CANNOT_SWITCH_RUNLEVEL 0x00003C28
#define ERROR_INVALID_RUNLEVEL_SETTING 0x00003C29
#define ERROR_RUNLEVEL_SWITCH_TIMEOUT 0x00003C2A
#define ERROR_RUNLEVEL_SWITCH_AGENT_TIMEOUT 0x00003C2B
#define ERROR_RUNLEVEL_SWITCH_IN_PROGRESS 0x00003C2C
#define ERROR_SERVICES_FAILED_AUTOSTART 0x00003C2D
#define ERROR_COM_TASK_STOP_PENDING 0x00003C8D
#define ERROR_INSTALL_OPEN_PACKAGE_FAILED 0x00003CF0
#define ERROR_INSTALL_PACKAGE_NOT_FOUND 0x00003CF1
#define ERROR_INSTALL_INVALID_PACKAGE 0x00003CF2
#define ERROR_INSTALL_RESOLVE_DEPENDENCY_FAILED 0x00003CF3
#define ERROR_INSTALL_OUT_OF_DISK_SPACE 0x00003CF4
#define ERROR_INSTALL_NETWORK_FAILURE 0x00003CF5
#define ERROR_INSTALL_REGISTRATION_FAILURE 0x00003CF6
#define ERROR_INSTALL_DEREGISTRATION_FAILURE 0x00003CF7
#define ERROR_INSTALL_CANCEL 0x00003CF8
#define ERROR_INSTALL_FAILED 0x00003CF9
#define ERROR_REMOVE_FAILED 0x00003CFA
#define ERROR_PACKAGE_ALREADY_EXISTS 0x00003CFB
#define ERROR_NEEDS_REMEDIATION 0x00003CFC
#define ERROR_INSTALL_PREREQUISITE_FAILED 0x00003CFD
#define ERROR_PACKAGE_REPOSITORY_CORRUPTED 0x00003CFE
#define ERROR_INSTALL_POLICY_FAILURE 0x00003CFF
#define ERROR_PACKAGE_UPDATING 0x00003D00
#define ERROR_DEPLOYMENT_BLOCKED_BY_POLICY 0x00003D01
#define ERROR_PACKAGES_IN_USE 0x00003D02
#define ERROR_RECOVERY_FILE_CORRUPT 0x00003D03
#define ERROR_INVALID_STAGED_SIGNATURE 0x00003D04
#define ERROR_DELETING_EXISTING_APPLICATIONDATA_STORE_FAILED 0x00003D05
#define ERROR_INSTALL_PACKAGE_DOWNGRADE 0x00003D06
#define ERROR_SYSTEM_NEEDS_REMEDIATION 0x00003D07
#define ERROR_APPX_INTEGRITY_FAILURE_CLR_NGEN 0x00003D08
#define ERROR_RESILIENCY_FILE_CORRUPT 0x00003D09
#define ERROR_INSTALL_FIREWALL_SERVICE_NOT_RUNNING 0x00003D0A
#define APPMODEL_ERROR_NO_PACKAGE 0x00003D54
#define APPMODEL_ERROR_PACKAGE_RUNTIME_CORRUPT 0x00003D55
#define APPMODEL_ERROR_PACKAGE_IDENTITY_CORRUPT 0x00003D56
#define APPMODEL_ERROR_NO_APPLICATION 0x00003D57
#define ERROR_STATE_LOAD_STORE_FAILED 0x00003DB8
#define ERROR_STATE_GET_VERSION_FAILED 0x00003DB9
#define ERROR_STATE_SET_VERSION_FAILED 0x00003DBA
#define ERROR_STATE_STRUCTURED_RESET_FAILED 0x00003DBB
#define ERROR_STATE_OPEN_CONTAINER_FAILED 0x00003DBC
#define ERROR_STATE_CREATE_CONTAINER_FAILED 0x00003DBD
#define ERROR_STATE_DELETE_CONTAINER_FAILED 0x00003DBE
#define ERROR_STATE_READ_SETTING_FAILED 0x00003DBF
#define ERROR_STATE_WRITE_SETTING_FAILED 0x00003DC0
#define ERROR_STATE_DELETE_SETTING_FAILED 0x00003DC1
#define ERROR_STATE_QUERY_SETTING_FAILED 0x00003DC2
#define ERROR_STATE_READ_COMPOSITE_SETTING_FAILED 0x00003DC3
#define ERROR_STATE_WRITE_COMPOSITE_SETTING_FAILED 0x00003DC4
#define ERROR_STATE_ENUMERATE_CONTAINER_FAILED 0x00003DC5
#define ERROR_STATE_ENUMERATE_SETTINGS_FAILED 0x00003DC6
#define ERROR_STATE_COMPOSITE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED 0x00003DC7
#define ERROR_STATE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED 0x00003DC8
#define ERROR_STATE_SETTING_NAME_SIZE_LIMIT_EXCEEDED 0x00003DC9
#define ERROR_STATE_CONTAINER_NAME_SIZE_LIMIT_EXCEEDED 0x00003DCA
#define ERROR_API_UNAVAILABLE 0x00003DE1

#ifndef FACILITY_WEBSERVICES
#define FACILITY_WEBSERVICES 61
#define WS_S_ASYNC 0x003D0000
#define WS_S_END 0x003D0001
#define WS_E_INVALID_FORMAT 0x803D0000
#define WS_E_OBJECT_FAULTED 0x803D0001
#define WS_E_NUMERIC_OVERFLOW 0x803D0002
#define WS_E_INVALID_OPERATION 0x803D0003
#define WS_E_OPERATION_ABORTED 0x803D0004
#define WS_E_ENDPOINT_ACCESS_DENIED 0x803D0005
#define WS_E_OPERATION_TIMED_OUT 0x803D0006
#define WS_E_OPERATION_ABANDONED 0x803D0007
#define WS_E_QUOTA_EXCEEDED 0x803D0008
#define WS_E_NO_TRANSLATION_AVAILABLE 0x803D0009
#define WS_E_SECURITY_VERIFICATION_FAILURE 0x803D000A
#define WS_E_ADDRESS_IN_USE 0x803D000B
#define WS_E_ADDRESS_NOT_AVAILABLE 0x803D000C
#define WS_E_ENDPOINT_NOT_FOUND 0x803D000D
#define WS_E_ENDPOINT_NOT_AVAILABLE 0x803D000E
#define WS_E_ENDPOINT_FAILURE 0x803D000F
#define WS_E_ENDPOINT_UNREACHABLE 0x803D0010
#define WS_E_ENDPOINT_ACTION_NOT_SUPPORTED 0x803D0011
#define WS_E_ENDPOINT_TOO_BUSY 0x803D0012
#define WS_E_ENDPOINT_FAULT_RECEIVED 0x803D0013
#define WS_E_ENDPOINT_DISCONNECTED 0x803D0014
#define WS_E_PROXY_FAILURE 0x803D0015
#define WS_E_PROXY_ACCESS_DENIED 0x803D0016
#define WS_E_NOT_SUPPORTED 0x803D0017
#define WS_E_PROXY_REQUIRES_BASIC_AUTH 0x803D0018
#define WS_E_PROXY_REQUIRES_DIGEST_AUTH 0x803D0019
#define WS_E_PROXY_REQUIRES_NTLM_AUTH 0x803D001A
#define WS_E_PROXY_REQUIRES_NEGOTIATE_AUTH 0x803D001B
#define WS_E_SERVER_REQUIRES_BASIC_AUTH 0x803D001C
#define WS_E_SERVER_REQUIRES_DIGEST_AUTH 0x803D001D
#define WS_E_SERVER_REQUIRES_NTLM_AUTH 0x803D001E
#define WS_E_SERVER_REQUIRES_NEGOTIATE_AUTH 0x803D001F
#define WS_E_INVALID_ENDPOINT_URL 0x803D0020
#define WS_E_OTHER 0x803D0021
#define WS_E_SECURITY_TOKEN_EXPIRED 0x803D0022
#define WS_E_SECURITY_SYSTEM_FAILURE 0x803D0023
#endif

#define NTE_BAD_UID WINPR_CXX_COMPAT_CAST(LONG, 0x80090001)
#define NTE_BAD_HASH WINPR_CXX_COMPAT_CAST(LONG, 0x80090002)
#define NTE_BAD_KEY WINPR_CXX_COMPAT_CAST(LONG, 0x80090003)
#define NTE_BAD_LEN WINPR_CXX_COMPAT_CAST(LONG, 0x80090004)
#define NTE_BAD_DATA WINPR_CXX_COMPAT_CAST(LONG, 0x80090005)
#define NTE_BAD_SIGNATURE WINPR_CXX_COMPAT_CAST(LONG, 0x80090006)
#define NTE_BAD_VER WINPR_CXX_COMPAT_CAST(LONG, 0x80090007)
#define NTE_BAD_ALGID WINPR_CXX_COMPAT_CAST(LONG, 0x80090008)
#define NTE_BAD_FLAGS WINPR_CXX_COMPAT_CAST(LONG, 0x80090009)
#define NTE_BAD_TYPE WINPR_CXX_COMPAT_CAST(LONG, 0x8009000A)
#define NTE_BAD_KEY_STATE WINPR_CXX_COMPAT_CAST(LONG, 0x8009000B)
#define NTE_BAD_HASH_STATE WINPR_CXX_COMPAT_CAST(LONG, 0x8009000C)
#define NTE_NO_KEY WINPR_CXX_COMPAT_CAST(LONG, 0x8009000D)
#define NTE_NO_MEMORY WINPR_CXX_COMPAT_CAST(LONG, 0x8009000E)
#define NTE_EXISTS WINPR_CXX_COMPAT_CAST(LONG, 0x8009000F)
#define NTE_PERM WINPR_CXX_COMPAT_CAST(LONG, 0x80090010)
#define NTE_NOT_FOUND WINPR_CXX_COMPAT_CAST(LONG, 0x80090011)
#define NTE_DOUBLE_ENCRYPT WINPR_CXX_COMPAT_CAST(LONG, 0x80090012)
#define NTE_BAD_PROVIDER WINPR_CXX_COMPAT_CAST(LONG, 0x80090013)
#define NTE_BAD_PROV_TYPE WINPR_CXX_COMPAT_CAST(LONG, 0x80090014)
#define NTE_BAD_PUBLIC_KEY WINPR_CXX_COMPAT_CAST(LONG, 0x80090015)
#define NTE_BAD_KEYSET WINPR_CXX_COMPAT_CAST(LONG, 0x80090016)
#define NTE_PROV_TYPE_NOT_DEF WINPR_CXX_COMPAT_CAST(LONG, 0x80090017)
#define NTE_PROV_TYPE_ENTRY_BAD WINPR_CXX_COMPAT_CAST(LONG, 0x80090018)
#define NTE_KEYSET_NOT_DEF WINPR_CXX_COMPAT_CAST(LONG, 0x80090019)
#define NTE_KEYSET_ENTRY_BAD WINPR_CXX_COMPAT_CAST(LONG, 0x8009001A)
#define NTE_PROV_TYPE_NO_MATCH WINPR_CXX_COMPAT_CAST(LONG, 0x8009001B)
#define NTE_SIGNATURE_FILE_BAD WINPR_CXX_COMPAT_CAST(LONG, 0x8009001C)
#define NTE_PROVIDER_DLL_FAIL WINPR_CXX_COMPAT_CAST(LONG, 0x8009001D)
#define NTE_PROV_DLL_NOT_FOUND WINPR_CXX_COMPAT_CAST(LONG, 0x8009001E)
#define NTE_BAD_KEYSET_PARAM WINPR_CXX_COMPAT_CAST(LONG, 0x8009001F)
#define NTE_FAIL WINPR_CXX_COMPAT_CAST(LONG, 0x80090020)
#define NTE_SYS_ERR WINPR_CXX_COMPAT_CAST(LONG, 0x80090021)
#define NTE_SILENT_CONTEXT WINPR_CXX_COMPAT_CAST(LONG, 0x80090022)
#define NTE_TOKEN_KEYSET_STORAGE_FULL WINPR_CXX_COMPAT_CAST(LONG, 0x80090023)
#define NTE_TEMPORARY_PROFILE WINPR_CXX_COMPAT_CAST(LONG, 0x80090024)
#define NTE_FIXEDPARAMETER WINPR_CXX_COMPAT_CAST(LONG, 0x80090025)
#define NTE_NO_MORE_ITEMS ERROR_NO_MORE_ITEMS
#define NTE_NOT_SUPPORTED ERROR_NOT_SUPPORTED
#define NTE_INVALID_PARAMETER WINPR_CXX_COMPAT_CAST(LONG, 0x80090027)

#define EXCEPTION_MAXIMUM_PARAMETERS 15

typedef struct s_EXCEPTION_RECORD EXCEPTION_RECORD;
typedef struct s_EXCEPTION_RECORD* PEXCEPTION_RECORD;

struct s_EXCEPTION_RECORD
{
	DWORD ExceptionCode;
	DWORD ExceptionFlags;
	PEXCEPTION_RECORD ExceptionRecord;
	PVOID ExceptionAddress;
	DWORD NumberParameters;
	ULONG_PTR ExceptionInformation[EXCEPTION_MAXIMUM_PARAMETERS];
};

typedef void* PCONTEXT;

typedef struct s_EXCEPTION_POINTERS
{
	PEXCEPTION_RECORD ExceptionRecord;
	PCONTEXT ContextRecord;
} EXCEPTION_POINTERS, *PEXCEPTION_POINTERS;

typedef LONG (*PTOP_LEVEL_EXCEPTION_FILTER)(PEXCEPTION_POINTERS ExceptionInfo);
typedef PTOP_LEVEL_EXCEPTION_FILTER LPTOP_LEVEL_EXCEPTION_FILTER;

typedef LONG (*PVECTORED_EXCEPTION_HANDLER)(PEXCEPTION_POINTERS ExceptionInfo);

#ifdef __cplusplus
extern "C"
{
#endif

	WINPR_API UINT GetErrorMode(void);

	WINPR_API UINT SetErrorMode(UINT uMode);

	WINPR_API DWORD GetLastError(void);

	WINPR_API VOID SetLastError(DWORD dwErrCode);

	WINPR_API VOID RestoreLastError(DWORD dwErrCode);

	WINPR_API VOID RaiseException(DWORD dwExceptionCode, DWORD dwExceptionFlags,
	                              DWORD nNumberOfArguments, CONST ULONG_PTR* lpArguments);

	WINPR_API LONG UnhandledExceptionFilter(PEXCEPTION_POINTERS ExceptionInfo);

	WINPR_API LPTOP_LEVEL_EXCEPTION_FILTER
	SetUnhandledExceptionFilter(LPTOP_LEVEL_EXCEPTION_FILTER lpTopLevelExceptionFilter);

	WINPR_API PVOID AddVectoredExceptionHandler(ULONG First, PVECTORED_EXCEPTION_HANDLER Handler);

	WINPR_API ULONG RemoveVectoredExceptionHandler(PVOID Handle);

	WINPR_API PVOID AddVectoredContinueHandler(ULONG First, PVECTORED_EXCEPTION_HANDLER Handler);

	WINPR_API ULONG RemoveVectoredContinueHandler(PVOID Handle);

#ifdef __cplusplus
}
#endif

#endif

#endif /* WINPR_ERROR_H */
