//  Copyright John Maddock 2014
//  Copyright Matt Borland 2024
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#define BOOST_MATH_OVERFLOW_ERROR_POLICY ignore_error

#include <boost/math/concepts/real_concept.hpp>
#include <boost/math/special_functions/math_fwd.hpp>
#define BOOST_TEST_MAIN
#include <boost/test/unit_test.hpp>
#include <boost/math/special_functions/trigamma.hpp>
#include "../include_private/boost/math/tools/test.hpp"
#include <boost/test/tools/floating_point_comparison.hpp>
#include <boost/math/tools/stats.hpp>
#include <boost/math/constants/constants.hpp>
#include <boost/type_traits/is_floating_point.hpp>
#include <boost/array.hpp>
#include "functor.hpp"

#include "handle_test_result.hpp"
#include "table_type.hpp"

#ifndef SC_
#ifndef BOOST_FLOAT128_C
#define SC_(x) static_cast<typename table_type<T>::type>(BOOST_JOIN(x, L))
#else
#define SC_(x) static_cast<typename table_type<T>::type>(BOOST_FLOAT128_C(x))
#endif
#endif

template <class Real, class T>
void do_test_trigamma(const T& data, const char* type_name, const char* test_name)
{
#if !(defined(ERROR_REPORTING_MODE) && !defined(TRIGAMMA_RATIO_FUNCTION_TO_TEST))
   typedef Real                   value_type;

   typedef value_type(*pg)(value_type);
#ifdef TRIGAMMA_RATIO_FUNCTION_TO_TEST
   pg funcp = TRIGAMMA_RATIO_FUNCTION_TO_TEST;
#elif defined(BOOST_MATH_NO_DEDUCED_FUNCTION_POINTERS)
   pg funcp = boost::math::trigamma<value_type>;
#else
   pg funcp = boost::math::trigamma;
#endif

   boost::math::tools::test_result<value_type> result;

   std::cout << "Testing " << test_name << " with type " << type_name
      << "\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n";

   //
   // test trigamma against data:
   //
   result = boost::math::tools::test_hetero<Real>(
      data,
      bind_func<Real>(funcp, 0),
      extract_result<Real>(1));
   handle_test_result(result, data[result.worst()], result.worst(), type_name, "trigamma", test_name);
   std::cout << std::endl;
#endif
}

template <class T>
void test_trigamma(T, const char* name)
{
   typedef typename table_type<T>::type value_type;

   std::array<std::array<value_type, 3>, 659> data =
   { { 
      {{ SC_(1.0), SC_(1.6449340668482264364724151666460252) }}, {{ SC_(2.0), SC_(0.64493406684822643647241516664602519) }}, {{ SC_(3.0), SC_(0.39493406684822643647241516664602519) }}, {{ SC_(4.0), SC_(0.28382295573711532536130405553491408) }}, {{ SC_(5.0), SC_(0.22132295573711532536130405553491408) }}, {{ SC_(6.0), SC_(0.18132295573711532536130405553491408) }}, {{ SC_(7.0), SC_(0.15354517795933754758352627775713630) }}, {{ SC_(8.0), SC_(0.13313701469403142513454668592040161) }}, {{ SC_(9.0), SC_(0.11751201469403142513454668592040161) }}, {{ SC_(10.0), SC_(0.10516633568168574612220100690805593) }}, {{ SC_(11.0), SC_(0.095166335681685746122201006908055927) }}, {{ SC_(12.0), SC_(0.086901872871768390750300180461774936) }}, {{ SC_(13.0), SC_(0.079957428427323946305855736017330491) }}, {{ SC_(14.0), SC_(0.074040268664010336838400114715555343) }}, {{ SC_(15.0), SC_(0.068938227847683806226155216756371670) }}, {{ SC_(16.0), SC_(0.064493783403239361781710772311927225) }}, {{ SC_(17.0), SC_(0.060587533403239361781710772311927225) }}, {{ SC_(18.0), SC_(0.057127325790782614376866481654487779) }}, {{ SC_(19.0), SC_(0.054040906037696194623780061901401359) }}, {{ SC_(20.0), SC_(0.051270822935203119831536294588381969) }}, {{ SC_(21.0), SC_(0.048770822935203119831536294588381969) }}, {{ SC_(22.0), SC_(0.046503249239057995114983006606522558) }}, {{ SC_(23.0), SC_(0.044437133536578656272007799994952310) }}, {{ SC_(24.0), SC_(0.042546774368336690298472828350339834) }}, {{ SC_(25.0), SC_(0.040810663257225579187361717239228723) }}, {{ SC_(26.0), SC_(0.039210663257225579187361717239228723) }}, {{ SC_(27.0), SC_(0.037731373316397176820497811913784936) }}, {{ SC_(28.0), SC_(0.036359631203914323596903847579079860) }}, {{ SC_(29.0), SC_(0.035084120999832690943842623089283942) }}, {{ SC_(30.0), SC_(0.033895060357739944213759388844337450) }}, {{ SC_(31.0), SC_(0.032783949246628833102648277733226339) }}, {{ SC_(32.0), SC_(0.031743366520302090126581680438741427) }}, {{ SC_(33.0), SC_(0.030766804020302090126581680438741427) }}, {{ SC_(34.0), SC_(0.029848530374755717307481588611376873) }}, {{ SC_(35.0), SC_(0.028983478471641530456270515947017011) }}, {{ SC_(36.0), SC_(0.028167151941029285558311332273547623) }}, {{ SC_(37.0), SC_(0.027395547002757680620039727335276018) }}, {{ SC_(38.0), SC_(0.026665086812838031240930888766977990) }}, {{ SC_(39.0), SC_(0.025972566037214762542869946938723143) }}, {{ SC_(40.0), SC_(0.025315103841291028157597100127414793) }}, {{ SC_(41.0), SC_(0.024690103841291028157597100127414793) }}, {{ SC_(42.0), SC_(0.024095219843670564148078956165487369) }}, {{ SC_(43.0), SC_(0.023528326419634282968940634170022516) }}, {{ SC_(44.0), SC_(0.022987493536995018501661023569698017) }}, {{ SC_(45.0), SC_(0.022470964611375183790917221916805455) }}, {{ SC_(46.0), SC_(0.021977137450881356630423394756311627) }}, {{ SC_(47.0), SC_(0.021504547658820865137039651845158508) }}, {{ SC_(48.0), SC_(0.021051854132338293837809230840178880) }}, {{ SC_(49.0), SC_(0.020617826354560516060031453062401102) }}, {{ SC_(50.0), SC_(0.020201333226697125805970645065733047) }}, {{ SC_(51.0), SC_(0.019801333226697125805970645065733047) }}, {{ SC_(52.0), SC_(0.019416865714201931649876834992684219) }}, {{ SC_(53.0), SC_(0.019047043228994831058160858661323273) }}, {{ SC_(54.0), SC_(0.018691044652989135080944767525687815) }}, {{ SC_(55.0), SC_(0.018348109124868421775046276442011546) }}, {{ SC_(56.0), SC_(0.018017530612471727560170243384160307) }}, {{ SC_(57.0), SC_(0.017698653061451319396904937261711327) }}, {{ SC_(58.0), SC_(0.017390866050063199975544518671375839) }}, {{ SC_(59.0), SC_(0.017093600889540013293023710110139216) }}, {{ SC_(60.0), SC_(0.016806327117635388185296045645904801) }}, {{ SC_(61.0), SC_(0.016528549339857610407518267868127023) }}, {{ SC_(62.0), SC_(0.016259804378825629757155462170733849) }}, {{ SC_(63.0), SC_(0.015999658697243944013138812847112621) }}, {{ SC_(64.0), SC_(0.015747706064338930155744003071350465) }}, {{ SC_(65.0), SC_(0.015503565439338930155744003071350465) }}, {{ SC_(66.0), SC_(0.015266879048806385777045778219279459) }}, {{ SC_(67.0), SC_(0.015037310637419792572270755262438320) }}, {{ SC_(68.0), SC_(0.014814543874220861852734110129892096) }}, {{ SC_(69.0), SC_(0.014598280898442315139931341963802130) }}, {{ SC_(70.0), SC_(0.014388240990859874476205234003289633) }}, {{ SC_(71.0), SC_(0.014184159358206813251715438084922286) }}, {{ SC_(72.0), SC_(0.013985786019583524221761063952805643) }}, {{ SC_(73.0), SC_(0.013792884785015622987193162718237741) }}, {{ SC_(74.0), SC_(0.013605232317385673653359422804557876) }}, {{ SC_(75.0), SC_(0.013422617269905761308582213162483370) }}, {{ SC_(76.0), SC_(0.013244839492127983530804435384705592) }}, {{ SC_(77.0), SC_(0.013071709298222166356289199927641880) }}, {{ SC_(78.0), SC_(0.012903046791897322369107550408330023) }}, {{ SC_(79.0), SC_(0.012738681242916388772789338705502935) }}, {{ SC_(80.0), SC_(0.012578450510661942369969277817824679) }}, {{ SC_(81.0), SC_(0.012422200510661942369969277817824679) }}, {{ SC_(82.0), SC_(0.012269784720386069789569948447301893) }}, {{ SC_(83.0), SC_(0.012121063720980953787190412456820037) }}, {{ SC_(84.0), SC_(0.011975904771931744903462730645236352) }}, {{ SC_(85.0), SC_(0.011834181415922674608678150146370139) }}, {{ SC_(86.0), SC_(0.011695773111424404712484378520072561) }}, {{ SC_(87.0), SC_(0.011560564890764588595664475869991436) }}, {{ SC_(88.0), SC_(0.011428447041643172292321894287219604) }}, {{ SC_(89.0), SC_(0.011299314810238213614635943873996463) }}, {{ SC_(90.0), SC_(0.011173068124213721757547192453721246) }}, {{ SC_(91.0), SC_(0.011049611334090264967423735663597789) }}, {{ SC_(92.0), SC_(0.010928852971573660692577702575806459) }}, {{ SC_(93.0), SC_(0.010810705523558537819231766848018180) }}, {{ SC_(94.0), SC_(0.010695085220633344155224367148630967) }}, {{ SC_(95.0), SC_(0.010581911839012701330416761897386060) }}, {{ SC_(96.0), SC_(0.010471108514912978338727011204865284) }}, {{ SC_(97.0), SC_(0.010362601570468533894282566760420840) }}, {{ SC_(98.0), SC_(0.010256320350360127049771991779020053) }}, {{ SC_(99.0), SC_(0.010152197068394279486256789779853040) }}, {{ SC_(100.0), SC_(0.010050166663333571395245668465701423) }}, {{ SC_(101.0), SC_(0.0099501666633335713952456684657014225) }}, {{ SC_(102.0), SC_(0.0098521370583928793062347871795530057) }}, {{ SC_(103.0), SC_(0.0097560201802690807672113346612907989) }}, {{ SC_(104.0), SC_(0.0096617605893557053312607266869294076) }}, {{ SC_(105.0), SC_(0.0095693049680539301833317326040891709) }}, {{ SC_(106.0), SC_(0.0094786020202081251946696010848147945) }}, {{ SC_(107.0), SC_(0.0093896023762067012003655783009059301) }}, {{ SC_(108.0), SC_(0.0093022585033793800369451922409880334) }}, {{ SC_(109.0), SC_(0.0092165246213492017104705694700689662) }}, {{ SC_(110.0), SC_(0.0091323566220225457050838175131629818) }}, {{ SC_(111.0), SC_(0.0090497119939233721513648092487001719) }}, {{ SC_(112.0), SC_(0.0089685497505989666647971605188892799) }}, {{ SC_(113.0), SC_(0.0088888303628438646239808339882770350) }}, {{ SC_(114.0), SC_(0.0088105156945064850327834027093985011) }}, {{ SC_(115.0), SC_(0.0087335689416594551774432980618146291) }}, {{ SC_(116.0), SC_(0.0086579545749297765385018991960301301) }}, {{ SC_(117.0), SC_(0.0085836382847989798678716970557209743) }}, {{ SC_(118.0), SC_(0.0085105869296963427139524918544644910) }}, {{ SC_(119.0), SC_(0.0084387684867201864370205757384058871) }}, {{ SC_(120.0), SC_(0.0083681520048333140410033453168254903) }}, {{ SC_(121.0), SC_(0.0082987075603888695965589008723810458) }}, {{ SC_(122.0), SC_(0.0082304062148523625273696378438993847) }}, {{ SC_(123.0), SC_(0.0081632199745943673647789364195510913) }}, {{ SC_(124.0), SC_(0.0080971217526365380303880315348924886) }}, {{ SC_(125.0), SC_(0.0080320853322411165943838692039871816) }}, {{ SC_(126.0), SC_(0.0079680853322411165943838692039871816) }}, {{ SC_(127.0), SC_(0.0079050971740148631300351667600466424) }}, {{ SC_(128.0), SC_(0.0078430970500146151295391657680446584) }}, {{ SC_(129.0), SC_(0.0077820618937646151295391657680446584) }}, {{ SC_(130.0), SC_(0.0077219693512491412998414312568974918) }}, {{ SC_(131.0), SC_(0.0076627977536160052051668750438797403) }}, {{ SC_(132.0), SC_(0.0076045260911254743503215863077920997) }}, {{ SC_(133.0), SC_(0.0075471339882788260491278305685818151) }}, {{ SC_(134.0), SC_(0.0074906016800646816656126516438263173) }}, {{ SC_(135.0), SC_(0.0074349099892649489857284903606897613) }}, {{ SC_(136.0), SC_(0.0073800403047656348567847317873015583) }}, {{ SC_(137.0), SC_(0.0073259745608209981785840397457790669) }}, {{ SC_(138.0), SC_(0.0072726952172225113119422367727916941) }}, {{ SC_(139.0), SC_(0.0072201852403269011460107097826635698) }}, {{ SC_(140.0), SC_(0.0071684280849001633995172570628250521) }}, {{ SC_(141.0), SC_(0.0071174076767368980933948080832332154) }}, {{ SC_(142.0), SC_(0.0070671083960166123934803168604577010) }}, {{ SC_(143.0), SC_(0.0070175150613607901359917233274285402) }}, {{ SC_(144.0), SC_(0.0069686129145565454296491148869180018) }}, {{ SC_(145.0), SC_(0.0069203876059145701210071395782760264) }}, {{ SC_(146.0), SC_(0.0068728251802308602518038102084781668) }}, {{ SC_(147.0), SC_(0.0068259120633233729183453752300582005) }}, {{ SC_(148.0), SC_(0.0067796350491163295567830632304284166) }}, {{ SC_(149.0), SC_(0.0067339812872463514705887608199097899) }}, {{ SC_(150.0), SC_(0.0066889382711659947299014043945235460) }}, {{ SC_(151.0), SC_(0.0066444938267215502854569599500791016) }}, {{ SC_(152.0), SC_(0.0066006361011831967044736697435092143) }}, {{ SC_(153.0), SC_(0.0065573535527067424108448608792432863) }}, {{ SC_(154.0), SC_(0.0065146349402072763935011042044600833) }}, {{ SC_(155.0), SC_(0.0064724693136260653967056918246321191) }}, {{ SC_(156.0), SC_(0.0064308460045729956776630279328527226) }}, {{ SC_(157.0), SC_(0.0063897546173277622785834750071459507) }}, {{ SC_(158.0), SC_(0.0063491850201838619175140604264327372) }}, {{ SC_(159.0), SC_(0.0063091273371202503168090452045131730) }}, {{ SC_(160.0), SC_(0.0062695719397862840971183684116647889) }}, {{ SC_(161.0), SC_(0.0062305094397862840971183684116647889) }}, {{ SC_(162.0), SC_(0.0061919306812507337711278587862645343) }}, {{ SC_(163.0), SC_(0.0061538267336817656260280264436338377) }}, {{ SC_(164.0), SC_(0.0061161888850611927779720213248864253) }}, {{ SC_(165.0), SC_(0.0060790086352099137773771373272659613) }}, {{ SC_(166.0), SC_(0.0060422776893880588646131336541713791) }}, {{ SC_(167.0), SC_(0.0060059879521257566436812132012754581) }}, {{ SC_(168.0), SC_(0.0059701315212748835395899944411908369) }}, {{ SC_(169.0), SC_(0.0059347006822726159658938493164742836) }}, {{ SC_(170.0), SC_(0.0058996879026080383950805024448661467) }}, {{ SC_(171.0), SC_(0.0058650858264834709210320595382917522) }}, {{ SC_(172.0), SC_(0.0058308872696625687631031241393655869) }}, {{ SC_(173.0), SC_(0.0057970852144976147338981484768453056) }}, {{ SC_(174.0), SC_(0.0057636728051287751468755282757026012) }}, {{ SC_(175.0), SC_(0.0057306433428484210710398828800096431) }}, {{ SC_(176.0), SC_(0.0056979902816239312751215155330708676) }}, {{ SC_(177.0), SC_(0.0056657072237726916057000279297650825) }}, {{ SC_(178.0), SC_(0.0056337879157832888159525096559612586) }}, {{ SC_(179.0), SC_(0.0056022262442771658516803218008924541) }}, {{ SC_(180.0), SC_(0.0055710162321052611046374704541804289) }}, {{ SC_(181.0), SC_(0.0055401520345743969071066062566495647) }}, {{ SC_(182.0), SC_(0.0055096279357984132680235501838801132) }}, {{ SC_(183.0), SC_(0.0054794383451692621993120419119322808) }}, {{ SC_(184.0), SC_(0.0054495777939434865714939523899997060) }}, {{ SC_(185.0), SC_(0.0054200409319397058531574684580526360) }}, {{ SC_(186.0), SC_(0.0053908225243429198779931149153207149) }}, {{ SC_(187.0), SC_(0.0053619174486116214619912649904739118) }}, {{ SC_(188.0), SC_(0.0053333206914838797479016427536355693) }}, {{ SC_(189.0), SC_(0.0053050273460787190416997414408243425) }}, {{ SC_(190.0), SC_(0.0052770326090892730575447625768507695) }}, {{ SC_(191.0), SC_(0.0052493317780643423096223249037205756) }}, {{ SC_(192.0), SC_(0.0052219202487751232640917747543277410) }}, {{ SC_(193.0), SC_(0.0051947935126640121529806636432166299) }}, {{ SC_(194.0), SC_(0.0051679471543725143946515809832794503) }}, {{ SC_(195.0), SC_(0.0051413768493454126835239372379292537) }}, {{ SC_(196.0), SC_(0.0051150783615084633081130233654769197) }}, {{ SC_(197.0), SC_(0.0050890475410170014172342228656851663) }}, {{ SC_(198.0), SC_(0.0050632803220729420495617757529020490) }}, {{ SC_(199.0), SC_(0.0050377727208077650268089954243641447) }}, {{ SC_(200.0), SC_(0.0050125208332291685267206138178390569) }}, {{ SC_(201.0), SC_(0.0049875208332291685267206138178390569) }}, {{ SC_(202.0), SC_(0.0049627689706515095578832088031116987) }}, {{ SC_(203.0), SC_(0.0049382615694163365356304884815745945) }}, {{ SC_(204.0), SC_(0.0049139950257001580309349122724940538) }}, {{ SC_(205.0), SC_(0.0048899658061692083961790491429285021) }}, {{ SC_(206.0), SC_(0.0048661704462643898357983233844514052) }}, {{ SC_(207.0), SC_(0.0048426055485360459768106713908610573) }}, {{ SC_(208.0), SC_(0.0048192677810268859030633260619152243) }}, {{ SC_(209.0), SC_(0.0047961538757014421160810775412051651) }}, {{ SC_(210.0), SC_(0.0047732606269205076136658397948165751) }}, {{ SC_(211.0), SC_(0.0047505848899590563665003069149979810) }}, {{ SC_(212.0), SC_(0.0047281235795662080477293898196946410) }}, {{ SC_(213.0), SC_(0.0047058736685658520491533841237174249) }}, {{ SC_(214.0), SC_(0.0046838321864965977124917869979266868) }}, {{ SC_(215.0), SC_(0.0046619962182897674216366904829472126) }}, {{ SC_(216.0), SC_(0.0046403629029841968429455060589342326) }}, {{ SC_(217.0), SC_(0.0046189294324766522613268503662044658) }}, {{ SC_(218.0), SC_(0.0045976930503067187312030422581537533) }}, {{ SC_(219.0), SC_(0.0045766510504750547298563542689272572) }}, {{ SC_(220.0), SC_(0.0045558007762939492483192720562961611) }}, {{ SC_(221.0), SC_(0.0045351396192691558598895199901804586) }}, {{ SC_(222.0), SC_(0.0045146650180120153427010922348109944) }}, {{ SC_(223.0), SC_(0.0044943744571809139710591800523582715) }}, {{ SC_(224.0), SC_(0.0044742654664511586974763611740377744) }}, {{ SC_(225.0), SC_(0.0044543356195123831872722795413847131) }}, {{ SC_(226.0), SC_(0.0044345825330926301008525264549649601) }}, {{ SC_(227.0), SC_(0.0044150038660082852030531686352453266) }}, {{ SC_(228.0), SC_(0.0043955973182390678691247011703226617) }}, {{ SC_(229.0), SC_(0.0043763606300273104052896750084266937) }}, {{ SC_(230.0), SC_(0.0043572915810007853580937786677772019) }}, {{ SC_(231.0), SC_(0.0043383879893183656983584289513310771) }}, {{ SC_(232.0), SC_(0.0043196477108378274775604678936297597) }}, {{ SC_(233.0), SC_(0.0043010686383051283099029173585524707) }}, {{ SC_(234.0), SC_(0.0042826487005645178731661935286790157) }}, {{ SC_(235.0), SC_(0.0042643858617888585846863922283648949) }}, {{ SC_(236.0), SC_(0.0042462781207295557327171753881657097) }}, {{ SC_(237.0), SC_(0.0042283235099855166634841963591510587) }}, {{ SC_(238.0), SC_(0.0042105200952905781742819673716312524) }}, {{ SC_(239.0), SC_(0.0041928659748188600752776597662361532) }}, {{ SC_(240.0), SC_(0.0041753592785075209880768054394561599) }}, {{ SC_(241.0), SC_(0.0041579981673964098769656943283450488) }}, {{ SC_(242.0), SC_(0.0041407808329841235871290868319176457) }}, {{ SC_(243.0), SC_(0.0041237054965999968198317710747972304) }}, {{ SC_(244.0), SC_(0.0041067704087915665331207344780724764) }}, {{ SC_(245.0), SC_(0.0040899738487270677424730591219854030) }}, {{ SC_(246.0), SC_(0.0040733141236125321323106268021186808) }}, {{ SC_(247.0), SC_(0.0040567895681230747987129005809540301) }}, {{ SC_(248.0), SC_(0.0040403985438479678472795055080959272) }}, {{ SC_(249.0), SC_(0.0040241394387491124882784649253696005) }}, {{ SC_(250.0), SC_(0.0040080106666325337234198336129714134) }}, {{ SC_(251.0), SC_(0.0039920106666325337234198336129714134) }}, {{ SC_(252.0), SC_(0.0039761379027081515707555901882638690) }}, {{ SC_(253.0), SC_(0.0039603908631515882046684145772787342) }}, {{ SC_(254.0), SC_(0.0039447680601082661718292825802158212) }}, {{ SC_(255.0), SC_(0.0039292680291082041717052823322153252) }}, {{ SC_(256.0), SC_(0.0039138893286083964054615299292933721) }}, {{ SC_(257.0), SC_(0.0038986305395458964054615299292933721) }}, {{ SC_(258.0), SC_(0.0038834902649013143527430936168586645) }}, {{ SC_(259.0), SC_(0.0038684671292724458953186599890718729) }}, {{ SC_(260.0), SC_(0.0038535597784577591732960306305351784) }}, {{ SC_(261.0), SC_(0.0038387668790494751496273915772807406) }}, {{ SC_(262.0), SC_(0.0038240871180359844492559936236394259) }}, {{ SC_(263.0), SC_(0.0038095192024133517355446714396175157) }}, {{ SC_(264.0), SC_(0.0037950618588056662116828258151325586) }}, {{ SC_(265.0), SC_(0.0037807138330940041363843868803299874) }}, {{ SC_(266.0), SC_(0.0037664738900537762972957432349045691) }}, {{ SC_(267.0), SC_(0.0037523408130002402014169485037156947) }}, {{ SC_(268.0), SC_(0.0037383134034419633284070872347962260) }}, {{ SC_(269.0), SC_(0.0037243904807420301584360469140120870) }}, {{ SC_(270.0), SC_(0.0037105708817867918394520638292012082) }}, {{ SC_(271.0), SC_(0.0036968534606619633072161241858541575) }}, {{ SC_(272.0), SC_(0.0036832370883358784227510433726844022) }}, {{ SC_(273.0), SC_(0.0036697206523497192532008703623037793) }}, {{ SC_(274.0), SC_(0.0036563030565145410004402000192158538) }}, {{ SC_(275.0), SC_(0.0036429832206149192837797492759690106) }}, {{ SC_(276.0), SC_(0.0036297600801190515151847079536549610) }}, {{ SC_(277.0), SC_(0.0036166325858951489737018262061229300) }}, {{ SC_(278.0), SC_(0.0036035997039339608961822443009762449) }}, {{ SC_(279.0), SC_(0.0035906604150772764595588811210166154) }}, {{ SC_(280.0), SC_(0.0035778137147522549413358367099735918) }}, {{ SC_(281.0), SC_(0.0035650586127114386148052244650756327) }}, {{ SC_(282.0), SC_(0.0035523941327783070688521590277078182) }}, {{ SC_(283.0), SC_(0.0035398193125982356438735362220139396) }}, {{ SC_(284.0), SC_(0.0035273332033947245499655089024070023) }}, {{ SC_(285.0), SC_(0.0035149348697307689855933605191497121) }}, {{ SC_(286.0), SC_(0.0035026233892752442087389437755362926) }}, {{ SC_(287.0), SC_(0.0034903978525741830321532916654086580) }}, {{ SC_(288.0), SC_(0.0034782573628268266237957785233285065) }}, {{ SC_(289.0), SC_(0.0034662010356663327966352846961680126) }}, {{ SC_(290.0), SC_(0.0034542279989450291723970691575609557) }}, {{ SC_(291.0), SC_(0.0034423373925241017050962368151114908) }}, {{ SC_(292.0), SC_(0.0034305283680676120557061729282891812) }}, {{ SC_(293.0), SC_(0.0034188000888407402223415641836841896) }}, {{ SC_(294.0), SC_(0.0034071517295121516540414092605051194) }}, {{ SC_(295.0), SC_(0.0033955824759603908136508312605976734) }}, {{ SC_(296.0), SC_(0.0033840915250842058093417246820282968) }}, {{ SC_(297.0), SC_(0.0033726780846167112877931490793986401) }}, {{ SC_(298.0), SC_(0.0033613413729432992776808022667151271) }}, {{ SC_(299.0), SC_(0.0033500806189232100925089631603685662) }}, {{ SC_(300.0), SC_(0.0033388950617146777494702946890986698) }}, {{ SC_(301.0), SC_(0.0033277839506035666383591835779875587) }}, {{ SC_(302.0), SC_(0.0033167465448354183839249058106339975) }}, {{ SC_(303.0), SC_(0.0033057821134508299886790832589915257) }}, {{ SC_(304.0), SC_(0.0032948899351240864232334297827528127) }}, {{ SC_(305.0), SC_(0.0032840692980049728498262275666863307) }}, {{ SC_(306.0), SC_(0.0032733194995636936238117153387906038) }}, {{ SC_(307.0), SC_(0.0032626398464388271194757761700948030) }}, {{ SC_(308.0), SC_(0.0032520296542882472724747469814561968) }}, {{ SC_(309.0), SC_(0.0032414882476429445232758938864992057) }}, {{ SC_(310.0), SC_(0.0032310149597636805859480485560146067) }}, {{ SC_(311.0), SC_(0.0032206091325004131561873825830697576) }}, {{ SC_(312.0), SC_(0.0032102701161544283131853457968495985) }}, {{ SC_(313.0), SC_(0.0031999972693431199634154575654229056) }}, {{ SC_(314.0), SC_(0.0031897899588673572221401561945811087) }}, {{ SC_(315.0), SC_(0.0031796475595813821318728025494028053) }}, {{ SC_(316.0), SC_(0.0031695694542651815775770101583723190) }}, {{ SC_(317.0), SC_(0.0031595550334992786774007563528924280) }}, {{ SC_(318.0), SC_(0.0031496036955418903065342933569426225) }}, {{ SC_(319.0), SC_(0.0031397148462083987516116241587305265) }}, {{ SC_(320.0), SC_(0.0031298878987530867951646552806731175) }}, {{ SC_(321.0), SC_(0.0031201222737530867951646552806731175) }}, {{ SC_(322.0), SC_(0.0031104173989944955547846123851266845) }}, {{ SC_(323.0), SC_(0.0031007727093606079732869849787766208) }}, {{ SC_(324.0), SC_(0.0030911876467222236314453110434374534) }}, {{ SC_(325.0), SC_(0.0030816616598299815951703529577797793) }}, {{ SC_(326.0), SC_(0.0030721942042086798200224239636969390) }}, {{ SC_(327.0), SC_(0.0030627847420535366080084226840100859) }}, {{ SC_(328.0), SC_(0.0030534327421283526074098946887983099) }}, {{ SC_(329.0), SC_(0.0030441376796655328572611736893931939) }}, {{ SC_(330.0), SC_(0.0030348990362679293613585120362303443) }}, {{ SC_(331.0), SC_(0.0030257162998124656331675111179566988) }}, {{ SC_(332.0), SC_(0.0030165889643555055835148062320940287) }}, {{ SC_(333.0), SC_(0.0030075165300399300282818261188700484) }}, {{ SC_(334.0), SC_(0.0029984985030038849742187540377799493) }}, {{ SC_(335.0), SC_(0.0029895343952911666981959493477587940) }}, {{ SC_(336.0), SC_(0.0029806237247632094694144835424569450) }}, {{ SC_(337.0), SC_(0.0029717660150126425759904472612778067) }}, {{ SC_(338.0), SC_(0.0029629607952783841075703678381958037) }}, {{ SC_(339.0), SC_(0.0029542076003622397148670311202937695) }}, {{ SC_(340.0), SC_(0.0029455059705469753158450943115294879) }}, {{ SC_(341.0), SC_(0.0029368554515158334473329835848858893) }}, {{ SC_(342.0), SC_(0.0029282555942734636706712761692289892) }}, {{ SC_(343.0), SC_(0.0029197059550682381311890423194974479) }}, {{ SC_(344.0), SC_(0.0029112060953159240443714748093613651) }}, {{ SC_(345.0), SC_(0.0029027555815246855370702308937312948) }}, {{ SC_(346.0), SC_(0.0028943539852213879105211865753107949) }}, {{ SC_(347.0), SC_(0.0028860008828791780137655315250251188) }}, {{ SC_(348.0), SC_(0.0028776958558463150217964926658036320) }}, {{ SC_(349.0), SC_(0.0028694384902762265028375813168803925) }}, {{ SC_(350.0), SC_(0.0028612283770587652340466846904159136) }}, {{ SC_(351.0), SC_(0.0028530651117526427850670928536812197) }}, {{ SC_(352.0), SC_(0.0028449482945190164346316256090971660) }}, {{ SC_(353.0), SC_(0.0028368775300562065172762537082707197) }}, {{ SC_(354.0), SC_(0.0028288524275355218155291888895176601) }}, {{ SC_(355.0), SC_(0.0028208726005381711180923093210667042) }}, {{ SC_(356.0), SC_(0.0028129376669932395568941343557820384) }}, {{ SC_(357.0), SC_(0.0028050472491167088158260873920148373) }}, {{ SC_(358.0), SC_(0.0027972009733515007718241729007281265) }}, {{ SC_(359.0), SC_(0.0027893984703085245850634600640501202) }}, {{ SC_(360.0), SC_(0.0027816393747087076997196157425442350) }}, {{ SC_(361.0), SC_(0.0027739233253259916503368996931615189) }}, {{ SC_(362.0), SC_(0.0027662499649312739916326233294135428) }}, {{ SC_(363.0), SC_(0.0027586189402372780818618593112211799) }}, {{ SC_(364.0), SC_(0.0027510299018443328519519411969454398) }}, {{ SC_(365.0), SC_(0.0027434825041870450847740641289584817) }}, {{ SC_(366.0), SC_(0.0027359764054818471114207145324112871) }}, {{ SC_(367.0), SC_(0.0027285112676754032044661921519281434) }}, {{ SC_(368.0), SC_(0.0027210867563938583121587282907368063) }}, {{ SC_(369.0), SC_(0.0027137025408929131325746073077500388) }}, {{ SC_(370.0), SC_(0.0027063582940087098731978400983435274) }}, {{ SC_(371.0), SC_(0.0026990536921095133794067517126605471) }}, {{ SC_(372.0), SC_(0.0026917884150481726451778518935659459) }}, {{ SC_(373.0), SC_(0.0026845621461153480411773894123542452) }}, {{ SC_(374.0), SC_(0.0026773745719934899095154066481569894) }}, {{ SC_(375.0), SC_(0.0026702253827115544809930010889474037) }}, {{ SC_(376.0), SC_(0.0026631142716004433698818899778362926) }}, {{ SC_(377.0), SC_(0.0026560409352491531933314146496334859) }}, {{ SC_(378.0), SC_(0.0026490050734616221475912771688941576) }}, {{ SC_(379.0), SC_(0.0026420063892142606515525324529007643) }}, {{ SC_(380.0), SC_(0.0026350445886141534398232908018401340) }}, {{ SC_(381.0), SC_(0.0026281193808579207528426813835575855) }}, {{ SC_(382.0), SC_(0.0026212304781912265305653479400018095) }}, {{ SC_(383.0), SC_(0.0026143775958689217691827104026536009) }}, {{ SC_(384.0), SC_(0.0026075604521158114473453538183153069) }}, {{ SC_(385.0), SC_(0.0026007787680880336695675760405375292) }}, {{ SC_(386.0), SC_(0.0025940322678350399100803100597650549) }}, {{ SC_(387.0), SC_(0.0025873206782621654704980393947807600) }}, {{ SC_(388.0), SC_(0.0025806437290937794894205133379866304) }}, {{ SC_(389.0), SC_(0.0025740011528370040616386024016490812) }}, {{ SC_(390.0), SC_(0.0025673926847459922390891875814985403) }}, {{ SC_(391.0), SC_(0.0025608180627867548952364591133854568) }}, {{ SC_(392.0), SC_(0.0025542770276025266392726702841653444) }}, {{ SC_(393.0), SC_(0.0025477693224796611665529701592174061) }}, {{ SC_(394.0), SC_(0.0025412946933140466271257158552076682) }}, {{ SC_(395.0), SC_(0.0025348528885780317852076040770118889) }}, {{ SC_(396.0), SC_(0.0025284436592878539290948016415047586) }}, {{ SC_(397.0), SC_(0.0025220667589715596734066065593702826) }}, {{ SC_(398.0), SC_(0.0025157219436374099738336126313585574) }}, {{ SC_(399.0), SC_(0.0025094089717427608488115172297272854) }}, {{ SC_(400.0), SC_(0.0025031276041634114728653862380877857) }}, {{ SC_(401.0), SC_(0.0024968776041634114728653862380877857) }}, {{ SC_(402.0), SC_(0.0024906587373653194211990408795390204) }}, {{ SC_(403.0), SC_(0.0024844707717209046789896896258571809) }}, {{ SC_(404.0), SC_(0.0024783134774822848980662186359489861) }}, {{ SC_(405.0), SC_(0.0024721866271734916425030385555647101) }}, {{ SC_(406.0), SC_(0.0024660899955624567392870653807437986) }}, {{ SC_(407.0), SC_(0.0024600233596334121131131713284736635) }}, {{ SC_(408.0), SC_(0.0024539864985596960025420239022893822) }}, {{ SC_(409.0), SC_(0.0024479791936769585938530581198979943) }}, {{ SC_(410.0), SC_(0.0024420012284567602449670519386819566) }}, {{ SC_(411.0), SC_(0.0024360523884805556048718704990626824) }}, {{ SC_(412.0), SC_(0.0024301324614140570641338923909529743) }}, {{ SC_(413.0), SC_(0.0024242412369819710993869793925553873) }}, {{ SC_(414.0), SC_(0.0024183785069431011992292719545097870) }}, {{ SC_(415.0), SC_(0.0024125440650658111807924356222733288) }}, {{ SC_(416.0), SC_(0.0024067377071038428254433283498099814) }}, {{ SC_(417.0), SC_(0.0024009592307724818786977662196324666) }}, {{ SC_(418.0), SC_(0.0023952084357250665735318270285392980) }}, {{ SC_(419.0), SC_(0.0023894851235298329479280175919421505) }}, {{ SC_(420.0), SC_(0.0023837890976470913367501363996500127) }}, {{ SC_(421.0), SC_(0.0023781201634067285249587531796953642) }}, {{ SC_(422.0), SC_(0.0023724781279860301538143791353151136) }}, {{ SC_(423.0), SC_(0.0023668628003878180741216498614892786) }}, {{ SC_(424.0), SC_(0.0023612739914188974407978175034031103) }}, {{ SC_(425.0), SC_(0.0023557115136688084411538160794088063) }}, {{ SC_(426.0), SC_(0.0023501751814888776453060652143569032) }}, {{ SC_(427.0), SC_(0.0023446648109715640611406659329092186) }}, {{ SC_(428.0), SC_(0.0023391802199300950682761188778603783) }}, {{ SC_(429.0), SC_(0.0023337212278783874955623447491155098) }}, {{ SC_(430.0), SC_(0.0023282876560112491948576104779476722) }}, {{ SC_(431.0), SC_(0.0023228793271848565501848143719444272) }}, {{ SC_(432.0), SC_(0.0023174960658975034459271930197768570) }}, {{ SC_(433.0), SC_(0.0023121376982706173005225290965944153) }}, {{ SC_(434.0), SC_(0.0023068040520300378531949525454367474) }}, {{ SC_(435.0), SC_(0.0023014949564875544706640005184240693) }}, {{ SC_(436.0), SC_(0.0022962102425226978185302972551131960) }}, {{ SC_(437.0), SC_(0.0022909497425647818181936252578065720) }}, {{ SC_(438.0), SC_(0.0022857132905751918847489300454946261) }}, {{ SC_(439.0), SC_(0.0022805007220299155143646594923368520) }}, {{ SC_(440.0), SC_(0.0022753118739023113612106181579726675) }}, {{ SC_(441.0), SC_(0.0022701465846461130141031801414437419) }}, {{ SC_(442.0), SC_(0.0022650046941786637517073676970404326) }}, {{ SC_(443.0), SC_(0.0022598860438643786224102607581980666) }}, {{ SC_(444.0), SC_(0.0022547904764984302608899472788937134) }}, {{ SC_(445.0), SC_(0.0022497178362906549179794692332805327) }}, {{ SC_(446.0), SC_(0.0022446679688496752436959191764695239) }}, {{ SC_(447.0), SC_(0.0022396407211672364253002144568893997) }}, {{ SC_(448.0), SC_(0.0022346359416027523430016192985131504) }}, {{ SC_(449.0), SC_(0.0022296534798680584654505988903498851) }}, {{ SC_(450.0), SC_(0.0022246931870123682655011938774779251) }}, {{ SC_(451.0), SC_(0.0022197549154074299938962556058729868) }}, {{ SC_(452.0), SC_(0.0022148385187328807045613949119727602) }}, {{ SC_(453.0), SC_(0.0022099438519617944801115554570428518) }}, {{ SC_(454.0), SC_(0.0022050707713464218600023009896461977) }}, {{ SC_(455.0), SC_(0.0022002191344041175265201841234155314) }}, {{ SC_(456.0), SC_(0.0021953887999034533555263427999038782) }}, {{ SC_(457.0), SC_(0.0021905796278505139895675862594298863) }}, {{ SC_(458.0), SC_(0.0021857914794753721406719726821563537) }}, {{ SC_(459.0), SC_(0.0021810242172187408788729985969939807) }}, {{ SC_(460.0), SC_(0.0021762777047188002102792478553514026) }}, {{ SC_(461.0), SC_(0.0021715518067981952953454104262398714) }}, {{ SC_(462.0), SC_(0.0021668463894512037039262095002137376) }}, {{ SC_(463.0), SC_(0.0021621613198310691487267192357884082) }}, {{ SC_(464.0), SC_(0.0021574964662374991829200960766562576) }}, {{ SC_(465.0), SC_(0.0021528516981043243910057084428869353) }}, {{ SC_(466.0), SC_(0.0021482268859873166444454124549114468) }}, {{ SC_(467.0), SC_(0.0021436219015521640352612314974430831) }}, {{ SC_(468.0), SC_(0.0021390366175626001416214789193671599) }}, {{ SC_(469.0), SC_(0.0021344709078686853195015285942886297) }}, {{ SC_(470.0), SC_(0.0021299246473952377537966991017876864) }}, {{ SC_(471.0), SC_(0.0021253977121304120408043948917378901) }}, {{ SC_(472.0), SC_(0.0021208899791144231117966821605475330) }}, {{ SC_(473.0), SC_(0.0021164013264284133444884374032938703) }}, {{ SC_(474.0), SC_(0.0021119316331834607455853001256052381) }}, {{ SC_(475.0), SC_(0.0021074807795097261232847428787252865) }}, {{ SC_(476.0), SC_(0.0021030486465457372036171528510244555) }}, {{ SC_(477.0), SC_(0.0020986351164278076788660759496756807) }}, {{ SC_(478.0), SC_(0.0020942400722795892100115563060258602) }}, {{ SC_(479.0), SC_(0.0020898633982017544382113427243308619) }}, {{ SC_(480.0), SC_(0.0020855049792618090927848496389625101) }}, {{ SC_(481.0), SC_(0.0020811647014840313150070718611847323) }}, {{ SC_(482.0), SC_(0.0020768424518395363482667828755735014) }}, {{ SC_(483.0), SC_(0.0020725381182364647758076310014666506) }}, {{ SC_(484.0), SC_(0.0020682515895102925173642410430888445) }}, {{ SC_(485.0), SC_(0.0020639827554142608255399121038087407) }}, {{ SC_(486.0), SC_(0.0020597315066099245517594891045527092) }}, {{ SC_(487.0), SC_(0.0020554977346578169800817299553715207) }}, {{ SC_(488.0), SC_(0.0020512813320082295550809920849078429) }}, {{ SC_(489.0), SC_(0.0020470821919921048574190732458860746) }}, {{ SC_(490.0), SC_(0.0020429002088120412076350726771363621) }}, {{ SC_(491.0), SC_(0.0020387352775334073050944645971696816) }}, {{ SC_(492.0), SC_(0.0020345872940755653349682414605475504) }}, {{ SC_(493.0), SC_(0.0020304561552032010015688099052563877) }}, {{ SC_(494.0), SC_(0.0020263417585177589713609094407410020) }}, {{ SC_(495.0), SC_(0.0020222440024489822335025606725264762) }}, {{ SC_(496.0), SC_(0.0020181627862465539098621158199604115) }}, {{ SC_(497.0), SC_(0.0020140980099718400701118556742788299) }}, {{ SC_(498.0), SC_(0.0020100495744897321307250317123988984) }}, {{ SC_(499.0), SC_(0.0020060173814605874395103738842993516) }}, {{ SC_(500.0), SC_(0.0020020013333322666697142686477741971) }},
      {{ SC_(0.25000000000000000000000000000000000), SC_(17.197329154507110739271319119335224)}}, {{ SC_(0.12500000000000000000000000000000000), SC_(65.388133444988034473142999334395961) }}, {{ SC_(0.062500000000000000000000000000000000), SC_(257.50642004291541426394984152786018) }}, {{ SC_(0.031250000000000000000000000000000000), SC_(1025.5728544782377088851896549789956) }}, {{ SC_(0.015625000000000000000000000000000000), SC_(4097.6081469812325471140472931934309) }}, {{ SC_(0.0078125000000000000000000000000000000), SC_(16385.626348148031663597978251925972) }}, {{ SC_(0.0039062500000000000000000000000000000), SC_(65537.635592296074077546680509110271) }}, {{ SC_(0.0019531250000000000000000000000000000), SC_(262145.64025088744769438583827382756) }}, {{ SC_(0.00097656250000000000000000000000000000), SC_(1.0485776425893921526170408061678298e6) }}, {{ SC_(0.00048828125000000000000000000000000000), SC_(4.1943056437609568090690403869475364e6) }}, {{ SC_(0.00024414062500000000000000000000000000), SC_(1.6777217644347318475117100572465400e7) }}, {{ SC_(0.00012207031250000000000000000000000000), SC_(6.7108865644640644300678883695523690e7) }}, {{ SC_(0.000061035156250000000000000000000000000), SC_(2.6843545764478734348137667988052034e8) }}, {{ SC_(0.000030517578125000000000000000000000000), SC_(1.0737418256448607021411790006489757e9) }}, {{ SC_(0.000015258789062500000000000000000000000), SC_(4.2949672976448973837387528780800407e9) }}, {{ SC_(7.6293945312500000000000000000000000e-6), SC_(1.7179869185644915725104496671656842e10) }}, {{ SC_(3.8146972656250000000000000000000000e-6), SC_(6.8719476737644924895929112616949761e10) }}, {{ SC_(1.9073486328125000000000000000000000e-6), SC_(2.7487790694564492948137685720609220e11) }}, {{ SC_(9.5367431640625000000000000000000000e-7), SC_(1.0995116277776449317741095887303350e12) }}, {{ SC_(4.7683715820312500000000000000000000e-7), SC_(4.3980465111056449329204781693093178e12) }}, {{ SC_(2.3841857910156250000000000000000000e-7), SC_(1.7592186044417644933493663013304205e13) }}, {{ SC_(1.1920928955078125000000000000000000e-7), SC_(7.0368744177665644933780255573728145e13) }}, {{ SC_(5.9604644775390625000000000000000000e-8), SC_(2.8147497671065764493392355188854676e14) }}, {{ SC_(2.9802322387695312500000000000000000e-8), SC_(1.1258999068426256449339952000546077e15) }}, {{ SC_(1.4901161193847656250000000000000000e-8), SC_(4.5035996273704976449340310241398011e15) }}, {{ SC_(7.4505805969238281250000000000000000e-9), SC_(1.8014398509481985644934048936182939e16) }}, {{ SC_(3.7252902984619140625000000000000000e-9), SC_(7.2057594037927937644934057892204642e16) }}, {{ SC_(1.8626451492309570312500000000000000e-9), SC_(2.8823037615171174564493406237021553e17) }}, {{ SC_(9.3132257461547851562500000000000000e-10), SC_(1.1529215046068469776449340646092210e18) }}, {{ SC_(4.6566128730773925781250000000000000e-10), SC_(4.6116860184273879056449340657287237e18) }}, {{ SC_(2.3283064365386962890625000000000000e-10), SC_(1.8446744073709551617644934066288475e19) }}, {{ SC_(1.1641532182693481445312500000000000e-10), SC_(7.3786976294838206465644934066568351e19) }}, {{ SC_(5.8207660913467407226562500000000000e-11), SC_(2.9514790517935282585764493406670829e20) }}, {{ SC_(2.9103830456733703613281250000000000e-11), SC_(1.1805916207174113034256449340667783e21) }}, {{ SC_(1.4551915228366851806640625000000000e-11), SC_(4.7223664828696452136976449340668132e21) }}, {{ SC_(7.2759576141834259033203125000000000e-12), SC_(1.8889465931478580854785644934066831e22) }}, {{ SC_(3.6379788070917129516601562500000000e-12), SC_(7.5557863725914323419137644934066839e22) }}, {{ SC_(1.8189894035458564758300781250000000e-12), SC_(3.0223145490365729367654564493406684e23) }}, {{ SC_(9.0949470177292823791503906250000000e-13), SC_(1.2089258196146291747061776449340668e24) }}, {{ SC_(4.5474735088646411895751953125000000e-13), SC_(4.8357032784585166988247056449340668e24) }}, {{ SC_(2.2737367544323205947875976562500000e-13), SC_(1.9342813113834066795298817644934067e25) }}, {{ SC_(1.1368683772161602973937988281250000e-13), SC_(7.7371252455336267181195265644934067e25) }}, {{ SC_(5.6843418860808014869689941406250000e-14), SC_(3.0948500982134506872478105764493407e26) }}, {{ SC_(2.8421709430404007434844970703125000e-14), SC_(1.2379400392853802748991242256449341e27) }}, {{ SC_(1.4210854715202003717422485351562500e-14), SC_(4.9517601571415210995964968976449341e27) }}, {{ SC_(7.1054273576010018587112426757812500e-15), SC_(1.9807040628566084398385987585644934e28) }}, {{ SC_(3.5527136788005009293556213378906250e-15), SC_(7.9228162514264337593543950337644934e28) }}, {{ SC_(1.7763568394002504646778106689453125e-15), SC_(3.1691265005705735037417580134564493e29) }}, {{ SC_(8.8817841970012523233890533447265625e-16), SC_(1.2676506002282294014967032053776449e30) }}, {{ SC_(4.4408920985006261616945266723632812e-16), SC_(5.0706024009129176059868128215056449e30) }}, {{ SC_(2.2204460492503130808472633361816406e-16), SC_(2.0282409603651670423947251286017645e31) }}, {{ SC_(1.1102230246251565404236316680908203e-16), SC_(8.1129638414606681695789005144065645e31) }}, {{ SC_(5.5511151231257827021181583404541016e-17), SC_(3.2451855365842672678315602057625764e32) }}, {{ SC_(2.7755575615628913510590791702270508e-17), SC_(1.2980742146337069071326240823050256e33) }}, {{ SC_(1.3877787807814456755295395851135254e-17), SC_(5.1922968585348276285304963292200976e33) }}, {{ SC_(6.9388939039072283776476979255676270e-18), SC_(2.0769187434139310514121985316880386e34) }}, {{ SC_(3.4694469519536141888238489627838135e-18), SC_(8.3076749736557242056487941267521538e34) }}, {{ SC_(1.7347234759768070944119244813919067e-18), SC_(3.3230699894622896822595176507008615e35) }}, {{ SC_(8.6736173798840354720596224069595337e-19), SC_(1.3292279957849158729038070602803446e36) }}, {{ SC_(4.3368086899420177360298112034797668e-19), SC_(5.3169119831396634916152282411213783e36) }}, {{ SC_(2.1684043449710088680149056017398834e-19), SC_(2.1267647932558653966460912964485513e37) }}, {{ SC_(1.0842021724855044340074528008699417e-19), SC_(8.5070591730234615865843651857942053e37) }}, {{ SC_(5.4210108624275221700372640043497086e-20), SC_(3.4028236692093846346337460743176821e38) }}, {{ SC_(2.7105054312137610850186320021748543e-20), SC_(1.3611294676837538538534984297270728e39) }}, {{ SC_(1.3552527156068805425093160010874271e-20), SC_(5.4445178707350154154139937189082914e39) }}, {{ SC_(6.7762635780344027125465800054371357e-21), SC_(2.1778071482940061661655974875633166e40) }}, {{ SC_(3.3881317890172013562732900027185678e-21), SC_(8.7112285931760246646623899502532662e40) }}, {{ SC_(1.6940658945086006781366450013592839e-21), SC_(3.4844914372704098658649559801013065e41) }}, {{ SC_(8.4703294725430033906832250067964196e-22), SC_(1.3937965749081639463459823920405226e42) }}, {{ SC_(4.2351647362715016953416125033982098e-22), SC_(5.5751862996326557853839295681620904e42) }}, {{ SC_(2.1175823681357508476708062516991049e-22), SC_(2.2300745198530623141535718272648362e43) }}, {{ SC_(1.0587911840678754238354031258495525e-22), SC_(8.9202980794122492566142873090593446e43) }}, {{ SC_(5.2939559203393771191770156292477623e-23), SC_(3.5681192317648997026457149236237378e44) }}, {{ SC_(2.6469779601696885595885078146238811e-23), SC_(1.4272476927059598810582859694494951e45) }}, {{ SC_(1.3234889800848442797942539073119406e-23), SC_(5.7089907708238395242331438777979805e45) }}, {{ SC_(6.6174449004242213989712695365597028e-24), SC_(2.2835963083295358096932575511191922e46) }}, {{ SC_(3.3087224502121106994856347682798514e-24), SC_(9.1343852333181432387730302044767689e46) }}, {{ SC_(1.6543612251060553497428173841399257e-24), SC_(3.6537540933272572955092120817907075e47) }}, {{ SC_(8.2718061255302767487140869206996285e-25), SC_(1.4615016373309029182036848327162830e48) }}, {{ SC_(4.1359030627651383743570434603498143e-25), SC_(5.8460065493236116728147393308651321e48) }}, {{ SC_(2.0679515313825691871785217301749071e-25), SC_(2.3384026197294446691258957323460528e49) }}, {{ SC_(1.0339757656912845935892608650874536e-25), SC_(9.3536104789177786765035829293842113e49) }}, {{ SC_(5.1698788284564229679463043254372678e-26), SC_(3.7414441915671114706014331717536845e50) }}, {{ SC_(2.5849394142282114839731521627186339e-26), SC_(1.4965776766268445882405732687014738e51) }}, {{ SC_(1.2924697071141057419865760813593170e-26), SC_(5.9863107065073783529622930748058952e51) }}, {{ SC_(6.4623485355705287099328804067965848e-27), SC_(2.3945242826029513411849172299223581e52) }}, {{ SC_(3.2311742677852643549664402033982924e-27), SC_(9.5780971304118053647396689196894324e52) }}, {{ SC_(1.6155871338926321774832201016991462e-27), SC_(3.8312388521647221458958675678757730e53) }}, {{ SC_(8.0779356694631608874161005084957310e-28), SC_(1.5324955408658888583583470271503092e54) }}, {{ SC_(4.0389678347315804437080502542478655e-28), SC_(6.1299821634635554334333881086012367e54) }}, {{ SC_(2.0194839173657902218540251271239327e-28), SC_(2.4519928653854221733733552434404947e55) }}, {{ SC_(1.0097419586828951109270125635619664e-28), SC_(9.8079714615416886934934209737619788e55) }}, {{ SC_(5.0487097934144755546350628178098319e-29), SC_(3.9231885846166754773973683895047915e56) }}, {{ SC_(2.5243548967072377773175314089049159e-29), SC_(1.5692754338466701909589473558019166e57) }}, {{ SC_(1.2621774483536188886587657044524580e-29), SC_(6.2771017353866807638357894232076664e57) }}, {{ SC_(6.3108872417680944432938285222622898e-30), SC_(2.5108406941546723055343157692830666e58) }}, {{ SC_(3.1554436208840472216469142611311449e-30), SC_(1.0043362776618689222137263077132266e59) }}, {{ SC_(1.5777218104420236108234571305655725e-30), SC_(4.0173451106474756888549052308529065e59) }}, {{ SC_(7.8886090522101180541172856528278623e-31), SC_(1.6069380442589902755419620923411626e60) }},
      {{ SC_(-0.5000000000), SC_(8.9348022005446793094172454999380756) }}, {{ SC_(-1.500000000), SC_(9.3792466449891237538616899443825200) }}, {{ SC_(-2.500000000), SC_(9.5392466449891237538616899443825200) }}, {{ SC_(-3.500000000), SC_(9.6208792980503482436576083117294588) }}, {{ SC_(-4.500000000), SC_(9.6702620140997309597069910277788415) }}, {{ SC_(-5.500000000), SC_(9.7033198653394003811945943335639655) }}, {{ SC_(-6.500000000), SC_(9.7269885043926548190644168187710661) }}, {{ SC_(-7.500000000), SC_(9.7447662821704325968421945965488438) }}, {{ SC_(-8.500000000), SC_(9.7586071126202595864615717591786016) }}, {{ SC_(-9.500000000), SC_(9.7696874450302318856305468284306792) }}, {{ SC_(-10.50000000), SC_(9.7787577398148123844967599803581168) }}, {{ SC_(-11.50000000), SC_(9.7863191764877802483908998669365667) }}, {{ SC_(-12.50000000), SC_(9.7927191764877802483908998669365667) }}, {{ SC_(-13.50000000), SC_(9.7982061449377116612852757242753870) }}, {{ SC_(-14.50000000), SC_(9.8029623875060826482056086612551730) }}, {{ SC_(-15.50000000), SC_(9.8071247184113896201098750504331127) }}, {{ SC_(-16.50000000), SC_(9.8107978129935751113862754177425709) }}, {{ SC_(-17.50000000), SC_(9.8140631191160240909781121524364484) }}, {{ SC_(-18.50000000), SC_(9.8169849598757026884945475067096405) }}, {{ SC_(-19.50000000), SC_(9.8196148086593976260356388939548739) }},
      {{ SC_(-0.99902343750000000000000000000000000), SC_(1.0485786445453819054093666757806164e6) }}, {{ SC_(-1.9990234375000000000000000000000000), SC_(1.0485788947897014608301106052600290e6) }}, {{ SC_(-2.9990234375000000000000000000000000), SC_(1.0485790059731858715118158380121717e6) }}, {{ SC_(-3.9990234375000000000000000000000000), SC_(1.0485790685037146291468005792240191e6) }}, {{ SC_(-4.9990234375000000000000000000000000), SC_(1.0485791085193442079759033312640482e6) }}, {{ SC_(-5.9990234375000000000000000000000000), SC_(1.0485791363061664391826604906353499e6) }}, {{ SC_(-6.9990234375000000000000000000000000), SC_(1.0485791567200251382893770121774963e6) }}, {{ SC_(-7.9990234375000000000000000000000000), SC_(1.0485791723488405341606371612842196e6) }}, {{ SC_(-8.9990234375000000000000000000000000), SC_(1.0485791846971991664479297252200281e6) }}, {{ SC_(-9.9990234375000000000000000000000000), SC_(1.0485791946991525775874820980861239e6) }}, {{ SC_(-10.999023437500000000000000000000000), SC_(1.0485792029650829946601990955306663e6) }}, {{ SC_(-11.999023437500000000000000000000000), SC_(1.0485792099106578577646272234105154e6) }}, {{ SC_(-12.999023437500000000000000000000000), SC_(1.0485792158287067176193079741367670e6) }}, {{ SC_(-13.999023437500000000000000000000000), SC_(1.0485792209314593886753540118883170e6) }}, {{ SC_(-14.999023437500000000000000000000000), SC_(1.0485792253764825933424418561922499e6) }}, {{ SC_(-15.999023437500000000000000000000000), SC_(1.0485792292832094741599436510420027e6) }}, {{ SC_(-16.999023437500000000000000000000000), SC_(1.0485792327438146631093348182751076e6) }}, {{ SC_(-17.999023437500000000000000000000000), SC_(1.0485792358305693414284737055180265e6) }}, {{ SC_(-18.999023437500000000000000000000000), SC_(1.0485792386009372194851110066962854e6) }}, {{ SC_(-19.999023437500000000000000000000000), SC_(1.0485792411011813779926686635901794e6) }},
      {{ SC_(-1.0009765625000000000000000000000000), SC_(1.0485786453346669585098368000725210e6)}}, {{ SC_(-2.0009765625000000000000000000000000), SC_(1.0485788950907050310998538089289880e6) }}, {{ SC_(-3.0009765625000000000000000000000000), SC_(1.0485790061295134851948027406034007e6) }}, {{ SC_(-4.0009765625000000000000000000000000), SC_(1.0485790685990070793038292171104670e6) }}, {{ SC_(-5.0009765625000000000000000000000000), SC_(1.0485791085833866557487460417106589e6) }}, {{ SC_(-6.0009765625000000000000000000000000), SC_(1.0485791363521243952566116253382220e6) }}, {{ SC_(-7.0009765625000000000000000000000000), SC_(1.0485791567545946099550113256489222e6) }}, {{ SC_(-8.0009765625000000000000000000000000), SC_(1.0485791723757806110676477942302156e6) }}, {{ SC_(-9.0009765625000000000000000000000000), SC_(1.0485791847187808756018779739003157e6) }}, {{ SC_(-10.000976562500000000000000000000000), SC_(1.0485791947168280366669245397313590e6) }}, {{ SC_(-11.000976562500000000000000000000000), SC_(1.0485792029798236302523575048817360e6) }}, {{ SC_(-12.000976562500000000000000000000000), SC_(1.0485792099231379319842508274674963e6) }}, {{ SC_(-13.000976562500000000000000000000000), SC_(1.0485792158394087991015231685011609e6) }}, {{ SC_(-14.000976562500000000000000000000000), SC_(1.0485792209407379096480891892744762e6) }}, {{ SC_(-15.000976562500000000000000000000000), SC_(1.0485792253846037068979581619480639e6) }}, {{ SC_(-16.000976562500000000000000000000000), SC_(1.0485792292903769133919482794005111e6) }}, {{ SC_(-17.000976562500000000000000000000000), SC_(1.0485792327501870178663179616292787e6) }}, {{ SC_(-18.000976562500000000000000000000000), SC_(1.0485792358362719002281530732844340e6) }}, {{ SC_(-19.000976562500000000000000000000000), SC_(1.0485792386060702710649859237414384e6) }}, {{ SC_(-20.000976562500000000000000000000000), SC_(1.0485792411058261483202152741904670e6) }},
   } };
   do_test_trigamma<T>(data, name, "Mathematica Data");
}

