/***************************************************************************
                          emoticonspage.cpp -  description
                             -------------------
    begin                : Sat May 3 2008
    copyright            : (C) 2008 by Valerio Pilo
    email                : valerio@kmess.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "accountsmanagerpage.h"

#include "../utils/kmessconfig.h"
#include "../account.h"
#include "../accountsmanager.h"
#include "../currentaccount.h"
#include "../kmess.h"
#include "../kmessapplication.h"
#include "../kmessdebug.h"

#include <QListWidget>

#include <KMessageBox>




/**
 * Constructor
 *
 * @param parent  Parent widget
 */
AccountsManagerPage::AccountsManagerPage( QWidget *parent )
: QWidget(parent)
, Ui::AccountsManagerPage()
{
  // First setup the user interface
  setupUi( this );

  // Connect the signals
  connect( addAccountButton_,       SIGNAL(              clicked()                               ),
           this,                      SLOT( showAddAccountDialog()                               ) );
  connect( configureAccountButton_, SIGNAL(              clicked()                               ),
           this,                      SLOT(     configureAccount()                               ) );
  connect( removeAccountButton_,    SIGNAL(              clicked()                               ),
           this,                      SLOT(        deleteAccount()                               ) );
  connect( accountsList_,           SIGNAL(        doubleClicked(QListWidgetItem*,const QPoint&) ),
           this,                      SLOT(     configureAccount()                               ) );
  connect( accountsList_,           SIGNAL( itemSelectionChanged()                               ),
           this,                      SLOT(      accountSelected()                               ) );

  accountsManager_ = AccountsManager::instance();

  connect( accountsManager_,        SIGNAL( accountChanged(Account*,QString,QString)             ),
           this,                    SLOT  ( readProperties()                                     ) );

  readProperties();
}



/**
 * Destructor
 */
AccountsManagerPage::~AccountsManagerPage()
{
}



/**
 * Update the buttons status when accounts get selected
 */
void AccountsManagerPage::accountSelected()
{
  bool hasSelection = ( ! accountsList_->selectedItems().isEmpty() );

  configureAccountButton_->setEnabled( hasSelection );
  removeAccountButton_   ->setEnabled( hasSelection );

  // Disallow removal of the currently connected account.
  if( hasSelection )
  {
    const KMessApplication *kmessApp = static_cast<KMessApplication*>( kapp );
    const QListWidgetItem *selection = accountsList_->selectedItems().first();

    const QString handle( selection->data( Qt::UserRole ).toString() );

    if( kmessApp->getContactListWindow()->isConnected()
    &&  CurrentAccount::instance()->getHandle() == handle )
    {
      removeAccountButton_->setEnabled( false );
    }
  }
}



/**
 * Open the currently selected account's settings
 */
void AccountsManagerPage::configureAccount()
{
  if( accountsList_->selectedItems().isEmpty() )
  {
    return;
  }

  const QListWidgetItem * selection = accountsList_->selectedItems().first();
  const QString accountHandle( selection->data( Qt::UserRole ).toString() );

#ifdef KMESSDEBUG_SETTINGSDIALOG
  kmDebug() << "Opening Account Settings for account" << accountHandle << ".";
#endif

  Account *account = accountsManager_->getAccountByHandle( accountHandle );
  accountsManager_->showAccountSettings( account, window() );
}



/**
 * Delete the currently selected account
 */
void AccountsManagerPage::deleteAccount()
{
  if( accountsList_->selectedItems().isEmpty() )
  {
    kmWarning() << "No account was selected for deletion!";
    return;
  }

  const QListWidgetItem *selection = accountsList_->selectedItems().first();
  const QString handle( selection->data( Qt::UserRole ).toString() );

  // Prevent deletion of the currently connected account
  const KMessApplication *kmessApp = static_cast<KMessApplication*>( kapp );
  if( kmessApp->getContactListWindow()->isConnected()
  &&  CurrentAccount::instance()->getHandle() == handle )
  {
    kmWarning() << "Cannot delete the current account!";
    return;
  }

  Account *account = accountsManager_->getAccountByHandle( handle );
  if( ! account )
  {
    return;
  }

  int result = KMessageBox::warningYesNo(
                                 this,
                                 i18n("<html>Are you sure you want to delete the account '%1' ?<br/>"
                                      "All settings of this account will be lost.</html>",
                                      handle ) );
  if( result != KMessageBox::Yes )
  {
    return;
  }

  accountsManager_->deleteAccount( account );

  // Refresh the accounts list
  readProperties();
}



/**
 * Load the settings of the dialog
 */
void AccountsManagerPage::readProperties()
{
  accountsList_->clear();
  const QList<Account*>& accountsList( AccountsManager::instance()->getAccounts() );

  foreach( Account *account, accountsList )
  {
    const QString text( account->getFriendlyName() + "\n" + account->getHandle() );
    QListWidgetItem *item = new QListWidgetItem( QIcon( account->getPicturePath() ), text, accountsList_ );
    item->setData( Qt::UserRole, account->getHandle() );
  }

  // Update the viewport and the buttons
  accountSelected();
}



/**
 * Show the Account Settings dialog for a new account
 */
void AccountsManagerPage::showAddAccountDialog()
{
  accountsManager_->showAccountSettings();
}



#include "accountsmanagerpage.moc"
