/* SPDX-License-Identifier: LGPL-2.1-or-later */
#include <string.h>
#include <errno.h>

#include "internal.h"

/**
 * SECTION:snd_dg00x
 * @Title: HinawaSndDg00x
 * @Short_description: A notification listener for Dg00x models
 * @include: snd_dg00x.h
 *
 * A #HinawaSndDg00x listen to Dg00x notification and generates signal when
 * received. This inherits #HinawaSndUnit.
 */

G_DEFINE_TYPE(HinawaSndDg00x, hinawa_snd_dg00x, HINAWA_TYPE_SND_UNIT)

/* This object has one signal. */
enum dg00x_sig_type {
	DG00X_SIG_TYPE_MESSAGE,
	DG00X_SIG_TYPE_COUNT,
};
static guint dg00x_sigs[DG00X_SIG_TYPE_COUNT] = { 0 };

static void hinawa_snd_dg00x_class_init(HinawaSndDg00xClass *klass)
{
	/**
	 * HinawaSndDg00x::message:
	 * @self: A #HinawaSndDg00x
	 * @message: A message
	 *
	 * When Dg00x models transfer notification, the #HinawaSndDg00x::message signal is
	 * generated.
	 *
	 * Since: 0.7
	 */
	dg00x_sigs[DG00X_SIG_TYPE_MESSAGE] =
		g_signal_new("message",
			     G_OBJECT_CLASS_TYPE(klass),
			     G_SIGNAL_RUN_LAST,
			     G_STRUCT_OFFSET(HinawaSndDg00xClass, message),
			     NULL, NULL,
			     g_cclosure_marshal_VOID__UINT,
			     G_TYPE_NONE, 1, G_TYPE_UINT);
}

static void hinawa_snd_dg00x_init(HinawaSndDg00x *self)
{
	return;
}

/**
 * hinawa_snd_dg00x_new:
 *
 * Instantiate #HinawaSndDg00x object and return the instance.
 *
 * Returns: an instance of #HinawaSndDg00x.
 * Since: 1.3.
 */
HinawaSndDg00x *hinawa_snd_dg00x_new(void)
{
	return g_object_new(HINAWA_TYPE_SND_DG00X, NULL);
}

/**
 * hinawa_snd_dg00x_open:
 * @self: A #HinawaSndUnit
 * @path: A full path of a special file for ALSA hwdep character device
 * @exception: A #GError. Error can be generated with three domains; #g_file_error_quark(),
 *	       #hinawa_fw_node_error_quark(), and #hinawa_snd_unit_error_quark().
 *
 * Open ALSA hwdep character device and check it for Dg00x  devices.
 *
 * Since: 0.7
 */
void hinawa_snd_dg00x_open(HinawaSndDg00x *self, gchar *path, GError **exception)
{
	g_return_if_fail(HINAWA_IS_SND_DG00X(self));
	g_return_if_fail(path != NULL && strlen(path) > 0);
	g_return_if_fail(exception == NULL || *exception == NULL);

	hinawa_snd_unit_open(&self->parent_instance, path, exception);
}

void hinawa_snd_dg00x_handle_msg(HinawaSndDg00x *self, const void *buf,
				 ssize_t len)
{
	const struct snd_firewire_event_digi00x_message *event = buf;

	g_return_if_fail(HINAWA_IS_SND_DG00X(self));

	g_signal_emit(self, dg00x_sigs[DG00X_SIG_TYPE_MESSAGE], 0,
		      event->message);
}
