% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/setupBasiliskEnv.R
\name{setupBasiliskEnv}
\alias{setupBasiliskEnv}
\title{Set up \pkg{basilisk}-managed environments}
\usage{
setupBasiliskEnv(
  envpath,
  packages,
  channels = "conda-forge",
  pip = NULL,
  paths = NULL
)
}
\arguments{
\item{envpath}{String containing the path to the environment to use.}

\item{packages}{Character vector containing the names of conda packages to install into the environment.
Version numbers must be included.}

\item{channels}{Character vector containing the names of Conda channels to search.
Defaults to the conda-forge repository.}

\item{pip}{Character vector containing the names of additional packages to install from PyPi using \code{pip}.
Version numbers must be included.}

\item{paths}{Character vector containing absolute paths to Python package directories, to be installed by \code{pip}.}
}
\value{
A conda environment is created at \code{envpath} containing the specified \code{packages}.
A \code{NULL} is invisibly returned.
}
\description{
Set up a conda environment for isolated execution of Python code with appropriate versions of all Python packages.
}
\details{
Developers of \pkg{basilisk} client packages should never need to call this function directly.
For typical usage, \code{setupBasiliskEnv} is automatically called by \code{\link{basiliskStart}} to perform lazy installation.
Developers should also create \code{configure(.win)} files to call \code{\link{configureBasiliskEnv}},
which will call \code{setupBasiliskEnv} during R package installation when \code{BASILISK_USE_SYSTEM_DIR=1}.

Pinned version numbers must be present for all desired conda packages in \code{packages}.
This improves predictability and simplifies debugging across different systems.
Note that the version notation for conda packages uses a single \code{=}, while the notation for Python packages uses \code{==}; any instances of the latter will be coerced to the former automatically.

For \code{channels}, we recommend using open-source repositories like conda-forge and bioconda.
This avoids problems with non-open-source licensing of the main Anaconda repositories (i.e., the \code{"defaults"} channel).
If a client package relies on non-free channels, its users may inadvertently violate the Anaconda license,
e.g., when used in a commercial environment.

After all conda packages are installed, additional packages can be installed from PyPi using the \code{pip} argument.
All packages listed here are also expected to have pinned versions, this time using the \code{==} notation.
However, some caution is required when mixing packages from conda and pip,
see \url{https://www.anaconda.com/using-pip-in-a-conda-environment} for more details.

After conda and PyPi, more Python packages can be installed from local directories via the \code{paths} argument.
This is useful for \pkg{basilisk} clients vendoring Python packages that are not available in standard repositories.
While \code{paths} expects absolute paths for general usage, this will be auto-generated in a package development context -
see \code{\link{BasiliskEnvironment}} for details.

It is also good practice to explicitly list the versions of the \emph{dependencies} of all desired packages.
This protects against future changes in the behavior of your code if conda's solver decides to use a different version of a dependency.
To identify appropriate versions of dependencies, we suggest:
\enumerate{
\item Creating a fresh conda environment with the desired packages, using \code{packages=} in \code{setupBasiliskEnv}.
\item Calling \code{\link{listPackages}} on the environment to identify any relevant dependencies and their versions.
\item Including those dependencies in the \code{packages=} argument for future use.
(It is helpful to mark dependencies in some manner, e.g., with comments, to distinguish them from the actual desired packages.)
}
The only reason that pinned dependencies are not mandatory is because some dependencies are OS-specific,
requiring some manual pruning of the output of \code{\link{listPackages}}.

If versions for the desired conda packages are not known beforehand, developers may use \code{\link{setBasiliskCheckVersions}(FALSE)} before running \code{setupBasiliskEnv}.
This instructs conda to create an environment with appropriate versions of all unpinned packages, 
which can then be read out via \code{\link{listPackages}} for insertion in the \code{packages=} argument as described above.
We stress that this option should \emph{not} be used in any release of the R package, it is a development-phase-only utility.

If no Python version is listed, the version in the base conda installation is used by default - check \code{\link{listPythonVersion}} for the version number.
However, it is often prudent to explicitly list the desired version of Python in \code{packages}, even if this is already version-compatible with the default (e.g., \code{"python=3.8"}).
This protects against changes to the Python version in the base installation, e.g., if administrators override the choice of conda installation with certain environment variables.
Of course, it is possible to specify an entirely different version of Python in \code{packages} by supplying, e.g., \code{"python=2.7.10"}.
}
\examples{
if (.Platform$OS.type != "windows") {
 \dontshow{basilisk.utils::installConda()}

 tmploc <- file.path(tempdir(), "my_package_A")
 if (!file.exists(tmploc)) {
     setupBasiliskEnv(tmploc, c('pandas=1.4.3'))
 }
}

}
\seealso{
\code{\link{listPackages}}, to list the packages in the conda environment.
}
