"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.uiToBackendReportDefinition = exports.getBackendReportState = exports.getBackendReportSource = void 0;
var _backendModel = require("../../../model/backendModel");
var _constants = require("../constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const uiToBackendReportDefinition = reportDefinition => {
  const {
    report_params: {
      report_name: reportName,
      description,
      report_source: reportSource,
      core_params: {
        base_url: baseUrl,
        time_duration: timeDuration,
        report_format: reportFormat,
        header,
        footer,
        limit,
        origin
      }
    },
    trigger,
    delivery
  } = reportDefinition;
  let backendReportDefinition = {
    name: reportName,
    isEnabled: getBackendIsEnabled(trigger),
    source: {
      description: description,
      type: getBackendReportSource(reportSource),
      id: getBackendReportSourceId(baseUrl),
      origin: origin
    },
    format: {
      duration: timeDuration,
      fileFormat: getBackendReportFormat(reportFormat),
      ...(limit && {
        limit: limit
      }),
      ...(header && {
        header: header
      }),
      ...(footer && {
        footer: footer
      })
    },
    trigger: getBackendTrigger(trigger),
    ...(getBackendDelivery(delivery) && {
      delivery: getBackendDelivery(delivery)
    })
  };
  return backendReportDefinition;
};
exports.uiToBackendReportDefinition = uiToBackendReportDefinition;
const getBackendIsEnabled = trigger => {
  let enabled = true;
  if (trigger.trigger_params) {
    enabled = trigger.trigger_params.enabled;
  }
  return enabled;
};
const getBackendDelivery = delivery => {
  const {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  } = delivery;
  let res = {
    configIds: configIds,
    title: title,
    textDescription: textDescription,
    htmlDescription: htmlDescription
  };
  return res;
};
const getBackendTrigger = trigger => {
  const {
    trigger_params: scheduleParams
  } = trigger;
  const {
    schedule
  } = {
    ...scheduleParams
  };
  let res = {
    triggerType: scheduleParams ? getBackendTriggerType(scheduleParams) : _backendModel.BACKEND_TRIGGER_TYPE.onDemand,
    schedule: schedule
  };
  return res;
};
const getBackendTriggerType = scheduleParams => {
  const {
    schedule_type: scheduleType
  } = scheduleParams;
  let res;
  switch (scheduleType) {
    case _constants.SCHEDULE_TYPE.cron:
      res = _backendModel.BACKEND_TRIGGER_TYPE.cronSchedule;
      break;
    case _constants.SCHEDULE_TYPE.recurring:
      res = _backendModel.BACKEND_TRIGGER_TYPE.intervalSchedule;
      break;
  }
  return res;
};
const getBackendReportFormat = reportFormat => {
  return _backendModel.REPORT_FORMAT_DICT[reportFormat];
};
const getBackendReportState = reportState => {
  return _backendModel.REPORT_STATE_DICT[reportState];
};
exports.getBackendReportState = getBackendReportState;
const getBackendReportSource = reportSource => {
  return _backendModel.REPORT_SOURCE_DICT[reportSource];
};
//TODO: tmp solution, we are extracting the id from the baseUrl, e.g. /app/dashboards#/view/<id>
// since currently dashboard/visualization id are not required in the UI model, will add in the future
exports.getBackendReportSource = getBackendReportSource;
const getBackendReportSourceId = baseUrl => {
  const id = baseUrl.split('/').pop() || '';
  return id;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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