/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.ssl;

import java.io.IOError;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;
import org.wildfly.security.ssl.Authentication;
import org.wildfly.security.ssl.Digest;
import org.wildfly.security.ssl.ElytronMessages;
import org.wildfly.security.ssl.Encryption;
import org.wildfly.security.ssl.KeyAgreement;
import org.wildfly.security.ssl.LinkedProperties;
import org.wildfly.security.ssl.Protocol;
import org.wildfly.security.ssl.SecurityLevel;

class MechanismDatabase {
    private static final MechanismDatabase INSTANCE = new MechanismDatabase("MechanismDatabase.properties");
    private static final MechanismDatabase TLS13_INSTANCE = new MechanismDatabase("TLS13MechanismDatabase.properties", true);
    private final Map<String, Entry> entriesByStdName;
    private final Map<String, Entry> entriesByOSSLName;
    private final Entry[][] algorithmsById;
    private final boolean isTLS13;

    static MechanismDatabase getInstance() {
        return INSTANCE;
    }

    static MechanismDatabase getTLS13Instance() {
        return TLS13_INSTANCE;
    }

    MechanismDatabase(String databaseFileName) {
        this(databaseFileName, false);
    }

    MechanismDatabase(String databaseFileName, boolean isTLS13) {
        String name;
        this.isTLS13 = isTLS13;
        LinkedProperties properties = new LinkedProperties();
        try (InputStream stream = this.getClass().getResourceAsStream(databaseFileName);
             InputStreamReader reader = new InputStreamReader(stream, StandardCharsets.UTF_8);){
            properties.load(reader);
        }
        catch (IOException e) {
            throw new IOError(e);
        }
        Pattern p = Pattern.compile("\\s*,\\s*");
        LinkedHashMap<String, Entry> entriesByStdName = new LinkedHashMap<String, Entry>();
        LinkedHashMap<String, Entry> entriesByOSSLName = new LinkedHashMap<String, Entry>();
        EnumMap entriesByKeyExchange = new EnumMap(KeyAgreement.class);
        EnumMap entriesByAuthentication = new EnumMap(Authentication.class);
        EnumMap entriesByEncryption = new EnumMap(Encryption.class);
        EnumMap entriesByDigest = new EnumMap(Digest.class);
        EnumMap entriesByProtocol = new EnumMap(Protocol.class);
        EnumMap entriesByLevel = new EnumMap(SecurityLevel.class);
        HashMap<String, String> aliases = new HashMap<String, String>();
        Entry[][] algorithms = new Entry[256][];
        for (Map.Entry<String, String> entry : properties.stringMapEntries()) {
            int byte2;
            int byte1;
            int algBits;
            int strBits;
            Protocol prot;
            Digest digest;
            Encryption enc;
            name = entry.getKey();
            String rawValue = entry.getValue();
            String[] strings = p.split(rawValue);
            if (strings.length == 1 && strings[0].startsWith("alias:")) {
                String target = strings[0].substring(6);
                aliases.put(name, target);
                if (!name.startsWith("TLS_")) continue;
                aliases.put("SSL_" + name.substring(4), target);
                continue;
            }
            if (strings.length != 11 && strings.length != 13) {
                ElytronMessages.log.warnInvalidStringCountForMechanismDatabaseEntry(name);
                continue;
            }
            boolean ok = true;
            String[] openSslNames = strings[0].split("\\|");
            KeyAgreement key = KeyAgreement.forName(strings[1]);
            if (!isTLS13 && key == null) {
                ElytronMessages.log.warnInvalidKeyExchangeForMechanismDatabaseEntry(strings[1], name);
                ok = false;
            }
            Authentication auth = Authentication.forName(strings[2]);
            if (!isTLS13 && auth == null || isTLS13 && !strings[2].equals("ANY")) {
                ElytronMessages.log.warnInvalidAuthenticationForMechanismDatabaseEntry(strings[2], name);
                ok = false;
            }
            if ((enc = Encryption.forName(strings[3])) == null) {
                ElytronMessages.log.warnInvalidEncryptionForMechanismDatabaseEntry(strings[3], name);
                ok = false;
            }
            if ((digest = Digest.forName(strings[4])) == null) {
                ElytronMessages.log.warnInvalidDigestForMechanismDatabaseEntry(strings[4], name);
                ok = false;
            }
            if ((prot = Protocol.forName(strings[5])) == null || isTLS13 ^ prot.equals((Object)Protocol.TLSv1_3)) {
                ElytronMessages.log.warnInvalidProtocolForMechanismDatabaseEntry(strings[5], name);
                ok = false;
            }
            boolean export = Boolean.parseBoolean(strings[6]);
            SecurityLevel level = SecurityLevel.forName(strings[7]);
            if (level == null) {
                ElytronMessages.log.warnInvalidLevelForMechanismDatabaseEntry(strings[7], name);
                ok = false;
            }
            boolean fips = Boolean.parseBoolean(strings[8]);
            try {
                strBits = Integer.parseInt(strings[9], 10);
            }
            catch (NumberFormatException ignored) {
                ElytronMessages.log.warnInvalidStrengthBitsForMechanismDatabaseEntry(strings[9], name);
                strBits = 0;
                ok = false;
            }
            try {
                algBits = Integer.parseInt(strings[10], 10);
            }
            catch (NumberFormatException ignored) {
                ElytronMessages.log.warnInvalidAlgorithmBitsForMechanismDatabaseEntry(strings[10], name);
                algBits = 0;
                ok = false;
            }
            try {
                byte1 = Integer.parseInt(strings[11], 16);
                byte2 = Integer.parseInt(strings[12], 16);
            }
            catch (ArrayIndexOutOfBoundsException | NumberFormatException ignored) {
                byte1 = -1;
                byte2 = -1;
            }
            if (!ok) continue;
            Entry entry2 = new Entry(name, Arrays.asList(openSslNames), new ArrayList<String>(0), key, auth, enc, digest, prot, export, level, fips, strBits, algBits);
            if (entriesByStdName.containsKey(name)) {
                ElytronMessages.log.warnInvalidDuplicateMechanismDatabaseEntry(name);
            } else {
                entriesByStdName.put(name, entry2);
            }
            for (String openSslName : openSslNames) {
                String newKey;
                if (entriesByOSSLName.containsKey(openSslName)) {
                    ElytronMessages.log.warnInvalidDuplicateOpenSslStyleAliasForMechanismDatabaseEntry(openSslName, name, ((Entry)entriesByOSSLName.get(openSslName)).getName());
                } else {
                    entriesByOSSLName.put(openSslName, entry2);
                }
                if (key != KeyAgreement.DHE || entriesByOSSLName.containsKey(newKey = MechanismDatabase.join("-", MechanismDatabase.replaceEdh(openSslName.split("-"))))) continue;
                entriesByOSSLName.put(newKey, entry2);
            }
            if (!isTLS13) {
                MechanismDatabase.addTo(entriesByKeyExchange, key, entry2);
                MechanismDatabase.addTo(entriesByAuthentication, auth, entry2);
            }
            MechanismDatabase.addTo(entriesByEncryption, enc, entry2);
            MechanismDatabase.addTo(entriesByDigest, digest, entry2);
            MechanismDatabase.addTo(entriesByProtocol, prot, entry2);
            MechanismDatabase.addTo(entriesByLevel, level, entry2);
            if (byte1 != -1 && byte2 != -1) {
                Entry[] row = algorithms[byte1];
                if (row == null) {
                    row = algorithms[byte1] = new Entry[256];
                }
                row[byte2] = entry2;
            }
            if (!name.startsWith("TLS_")) continue;
            aliases.put("SSL_" + name.substring(4), name);
        }
        for (Map.Entry<String, String> entry : aliases.entrySet()) {
            name = entry.getKey();
            String value = entry.getValue();
            while (aliases.containsKey(value)) {
                value = (String)aliases.get(value);
            }
            if (entriesByStdName.containsKey(name)) {
                ElytronMessages.log.warnInvalidDuplicateMechanismDatabaseEntry(name);
                continue;
            }
            if (!entriesByStdName.containsKey(value)) {
                ElytronMessages.log.warnInvalidAliasForMissingMechanismDatabaseEntry(value, name);
                continue;
            }
            Entry dbEntry = (Entry)entriesByStdName.get(value);
            dbEntry.getAliases().add(name);
            entriesByStdName.put(name, dbEntry);
        }
        this.entriesByStdName = entriesByStdName;
        this.entriesByOSSLName = entriesByOSSLName;
        this.algorithmsById = algorithms;
    }

    static String[] replaceEdh(String ... strings) {
        int stringsLength = strings.length;
        for (int i = 0; i < stringsLength; ++i) {
            if ("EDH".equals(strings[i])) {
                strings[i] = "DHE";
                continue;
            }
            if (!"DHE".equals(strings[i])) continue;
            strings[i] = "EDH";
        }
        return strings;
    }

    static String join(String joiner, String ... strings) {
        StringBuilder b = new StringBuilder();
        int length = strings.length;
        int i = 0;
        if (length == 0) {
            return "";
        }
        while (true) {
            b.append(strings[i++]);
            if (i == length) {
                return b.toString();
            }
            b.append(joiner);
        }
    }

    static <T> void addTo(Map<T, List<Entry>> map, T item, Entry entry) {
        List<Entry> list = map.get(item);
        if (list == null) {
            list = new ArrayList<Entry>(1);
            map.put(item, list);
        }
        list.add(entry);
    }

    Entry getCipherSuite(String cipherSuite) {
        return this.entriesByStdName.get(cipherSuite);
    }

    Entry getCipherSuiteOpenSSLName(String cipherSuite) {
        return this.entriesByOSSLName.get(cipherSuite);
    }

    Entry getCipherSuiteById(int byte1, int byte2) {
        if (byte1 < 0 || byte1 > 255 || byte2 < 0 || byte2 > 255) {
            return null;
        }
        Entry[] row = this.algorithmsById[byte1];
        if (row == null) {
            return null;
        }
        return row[byte2];
    }

    boolean isTLS13() {
        return this.isTLS13;
    }

    static final class Entry {
        private final String name;
        private final List<String> openSslNames;
        private final List<String> aliases;
        private final KeyAgreement keyAgreement;
        private final Authentication authentication;
        private final Encryption encryption;
        private final Digest digest;
        private final Protocol protocol;
        private final boolean export;
        private final SecurityLevel level;
        private final boolean fips;
        private final int strengthBits;
        private final int algorithmBits;

        Entry(String name, List<String> openSslNames, List<String> aliases, KeyAgreement keyAgreement, Authentication authentication, Encryption encryption, Digest digest, Protocol protocol, boolean export, SecurityLevel level, boolean fips, int strengthBits, int algorithmBits) {
            this.name = name;
            this.openSslNames = openSslNames;
            this.aliases = aliases;
            this.keyAgreement = keyAgreement;
            this.authentication = authentication;
            this.encryption = encryption;
            this.digest = digest;
            this.protocol = protocol;
            this.export = export;
            this.level = level;
            this.fips = fips;
            this.strengthBits = strengthBits;
            this.algorithmBits = algorithmBits;
        }

        public String getName() {
            return this.name;
        }

        public List<String> getOpenSslNames() {
            return this.openSslNames;
        }

        public List<String> getAliases() {
            return this.aliases;
        }

        public KeyAgreement getKeyAgreement() {
            return this.keyAgreement;
        }

        public Authentication getAuthentication() {
            return this.authentication;
        }

        public Encryption getEncryption() {
            return this.encryption;
        }

        public Digest getDigest() {
            return this.digest;
        }

        public Protocol getProtocol() {
            return this.protocol;
        }

        public boolean isExport() {
            return this.export;
        }

        public SecurityLevel getLevel() {
            return this.level;
        }

        public boolean isFips() {
            return this.fips;
        }

        public int getStrengthBits() {
            return this.strengthBits;
        }

        public int getAlgorithmBits() {
            return this.algorithmBits;
        }

        public String toString() {
            return this.getName() + "/" + String.join((CharSequence)"|", this.openSslNames);
        }
    }
}

