/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*- */
/*
    This file is part of Déjà Dup.
    For copyright information, see AUTHORS.

    Déjà Dup is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Déjà Dup is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Déjà Dup.  If not, see <http://www.gnu.org/licenses/>.
*/

using GLib;

namespace DejaDup {

public class ConfigDelete : ConfigChoice
{
  public const int SEMIANNUALLY = 365 / 2;
  public const int ANNUALLY = 365;
  public const int FOREVER = int.MAX;

  public Hdy.ComboRow row { construct set; get; }

  public ConfigDelete(Hdy.ComboRow row, string key, string ns="") {
    Object(key: key, ns: ns, row: row);
  }

  construct {
    var store = new ListStore(typeof(Hdy.ValueObject));

    var val = Value (typeof (int));
    val.set_int (SEMIANNUALLY);
    store.append(new Hdy.ValueObject (val));

    val = Value (typeof (int));
    val.set_int (ANNUALLY);
    store.append(new Hdy.ValueObject (val));

    val = Value (typeof (int));
    val.set_int (FOREVER);
    store.append(new Hdy.ValueObject (val));

    row.bind_name_model(store, days_to_name);

    init(row);
  }

  public static string days_to_name(Object object)
  {
    var val_object = object as Hdy.ValueObject;
    var val = val_object.get_value();
    var days = val.get_int();
    if (days == SEMIANNUALLY)
      return _("At least six months");
    else if (days == ANNUALLY)
      return _("At least a year");
    else if (days == FOREVER)
      return _("Forever");
    else
      return dngettext(Config.GETTEXT_PACKAGE, "At least %d day", "At least %d days", days);
  }

  protected override void handle_changed()
  {
    Value? val = get_current_value();
    int intval = val == null ? 0 : val.get_int();
    if (intval == int.MAX)
      intval = 0; // forever

    settings.set_int(key, intval);

    choice_changed(intval.to_string());
  }

  protected override async void set_from_config()
  {
    var confval = settings.get_int(key);
    if (confval <= 0)
      confval = int.MAX;

    bool valid = false;
    var model = row.get_model();
    for (int i = 0; i < model.get_n_items(); i++) {
      var val_object = model.get_object(i) as Hdy.ValueObject;
      var val = val_object.get_value();

      if (val.get_int() != confval)
        continue;

      valid = true;
      row.selected_index = i;

      break;
    }

    // If we didn't find the time, user must have set it to something non
    // standard.  Let's add an entry to the combo.
    if (!valid) {
      var store = row.get_model() as ListStore;
      var val = Value (typeof (int));
      val.set_int (confval);
      store.append(new Hdy.ValueObject (val));
      row.selected_index = (int) store.get_n_items() - 1;
    }
  }
}

}
