/*
 * Copyright (C) 2018 Purism SPC
 *
 * This file is part of Hægtesse.
 *
 * Hægtesse is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * Hægtesse is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Hægtesse.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Bob Ham <bob.ham@puri.sm>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 */

#include "haeg-port-monitor.h"
#include "util.h"

#include <glib/gi18n.h>
#include <glib-object.h>
#include <gio/gio.h>

#include <libudev.h>

#define ID_VENDOR          "1e0e"
#define ID_PRODUCT         "9001"
#define B_INTERFACE_NUMBER "04"

/**
 * SECTION:haeg-port-monitor
 * @short_description: Monitor for the appearance of an appropriate TTY port
 * @Title: HaegPortMonitor
 */

struct _HaegPortMonitor
{
  GObject parent_instance;

  struct udev         *ctx;
  struct udev_monitor *monitor;
  GIOChannel          *channel;
  guint                watch_id;
  gchar               *port_node;
};

G_DEFINE_TYPE (HaegPortMonitor, haeg_port_monitor, G_TYPE_OBJECT);


enum {
  SIGNAL_PORT,
  SIGNAL_LAST_SIGNAL,
};
static guint signals [SIGNAL_LAST_SIGNAL];

static gboolean
check_device_sysattr(struct udev_device *device,
                     const char *sysattr,
                     const char *match_value)
{
  const char *value;

  value = udev_device_get_sysattr_value (device, sysattr);
  if (!value)
    {
      return FALSE;
    }

  return strcmp (value, match_value) == 0;
}

static gboolean
check_device (struct udev_device *device)
{
  struct udev_device *parent;
  gboolean match;

  /* Check USB vendor/product */
  parent = udev_device_get_parent_with_subsystem_devtype
    (device, "usb", "usb_device");
  if (!parent)
    {
      //g_debug ("No usb_device parent");
      return FALSE;
    }

  match = check_device_sysattr (parent, "idVendor", ID_VENDOR);
  if (!match)
    {
      //g_debug ("idVendor does not match");
      return FALSE;
    }

  match = check_device_sysattr (parent, "idProduct", ID_PRODUCT);
  if (!match)
    {
      //g_debug ("idProduct does not match");
      return FALSE;
    }

  /* Check USB interface number */
  parent = udev_device_get_parent_with_subsystem_devtype
    (device, "usb", "usb_interface");
  if (!parent)
    {
      //g_debug ("No usb_interface parent");
      return FALSE;
    }

  return check_device_sysattr (parent, "bInterfaceNumber",
                               B_INTERFACE_NUMBER);
}


static void
check_monitor_device (HaegPortMonitor *self,
                      struct udev_device *device)
{
  const char *action;
  gboolean match;
  const char *devnode;

  action = udev_device_get_action (device);
  if (!action)
    {
      g_warning ("No action from udev device");
      return;
    }

  if (strcmp (action, "add")    != 0 &&
      strcmp (action, "remove") != 0)
    {
      return;
    }

  devnode = udev_device_get_devnode (device);
  if (strcmp (action, "add") == 0)
    {
      match = check_device (device);
      if (!match)
        {
          //g_debug ("udev device `%s' is not what we're looking for",
          //       devnode);
          return;
        }

      if (!self->port_node)
        {
          g_debug ("TTY port `%s' added", devnode);
          self->port_node = g_strdup (devnode);
          g_signal_emit (self, signals[SIGNAL_PORT],
                         0, self->port_node);
        }
      else
        {
          g_warning ("Extraneous TTY port `%s' addition from udev, ignoring", devnode);
        }
    }
  else
    {
      if (!self->port_node
          || strcmp (self->port_node, devnode) != 0)
        {
          return;
        }

      g_debug ("TTY port `%s' removed", devnode);
      g_free (self->port_node);
      self->port_node = NULL;
    }
}

static void
read_device (HaegPortMonitor *self)
{
  struct udev_device *device;

  device = udev_monitor_receive_device (self->monitor);
  if (!device)
    {
      haeg_error ("Error receiving udev device from monitor");
    }

  check_monitor_device (self, device);
  udev_device_unref (device);
}


static gboolean
watch_cb (GIOChannel      *source,
          GIOCondition     condition,
          HaegPortMonitor *self)
{
  const gchar *err_msg = NULL;

  /* Check what happened */
  switch (condition)
    {
    case G_IO_IN:
    case G_IO_PRI:
      read_device (self);
    case G_IO_OUT:
      return TRUE;

    case G_IO_ERR:
      err_msg = "error condition";
      break;
    case G_IO_HUP:
      err_msg = "file hung up";
      break;
    case G_IO_NVAL:
      err_msg = "invalid request";
      break;
    }

  haeg_error ("Error during udev monitor IO watch: %s", err_msg);
  return FALSE;
}


static void
enumerate (HaegPortMonitor *self)
{
  struct udev_enumerate *enumerator;
  int err;
  struct udev_list_entry *devices, *node;
  const gchar *name;
  struct udev_device *device;
  gboolean match;

  enumerator = udev_enumerate_new (self->ctx);
  if (!enumerator)
    {
      haeg_error ("Error creating udev enumerator");
    }

  err = udev_enumerate_add_match_subsystem (enumerator, "tty");
  if (err < 0)
    {
      haeg_error ("Error adding subsystem match to udev enumerator");
    }

  err = udev_enumerate_scan_devices (enumerator);
  if (err < 0)
    {
      haeg_error ("Error enumerating udev devices");
    }

  devices = udev_enumerate_get_list_entry (enumerator);
  udev_list_entry_foreach (node, devices)
    {
      name = udev_list_entry_get_name (node);

      device = udev_device_new_from_syspath (self->ctx, name);
      if (!device)
        {
          haeg_error ("Error creating udev device from name `%s'",
                      name);
        }

      match = check_device (device);
      if (match)
        {
          const char *devnode;

          devnode = udev_device_get_devnode (device);
          g_debug ("Found TTY port `%s' while enumerating devices",
                   devnode);
          self->port_node = g_strdup (devnode);
        }

      udev_device_unref (device);

      if (match)
        {
          break;
        }
    }

  udev_enumerate_unref (enumerator);
}


static void
haeg_port_monitor_init (HaegPortMonitor *self)
{
}


static void
constructed (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegPortMonitor *self = HAEG_PORT_MONITOR (object);

  // Create context
  self->ctx = udev_new();
  if (!self->ctx)
    {
      haeg_error ("Error creating udev context");
    }

  // Create monitor
  self->monitor = udev_monitor_new_from_netlink
    (self->ctx, "udev");
  if (!self->monitor)
    {
      haeg_error ("Error creating udev monitor");
    }

  udev_monitor_filter_add_match_subsystem_devtype
    (self->monitor, "tty", NULL);
  udev_monitor_enable_receiving (self->monitor);

  // Create channel
  self->channel = g_io_channel_unix_new
    (udev_monitor_get_fd (self->monitor));

  self->watch_id =
    g_io_add_watch (self->channel, G_IO_IN|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
                    (GIOFunc)watch_cb, self);

  // Enumerate
  enumerate (self);

  parent_class->constructed (object);
}


static void
dispose (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegPortMonitor *self = HAEG_PORT_MONITOR (object);

  if (self->watch_id)
    {
      g_source_remove (self->watch_id);
      self->watch_id = 0;
    }

  if (self->channel)
    {
      g_io_channel_unref (self->channel);
      self->channel = NULL;
    }

  if (self->monitor)
    {
      udev_monitor_unref (self->monitor);
      self->monitor = NULL;
    }

  if (self->ctx)
    {
      udev_unref (self->ctx);
      self->ctx = NULL;
    }

  parent_class->dispose (object);
}


static void
finalize (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegPortMonitor *self = HAEG_PORT_MONITOR (object);

  g_free (self->port_node);

  parent_class->finalize (object);
}


static void
haeg_port_monitor_class_init (HaegPortMonitorClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->constructed  = constructed;
  object_class->dispose      = dispose;
  object_class->finalize     = finalize;

  /**
   * HaegPortMonitor::port:
   * @self: The #HaegPortMonitor instance.
   *
   * The TTY port has been added or requested.
   */
  signals[SIGNAL_PORT] =
    g_signal_new ("port",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0, NULL, NULL, NULL,
                  G_TYPE_NONE,
                  1,
                  G_TYPE_STRING);
}


HaegPortMonitor *
haeg_port_monitor_new ()
{
  return g_object_new (HAEG_TYPE_PORT_MONITOR, NULL);
}


static gboolean
request_port (HaegPortMonitor *self)
{
  if (self->port_node)
    {
      g_signal_emit (self, signals[SIGNAL_PORT],
                     0, self->port_node);
    }

  return FALSE;
}


void
haeg_port_monitor_request_port (HaegPortMonitor *self)
{
  g_idle_add ((GSourceFunc)request_port, self);
}
