/*
 * Copyright (C) 2018 Purism SPC
 *
 * This file is part of Hægtesse.
 *
 * Hægtesse is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * Hægtesse is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Hægtesse.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Bob Ham <bob.ham@puri.sm>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 */

#include "haeg-port.h"
#include "haeg-sample-buffer.h"

#include <glib/gi18n.h>
#include <glib-object.h>
#include <gio/gio.h>

#include <termios.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/**
 * SECTION:haeg-port
 * @short_description: Abstraction of the TTY port used to interface
 * with the modem.
 * @Title: HaegPort
 */

struct _HaegPort
{
  GObject parent_instance;

  gchar            *file_name;
  gsize             sample_len;
  int               fd;
  GIOChannel       *channel;
  HaegSampleBuffer *read_buffer;
  guint             read_watch_id;
};

static void haeg_port_initable_interface_init (GInitableIface *iface);

G_DEFINE_TYPE_WITH_CODE (HaegPort, haeg_port, G_TYPE_OBJECT,
                         G_IMPLEMENT_INTERFACE (G_TYPE_INITABLE,
                                                haeg_port_initable_interface_init))


enum {
  PROP_0,
  PROP_FILE_NAME,
  PROP_SAMPLE_LEN,
  PROP_LAST_PROP,
};
static GParamSpec *props[PROP_LAST_PROP];

enum {
  SIGNAL_READ_READY,
  SIGNAL_ERROR,
  SIGNAL_LAST_SIGNAL,
};
static guint signals [SIGNAL_LAST_SIGNAL];


static void
set_property (GObject      *object,
              guint         property_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  HaegPort *self = HAEG_PORT (object);

  switch (property_id) {
  case PROP_FILE_NAME:
    self->file_name = g_value_dup_string (value);
    break;

  case PROP_SAMPLE_LEN:
    self->sample_len = g_value_get_ulong (value);
    break;

  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}


static void
haeg_port_init (HaegPort *self)
{
  self->fd = -1;
}


static gboolean
set_up_tty (HaegPort *self, GError **error)
{
  struct termios tty;
  int err;

  memset (&tty, 0, sizeof (tty));

#define try(call,text)                                  \
  err = call ;                                          \
  if (err != 0)                                         \
    {                                                   \
      g_set_error (error,                               \
                   G_IO_ERROR,                          \
                   g_io_error_from_errno (errno),       \
                   "Error " text                        \
                   " of TTY port `%s': %s",             \
                   self->file_name,                     \
                   g_strerror (errno));                 \
      return FALSE;                                     \
    }

  try (tcgetattr (self->fd, &tty), "getting attributes");
  try (cfsetospeed (&tty, B115200), "setting output speed");
  try (cfsetispeed (&tty, B115200), "setting input speed");

  tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8;
  tty.c_cflag |= (CLOCAL | CREAD);
  tty.c_cflag &= ~(PARENB | PARODD | CSTOPB | CRTSCTS);
  tty.c_iflag &= ~(ICANON | IGNBRK | IXON | IXOFF | IXANY);
  tty.c_lflag = tty.c_oflag = 0;
  tty.c_cc[VMIN]  = 0;
  tty.c_cc[VTIME] = 5;

  cfmakeraw(&tty);

  try (tcsetattr (self->fd, TCSANOW, &tty), "setting attributes");

#undef try

return TRUE;
}

static gboolean
haeg_port_initable_init (GInitable    *initable,
                         GCancellable *cancellable,
                         GError      **error)
{
  HaegPort *self = HAEG_PORT (initable);
  gboolean ok;
  //GIOStatus status;

  g_return_val_if_fail (self->file_name != NULL, FALSE);

  self->fd = open (self->file_name, O_RDWR | O_NONBLOCK);
  if (self->fd == -1)
    {
      g_set_error (error,
                   G_IO_ERROR,
                   g_io_error_from_errno (errno),
                   "Error opening TTY port `%s': %s",
                   self->file_name,
                   g_strerror (errno));
      return FALSE;
    }

  ok = set_up_tty (self, error);
  if (!ok)
    {
      int err;

      err = close (self->fd);
      if (err == -1)
        {
          g_warning ("After TTY set up error,"
                     " error closing TTY port `%s': %s",
                     self->file_name,
                     g_strerror (errno));
        }

      self->fd = -1;
      return FALSE;
    }

  self->channel = g_io_channel_unix_new (self->fd);

  /*
  status = g_io_channel_set_encoding (self->channel, NULL, error);
  if (status != G_IO_STATUS_NORMAL)
    {
      return FALSE;
    }
  */

  self->read_buffer = haeg_sample_buffer_new (self->fd, self->sample_len);

  g_debug ("TTY port `%s' initialised", self->file_name);
  return TRUE;
}


static void
dispose (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegPort *self = HAEG_PORT (object);

  g_debug ("Disposing TTY port `%s'", self->file_name);


  g_debug ("Stopping TTY port");
  haeg_port_stop (self);

  if (self->read_buffer)
    {
      g_debug ("Clearing read buffer");
      g_clear_object (&self->read_buffer);
    }

  if (self->channel)
    {
      g_debug ("Unref'ing channel");
      g_io_channel_unref (self->channel);
      self->channel = NULL;
    }

  if (self->fd != -1)
    {
      int err;

      g_debug ("Closing file");
      err = close (self->fd);
      if (err == -1)
        {
          g_warning ("Error closing TTY port `%s': %s",
                     self->file_name, g_strerror (errno));
        }

      self->fd = -1;
    }

  g_debug ("TTY port `%s' disposed", self->file_name);

  parent_class->dispose (object);
}


static void
finalize (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegPort *self = HAEG_PORT (object);

  g_free (self->file_name);

  parent_class->finalize (object);
}


static void
haeg_port_class_init (HaegPortClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->set_property = set_property;
  object_class->dispose = dispose;
  object_class->finalize = finalize;

  props[PROP_FILE_NAME] =
    g_param_spec_string ("file-name",
                         _("File name"),
                         _("The file name of the TTY port device file"),
                         NULL,
                         G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);

  props[PROP_SAMPLE_LEN] =
    g_param_spec_ulong ("sample-len",
                      _("Sample len"),
                      _("The size, in bytes, of a sample"),
                      0, G_MAXULONG, 0,
                      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);


  g_object_class_install_properties (object_class, PROP_LAST_PROP, props);

  /**
   * HaegPort::read-ready:
   * @self: The #HaegPort instance.
   *
   * The TTY port has data to read.
   */
  signals[SIGNAL_READ_READY] =
    g_signal_new ("read-ready",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0, NULL, NULL, NULL,
                  G_TYPE_NONE,
                  0);

  /**
   * HaegPort::error:
   * @self: The #HaegPort instance.
   * @text: Some string describing the error.
   *
   * An error occurred on the TTY port.  The #HaegPort instance should
   * be immediately destroyed (unref'd).
   */
  signals[SIGNAL_ERROR] =
    g_signal_new ("error",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  0, NULL, NULL, NULL,
                  G_TYPE_NONE,
                  1,
                  G_TYPE_STRING);
}


static void
haeg_port_initable_interface_init (GInitableIface *iface)
{
  iface->init = haeg_port_initable_init;
}


HaegPort *
haeg_port_new (const gchar *file_name,
               gsize        sample_len,
               GError     **error)
{
  return g_initable_new (HAEG_TYPE_PORT,
                         NULL,
                         error,
                         "file-name", file_name,
                         "sample-len", sample_len,
                         NULL);
}


static void
emit_error (HaegPort    *self,
            const gchar *message)
{
  gchar *str =
    g_strdup_printf ("Error on TTY port `%s': %s",
                     self->file_name, message);

  g_signal_emit (self, signals[SIGNAL_ERROR], 0, str);

  g_free (str);
}


static gboolean
read_watch_cb (GIOChannel  *source,
               GIOCondition condition,
               HaegPort    *self)
{
  const gchar *err_msg = NULL;

  switch (condition)
    {
    case G_IO_IN:
    case G_IO_PRI:
      g_signal_emit (self, signals[SIGNAL_READ_READY], 0);
    case G_IO_OUT:
      return TRUE;

    case G_IO_ERR:
      err_msg = "error condition during IO watch";
      break;
    case G_IO_HUP:
      err_msg = "file hung up during IO watch";
      break;
    case G_IO_NVAL:
      err_msg = "invalid request during IO watch";
      break;
    }

    emit_error (self, err_msg);
    self->read_watch_id = 0;
    return FALSE;
}


void
haeg_port_start (HaegPort *self)
{
  if (self->read_watch_id)
    {
      return;
    }

  g_return_if_fail (self->channel != NULL);

  self->read_watch_id =
    //g_io_add_watch (self->channel, G_IO_IN|G_IO_ERR|G_IO_HUP|G_IO_NVAL,
    g_io_add_watch (self->channel, G_IO_IN,
                    (GIOFunc)read_watch_cb, self);
}


void
haeg_port_stop (HaegPort *self)
{
  if (!self->read_watch_id)
    {
      return;
    }

  g_source_remove (self->read_watch_id);
  self->read_watch_id = 0;
}


gsize
haeg_port_read  (HaegPort    *self,
                 void        *buf,
                 size_t       buf_size,
                 GError     **error)
{
  return haeg_sample_buffer_read (self->read_buffer,
                                  buf, buf_size,
                                  error);
}


gsize
haeg_port_write (HaegPort    *self,
                 void        *buf,
                 size_t       buf_size,
                 gboolean    *not_ready,
                 GError     **error)
{
  int written;

  written = write (self->fd, buf, buf_size);
  if (written == -1)
    {
      if (errno == EWOULDBLOCK || errno == EAGAIN)
        {
          g_debug ("TTY port `%s' not ready for writing",
                   self->file_name);
          g_assert (not_ready != NULL);
          *not_ready = TRUE;
          return 0;
        }
      else
        {
          g_set_error (error,
                       G_IO_ERROR,
                       g_io_error_from_errno (errno),
                       "Error writing to TTY port `%s': %s",
                       self->file_name,
                       g_strerror (errno));
          return 0;
        }
    }

  return written;
}
