/*
 * Copyright (C) 2018 Purism SPC
 *
 * This file is part of Hægtesse.
 *
 * Hægtesse is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * Hægtesse is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Hægtesse.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Bob Ham <bob.ham@puri.sm>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 */

#include "util.h"
#include "haeg-port.h"
#include "haeg-audio.h"
#include "haeg-port-monitor.h"
#include "config.h"
#include "mchk-machine-check.h"

#include <glib.h>
#include <glib/gi18n.h>

#include <stdio.h>
#include <locale.h>
#include <errno.h>


#define TTY_CHUNK_SIZE   320
#define SAMPLE_LEN       2

static GMainLoop *main_loop = NULL;

struct haegtesse_data
{
  HaegPortMonitor     *monitor;
  HaegPort            *port;
  HaegAudio           *audio;
  guint                timeout_id;
};

static void
clear_port (struct haegtesse_data *data)
{
  haeg_audio_cork (data->audio);
  g_clear_object (&data->port);
  haeg_port_monitor_request_port (data->monitor);
}

static gboolean
timeout_cb (struct haegtesse_data *data)
{
  g_debug ("Timeout");

  /* Reopen port */
  /* Without this, the audio on repeated calls can get corrupted,
     probably because the sample stream loses a byte and so every
     successive (2-byte) sample is actually byte 1 of one sample and
     byte 0 of the next.  Re-opening the port seems to reset this. */
  clear_port (data);

  data->timeout_id = 0;
  return FALSE;
}


static void
stop_timeout (struct haegtesse_data *data)
{
  if (!data->timeout_id)
    {
      return;
    }

  g_source_remove (data->timeout_id);
  data->timeout_id = 0;
}


static void
start_timeout (struct haegtesse_data *data)
{
  if (data->timeout_id)
    {
      g_source_remove (data->timeout_id);
    }

  data->timeout_id =
    g_timeout_add (1000, (GSourceFunc)timeout_cb, data);
}


static void
reopen_port (struct haegtesse_data *data)
{
  stop_timeout (data);
  clear_port (data);
}


static gboolean
transfer_spk (struct haegtesse_data *data,
              gsize *written)
{
  void *buf = NULL;
  size_t buf_size = 320;
  gsize bytes_read;
  GError *error = NULL;

  /* Get the write buffer */
  haeg_audio_begin_write (data->audio, &buf, &buf_size);

  /* Read audio data from the TTY */
  bytes_read = haeg_port_read (data->port, buf, buf_size, &error);
  if (error || bytes_read == 0)
    {
      if (error)
        {
          g_warning ("%s", error->message);
          g_error_free (error);
        }
      else
        {
          g_warning ("EOF from TTY port");
        }

      haeg_audio_cancel_write (data->audio);
      reopen_port (data);

      return FALSE;
    }

  /* Write audio data to speaker stream */
  haeg_audio_write (data->audio, buf, bytes_read);

  g_debug ("Wrote %zi bytes to PA speaker stream", (size_t)bytes_read);

  if (written)
    {
      *written = bytes_read;
    }
  return TRUE;
}


static gboolean
transfer_mic (struct haegtesse_data *data)
{
  void *buf = NULL;
  size_t write_len = 0;
  gsize written;
  gboolean not_ready = FALSE;
  GError *error = NULL;

  haeg_audio_get_mic_buffer (data->audio, &buf, &write_len);

  // Check we actually have data
  if (write_len == 0)
    {
      g_debug ("Empty mic buffer");
      return TRUE;
    }

  // Write audio data to the TTY
  written = haeg_port_write (data->port,
                             buf, write_len,
                             &not_ready, &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      reopen_port (data);
      return FALSE;
    }
  if (not_ready)
    {
      return TRUE;
    }
  g_debug ("Wrote %" G_GSIZE_FORMAT" bytes (of %zi) to TTY port from mic buffer",
           written, write_len);

  // Update buffer
  haeg_audio_update_mic_buffer (data->audio, written);

  return TRUE;
}


static void
port_error_cb (struct haegtesse_data *data,
               const gchar *message)
{
  g_warning ("Error on TTY port: %s", message);
  reopen_port (data);
}


static void
port_read_ready_cb (struct haegtesse_data *data)
{
  gboolean ok;

  /* Cancel any running timeout */
  stop_timeout (data);

  /* "Uncork" the streams if this is the first chunk */
  if (haeg_audio_is_corked (data->audio))
    {
      /* Uncork */
      haeg_audio_uncork (data->audio);
    }

  /* Move data from modem to speaker stream */
  ok = transfer_spk (data, NULL);
  if (!ok)
    {
      return;
    }

  /* Move data from microphone stream to modem */
  /* We have to do this here because for unknown reasons, having a
     write watch on the GIOChannel doesn't give us callbacks quickly
     enough.  We basically use the read watch to provide appropriate
     timing for the writing.  If there is mic data from PulseAudio in
     the buffer and we can write some of it without blocking, great.
     If not, so be it.  This works. */
  ok = transfer_mic (data);
  if (!ok)
    {
      return;
    }

  /* Start the timeout running */
  start_timeout (data);
}


static void
port_cb (struct haegtesse_data *data,
         const gchar *port_file_name)
{
  GError *error;

  if (data->port)
    {
      g_warning ("Received port signal while port object exists");
      return;
    }

  g_debug ("Creating new TTY port `%s'", port_file_name);

  error = NULL;
  data->port = haeg_port_new (port_file_name,
                              SAMPLE_LEN,
                              &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      return;
    }

  g_signal_connect_swapped (data->port, "error",
                            G_CALLBACK (port_error_cb), data);
  g_signal_connect_swapped (data->port, "read-ready",
                            G_CALLBACK (port_read_ready_cb), data);

  haeg_port_start (data->port);
}


static void
set_up (struct haegtesse_data *data)
{
  data->audio = haeg_audio_new (TTY_CHUNK_SIZE * 2);

  data->monitor = haeg_port_monitor_new ();
  g_signal_connect_swapped (data->monitor, "port",
                            G_CALLBACK (port_cb), data);
  haeg_port_monitor_request_port (data->monitor);
}


static void
tear_down (struct haegtesse_data *data)
{
  if (data->port)
    {
      haeg_port_stop (data->port);
      stop_timeout (data);
    }
  g_clear_object (&data->port);
  g_clear_object (&data->monitor);
  g_clear_object (&data->audio);
}


static void
run ()
{
  struct haegtesse_data data;

  memset (&data, 0, sizeof (struct haegtesse_data));
  set_up (&data);

  main_loop = g_main_loop_new (NULL, FALSE);

  printf (APPLICATION_NAME " started\n");
  g_main_loop_run (main_loop);

  g_main_loop_unref (main_loop);
  main_loop = NULL;

  tear_down (&data);
}


static void
check_machine ()
{
  gboolean ok, passed;
  GError *error = NULL;

  ok = mchk_check_machine (APP_DATA_NAME,
                           NULL,
                           &passed,
                           &error);
  if (!ok)
    {
      g_warning ("Error checking machine name against"
                 " whitelist/blacklist, continuing anyway");
      g_error_free (error);
    }
  else if (!passed)
    {
      g_message ("Machine name did not pass"
                 " whitelist/blacklist check, exiting");
      exit (EXIT_SUCCESS);
    }
}


static void
terminate (int signal)
{
  if (main_loop)
    {
      g_main_loop_quit (main_loop);
    }
}


/** Ignore signals which make systemd refrain from restarting us due
 * to "Restart=on-failure": SIGHUP, SIGINT, and SIGPIPE.
 */
static void
setup_signals ()
{
  void (*ret)(int);

#define try_setup(NUM,handler)                          \
  ret = signal (SIG##NUM, handler);                     \
  if (ret == SIG_ERR)                                   \
    {                                                   \
      g_error ("Error setting signal handler: %s",      \
               g_strerror (errno));                     \
    }

  try_setup (HUP,  SIG_IGN);
  try_setup (INT,  SIG_IGN);
  try_setup (PIPE, SIG_IGN);
  try_setup (TERM, terminate);

#undef try_setup
}


int
main (int argc, char **argv)
{
  GError *error = NULL;
  GOptionContext *context;
  gboolean ok;

  setlocale(LC_ALL, "");

  // Check the machine whitelist/blacklist
  check_machine ();

  GOptionEntry options[] =
    {
      { NULL }
    };

  context = g_option_context_new ("- transfer audio data between modem and PulseAudio");
  g_option_context_add_main_entries (context, options, NULL);
  ok = g_option_context_parse (context, &argc, &argv, &error);
  if (!ok)
    {
      g_print ("Error parsing options: %s\n", error->message);
    }

  setup_signals ();

  run ();

  return 0;
}
