/*
 * Copyright (C) 2018 Purism SPC
 *
 * This file is part of Hægtesse.
 *
 * Hægtesse is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * Hægtesse is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Hægtesse.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Author: Bob Ham <bob.ham@puri.sm>
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 */

#include "haeg-sample-buffer.h"
#include "util.h"

#include <glib/gi18n.h>
#include <gio/gio.h>

#include <unistd.h>
#include <errno.h>


/**
 * SECTION:haeg-sample-buffer
 * @short_description: Buffer data sample data from a TTY port and
 * deliver it efficiently.
 * @Title: HaegSampleBuffer
 */

struct _HaegSampleBuffer
{
  GObject parent_instance;

  gint fd;
  gsize sample_len;

  GByteArray *partial;
};

G_DEFINE_TYPE (HaegSampleBuffer, haeg_sample_buffer, G_TYPE_OBJECT);

enum {
  PROP_0,
  PROP_FILE_DES,
  PROP_SAMPLE_LEN,
  PROP_LAST_PROP,
};
static GParamSpec *props[PROP_LAST_PROP];


static void
set_property (GObject      *object,
              guint         property_id,
              const GValue *value,
              GParamSpec   *pspec)
{
  HaegSampleBuffer *self = HAEG_SAMPLE_BUFFER (object);

  switch (property_id) {
  case PROP_FILE_DES:
    self->fd = g_value_get_int (value);
    break;

  case PROP_SAMPLE_LEN:
    self->sample_len = g_value_get_ulong (value);
    break;

  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}


static void
haeg_sample_buffer_init (HaegSampleBuffer *self)
{
  self->partial = g_byte_array_new ();
}


static void
dispose (GObject *object)
{
  GObjectClass *parent_class = g_type_class_peek (G_TYPE_OBJECT);
  HaegSampleBuffer *self = HAEG_SAMPLE_BUFFER (object);

  g_byte_array_unref (self->partial);
  self->partial = NULL;

  parent_class->dispose (object);
}


static void
haeg_sample_buffer_class_init (HaegSampleBufferClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->set_property = set_property;
  object_class->dispose = dispose;

  props[PROP_FILE_DES] =
    g_param_spec_int ("file-des",
                      _("File descriptor"),
                      _("The file descriptor to read samples from"),
                      G_MININT, G_MAXINT, 0,
                      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);

  props[PROP_SAMPLE_LEN] =
    g_param_spec_ulong ("sample-len",
                      _("Sample len"),
                      _("The size, in bytes, of a sample"),
                      0, G_MAXULONG, 0,
                      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY);

  g_object_class_install_properties (object_class, PROP_LAST_PROP, props);
}


HaegSampleBuffer *
haeg_sample_buffer_new (gint fd, gsize sample_len)
{
  return g_object_new (HAEG_TYPE_SAMPLE_BUFFER,
                       "file-des", fd,
                       "sample-len", sample_len,
                       NULL);
}

gsize
haeg_sample_buffer_read (HaegSampleBuffer *self,
                         gchar            *buf,
                         gsize             buf_len,
                         GError          **error)
{
  guint partial_len = self->partial->len;
  gsize total;
  int bytes_read;

  if (partial_len > 0)
    {
      // Copy partial frame
      const gboolean partial_only = buf_len <= partial_len;
      const gsize copy_len = partial_only ? buf_len : partial_len;

      g_debug ("Using %" G_GSIZE_FORMAT
               " bytes out of partial sample of length %u",
               copy_len, partial_len);

      memcpy (buf, self->partial->data, copy_len);

      if (buf_len < partial_len)
        {
          g_byte_array_remove_range (self->partial, 0, copy_len);
        }
      else
        {
          g_byte_array_set_size (self->partial, 0);
        }

      if (partial_only)
        {
          return copy_len;
        }

      total = copy_len;
    }
  else
    {
      total = 0;
    }

  // Read more samples
  bytes_read = read (self->fd, buf + total, buf_len - total);
  if (bytes_read == -1)
    {
      g_set_error (error,
                   G_IO_ERROR,
                   g_io_error_from_errno (errno),
                   "Error reading from TTY port: %s",
                   g_strerror (errno));
      return 0;
    }
  else if (bytes_read == 0)
    {
      // EOF
      return 0;
    }
  g_debug ("Read %i bytes from TTY port", bytes_read);

  total += bytes_read;

  partial_len = total % self->sample_len;
  if (partial_len != 0)
    {
      // Store partial

      g_assert (self->partial->len == 0);

      g_debug ("Storing partial sample of length %u" 
               " from buffer with total length %" G_GSIZE_FORMAT,
               partial_len, total);

      total -= partial_len;
      g_byte_array_append (self->partial,
                           (const guint8 *)(buf + total),
                           partial_len);
    }

  g_debug ("Returning buffer of %" G_GSIZE_FORMAT " bytes", total);
  return total;
}
