/* $Id: hdrl_imagelist_view.c,v 1.1 2013-10-16 11:31:04 cgarcia Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-10-16 11:31:04 $
 * $Revision: 1.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "hdrl_imagelist.h"
#include "hdrl_imagelist_defs.h"
#include "hdrl_imagelist_view.h"
#include "hdrl_image.h"
#include "hdrl_image_defs.h"

#include <cpl.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                            Static Prototypes
 -----------------------------------------------------------------------------*/

static void hdrl_image_view_delete(hdrl_image * view);

/* TODO: internal representation of cpl_image in 6.3 required for mask views,
   can be removed in 6.4 which has cpl_image_set_bpm */
struct _cpl_image_ {
    /* Size of the image in x and y */
    cpl_size            nx, ny;
    /* Type of the pixels used for the cpl_image */
    cpl_type            type;
    /* Pointer to pixel buffer as a 1d buffer */
    void            *   pixels;
    /* Bad Pixels mask */
    cpl_mask        *   bpm;
};


/**@{*/

/*-----------------------------------------------------------------------------
                            Function codes
 -----------------------------------------------------------------------------*/

static hdrl_image *
hdrl_image_row_view_create(hdrl_image * himg,
                           cpl_size lly,
                           cpl_size ury)
{
    const cpl_type type = CPL_TYPE_DOUBLE; // TODO cpl_image_get_type(img);
    const size_t sz = cpl_type_get_sizeof(type);
    const cpl_size nx = hdrl_image_get_size_x(himg);
    char * d = cpl_image_get_data(himg->image);
    char * e = cpl_image_get_data(himg->error);
    size_t offset = (lly - 1) * nx;
    cpl_size nny = ury - lly + 1;
    cpl_image * img = cpl_image_wrap(nx, nny, type,
                                     d + offset * sz);
    cpl_image * err = cpl_image_wrap(nx, nny, type,
                                     e + offset * sz);

    /* we must create a mask in the original in order for views to work */
    cpl_mask * omask = cpl_image_get_bpm(himg->image);
    cpl_mask * mask = cpl_mask_wrap(nx, nny, cpl_mask_get_data(omask) + offset);
    /* TODO replace with 6.4 cpl_image_set_bpm */
    ((struct _cpl_image_ *)img)->bpm = mask;

    hdrl_image * h = hdrl_image_wrap(img, err);
    h->fp_free = (hdrl_free*)&hdrl_image_view_delete;

    return h;
}

static void hdrl_image_view_delete(hdrl_image * view)
{
    if (view) {
        hdrl_image * himg = (hdrl_image*)view;
        cpl_mask_unwrap(cpl_image_unset_bpm(himg->image));
        cpl_image_unwrap(himg->image);
        cpl_image_unwrap(himg->error);
        hdrl_image_unwrap(himg);
    }
}

hdrl_imagelist * hdrl_imagelist_row_view(
    const hdrl_imagelist * hl,
    cpl_size lly,
    cpl_size ury)
{
    cpl_ensure(hl, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(lly > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);
    size_t n = hdrl_imagelist_get_size(hl);
    cpl_ensure(n > 0, CPL_ERROR_ILLEGAL_INPUT, NULL);
    cpl_ensure(ury <= hdrl_image_get_size_y(hdrl_imagelist_get_const(hl, 0)),
               CPL_ERROR_ACCESS_OUT_OF_RANGE, NULL);
    cpl_ensure(ury >= lly, CPL_ERROR_ILLEGAL_INPUT, NULL);

    hdrl_imagelist * viewlist = hdrl_imagelist_new();
    for (size_t i = 0; i < n; i++) {
        hdrl_image * img = hdrl_imagelist_get(hl, i);
        hdrl_image * view = hdrl_image_row_view_create(img, lly, ury);
        hdrl_imagelist_set(viewlist, view, i);
    }

    return (hdrl_imagelist*)viewlist;
}
/**@}*/
