/*******************************************************************
 * libfaxophone                                                    *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file phone.c
 * \brief Phone handling functions
 */

#include <string.h>
#include <math.h>
#include <fcntl.h>
#include <stdint.h>

#include "faxophone.h"
#include "phone.h"
#include "isdn-convert.h"

#include <speex/speex_echo.h>
#include <speex/speex_preprocess.h>

/** Speex Echo Cancellation (does not work well at the moment) */
//#define SEC 1

#ifdef SEC
static SpeexEchoState *psEchoState = NULL;
#endif
static SpeexPreprocessState *psPreProcessState = NULL;

/* Close recording */
int recordingClose( struct sRecorder *psRecorder );

/**
 * \brief Input audio handler
 * \param pData capi connection pointer
 * \return NULL
 */
gpointer phoneInputThread( gpointer pData ) {
	struct sSession *psSession = faxophoneGetSession();
	struct sCapiConnection *psConnection = pData;
	unsigned char anAudioBufferRx[ CAPI_PACKETS * 2 ];
	unsigned char anAudioBuffer[ CAPI_PACKETS * 2 ];
	unsigned int nAudioBufLen;
	short anRecBuffer[ 16384 ];
	_cmsg sCmsg;

	while ( psSession -> nInputThreadState == 1 ) {
		static float fMean = 0.0f;
		short *pnTmp = NULL;
		int nLen = 160;
		
		nLen = sizeof( anAudioBufferRx );
		memset( anAudioBufferRx, 0, nLen );
		nLen = psSession -> psHandlers -> AudioInput( anAudioBufferRx, nLen );

		/* Check if we have some audio data to process */
		if ( nLen >= 0 ) {
			int nIndex;

			/* Remove DC offset */
			pnTmp = ( short * ) anAudioBufferRx;
			for ( nIndex = 0; nIndex < nLen / sizeof( short ); nIndex++ ) {
				fMean = 0.999 * fMean + 0.001 * pnTmp[ nIndex ];
				((short*)anAudioBufferRx)[ nIndex ] = pnTmp[ nIndex ] - ( short ) fMean;
			}

			speex_preprocess( psPreProcessState, ( spx_int16_t * ) anAudioBufferRx, NULL );

			/* convert audio data to isdn format */
			convertAudioToIsdn( psConnection, anAudioBufferRx, nLen, anAudioBuffer, &nAudioBufLen, anRecBuffer );

			/* Overflow check */
			if ( nAudioBufLen > CAPI_PACKETS ) {
				nAudioBufLen = CAPI_PACKETS;
			}

			isdnLock();
			DATA_B3_REQ( &sCmsg, psSession -> nApplId, 0, psConnection -> nNcci, anAudioBuffer, nAudioBufLen, psSession -> nMessageNumber++, 0 );
			isdnUnlock();
		}
	}

	psSession -> nInputThreadState = 0;

	if ( psConnection -> nRecording == 1 ) {
		recordingClose( &psConnection -> sRecorder );
	}

	return NULL;
}

/**
 * \brief Phone transfer routine which accepts incoming data, converts and outputs the audio
 * \param psConnection active capi connection
 * \param sCapiMessage current capi message
 */
void phoneTransfer( struct sCapiConnection *psConnection, _cmsg sCapiMessage ) {
	struct sSession *psSession = faxophoneGetSession();
	_cmsg sCmsg;
	unsigned char anAudioBuffer[ CAPI_PACKETS * 2 ];
	unsigned char anAudioBufferTx[ CAPI_PACKETS * 2 ];
	unsigned char anAudioBufferRx[ CAPI_PACKETS * 2 ];
	unsigned int nLen = DATA_B3_IND_DATALENGTH( &sCapiMessage );
	unsigned int nAudioBufLen;
	short anRecBuffer[ 8192 ];

	if ( psSession -> nInputThreadState == 0 ) {
		int nI = 1;
		int nLen = 320;

		psSession -> nInputThreadState = 1;

#ifdef SEC
		if ( psEchoState == NULL ) {
			psEchoState = speex_echo_state_init( ( nLen ) / ( sizeof( short ) ), 16 * nLen );
		}
#endif

		if ( psPreProcessState == NULL ) {
			psPreProcessState = speex_preprocess_state_init( nLen / sizeof( short ), 8000 );
			speex_preprocess_ctl( psPreProcessState, SPEEX_PREPROCESS_SET_DENOISE, &nI );
		}

		CREATE_THREAD( "phone-input", phoneInputThread, psConnection );
	}

	/* Clear buffer */
	memset( anAudioBufferRx, 0, sizeof( anAudioBufferRx ) );
	memset( anAudioBufferTx, 0, sizeof( anAudioBufferTx ) );
	memset( anAudioBuffer, 0, sizeof( anAudioBuffer ) );

	/* Overflow check */
	if ( nLen > CAPI_PACKETS ) {
		FopDebug( FOP_DEBUG, "Warning: Buffer overflow! %d > %d\n", nLen, CAPI_PACKETS );
		nLen = CAPI_PACKETS;
	}
	
	/* Save incoming data and send response */
	memcpy( anAudioBufferRx, ( uint8_t * ) DATA_B3_IND_DATA( &sCapiMessage ), nLen );
	isdnLock();
	DATA_B3_RESP( &sCmsg, psSession -> nApplId, psSession -> nMessageNumber++, psConnection -> nNcci, DATA_B3_IND_DATAHANDLE( &sCapiMessage ) );
	isdnUnlock();

	/* convert isdn to audio format */
	convertIsdnToAudio( &psConnection -> sRecorder, anAudioBufferRx, nLen, anAudioBuffer, &nAudioBufLen, anRecBuffer );

	/* Send data */
	psSession -> psHandlers -> AudioOutput( anAudioBuffer, nAudioBufLen );
}

/**
 * \brief Dial a given number and set connection type to SESSION_PHONE
 * \param nController capi controller
 * \param pnSource source number (own MSN)
 * \param pnTarget remote number (we want to dial)
 * \param nAnonymous anonymous flag (suppress number)
 * \return seee capiCall
 */
struct sCapiConnection *phoneCall( unsigned char nController, const char *pnSource, const char *pnTarget, unsigned char nAnonymous ) {
	return capiCall( nController, pnSource, pnTarget, nAnonymous, SESSION_PHONE, PHONE_CIP );
}

/**
 * \brief Mute/unmute active capi connection
 * \param psConnection active capi connection
 * \param nMute mute flag
 */
void phoneMute( struct sCapiConnection *psConnection, unsigned char nMute ) {
	/* Just set the flag, the audio input thread handle the mute case */
	psConnection -> nMute = nMute;
}

/**
 * \brief Return current time in microseconds
 * \return time in microseconds
 */
unsigned long microsec_time( void ) {
	struct timeval sTime;

	gettimeofday( &sTime, 0 );

	return sTime.tv_sec * ( ( unsigned long ) 1000000 ) + sTime.tv_usec;
}

/**
 * \brief Initialize recording structure
 * \param psRecorder pointer to recorder structure
 * \return 0
 */
int recordingInit( struct sRecorder *psRecorder ) {
	memset( psRecorder, 0, sizeof( struct sRecorder ) );

	return 0;
}

/**
 * \brief Open recording file
 * \param psRecorder pointer to recorder structure
 * \param pnFile record file
 * \return 0 on success, otherwise error
 */
int recordingOpen( struct sRecorder *psRecorder, char *pnFile ) {
	SF_INFO sInfo;

	if ( access( pnFile, F_OK ) ) {
		/* File not present */
		sInfo.format = SF_FORMAT_WAV | SF_FORMAT_ULAW;
		sInfo.channels = 2;
		sInfo.samplerate = 8000;

		if ( !( psRecorder -> psFile = sf_open( pnFile, SFM_WRITE, &sInfo ) ) ) {
			printf( "Error creating record file\n" );
			return -1;
		}
	} else {
		sInfo.format = 0;
		if ( !( psRecorder -> psFile = sf_open( pnFile, SFM_RDWR, &sInfo ) ) ) {
			printf( "Error opening record file\n" );
			return -1;
		}
		if ( sf_seek( psRecorder -> psFile, 0, SEEK_END ) == -1 ) {
			printf( "Error seeking record file\n" );
			return -1;
		}
	}

	psRecorder -> pnFileName = strdup( pnFile );
	psRecorder -> nLastWrite = 0;

	memset( &psRecorder -> sLocal, 0, sizeof( struct sRecordChannel ) );
	memset( &psRecorder -> sRemote, 0, sizeof( struct sRecordChannel ) );

	psRecorder -> nStartTime = microsec_time();

	return 0;
}

/**
 * \brief Write audio data to record file
 * \param psRecorder recorder structure
 * \param pnBuf audio buffer
 * \param nSize size of audio buffer
 * \param nChannel channel type (local/remote)
 * \return 0 on success, otherwise error
 */
int recordingWrite( struct sRecorder *psRecorder, short *pnBuf, int nSize, int nChannel ) {
	unsigned long nStart = psRecorder -> nStartTime;
	unsigned long nCurrent, nStartPos, nPosition, nEndPos;
	int nBufPos, nSplit, nDelta;
	struct sRecordChannel *psBuffer;

	if ( nStart == 0 ) {
		return 0;
	}

	if ( nSize < 1 ) {
		printf( "Warning: Illegal size!\n" );
		return -1;
	}

	switch ( nChannel ) {
		case RECORDING_LOCAL:
			psBuffer = &psRecorder -> sLocal;
			break;
		case RECORDING_REMOTE:
			psBuffer = &psRecorder -> sRemote;
			break;
		default:
			printf( "Recording to unknown channel!\n" );
			return -1;
	}

	nCurrent = microsec_time() - nStart;

	nEndPos = nCurrent * 8000 / 1000000LL;
	nStartPos = nEndPos - nSize;
	nPosition = psBuffer -> nPosition;

	if ( nStartPos >= nPosition - RECORDING_JITTER && nStartPos <= nPosition + RECORDING_JITTER ) {
		nStartPos = nPosition;
		nEndPos = nPosition + nSize;
	}

	if ( nStartPos < nPosition ) {
		nDelta = ( int ) nPosition - nStartPos;
		nStartPos = nPosition;
		pnBuf += nDelta;
		nSize -= nDelta;
		if ( nSize <= 0 ) {
			return 0;
		}
	}

	nBufPos = nStartPos % RECORDING_BUFSIZE;

	if ( nBufPos + nSize <= RECORDING_BUFSIZE ) {
		memcpy( psBuffer -> anBuffer + nBufPos, pnBuf, nSize * sizeof( short ) );
	} else {
		nSplit = RECORDING_BUFSIZE - nBufPos;
		memcpy( psBuffer -> anBuffer + nBufPos, pnBuf, nSplit * sizeof( short ) );
		pnBuf += nSplit;
		nSize -= nSplit;
		memcpy( psBuffer -> anBuffer, pnBuf, nSize * sizeof( short ) );
	}

	psBuffer -> nPosition = nEndPos;

	return 0;
}

/**
 * \brief Flush recording buffer
 * \param psRecorder recording structure
 * \param nLast last call flag
 * \return 0 on success, otherwise error
 */
int recordingFlush( struct sRecorder *psRecorder, unsigned int nLast ) {
	unsigned long nMaxPosition = psRecorder -> sLocal.nPosition;
	unsigned long nTmp = psRecorder -> sRemote.nPosition;
	unsigned long nStartPosition = psRecorder -> nLastWrite;
	short anRecBuf[ RECORDING_BUFSIZE * 2 ];
	int nSrcPtr, nDstPtr, nSize;

	if ( psRecorder -> nStartTime == 0 ) {
		return 0;
	}

	if ( nTmp > nMaxPosition ) {
		nMaxPosition = nTmp;
	}

	if ( nStartPosition + ( RECORDING_BUFSIZE * 7 / 8 ) < nMaxPosition ) {
		nStartPosition = nMaxPosition - ( RECORDING_BUFSIZE * 7 / 8 );
	}

	if ( !nLast ) {
		nMaxPosition -= RECORDING_BUFSIZE / 8;
	}

	nSize = ( int ) ( nMaxPosition - nStartPosition );
	if ( nMaxPosition == 0 || nStartPosition >= nMaxPosition || ( !nLast && nSize < RECORDING_BUFSIZE / 8 ) ) {
		return 0;
	}

	nDstPtr = 0;
	nSrcPtr = nStartPosition % RECORDING_BUFSIZE;

	while  ( --nSize ) {
		anRecBuf[ nDstPtr++ ] = psRecorder -> sLocal.anBuffer[ nSrcPtr ];
		psRecorder -> sLocal.anBuffer[ nSrcPtr ] = 0;
		anRecBuf[ nDstPtr++ ] = psRecorder -> sRemote.anBuffer[ nSrcPtr ];
		psRecorder -> sRemote.anBuffer[ nSrcPtr ] = 0;

		if ( ++nSrcPtr >= RECORDING_BUFSIZE ) {
			nSrcPtr = 0;
		}
	}

	sf_writef_short( psRecorder -> psFile, anRecBuf, nDstPtr / 2 );

	psRecorder -> nLastWrite = nMaxPosition;

	return 0;
}

/**
 * \brief Close recording structure
 * \param psRecorder recorder structure
 * \return 0 on success, otherwise error
 */
int recordingClose( struct sRecorder *psRecorder ) {
	int nResult = 0;

	if ( psRecorder -> nStartTime ) {
		if ( recordingFlush( psRecorder, 1 ) < 0 ) {
			nResult = -1;
		}
		psRecorder -> nStartTime = 0;

		if ( psRecorder -> pnFileName ) {
			free( psRecorder -> pnFileName );
			psRecorder -> pnFileName = NULL;
		}

		if ( sf_close( psRecorder -> psFile ) != 0 ) {
			printf( "Error closing record file!\n" );
			nResult = -1;
		}
	}

	return nResult;
}

/**
 * \brief Flush connection recorder
 * \param psConnection capi connection
 */
void phoneFlush( struct sCapiConnection *psConnection ) {
	if ( psConnection != NULL ) {
		recordingFlush( &psConnection -> sRecorder, 0 );
	}
}

/**
 * \brief Start/stop recording of active capi connection
 * \param psConnection active capi connection
 * \param nRecord record flag
 */
void phoneRecord( struct sCapiConnection *psConnection, unsigned char nRecord/*, gchar *pnFile*/ ) {
	if ( nRecord == 1 ) {
		gchar *pnFile = NULL;

		if ( psConnection -> nRecording == 0 ) {
			recordingInit( &psConnection -> sRecorder );
		}

		pnFile = g_strdup_printf( "%s/%s-%s.wav", g_get_tmp_dir(), psConnection -> pnSource, psConnection -> pnTarget ); 

		recordingOpen( &psConnection -> sRecorder, pnFile );
		g_free( pnFile );
	} else {
		if ( psConnection -> nRecording == 1 ) {
			recordingClose( &psConnection -> sRecorder );
		}
	}

	psConnection -> nRecording = nRecord;
}

/**
 * \brief Hold and retrieve active capi connection
 * \param psConnection active capi connection
 * \param nHold hold flag
 */
void phoneHold( struct sCapiConnection *psConnection, unsigned char nHold ) {
	struct sSession *psSession = faxophoneGetSession();
	_cmsg sMessage;
	_cbyte anFac[ 9 ];

	/* Save state */
	psConnection -> nHold = nHold;

	/* Generate facility structure */
	anFac[ 0 ] = 3;
	anFac[ 1 ] = ( _cbyte )( 0x0003 - nHold ) & 0xFF;
	anFac[ 2 ] = ( _cbyte )( ( 0x0003 - nHold ) >> 8 ) & 0xFF;
	anFac[ 3 ] = 0;

	isdnLock();
	if ( nHold == 1 ) {
		/* Hold active connection */
		FACILITY_REQ( &sMessage, psSession -> nApplId, 0, psConnection -> nNcci, 3, ( unsigned char * ) anFac );
	} else {
		/* Retrieve active connection */
		FACILITY_REQ( &sMessage, psSession -> nApplId, 0, psConnection -> nPlci, 3, ( unsigned char * ) anFac );
	}
	isdnUnlock();
}

/**
 * \brief Send DTMF codes on connection
 * \param psConnection active capi connection
 * \param nCode DTMF code
 */
void phoneSendDtmfCode( struct sCapiConnection *psConnection, unsigned char nCode ) {
	capiSendDtmfCode( psConnection, nCode );
}

/**
 * \brief Hangup phone connection
 * \param psConnection active capi connection
 */
void phoneHangup( struct sCapiConnection *psConnection ) {
	if ( psConnection == NULL ) {
		return;
	}

	/* Hangup */
	capiHangup( psConnection );
}

/**
 * \brief Pickup a phone call
 * \param psConnection active capi connection
 * \return see capiPickup
 */
int phonePickup( struct sCapiConnection *psConnection ) {
	if ( psConnection == NULL ) {
		return -1;
	}

	/* Pickup connection and set connection type to SESSION_PHONE */
	return capiPickup( psConnection, SESSION_PHONE );
}
