var optionator, assert, equal, deepEqual, throws, q, eq;
optionator = require('../..');
assert = require('assert'), equal = assert.strictEqual, deepEqual = assert.deepEqual, throws = assert.throws;
q = function(args, options, more, parseOptions){
  var parse;
  more == null && (more = {});
  more.options = options;
  parse = optionator(more).parse;
  return parse(args, parseOptions);
};
eq = function(expectedOptions, expectedPositional, args, options, more, parseOptions){
  var result;
  result = q(args, options, more, parseOptions);
  deepEqual(result._, expectedPositional);
  delete result._;
  return deepEqual(result, expectedOptions);
};
describe('misc', function(){
  return it('version', function(){
    return equal(optionator.VERSION, require('../../package.json').version);
  });
});
describe('boolean flags', function(){
  var opts;
  opts = [
    {
      option: 'help',
      alias: 'h',
      type: 'Boolean'
    }, {
      option: 'match',
      alias: ['m', 't'],
      type: 'Boolean'
    }
  ];
  it('long', function(){
    return eq({
      help: true
    }, [], '--help', opts);
  });
  it('short', function(){
    eq({
      help: true
    }, [], '-h', opts);
    eq({
      match: true
    }, [], '-m', opts);
    return eq({
      match: true
    }, [], '-m', opts);
  });
  it('short with --', function(){
    return eq({
      help: true
    }, [], '--h', opts);
  });
  it('multiple', function(){
    return eq({
      help: true,
      match: true
    }, [], '-hm', opts);
  });
  it('negative', function(){
    return eq({
      match: false
    }, [], '--no-match', opts);
  });
  it('redefine', function(){
    eq({
      match: false
    }, [], '--match --no-match', opts);
    return eq({
      match: true
    }, [], '--no-match --match', opts);
  });
  it('using = true', function(){
    eq({
      match: true
    }, [], '--match=true', opts);
    return eq({
      match: true
    }, [], '--no-match=false', opts);
  });
  it('using = true, short', function(){
    return eq({
      match: true
    }, [], '-m=true', opts);
  });
  it('using = negated', function(){
    eq({
      match: false
    }, [], '--match=false', opts);
    return eq({
      match: false
    }, [], '--no-match=true', opts);
  });
  return it('using = false, short', function(){
    return eq({
      match: false
    }, [], '-m=false', opts);
  });
});
describe('argument', function(){
  var opts;
  opts = [
    {
      option: 'context',
      alias: 'C',
      type: 'Number'
    }, {
      option: 'name',
      alias: 'A',
      type: 'String'
    }, {
      option: 'destroy',
      alias: 'd',
      type: 'Boolean'
    }
  ];
  it('simple long', function(){
    return eq({
      context: 2
    }, [], '--context 2', opts);
  });
  it('simple short', function(){
    return eq({
      context: 2
    }, [], '-C 2', opts);
  });
  it('grouped short, when last', function(){
    return eq({
      destroy: true,
      context: 2
    }, [], '-dC 2', opts);
  });
  it('multiple', function(){
    return eq({
      context: 2,
      name: 'Arnie'
    }, [], '--context 2 --name Arnie', opts);
  });
  it('with boolean flag', function(){
    eq({
      context: 2,
      destroy: true
    }, [], '--destroy --context 2', opts);
    return eq({
      context: 2,
      destroy: true
    }, [], '--context 2 --destroy', opts);
  });
  it('using =', function(){
    return eq({
      context: 2
    }, [], '--context=2', opts);
  });
  it('using = complex', function(){
    return eq({
      name: 'Arnie S'
    }, [], '--name="Arnie S"', opts);
  });
  it('using = no value', function(){
    return throws(function(){
      return q('--context=', opts);
    }, /No value for 'context' specified/);
  });
  it('using = short', function(){
    eq({
      context: 2
    }, [], '-C=2', opts);
    return eq({
      context: 2,
      destroy: true
    }, [], '-dC=2', opts);
  });
  it('using = short no value', function(){
    return throws(function(){
      return q('-C=', opts);
    }, /No value for 'C' specified/);
  });
  it('value for prop required', function(){
    throws(function(){
      return q('--context', opts);
    }, /Value for 'context' of type 'Number' required./);
    return throws(function(){
      return q('--context --destroy', opts);
    }, /Value for 'context' of type 'Number' required./);
  });
  it('can\'t set flag val when not last', function(){
    return throws(function(){
      return q('-Cd 2', opts);
    }, /Can't set argument 'C' when not last flag in a group of short flags./);
  });
  it('no- prefix only on boolean options', function(){
    return throws(function(){
      return q('--no-context 2', opts);
    }, /Only use 'no-' prefix for Boolean options, not with 'context'./);
  });
  it('redefine', function(){
    eq({
      context: 5
    }, [], '--context 4 --context 5', opts);
    return eq({
      context: 5
    }, [], '-C 4 --context 5', opts);
  });
  return it('invalid type', function(){
    throws(function(){
      return q('--ends 20-11', [{
        option: 'ends',
        type: 'Date'
      }]);
    }, /expected type Date/);
    throws(function(){
      return q('--pair "true"', [{
        option: 'pair',
        type: '(Boolean, Number)'
      }]);
    }, /expected type \(Boolean, Number\)/);
    throws(function(){
      return q('--pair "true, 2, hider"', [{
        option: 'pair',
        type: '(Boolean, Number)'
      }]);
    }, /expected type \(Boolean, Number\)/);
    return throws(function(){
      return q('--props "x:1,fake:yo"', [{
        option: 'props',
        type: '{x:Number}'
      }]);
    }, /expected type {x:Number}/);
  });
});
describe('enum', function(){
  var enumOpt;
  enumOpt = [{
    option: 'size',
    type: 'String',
    'enum': ['small', 'medium', 'large']
  }];
  it('enum', function(){
    return eq({
      size: 'medium'
    }, [], '--size medium', enumOpt);
  });
  return it('invalid enum', function(){
    return throws(function(){
      return q('--size Hello', enumOpt);
    }, /Option size: 'Hello' not one of small, medium, or large/);
  });
});
describe('argument names', function(){
  var opts;
  opts = [
    {
      option: 'after-context',
      type: 'Number'
    }, {
      option: 'is-JSON',
      type: 'Boolean'
    }, {
      option: 'HiThere',
      type: 'Boolean'
    }, {
      option: 'context2',
      type: 'Number'
    }
  ];
  it('dash to camel', function(){
    eq({
      afterContext: 99
    }, [], '--after-context 99', opts);
    return eq({
      isJSON: true
    }, [], '--is-JSON', opts);
  });
  it('preserve PascalCase', function(){
    return eq({
      HiThere: true
    }, [], '--HiThere', opts);
  });
  return it('numbers', function(){
    return eq({
      context2: 1
    }, [], '--context2 1', opts);
  });
});
describe('-NUM', function(){
  it('no -NUM option defined', function(){
    return throws(function(){
      return q('-1', []);
    }, /No -NUM option defined./);
  });
  it('no aliases allowed', function(){
    return throws(function(){
      return q('', [{
        option: 'NUM',
        type: 'Number',
        alias: 'n'
      }]);
    }, /-NUM option can't have aliases/);
  });
  it('number', function(){
    var opts;
    opts = [{
      option: 'NUM',
      type: 'Number'
    }];
    it('0', function(){
      return eq({
        NUM: 0
      }, [], '-0', opts);
    });
    it('1', function(){
      return eq({
        NUM: 1
      }, [], '-1', opts);
    });
    it('multi digit', function(){
      return eq({
        NUM: 10
      }, [], '-10', opts);
    });
    return it('float', function(){
      return eq({
        NUM: 1.0
      }, [], '-1.0', opts);
    });
  });
  it('float', function(){
    var opts;
    opts = [{
      option: 'NUM',
      type: 'Float'
    }];
    it('float basic', function(){
      return eq({
        NUM: 1.2
      }, [], '-1.2', opts);
    });
    return it('float from int', function(){
      return eq({
        NUM: 1.0
      }, [], '-1', opts);
    });
  });
  return it('int', function(){
    var opts;
    opts = [{
      option: 'NUM',
      type: 'Int'
    }];
    return it('int basic', function(){
      return eq({
        NUM: 1
      }, [], '-1', opts);
    });
  });
});
describe('positional', function(){
  var opts;
  opts = [
    {
      option: 'flag',
      alias: 'f',
      type: 'Boolean'
    }, {
      option: 'cc',
      type: 'Number'
    }, {
      option: 'help',
      alias: 'h',
      type: 'Boolean',
      restPositional: true
    }, {
      option: 'help-two',
      alias: 'H',
      type: 'String',
      restPositional: true
    }
  ];
  it('basic', function(){
    return eq({}, ['boom'], 'boom', opts);
  });
  it('anywhere', function(){
    return eq({
      flag: true,
      cc: 42
    }, ['boom', '2', 'hi'], 'boom --flag 2 --cc 42 hi', opts);
  });
  it('not anywhere', function(){
    return eq({
      flag: true,
      cc: 42
    }, ['hi', '--xx'], '--flag --cc 42 hi --xx', opts, {
      positionalAnywhere: false
    });
  });
  it('--', function(){
    return eq({
      flag: true
    }, ['--flag', '2', 'boom'], '--flag -- --flag 2 boom', opts);
  });
  it('rest positional boolean', function(){
    return eq({
      help: true
    }, ['--flag', '2', 'boom'], '--help --flag 2 boom', opts);
  });
  it('rest positional value', function(){
    return eq({
      helpTwo: 'lalala'
    }, ['--flag', '2', 'boom'], '--help-two lalala --flag 2 boom', opts);
  });
  it('rest positional flags simple', function(){
    eq({
      help: true
    }, ['--flag', '2', 'boom'], '-h --flag 2 boom', opts);
    return eq({
      helpTwo: 'lalala'
    }, ['--flag', '2', 'boom'], '-H lalala --flag 2 boom', opts);
  });
  it('rest positional flags grouped', function(){
    eq({
      help: true,
      flag: true
    }, ['--cc', '2', 'boom'], '-fh --cc 2 boom', opts);
    return eq({
      helpTwo: 'lalala',
      flag: true
    }, ['--cc', '2', 'boom'], '-fH lalala --cc 2 boom', opts);
  });
  return it('rest positional flags grouped complex', function(){
    return eq({
      help: true
    }, ['f', '--cc', '2', 'boom'], '-hf --cc 2 boom', opts);
  });
});
describe('defaults', function(){
  it('basic', function(){
    var opt;
    opt = [{
      option: 'go',
      type: 'String',
      'default': 'boom'
    }];
    eq({
      go: 'boom'
    }, [], '', opt);
    return eq({
      go: 'haha'
    }, [], '--go haha', opt);
  });
  return it('array', function(){
    var opt;
    opt = [{
      option: 'list',
      type: 'Array',
      'default': '1,2'
    }];
    eq({
      list: [1, 2]
    }, [], '', opt);
    return eq({
      list: [8, 9]
    }, [], '--list 8,9', opt);
  });
});
describe('array/object input', function(){
  var opts;
  opts = [
    {
      option: 'el',
      type: 'Number'
    }, {
      option: 'hasta-la-vista',
      alias: 'h',
      type: 'String'
    }, {
      option: 'is-JSON',
      type: 'Boolean'
    }, {
      option: 'it',
      type: 'RegExp'
    }, {
      option: 'HiThere',
      type: 'Boolean'
    }, {
      option: 'day',
      type: 'Date'
    }, {
      option: 'list',
      alias: 'l',
      type: '[Int]'
    }, {
      option: 'pair',
      type: '(Int,String)'
    }, {
      option: 'map',
      type: '{a:Int,b:Boolean}'
    }
  ];
  it('array', function(){
    return eq({
      el: 5
    }, [], ['node', 'cmd.js', '--el', '5'], opts);
  });
  it('object', function(){
    return eq({
      el: 5
    }, [], {
      el: 5
    }, opts);
  });
  it('object set positional', function(){
    return eq({
      el: 5
    }, ['haha'], {
      el: 5,
      _: ['haha']
    }, opts);
  });
  it('object - camelCase keys', function(){
    eq({
      hastaLaVista: 'baby'
    }, [], {
      hastaLaVista: 'baby'
    }, opts);
    return eq({
      isJSON: true
    }, [], {
      isJSON: true
    }, opts);
  });
  it('object - dashed-case keys', function(){
    return eq({
      hastaLaVista: 'baby'
    }, [], {
      'hasta-la-vista': 'baby'
    }, opts);
  });
  it('object - PascalCase keys', function(){
    return eq({
      HiThere: true
    }, [], {
      HiThere: true
    }, opts);
  });
  it('object -aliases', function(){
    return eq({
      hastaLaVista: 'lala',
      list: [1, 2, 3]
    }, [], {
      h: 'lala',
      l: [1, 2, 3]
    }, opts);
  });
  it('regexp object', function(){
    return eq({
      it: /I'll be back/g
    }, [], {
      it: /I'll be back/g
    }, opts);
  });
  it('date object', function(){
    return eq({
      day: new Date('2011-11-11')
    }, [], {
      day: new Date('2011-11-11')
    }, opts);
  });
  it('array object', function(){
    return eq({
      list: [1, 2, 3]
    }, [], {
      list: [1, 2, 3]
    }, opts);
  });
  it('tuple object', function(){
    return eq({
      pair: [1, '52']
    }, [], {
      pair: [1, '52']
    }, opts);
  });
  it('object object', function(){
    return eq({
      map: {
        a: 1,
        b: true
      }
    }, [], {
      map: {
        a: 1,
        b: true
      }
    }, opts);
  });
  return it('invalid object', function(){
    return throws(function(){
      return q({
        el: 'hi'
      }, opts);
    }, /Option 'el': Invalid type for 'hi' - expected type 'Number'/);
  });
});
describe('slicing', function(){
  it('string slice', function(){
    return eq({
      b: 2
    }, ['c'], 'cmd -b 2 c', [{
      option: 'b',
      type: 'Number'
    }], void 8, {
      slice: 3
    });
  });
  return it('array slice', function(){
    return eq({
      b: 2
    }, ['c'], ['cmd', '-b', '2', 'c'], [{
      option: 'b',
      type: 'Number'
    }], void 8, {
      slice: 1
    });
  });
});
describe('parse-argv', function(){
  return it('slices two', function(){
    var parseArgv, o;
    parseArgv = optionator({
      options: [{
        option: 'a',
        type: 'Boolean'
      }]
    }).parseArgv;
    o = parseArgv(['a', 'b', 'c', 'd', '-a']);
    deepEqual(o._, ['c', 'd']);
    return equal(o.a, true);
  });
});
describe('errors in defining options', function(){
  it('no options defined', function(){
    return throws(function(){
      return q('');
    }, /No options defined/);
  });
  it('option already defined', function(){
    throws(function(){
      return q('', [
        {
          option: 'opt',
          type: '*'
        }, {
          option: 'opt',
          type: '*'
        }
      ]);
    }, /Option 'opt' already defined/);
    return throws(function(){
      return q('', [
        {
          option: 'opt',
          type: '*'
        }, {
          option: 'top',
          type: '*',
          alias: 'opt'
        }
      ]);
    }, /Option 'opt' already defined/);
  });
  it('no type defined', function(){
    return throws(function(){
      return q('', [{
        option: 'opt'
      }]);
    }, /No type defined for option 'opt'./);
  });
  it('error parsing type', function(){
    return throws(function(){
      return q('', [{
        option: 'opt',
        type: '[Int'
      }]);
    }, /Option 'opt': Error parsing type '\[Int'/);
  });
  it('error parsing default value', function(){
    return throws(function(){
      return q('', [{
        option: 'opt',
        type: 'Number',
        'default': 'hi'
      }]);
    }, /Option 'opt': Error parsing default value 'hi' for type 'Number':/);
  });
  return it('error parsing enum value', function(){
    return throws(function(){
      return q('', [{
        option: 'opt',
        type: 'Number',
        'enum': ['hi']
      }]);
    }, /Option 'opt': Error parsing enum value 'hi' for type 'Number':/);
  });
});
describe('errors parsing options', function(){
  it('invalid argument to parse', function(){
    return throws(function(){
      return q(2, []);
    }, /Invalid argument to 'parse': 2./);
  });
  it('invalid option', function(){
    var opts;
    opts = [
      {
        option: 'rake',
        type: 'Boolean'
      }, {
        option: 'kare',
        type: 'Boolean'
      }
    ];
    throws(function(){
      return q('--fake', opts);
    }, /Invalid option '--fake' - perhaps you meant '--rake'\?/);
    throws(function(){
      return q('--arket', opts);
    }, /Invalid option '--arket' - perhaps you meant '--rake'\?/);
    return throws(function(){
      return q('-k', opts);
    }, /Invalid option '-k' - perhaps you meant '--rake'\?/);
  });
  it('invalid option - no additional help', function(){
    return throws(function(){
      return q('--fake', []);
    }, /Invalid option '--fake'/);
  });
  it('is required', function(){
    var opts;
    opts = [{
      option: 'req-opt',
      type: 'Boolean',
      required: true
    }];
    eq({
      reqOpt: true
    }, [], {
      reqOpt: true
    }, opts);
    return throws(function(){
      return q('', opts);
    }, /Option --req-opt is required/);
  });
  it('override required', function(){
    var opts;
    opts = [
      {
        option: 'req-opt',
        type: 'Boolean',
        required: true
      }, {
        option: 'help',
        type: 'Boolean',
        overrideRequired: true
      }
    ];
    throws(function(){
      return q('', opts);
    }, /Option --req-opt is required/);
    return eq({
      help: true
    }, [], '--help', opts);
  });
  return it('is mutually exclusive', function(){
    var opts, more;
    opts = [
      {
        option: 'aa-aa',
        type: 'Boolean'
      }, {
        option: 'bb',
        type: 'Boolean'
      }, {
        option: 'cc',
        type: 'Boolean'
      }, {
        option: 'dd',
        type: 'Boolean'
      }, {
        option: 'ee',
        type: 'Boolean'
      }
    ];
    more = {
      mutuallyExclusive: [['aa-aa', 'bb'], [['bb', 'cc'], ['dd', 'ee']]]
    };
    throws(function(){
      return q('--aa-aa --bb', opts, more);
    }, /The options --aa-aa and --bb are mutually exclusive - you cannot use them at the same time/);
    throws(function(){
      return q('--bb --ee', opts, more);
    }, /The options --bb and --ee are mutually exclusive - you cannot use them at the same time/);
    throws(function(){
      return q('--cc --dd', opts, more);
    }, /The options --cc and --dd are mutually exclusive - you cannot use them at the same time/);
    return throws(function(){
      return q({
        aaAa: true,
        bb: true
      }, opts, more);
    }, /The options --aa-aa and --bb are mutually exclusive - you cannot use them at the same time/);
  });
});
describe('concat repeated arrays', function(){
  var opts, more;
  opts = [
    {
      option: 'nums',
      alias: 'n',
      type: '[Number]'
    }, {
      option: 'x',
      type: 'Number'
    }
  ];
  more = {
    concatRepeatedArrays: true
  };
  it('basic', function(){
    return eq({
      nums: [1, 2, 3]
    }, [], '-n 1 -n 2 -n 3', opts, more);
  });
  it('overwrites non-array', function(){
    return eq({
      x: 3
    }, [], '-x 1 -x 2 -x 3', opts, more);
  });
  it('per option', function(){
    var opts;
    opts = [
      {
        option: 'x',
        type: '[Number]',
        concatRepeatedArrays: true
      }, {
        option: 'y',
        type: '[Number]'
      }
    ];
    eq({
      x: [1, 2, 3]
    }, [], '-x 1 -x 2 -x 3', opts);
    return eq({
      y: [3]
    }, [], '-y 1 -y 2 -y 3', opts);
  });
  it('using defaults', function(){
    var opts, more;
    opts = [
      {
        option: 'x',
        type: '[Number]'
      }, {
        option: 'y',
        type: '[Number]',
        concatRepeatedArrays: false
      }
    ];
    more = {
      defaults: {
        concatRepeatedArrays: true
      }
    };
    eq({
      x: [1, 2, 3]
    }, [], '-x 1 -x 2 -x 3', opts, more);
    return eq({
      y: [3]
    }, [], '-y 1 -y 2 -y 3', opts, more);
  });
  it('one value per flag', function(){
    var opts;
    opts = [{
      option: 'x',
      type: '[String]',
      concatRepeatedArrays: [
        true, {
          oneValuePerFlag: true
        }
      ]
    }];
    return eq({
      x: ['a,b', 'c,d', 'e,f']
    }, [], '-x "a,b" -x "c,d" -x "e,f"', opts);
  });
  it('set with array, len is 1', function(){
    var opts;
    opts = [{
      option: 'x',
      type: '[String]',
      concatRepeatedArrays: [true]
    }];
    return eq({
      x: ['a', 'b', 'c', 'd']
    }, [], '-x "a,b" -x "c,d"', opts);
  });
  return it('invalid setting', function(){
    var opts;
    opts = [{
      option: 'x',
      type: '[String]',
      concatRepeatedArrays: []
    }];
    return throws(function(){
      return q('', opts);
    }, /Invalid setting for concatRepeatedArrays/);
  });
});
describe('merge repeated objects', function(){
  var opts, more, opts2;
  opts = [
    {
      option: 'config',
      alias: 'c',
      type: 'Object'
    }, {
      option: 'x',
      type: 'Number'
    }
  ];
  more = {
    mergeRepeatedObjects: true
  };
  it('basic', function(){
    return eq({
      config: {
        a: 1,
        b: 2,
        c: 3
      }
    }, [], '-c a:1 -c b:2 -c c:3', opts, more);
  });
  it('same properties', function(){
    return eq({
      config: {
        a: 3
      }
    }, [], '-c a:1 -c a:2 -c a:3', opts, more);
  });
  it('multiple properties in one go', function(){
    return eq({
      config: {
        a: 1,
        b: 2,
        c: 3,
        d: 4
      }
    }, [], '-c "a:1,b:2" -c "c: 3, d: 4"', opts, more);
  });
  it('overwrites non-array', function(){
    return eq({
      x: 3
    }, [], '-x 1 -x 2 -x 3', opts, more);
  });
  opts2 = [
    {
      option: 'c',
      type: 'Object',
      mergeRepeatedObjects: true
    }, {
      option: 'd',
      type: 'Object'
    }
  ];
  return it('per option', function(){
    eq({
      c: {
        a: 1,
        b: 2,
        c: 3
      }
    }, [], '-c a:1 -c b:2 -c c:3', opts2);
    return eq({
      d: {
        c: 3
      }
    }, [], '-d a:1 -d b:2 -d c:3', opts2);
  });
});
describe('dependency check', function(){
  var opts;
  opts = [
    {
      option: 'aa',
      type: 'Boolean'
    }, {
      option: 'bb',
      type: 'Boolean',
      dependsOn: ['or', 'aa', 'dd']
    }, {
      option: 'cc',
      type: 'Boolean',
      dependsOn: ['and', 'aa', 'dd']
    }, {
      option: 'dd',
      type: 'Boolean'
    }, {
      option: 'ff',
      type: 'Boolean',
      dependsOn: 'aa'
    }, {
      option: 'gg',
      type: 'Boolean',
      dependsOn: ['aa']
    }
  ];
  it('"and" should pass', function(){
    return eq({
      cc: true,
      aa: true,
      dd: true
    }, [], '--cc --aa --dd', opts);
  });
  it('"and" should fail', function(){
    throws(function(){
      return q('--cc', opts);
    }, /The option 'cc' did not have its dependencies met/);
    throws(function(){
      return q('--cc --aa', opts);
    }, /The option 'cc' did not have its dependencies met/);
    return throws(function(){
      return q('--cc --dd', opts);
    }, /The option 'cc' did not have its dependencies met/);
  });
  it('"or" should pass', function(){
    eq({
      bb: true,
      aa: true
    }, [], '--bb --aa', opts);
    return eq({
      bb: true,
      dd: true
    }, [], '--bb --dd', opts);
  });
  it('"or" should fail', function(){
    return throws(function(){
      return q('--bb', opts);
    }, /The option 'bb' did not meet any of its dependencies/);
  });
  it('single dependency, as string', function(){
    return eq({
      ff: true,
      aa: true
    }, [], '--ff --aa', opts);
  });
  it('single dependency, in array', function(){
    return eq({
      gg: true,
      aa: true
    }, [], '--gg --aa', opts);
  });
  it('just "and"', function(){
    var opts;
    opts = [{
      option: 'xx',
      type: 'Boolean',
      dependsOn: ['and']
    }];
    return eq({
      xx: true
    }, [], '--xx', opts);
  });
  it('empty array', function(){
    var opts;
    opts = [{
      option: 'xx',
      type: 'Boolean',
      dependsOn: []
    }];
    return eq({
      xx: true
    }, [], '--xx', opts);
  });
  return it('not using "and" or "or"', function(){
    var opts;
    opts = [{
      option: 'fail',
      type: 'Boolean',
      dependsOn: ['blerg', 'grr']
    }];
    return throws(function(){
      return q('--fail', opts);
    }, /Option 'fail': If you have more than one dependency, you must specify either 'and' or 'or'/);
  });
});
describe('option defaults', function(){
  var opts, more;
  opts = [
    {
      option: 'a'
    }, {
      option: 'b'
    }
  ];
  more = {
    defaults: {
      type: 'Number'
    }
  };
  return it('basic', function(){
    eq({
      a: 5
    }, [], '-a 5', opts, more);
    return eq({
      b: 5
    }, [], '-b 5', opts, more);
  });
});
describe('heading', function(){
  var opts;
  opts = [
    {
      option: 'aaa',
      type: 'Number'
    }, {
      heading: 'mooo'
    }, {
      option: 'bbb',
      type: 'String'
    }, {
      option: 'ccc',
      type: 'Boolean'
    }
  ];
  return it('basic', function(){
    eq({
      aaa: 5
    }, [], '--aaa 5', opts);
    eq({
      bbb: 'hi'
    }, [], '--bbb hi', opts);
    return eq({
      ccc: true
    }, [], '--ccc', opts);
  });
});
describe('type-aliases', function(){
  var opts, typeAliases;
  opts = [
    {
      option: 'x',
      type: 'Path'
    }, {
      option: 'y',
      type: 'Rules'
    }
  ];
  typeAliases = {
    Path: 'String',
    Rules: '[String]'
  };
  return it('basic', function(){
    return eq({
      x: 'a',
      y: ['c', 'd']
    }, [], '-x a -y c,d', opts, {
      typeAliases: typeAliases
    });
  });
});
