/*
 * Copyright (c) 2015, 2024, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLException;

/**
 * Testing SSLEngines handshake using each of the supported cipher suites with
 * different maximum fragment length. Testing of MFLN extension.
 */
public class MFLNTest extends SSLEngineTestCase {

    public static void main(String[] args) {
        setUpAndStartKDCIfNeeded();
        System.setProperty("jsse.enableMFLNExtension", "true");
        String testMode = System.getProperty("test.mode", "norm");
        int mflLen;
        if (testMode.equals("norm_sni")) {
            mflLen = 512;
        } else {
            mflLen = 256;
        }

        for (int mfl = 4096; mfl >= mflLen; mfl /= 2) {
            System.out.println("=============================================="
                    + "==============");
            System.out.printf("Testsing DTLS handshake with MFL = %d%n", mfl);
            MFLNTest test = new MFLNTest(mfl);
            test.runTests();
        }
    }

    protected MFLNTest(int maxPacketSize) {
        super(maxPacketSize);
    }

    @Override
    protected void testOneCipher(String cipher) throws SSLException {
        SSLContext context = getContext();
        int maxPacketSize = getMaxPacketSize();
        boolean useSNI = !TEST_MODE.equals("norm");
        SSLEngine clientEngine = getClientSSLEngine(context, useSNI);
        SSLEngine serverEngine = getServerSSLEngine(context, useSNI);
        clientEngine.setEnabledCipherSuites(new String[]{cipher});
        serverEngine.setEnabledCipherSuites(new String[]{cipher});
        serverEngine.setNeedClientAuth(!cipher.contains("anon"));
        doHandshake(clientEngine, serverEngine, maxPacketSize,
                HandshakeMode.INITIAL_HANDSHAKE);
    }
}
