///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/ObjectNode.h>
#include <atomviz/atoms/datachannels/BondsDataChannel.h>
#include "Atoms2Tachyon.h"

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(Atoms2TachyonExportInterface, TachyonExportInterface)

/// Exports the given scene object in the Tachyon format and returns true.
/// Return false if the scene object type is not supported by this interface class.
/// Throws an exception if an error occurred.
bool Atoms2TachyonExportInterface::exportSceneObject(SceneObject* sceneObj, TachyonWriter& writer, ObjectNode* contextNode, const AffineTransformation& objToWorldTM)
{
	AtomsObject* atomsObj = dynamic_object_cast<AtomsObject>(sceneObj);
	if(!atomsObj) return false;

	writer.setTransformation(objToWorldTM);

	// Render simulation cell.
	FloatType lineWidth = atomsObj->simulationCell()->simulationCellLineWidth() * 0.5;
	if(atomsObj->simulationCell()->renderSimulationCell() && lineWidth > 0) {
		Color lineColor = atomsObj->simulationCell()->simulationCellRenderingColor();
		AffineTransformation simBox = objToWorldTM * atomsObj->simulationCell()->cellMatrix();
		Point3 corners[8];
		corners[0] = ORIGIN + simBox.getTranslation();
		corners[1] = corners[0] + simBox.column(0);
		corners[2] = corners[0] + simBox.column(0) + simBox.column(1);
		corners[3] = corners[0] + simBox.column(1);
		corners[4] = corners[0] + simBox.column(2);
		corners[5] = corners[1] + simBox.column(2);
		corners[6] = corners[2] + simBox.column(2);
		corners[7] = corners[3] + simBox.column(2);

		writer.cylinder(corners[0], corners[1], lineWidth, lineColor);
		writer.cylinder(corners[0], corners[3], lineWidth, lineColor);
		writer.cylinder(corners[1], corners[2], lineWidth, lineColor);
		writer.cylinder(corners[3], corners[2], lineWidth, lineColor);
		writer.cylinder(corners[0], corners[4], lineWidth, lineColor);
		writer.cylinder(corners[1], corners[5], lineWidth, lineColor);
		writer.cylinder(corners[2], corners[6], lineWidth, lineColor);
		writer.cylinder(corners[3], corners[7], lineWidth, lineColor);
		writer.cylinder(corners[4], corners[5], lineWidth, lineColor);
		writer.cylinder(corners[4], corners[7], lineWidth, lineColor);
		writer.cylinder(corners[5], corners[6], lineWidth, lineColor);
		writer.cylinder(corners[7], corners[6], lineWidth, lineColor);

		for(int i = 0; i<8; i++)
			writer.sphere(corners[i], lineWidth, lineColor);
	}

	DataChannel* posChannel = atomsObj->getStandardDataChannel(DataChannel::PositionChannel);
	if(!posChannel || !posChannel->size()) return true;

	TimeInterval interval = TimeForever;

	QVector<Color> atomColors = atomsObj->getAtomColors(writer.time(), interval);
	QVector<FloatType> atomRadii = atomsObj->getAtomRadii(writer.time(), interval);

	writer.sphereArray(posChannel->size(), posChannel->constDataPoint3(), atomRadii.constData(), atomColors.constData());

	// Render atomic bonds.
	BondsDataChannel* bondsChannel = dynamic_object_cast<BondsDataChannel>(atomsObj->getStandardDataChannel(DataChannel::BondsChannel));
	if(bondsChannel && bondsChannel->isVisible()) {
		FloatType bondRadius = 0.5 * bondsChannel->bondWidth();
		if(bondRadius > 0) {

			const AffineTransformation simCell = atomsObj->simulationCell()->cellMatrix();
			const AffineTransformation simCellInverse = simCell.inverse();
			const array<bool,3> pbc = atomsObj->simulationCell()->periodicity();

			const Point3* p = posChannel->constDataPoint3();
			const int* b = bondsChannel->constDataInt();
			QVector<Color>::const_iterator color1 = atomColors.constBegin();
			for(size_t i = bondsChannel->size(); i != 0; --i, ++p, ++color1) {
				for(size_t j = bondsChannel->componentCount(); j != 0; --j, ++b) {
					if(*b < 0 || *b >= atomsObj->atomsCount()) continue;
					const Point3& p2 = posChannel->getPoint3(*b);
					Vector3 delta = (p2 - *p);

					// Use minimum image convention to handle periodic boundary conditions.
					bool renderCap = false;
					Vector3 reduced = simCellInverse * delta;
					for(int k = 0; k < 3; k++) {
						if(pbc[k]) {
							while(reduced[k] < -0.5) {
								reduced[k] += 1.0;
								delta += simCell.column(k);
								renderCap = true;
							}
							while(reduced[k] > 0.5) {
								reduced[k] -= 1.0;
								delta -= simCell.column(k);
								renderCap = true;
							}
						}
					}

					writer.cylinder(*p, (*p) + delta*0.5, bondRadius, *color1, renderCap, false);
				}
			}
		}
	}

	return true;
}

};	// End of namespace AtomViz
