///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/PluginManager.h>
#include <core/plugins/Plugin.h>
#include <core/plugins/NativePlugin.h>
#include <core/utilities/PathManager.h>

namespace Core {

///////////////////////////// SINGLETON CLASS METHODS ///////////////////////////////

/// The singleton instance of this class.
PluginManager* PluginManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the plugin manager.
******************************************************************************/
PluginManager::PluginManager() : _corePlugin(NULL) {}

/******************************************************************************
* Unloads all plugins.
******************************************************************************/
PluginManager::~PluginManager()
{
	// Unload plugins in reverse order.
	for(int i=plugins().size()-1; i >= 0; --i)
		delete plugins()[i];
}


/******************************************************************************
* Returns the plugin with the given identifier.
* Returns NULL when no such plugin is installed.
******************************************************************************/
Plugin* PluginManager::plugin(const QString& pluginId)
{
	Q_FOREACH(Plugin* plugin, plugins()) {
		if(plugin->pluginId() == pluginId)
			return plugin;
	}
	return NULL;
}

/******************************************************************************
* Registers a new plugin with the manager.
******************************************************************************/
void PluginManager::registerPlugin(Plugin* plugin)
{
	CHECK_POINTER(plugin);

	// Make sure the plugin id is unique.
	if(this->plugin(plugin->pluginId()) != NULL) {
		QString id = plugin->pluginId();
		delete plugin;
		throw Exception(QString("Non-unique plugin identifier detected: %1.").arg(id));
	}

	_plugins.push_back(plugin);
}


/******************************************************************************
* Searches the plugin directory for installed plugins and
* loads their XML manifests.
******************************************************************************/
void PluginManager::registerPlugins()
{
	// Register the built-in classes of the core.
	_corePlugin = loadPluginManifest(":/core/Core1.0.manifest.xml");

	// Scan the plugins directory for installed plugins.
	QDir pluginDir = QDir(PATH_MANAGER.pluginsDirectory());
	if(!pluginDir.exists())
		throw Exception(QString("Failed to scan the plugin directory: %1").arg(pluginDir.path()));

	// List all manifest files.
	pluginDir.setNameFilters(QStringList("*.manifest.xml"));
	pluginDir.setFilter(QDir::Files);
	QStringList files = pluginDir.entryList();

	// Load each manifest file in the plugins directory.
	for(int i=0; i<files.size(); i++) {
		try {
			loadPluginManifest(pluginDir.absoluteFilePath(files[i]));
		}
		catch(Exception& ex) {
			ex.prependGeneralMessage(tr("Failed to load plugin manifest:\n\n%1").arg(files[i]));
			ex.showError();
		}
	}

	// Parse the manifest of each plugin.
	Q_FOREACH(Plugin* plugin, plugins()) {
		try {
			plugin->parseManifest();
			VerboseLogger() << "Found plugin" << plugin->pluginId() << "(Version:" << plugin->pluginVersion() << ")" << endl;
		}
		catch(Exception& ex) {
			ex.prependGeneralMessage(tr("Failed to load plugin manifest:\n\n%1").arg(plugin->manifestFile()));
			_plugins.remove(_plugins.indexOf(plugin));
			if(plugin->isCore()) {
				delete plugin;
				throw ex;	// This is a fatal error.
			}
			else {
				delete plugin;
				ex.showError();
			}
		}
	}

	// Load the core plugin by default.
	corePlugin()->loadPlugin();
}

/******************************************************************************
* Loads the given plugin manifest file.
******************************************************************************/
Plugin* PluginManager::loadPluginManifest(const QString& file)
{
	// Check if manifest has already been loaded.
	Q_FOREACH(Plugin* p, plugins())
		if(p->manifestFile() == file) return p;

	// Create Plugin object and load XML file into DOM.
	Plugin* plugin = new NativePlugin(file);

	// Add it to the list of plugins.
	registerPlugin(plugin);

	return plugin;
}

/******************************************************************************
* Returns all installed plugin classes derived from the given type.
******************************************************************************/
QVector<PluginClassDescriptor*> PluginManager::listClasses(const PluginClassDescriptor* baseClass, bool skipAbstract)
{
	CHECK_POINTER(baseClass);
	QVector<PluginClassDescriptor*> result;

	Q_FOREACH(Plugin* plugin, plugins()) {
		Q_FOREACH(PluginClassDescriptor* clazz, plugin->classes()) {
			if(!skipAbstract || !clazz->isAbstract()) {
				if(clazz->isKindOf(baseClass))
					result.push_back(clazz);
			}
		}
	}

	return result;
}

};
