static char help[] = "Directly check a DMStag local-to-global scatter";

#include <petscdm.h>
#include <petscdmstag.h>

static PetscErrorCode Test_3d_4x4x4_3x3x3(DM dmstag);

int main(int argc,char **argv)
{
  PetscErrorCode ierr;
  DM             dmstag;
  PetscInt       dim,dof[4],i,elx,ely,elz;

  ierr = PetscInitialize(&argc,&argv,(char*)0,help);if (ierr) return ierr;
  dim = 3;
  for (i=0; i<4; ++i) dof[i] = 1;
  elx = ely = elz = 4;
  switch (dim) {
    case 3:
      ierr = DMStagCreate3d(PETSC_COMM_WORLD,DM_BOUNDARY_NONE,DM_BOUNDARY_NONE,DM_BOUNDARY_NONE,elx,ely,elz,PETSC_DECIDE,PETSC_DECIDE,PETSC_DECIDE,dof[0],dof[1],dof[2],dof[3],DMSTAG_STENCIL_BOX,1,NULL,NULL,NULL,&dmstag);CHKERRQ(ierr);
      break;
    default:
      SETERRQ1(PETSC_COMM_WORLD,PETSC_ERR_SUP,"No support for dimension %D",dim);
  }
  ierr = DMSetFromOptions(dmstag);CHKERRQ(ierr);
  ierr = DMSetUp(dmstag);CHKERRQ(ierr);
  ierr = Test_3d_4x4x4_3x3x3(dmstag);CHKERRQ(ierr);
  ierr = DMDestroy(&dmstag);CHKERRQ(ierr);
  ierr = PetscFinalize();
  return ierr;
}

static PetscErrorCode Test_3d_4x4x4_3x3x3(DM dmstag)
{
  PetscErrorCode ierr;
  Vec            vecLocal,vecGlobal;
  PetscInt       i,low,high,n;
  PetscScalar    *arr;

  PetscFunctionBeginUser;

  /* Check that grid and rank grid is as expected for this test */
  {
    PetscInt nRanks[3],n[3],dim;
    ierr = DMGetDimension(dmstag,&dim);CHKERRQ(ierr);
    if (dim != 3) SETERRQ(PetscObjectComm((PetscObject)dmstag),PETSC_ERR_SUP,"This is a 3d test");
    ierr = DMStagGetNumRanks(dmstag,&nRanks[0],&nRanks[1],&nRanks[2]);CHKERRQ(ierr);
    for (i=0; i<3; ++i) if (nRanks[i] != 3) SETERRQ(PetscObjectComm((PetscObject)dmstag),PETSC_ERR_SUP,"This test requires a 3x3x3 rank grid (run on 27 ranks)");
    ierr = DMStagGetGlobalSizes(dmstag,&n[0],&n[1],&n[2]);CHKERRQ(ierr);
    for (i=0; i<3; ++i) if (n[i] != 4) SETERRQ(PetscObjectComm((PetscObject)dmstag),PETSC_ERR_SUP,"This test requires a 4x4x4 element grid");
  }

  /* Populate global vector by converting the global index number to a scalar value. */
  ierr = DMCreateGlobalVector(dmstag,&vecGlobal);CHKERRQ(ierr);
  ierr = VecGetOwnershipRange(vecGlobal,&low,&high);CHKERRQ(ierr);
  n = high-low;
  ierr = VecGetArray(vecGlobal,&arr);CHKERRQ(ierr);
  for (i=0; i<n; ++i) {
   arr[i] = (PetscScalar) (i + low);
  }
  ierr = VecRestoreArray(vecGlobal,&arr);CHKERRQ(ierr);

  /* Populate a local vector initially with -1, then glocal->local scatter */
  ierr = DMCreateLocalVector(dmstag,&vecLocal);CHKERRQ(ierr);
  ierr = VecSet(vecLocal,-1.0);CHKERRQ(ierr);
  ierr = DMGlobalToLocalBegin(dmstag,vecGlobal,INSERT_VALUES,vecLocal);CHKERRQ(ierr);
  ierr = DMGlobalToLocalEnd  (dmstag,vecGlobal,INSERT_VALUES,vecLocal);CHKERRQ(ierr);

  /* Check that entries are as expected */
  {
    PetscInt          entriesGhost,nerr;
    const PetscInt    maxErrPerRank = 3;
    PetscScalar       *arrLocalExpected;
    const PetscScalar *arrLocal;
    PetscMPIInt       rank;

    ierr = MPI_Comm_rank(PetscObjectComm((PetscObject)dmstag),&rank);CHKERRMPI(ierr);
    ierr = VecGetSize(vecLocal,&entriesGhost);CHKERRQ(ierr); /* entriesGhost happens to always be 216 here */
    ierr = PetscMalloc1(entriesGhost,&arrLocalExpected);CHKERRQ(ierr);

    /* Hand-computed expected entries (27 blocks of 8 in all cases) */
    if (rank == 0) {
          const PetscScalar arrLocalExpectedHere[] = {
            0,  1,  2,  3,  4,  5,  6,  7,
            8,  9,  10, 11, 12, 13, 14, 15,
            64, 65, 66, 67, 68, 69, 70, 71,
            16, 17, 18, 19, 20, 21, 22, 23,
            24, 25, 26, 27, 28, 29 ,30, 31,
            72, 73, 74, 75, 76, 77, 78, 79,
            144,145,146,147,148,149,150,151,
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            32, 33, 34, 35, 36, 37, 38, 39,
            40, 41, 42, 43, 44, 45, 46, 47,
            80, 81, 82, 83, 84, 85, 86, 87,
            48, 49, 50, 51, 52, 53, 54, 55,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            324,325,326,327,328,329,330,331,
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 1) {
          const PetscScalar arrLocalExpectedHere[] = {
            8,  9,  10, 11, 12, 13, 14, 15,
            64, 65, 66, 67, 68, 69, 70, 71,
            96, 97, 98, 99, 100,101,102,103,
            24, 25, 26, 27, 28, 29 ,30, 31,
            72, 73, 74, 75, 76, 77, 78, 79,
            108,109,110,111,112,113,114,115,
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            40, 41, 42, 43, 44, 45, 46, 47,
            80, 81, 82, 83, 84, 85, 86, 87,
            120,121,122,123,124,125,126,127,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 2) {
          const PetscScalar arrLocalExpectedHere[] = {
            64, 65, 66, 67, 68, 69, 70, 71,
            96, 97, 98, 99, 100,101,102,103,
            104,-1, 105, -1,106,-1, 107,-1,
            72, 73, 74, 75, 76, 77, 78, 79,
            108,109,110,111,112,113,114,115,
            116,-1, 117,-1, 118,-1, 119,-1,
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            200,-1, 201,-1, 202,-1, 203,-1,
            80, 81, 82, 83, 84, 85, 86, 87,
            120,121,122,123,124,125,126,127,
            128,-1, 129,-1, 130,-1, 131, -1,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            140,-1 ,141,-1, 142,-1, 143,-1,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            380,-1, 381,-1, 382,-1, 383,-1,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 3) {
          const PetscScalar arrLocalExpectedHere[] = {
            16, 17, 18, 19, 20, 21, 22, 23,
            24, 25, 26, 27, 28, 29 ,30, 31,
            72, 73, 74, 75, 76, 77, 78, 79,
            144,145,146,147,148,149,150,151,
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            216,217,218,219,220,221,222,223,
            224,225,226,227,228,229,230,231,
            264,265,266,267,268,269,270,271,
            48, 49, 50, 51, 52, 53, 54, 55,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            240,241,242,243,244,245,246,247,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 4) {
          const PetscScalar arrLocalExpectedHere[] = {
            24, 25, 26, 27, 28, 29 ,30, 31,
            72, 73, 74, 75, 76, 77, 78, 79,
            108,109,110,111,112,113,114,115,
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            224,225,226,227,228,229,230,231,
            264,265,266,267,268,269,270,271,
            288,289,290,291,292,293,294,295,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 5) {
          const PetscScalar arrLocalExpectedHere[] = {
            72, 73, 74, 75, 76, 77, 78, 79,
            108,109,110,111,112,113,114,115,
            116,-1, 117,-1, 118,-1, 119,-1,
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            200,-1, 201,-1, 202,-1, 203,-1,
            264,265,266,267,268,269,270,271,
            288,289,290,291,292,293,294,295,
            296,-1, 297,-1, 298,-1, 299,-1,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            140,-1 ,141,-1, 142,-1, 143,-1,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            314,-1, 315,-1, 316,-1, 317,-1,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1, 431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 6) {
          const PetscScalar arrLocalExpectedHere[] = {
            144,145,146,147,148,149,150,151,
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            216,217,218,219,220,221,222,223,
            224,225,226,227,228,229,230,231,
            264,265,266,267,268,269,270,271,
            232,233,-1, -1, 234,235,-1, -1,
            236,237,-1, -1, 238,239,-1, -1,
            272,273,-1, -1, 274,275,-1, -1,
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            240,241,242,243,244,245,246,247,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            256,257,-1, -1, 258,259,-1, -1,
            260,261,-1, -1, 262,263,-1, -1,
            284,285,-1, -1, 286,287,-1, -1,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            448,449,-1, -1, 450,451,-1, -1,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 7) {
          const PetscScalar arrLocalExpectedHere[] = {
            152,153,154,155,156,157,158,159,
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            224,225,226,227,228,229,230,231,
            264,265,266,267,268,269,270,271,
            288,289,290,291,292,293,294,295,
            236,237,-1, -1, 238,239,-1, -1,
            272,273,-1, -1, 274,275,-1, -1,
            300,301,-1, -1, 302,303,-1, -1,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            260,261,-1, -1, 262,263,-1, -1,
            284,285,-1, -1, 286,287,-1, -1,
            318,319,-1, -1, 320,321,-1, -1,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 8) {
          const PetscScalar arrLocalExpectedHere[] = {
            176,177,178,179,180,181,182,183,
            192,193,194,195,196,197,198,199,
            200,-1, 201,-1, 202,-1, 203,-1,
            264,265,266,267,268,269,270,271,
            288,289,290,291,292,293,294,295,
            296,-1, 297,-1, 298,-1, 299,-1,
            272,273,-1, -1, 274,275,-1, -1,
            300,301,-1, -1, 302,303,-1, -1,
            304,-1, -1, -1, 305,-1, -1, -1,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            314,-1, 315,-1, 316,-1, 317,-1,
            284,285,-1, -1, 286,287,-1, -1,
            318,319,-1, -1, 320,321,-1, -1,
            322,-1, -1, -1, 323,-1, -1, -1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
            484,-1, -1, -1, 485,-1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 9) {
          const PetscScalar arrLocalExpectedHere[] = {
            32, 33, 34, 35, 36, 37, 38, 39,
            40, 41, 42, 43, 44, 45, 46, 47,
            80, 81, 82, 83, 84, 85, 86, 87,
            48, 49, 50, 51, 52, 53, 54, 55,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            324,325,326,327,328,329,330,331,
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            486,487,488,489,490,491,492,493,
            494,495,496,497,498,499,500,501,
            534,535,536,537,538,539,540,541,
            502,503,504,505,506,507,508,509,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 10) {
          const PetscScalar arrLocalExpectedHere[] = {
            40, 41, 42, 43, 44, 45, 46, 47,
            80, 81, 82, 83, 84, 85, 86, 87,
            120,121,122,123,124,125,126,127,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            494,495,496,497,498,499,500,501,
            534,535,536,537,538,539,540,541,
            558,559,560,561,562,563,564,565,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 11) {
          const PetscScalar arrLocalExpectedHere[] = {
            80, 81, 82, 83, 84, 85, 86, 87,
            120,121,122,123,124,125,126,127,
            128,-1, 129,-1, 130,-1, 131, -1,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            140,-1 ,141,-1, 142,-1, 143,-1,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            380,-1, 381,-1, 382,-1, 383,-1,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            534,535,536,537,538,539,540,541,
            558,559,560,561,562,563,564,565,
            566,-1, 567,-1, 568,-1, 569,-1,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            578,-1, 579,-1, 580,-1, 581,-1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 12) {
          const PetscScalar arrLocalExpectedHere[] = {
            48, 49, 50, 51, 52, 53, 54, 55,
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            240,241,242,243,244,245,246,247,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            502,503,504,505,506,507,508,509,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            648,649,650,651,652,653,654,655,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 13) {
          const PetscScalar arrLocalExpectedHere[] = {
            56, 57, 58, 59, 60, 61, 62, 63,
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 14) {
          const PetscScalar arrLocalExpectedHere[] = {
            88, 89, 90, 91, 92, 93, 94, 95,
            132,133,134,135,136,137,138,139,
            140,-1 ,141,-1, 142,-1, 143,-1,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            314,-1, 315,-1, 316,-1, 317,-1,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1, 431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            578,-1, 579,-1, 580,-1, 581,-1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            710,-1, 711,-1, 712,-1, 713,-1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 15) {
          const PetscScalar arrLocalExpectedHere[] = {
            160,161,162,163,164,165,166,167,
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            240,241,242,243,244,245,246,247,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            256,257,-1, -1, 258,259,-1, -1,
            260,261,-1, -1, 262,263,-1, -1,
            284,285,-1, -1, 286,287,-1, -1,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            448,449,-1, -1, 450,451,-1, -1,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            648,649,650,651,652,653,654,655,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            664,665,-1, -1, 666,667,-1, -1,
            668,669,-1, -1, 670,671,-1, -1,
            692,693,-1, -1, 694,695,-1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 16) {
          const PetscScalar arrLocalExpectedHere[] = {
            168,169,170,171,172,173,174,175,
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            248,249,250,251,252,253,254,255,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            260,261,-1, -1, 262,263,-1, -1,
            284,285,-1, -1, 286,287,-1, -1,
            318,319,-1, -1, 320,321,-1, -1,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            668,669,-1, -1, 670,671,-1, -1,
            692,693,-1, -1, 694,695,-1, -1,
            714,715,-1, -1, 716,717,-1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 17) {
          const PetscScalar arrLocalExpectedHere[] = {
            184,185,186,187,188,189,190,191,
            204,205,206,207,208,209,210,211,
            212,-1, 213,-1, 214,-1, 215,-1,
            276,277,278,279,280,281,282,283,
            306,307,308,309,310,311,312,313,
            314,-1, 315,-1, 316,-1, 317,-1,
            284,285,-1, -1, 286,287,-1, -1,
            318,319,-1, -1, 320,321,-1, -1,
            322,-1, -1, -1, 323,-1, -1, -1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
            484,-1, -1, -1, 485,-1, -1, -1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            710,-1, 711,-1, 712,-1, 713,-1,
            692,693,-1, -1, 694,695,-1, -1,
            714,715,-1, -1, 716,717,-1, -1,
            718,-1, -1, -1, 719,-1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 18) {
          const PetscScalar arrLocalExpectedHere[] = {
            324,325,326,327,328,329,330,331,
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            486,487,488,489,490,491,492,493,
            494,495,496,497,498,499,500,501,
            534,535,536,537,538,539,540,541,
            502,503,504,505,506,507,508,509,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            518,519,520,521,-1, -1, -1, -1,
            522,523,524,525,-1, -1, -1, -1,
            550,551,552,553,-1, -1, -1, -1,
            526,527,528,529,-1, -1, -1, -1,
            530,531,532,533,-1, -1, -1, -1,
            554,555,556,557,-1, -1, -1, -1,
            610,611,612,613,-1, -1, -1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 19) {
          const PetscScalar arrLocalExpectedHere[] = {
            332,333,334,335,336,337,338,339,
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            494,495,496,497,498,499,500,501,
            534,535,536,537,538,539,540,541,
            558,559,560,561,562,563,564,565,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            522,523,524,525,-1, -1, -1, -1,
            550,551,552,553,-1, -1, -1, -1,
            582,583,584,585,-1, -1, -1, -1,
            530,531,532,533,-1, -1, -1, -1,
            554,555,556,557,-1, -1, -1, -1,
            588,589,590,591,-1, -1, -1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 20) {
          const PetscScalar arrLocalExpectedHere[] = {
            356,357,358,359,360,361,362,363,
            372,373,374,375,376,377,378,379,
            380,-1, 381,-1, 382,-1, 383,-1,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            534,535,536,537,538,539,540,541,
            558,559,560,561,562,563,564,565,
            566,-1, 567,-1, 568,-1, 569,-1,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            578,-1, 579,-1, 580,-1, 581,-1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
            550,551,552,553,-1, -1, -1, -1,
            582,583,584,585,-1, -1, -1, -1,
            586,-1, 587,-1, -1, -1, -1, -1,
            554,555,556,557,-1, -1, -1, -1,
            588,589,590,591,-1, -1, -1, -1,
            592,-1, 593,-1, -1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
            646,-1, 647,-1, -1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 21) {
          const PetscScalar arrLocalExpectedHere[] = {
            340,341,342,343,344,345,346,347,
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            502,503,504,505,506,507,508,509,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            648,649,650,651,652,653,654,655,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            526,527,528,529,-1, -1, -1, -1,
            530,531,532,533,-1, -1, -1, -1,
            554,555,556,557,-1, -1, -1, -1,
            610,611,612,613,-1, -1, -1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            672,673,674,675,-1, -1, -1, -1,
            676,677,678,679,-1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 22) {
          const PetscScalar arrLocalExpectedHere[] = {
            348,349,350,351,352,353,354,355,
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            510,511,512,513,514,515,516,517,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            530,531,532,533,-1, -1, -1, -1,
            554,555,556,557,-1, -1, -1, -1,
            588,589,590,591,-1, -1, -1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
            676,677,678,679,-1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
            720,721,722,723,-1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 23) {
          const PetscScalar arrLocalExpectedHere[] = {
            364,365,366,367,368,369,370,371,
            384,385,386,387,388,389,390,391,
            392,-1, 393,-1, 394,-1, 395,-1,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
            542,543,544,545,546,547,548,549,
            570,571,572,573,574,575,576,577,
            578,-1, 579,-1, 580,-1, 581,-1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            710,-1, 711,-1, 712,-1, 713,-1,
            554,555,556,557,-1, -1, -1, -1,
            588,589,590,591,-1, -1, -1, -1,
            592,-1, 593,-1, -1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
            646,-1, 647,-1, -1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
            720,721,722,723,-1, -1, -1, -1,
            724,-1, 725,-1, -1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 24) {
          const PetscScalar arrLocalExpectedHere[] = {
            396,397,398,399,400,401,402,403,
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            432,433,434,435,436,437,438,439,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            448,449,-1, -1, 450,451,-1, -1,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
            594,595,596,597,598,599,600,601,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            648,649,650,651,652,653,654,655,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            664,665,-1, -1, 666,667,-1, -1,
            668,669,-1, -1, 670,671,-1, -1,
            692,693,-1, -1, 694,695,-1, -1,
            610,611,612,613,-1, -1, -1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            672,673,674,675,-1, -1, -1, -1,
            676,677,678,679,-1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
            680,681,-1, -1, -1, -1, -1, -1,
            682,683,-1, -1, -1, -1, -1, -1,
            700,701,-1, -1, -1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 25) {
          const PetscScalar arrLocalExpectedHere[] = {
            404,405,406,407,408,409,410,411,
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            440,441,442,443,444,445,446,447,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            452,453,-1, -1, 454,455,-1, -1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
            602,603,604,605,606,607,608,609,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            656,657,658,659,660,661,662,663,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            668,669,-1, -1, 670,671,-1, -1,
            692,693,-1, -1, 694,695,-1, -1,
            714,715,-1, -1, 716,717,-1, -1,
            614,615,616,617,-1, -1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
            676,677,678,679,-1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
            720,721,722,723,-1, -1, -1, -1,
            682,683,-1, -1, -1, -1, -1, -1,
            700,701,-1, -1, -1, -1, -1, -1,
            726,727,-1, -1, -1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    } else if (rank == 26) {
          const PetscScalar arrLocalExpectedHere[] = {
            412,413,414,415,416,417,418,419,
            420,421,422,423,424,425,426,427,
            428,-1, 429,-1, 430,-1 ,431,-1,
            456,457,458,459,460,461,462,463,
            468,469,470,471,472,473,474,475,
            476,-1, 477,-1, 478,-1, 479,-1,
            464,465,-1, -1, 466,467,-1, -1,
            480,481,-1, -1, 482,483,-1, -1,
            484,-1, -1, -1, 485,-1, -1, -1,
            618,619,620,621,622,623,624,625,
            630,631,632,633,634,635,636,637,
            638,-1, 639,-1, 640,-1, 641,-1,
            684,685,686,687,688,689,690,691,
            702,703,704,705,706,707,708,709,
            710,-1, 711,-1, 712,-1, 713,-1,
            692,693,-1, -1, 694,695,-1, -1,
            714,715,-1, -1, 716,717,-1, -1,
            718,-1, -1, -1, 719,-1, -1, -1,
            626,627,628,629,-1, -1, -1, -1,
            642,643,644,645,-1, -1, -1, -1,
            646,-1, 647,-1, -1, -1, -1, -1,
            696,697,698,699,-1, -1, -1, -1,
            720,721,722,723,-1, -1, -1, -1,
            724,-1, 725,-1, -1, -1, -1, -1,
            700,701,-1, -1, -1, -1, -1, -1,
            726,727,-1, -1, -1, -1, -1, -1,
            728,-1, -1, -1, -1, -1, -1, -1,
          };
         for (i=0; i<entriesGhost; ++i) arrLocalExpected[i] = arrLocalExpectedHere[i];
    }

    ierr = VecGetArrayRead(vecLocal,&arrLocal);CHKERRQ(ierr);
    for (i=0, nerr=0; i<entriesGhost; ++i) {
      if (arrLocal[i] != arrLocalExpected[i]) {
        ++nerr;
          if (nerr <= maxErrPerRank) {
            ierr = PetscPrintf(PETSC_COMM_SELF,"[%d] Entry %D has value %g instead of the expected %g\n",rank,i,(double)PetscRealPart(arrLocal[i]),(double)PetscRealPart(arrLocalExpected[i]));CHKERRQ(ierr);
            if (nerr == maxErrPerRank + 1) {
              ierr = PetscPrintf(PETSC_COMM_SELF,"[%d] Skipping additional errors on this rank\n",rank);CHKERRQ(ierr);
            }
          }
      }
    }
    if (nerr > 0) {
      ierr = PetscPrintf(PETSC_COMM_SELF,"[%d] %D incorrect values on this rank\n",rank,nerr);CHKERRQ(ierr);
    }
    ierr = VecRestoreArrayRead(vecLocal,&arrLocal);CHKERRQ(ierr);
    ierr = PetscFree(arrLocalExpected);CHKERRQ(ierr);
  }

  ierr = VecDestroy(&vecLocal);CHKERRQ(ierr);
  ierr = VecDestroy(&vecGlobal);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

/*TEST

   test:
      nsize: 27

TEST*/
